﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

extern nn::Result ClientSendLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReceiveLightRequest(nn::svc::Handle handle);
extern nn::Result ServerReplyLightRequest(nn::svc::Handle handle);

namespace {

extern "C" void nnMain();
const size_t StackSize = DefaultStackSize;
char g_Buffer[StackSize] __attribute__((aligned(0x1000)));

void LightSessionClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);

    result = nn::svc::SendSyncRequest(*clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = ClientSendLightRequest(*clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // Session を閉じないとサーバーが復帰しない
    result = nn::svc::CloseHandle(*clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void NormalSessionClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);

    result = ClientSendLightRequest(*clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(*clientSession);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Reply);
}


} // namespace

#ifdef INVALID_POINTER_TEST
TEST(CreateSession, InvalidPointerTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t addr;

    // pServerSession
    {
#ifdef INVALID_POINTER_TEST
        // TEST 64-1
        // NULL アドレスを受け付けない
        result = nn::svc::CreateSession(NULL, &handle, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 64-2
        // MemoryPermission_None の領域を受け付けない
        result = nn::svc::CreateSession(
                reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin), &handle, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 64-3
        // MemoryPermission_Read の領域を受け付けない
        {
            TestHeap heap(HeapAlign);
            addr = heap.GetAddress();
            {
                TestMemoryPermission perm(addr, 0x1000, nn::svc::MemoryPermission_Read);
                result = nn::svc::CreateSession(
                        reinterpret_cast<nn::svc::Handle*>(addr), &handle, false, 0);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
            }
        }
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 64-4
        // MemoryPermission_ReadExecute の領域を受け付けない
        addr = reinterpret_cast<uintptr_t>(nnMain);
        result = nn::svc::CreateSession(
                reinterpret_cast<nn::svc::Handle*>(addr), &handle, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
    }

    // pClientSession
    {
#ifdef INVALID_POINTER_TEST
        // TEST 64-7
        // NULL アドレスを受け付けない
        result = nn::svc::CreateSession(&handle, NULL, false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 64-8
        // MemoryPermission_None の領域を受け付けない
        result = nn::svc::CreateSession(
                &handle, reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin), false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 64-9
        // MemoryPermission_Read の領域を受け付けない
        {
            TestHeap heap(HeapAlign);
            addr = heap.GetAddress();
            {
                TestMemoryPermission perm(addr, 0x1000, nn::svc::MemoryPermission_Read);
                result = nn::svc::CreateSession(
                        &handle, reinterpret_cast<nn::svc::Handle*>(addr), false, 0);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
            }
        }
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 64-10
        // MemoryPermission_ReadExecute の領域を受け付けない
        addr = reinterpret_cast<uintptr_t>(nnMain);
        result = nn::svc::CreateSession(
                &handle, reinterpret_cast<nn::svc::Handle*>(addr), false, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
    }
}
#endif // INVALID_POINTER_TEST

// TEST 64-20
// Normal Session を作成することが出来る
TEST(CreateSession, NormalSessionTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    // Thread の設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(NormalSessionClientThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);
    uintptr_t priority = TestLowestThreadPriority;
    uintptr_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    result = ServerReceiveLightRequest(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    int32_t index;
    result = nn::svc::ReplyAndReceive(
            &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(nullptr), DefaultIpcTag_Send);

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceive(
            &index, &serverSession, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}

// TEST 64-21
// Light Session を作成することが出来る
TEST(CreateSession, LightSessionTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    result = nn::svc::CreateSession(&serverSession, &clientSession, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    // Thread の設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(LightSessionClientThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);
    uintptr_t priority = TestLowestThreadPriority;
    uintptr_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Receive);
    int32_t index;
    result = nn::svc::ReplyAndReceive(
            &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());


    result = ServerReceiveLightRequest(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    result = ServerReplyLightRequest(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    thread.Wait();
}

// TEST 64-22
// uintptr_tの範囲の値を受け付ける
TEST(CreateSession, TrackingIdTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    uintptr_t testData[] = { 0, static_cast<uintptr_t>(-1), 1ull << (sizeof(uintptr_t) * 8 / 2)};

    for (int32_t i = 0; i < static_cast<int>(sizeof(testData) / sizeof(uintptr_t)); i++)
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;

        result = nn::svc::CreateSession(&serverSession, &clientSession, false, testData[i]);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);
    }
}

