﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Result.h>

namespace {

extern "C" void nnMain();
const int64_t WaitTime = 1 * 1000 * 1000 * 1000; // (1 sec)

}

TEST(CreateEventTest, SignalTest)
{
    TestEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle writeHandle;
    nn::svc::Handle readHandle;
    int32_t index;

    // TEST 69-5
    // 作成直後のpReadHandle は非シグナル状態
    {
        result = nn::svc::CreateEvent(&writeHandle, &readHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readHandle, 1, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        result = nn::svc::CloseHandle(writeHandle);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

    {
        result = nn::svc::CreateEvent(&writeHandle, &readHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 69-7
        // pReadHandle に対して SignalEvent を行うことが出来ない
        result = nn::svc::SignalEvent(readHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        // TEST 69-6
        // pWriteHandle に対して SignalEvent を行うことが出来る
        result = nn::svc::SignalEvent(writeHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 69-9
        // pWriteHandle はシグナルを受信できない
        result = nn::svc::WaitSynchronization(&index, &writeHandle, 1, WaitTime);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        // TEST 69-8
        // pReadHandle はシグナルを受信できる
        result = nn::svc::WaitSynchronization(&index, &readHandle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(writeHandle);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

    {
        result = nn::svc::CreateEvent(&writeHandle, &readHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SignalEvent(writeHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 69-12
        // pReadHandle がシグナル状態でもpWriteHandle をクローズできる
        result = nn::svc::CloseHandle(writeHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 69-13
        // pReadHandle がシグナル状態でもpReadHandle をクローズできる
        result = nn::svc::CloseHandle(readHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 69-14
    // pReadHandle がClose されている状態でもシグナルを送信できる
    {
        result = nn::svc::CreateEvent(&writeHandle, &readHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(readHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SignalEvent(writeHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(writeHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 69-15
    // pReadHandle がシグナル状態であれば、pWriteHandleがクローズされていても、
    // シグナルを受信することが出来る
    {
        result = nn::svc::CreateEvent(&writeHandle, &readHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SignalEvent(writeHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(writeHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readHandle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(readHandle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(CreateEventTest, AutoClearTest)
{
    TestEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle writeHandle;
    nn::svc::Handle readHandle;
    int32_t index;

    // TEST 69-17
    // 自動でシグナルがクリアされない
    {
        result = nn::svc::CreateEvent(&writeHandle, &readHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::SignalEvent(writeHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readHandle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &readHandle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(readHandle);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(writeHandle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

#ifdef INVALID_POINTER_TEST
TEST(CreateEventTest, InvalidPointerTest)
{
    TestEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::Handle* invalidHandle;

    // pWriteHandle
    {
#ifdef INVALID_POINTER_TEST
        // TEST 69-18
        // pWriteHandle に NULL を渡すとデータアボートを起こす
        result = nn::svc::CreateEvent(NULL, &handle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 69-25
        // pWriteHandle に MemoryPermission_None を渡すとデータアボートを起こす
        invalidHandle = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
        result = nn::svc::CreateEvent(&handle, invalidHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 69-26
        // pWriteHandle に MemoryPermission_Read を渡すとデータアボートを起こす
        {
            TestHeap heap(HeapAlign);
            uintptr_t addr = heap.GetAddress();
            {
                TestMemoryPermission perm(addr, 0x1000, nn::svc::MemoryPermission_Read);
                invalidHandle = reinterpret_cast<nn::svc::Handle*>(addr);
                result = nn::svc::CreateEvent(&handle, invalidHandle);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
            }
        }
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 69-27
        // pWriteHandle に MemoryPermission_ReadExecute を渡すとデータアボートを起こす
        invalidHandle = reinterpret_cast<nn::svc::Handle*>(nnMain);
        result = nn::svc::CreateEvent(&handle, invalidHandle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

    }

    // pReadHandle
    {
#ifdef INVALID_POINTER_TEST
        // TEST 69-19
        // pReadHandle に NULL を渡すとデータアボートを起こす
        result = nn::svc::CreateEvent(NULL, &handle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 69-35
        // pReadHandle に MemoryPermission_None を渡すとデータアボートを起こす
        invalidHandle = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
        result = nn::svc::CreateEvent(invalidHandle, &handle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 69-36
        // pReadHandle に MemoryPermission_Read を渡すとデータアボートを起こす
        {
            TestHeap heap(HeapAlign);
            uintptr_t addr = heap.GetAddress();
            {
                TestMemoryPermission perm(addr, 0x1000, nn::svc::MemoryPermission_Read);
                invalidHandle = reinterpret_cast<nn::svc::Handle*>(addr);
                result = nn::svc::CreateEvent(invalidHandle, &handle);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
            }
        }
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
        // TEST 69-37
        // pReadHandle に MemoryPermission_ReadExecute を渡すとデータアボートを起こす
        invalidHandle = reinterpret_cast<nn::svc::Handle*>(nnMain);
        result = nn::svc::CreateEvent(invalidHandle, &handle);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
    }

}
#endif // INVALID_POINTER_TEST

