﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

namespace {
void CloseServerSession(nn::svc::Handle serverPort)
{
    nn::Result result;
    int32_t index;

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}
} //namespace

#ifdef ENABLE_MAX_HANDLE_TEST
TEST(ConnectToNamedPort, MaxHandleTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    const char* portName = "NamedPort";
    nn::svc::Handle portHandle;
    nn::svc::Handle sessionHandle;

    // 前準備
    result = nn::svc::ManageNamedPort(&portHandle, portName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose portCloser(portHandle);

    {
        ConsumeHandle maxHandle;

        // TEST 26-12
        // プロセスのハンドル上限数を超えてハンドルを生成できない
        result = nn::svc::ConnectToNamedPort(&sessionHandle, portName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());
    }

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &portHandle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 26-16
    // 解放すると再度接続できるようになる
    result = nn::svc::ConnectToNamedPort(&sessionHandle, portName);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    CloseServerSession(portHandle);

    result = nn::svc::ManageNamedPort(&portHandle, portName, 0);
    ASSERT_RESULT_SUCCESS(result);
}
#endif // ENABLE_MAX_HANDLE_TEST

// TEST 26-15
// 接続に成功すると、サーバーポートのハンドルがシグナルされる
TEST(ConnectToNamedPort, RaiseSignalTest)
{
    TestNamedPortLeak leakTest;
    const char* portName = "Test2615";

    nn::Result result;
    nn::svc::Handle namedPort;

    result = nn::svc::ManageNamedPort(&namedPort, portName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose portCloser(namedPort);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToNamedPort(&clientSession, portName);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &namedPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    CloseServerSession(namedPort);

    result = nn::svc::ManageNamedPort(&namedPort, portName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(ConnectToNamedPort, MaxSessionTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    const char* portName = "MaxSession";

    // ポートを作成
    nn::svc::Handle portHandle;
    result = nn::svc::ManageNamedPort(&portHandle, portName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(portHandle);

    nn::svc::Handle clientSession;

    result = nn::svc::ConnectToNamedPort(&clientSession, portName);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 26-17
    // ManageNamedPort のセッション数以上のセッションを作ることが出来ない
    nn::svc::Handle tmpHandle;
    result = nn::svc::ConnectToNamedPort(&tmpHandle, portName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 26-19
    // クライアントセッション を閉じてもサーバーセッションを閉じなければ、セッションが解放されない
    result = nn::svc::ConnectToNamedPort(&clientSession, portName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    CloseServerSession(portHandle);

    // TEST 26-18
    // セッションが解放されたら、再度セッションが作れるようになる
    result = nn::svc::ConnectToNamedPort(&clientSession, portName);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);
    CloseServerSession(portHandle);

    result = nn::svc::ManageNamedPort(&portHandle, portName, 0);
    ASSERT_RESULT_SUCCESS(result);
}
