﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestLoader.h"
#include "test_Break.h"
#include "util_TestIpc.h"
#include <nn/nn_Assert.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstring>

namespace {

} // namespace

extern char BinBreakPanic_begin[];
extern char BinBreakPanic_end[];
extern char BinBreakAssert_begin[];
extern char BinBreakAssert_end[];
extern char BinBreakUser_begin[];
extern char BinBreakUser_end[];

TEST(Break, BreakTest)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle handles[1];
    nn::Bit32* pMsgBuffer;
    int32_t index;

    NamedPortManager namedPort(PortName, 1);
    serverPort = namedPort.GetHandle();
    handles[0] = serverPort;

    pMsgBuffer = nn::svc::ipc::GetMessageBuffer();

    TestLoader panicLoader(BinBreakPanic_begin, reinterpret_cast<uintptr_t>(BinBreakPanic_end) - reinterpret_cast<uintptr_t>(BinBreakPanic_begin));
    TestLoader assertLoader(BinBreakAssert_begin, reinterpret_cast<uintptr_t>(BinBreakAssert_end) - reinterpret_cast<uintptr_t>(BinBreakAssert_begin));
    TestLoader userLoader(BinBreakUser_begin, reinterpret_cast<uintptr_t>(BinBreakUser_end) - reinterpret_cast<uintptr_t>(BinBreakUser_begin));
    TestLoader* loaders[3] = {&panicLoader, &assertLoader, &userLoader};

    uint32_t flag;
    MakeNo16Flag(&flag, 0x1);

    for(int32_t i = 0; i < 3; i++)
    {
        nn::svc::Handle process;
        loaders[i]->SetAdditionalCapability(&flag, 1);
        loaders[i]->SpawnProcess(&process);
        loaders[i]->StartProcess(process);

        // 使わないので、閉じてしまう
        // TEST 121-83
        // 作成したハンドルを、使っている途中で閉じることが出来る
        result = nn::svc::CloseHandle(process);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &serverPort, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::AcceptSession(&serverSession, serverPort);
        ASSERT_RESULT_SUCCESS(result);

        handles[0] = serverSession;

        result = nn::svc::ReplyAndReceive(&index, handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
#ifdef SUPPORT_BREAK_DETAIL_INFO
        nn::Bit16 tag = ipcHeader.GetTag();
#endif
        ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
        ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
        ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
        ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());

        nn::Bit64 id;
        id = pMsgBuffer[nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial)];

        nn::svc::Handle targetProcess;
        result = nn::svc::DebugActiveProcess(&targetProcess, id);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::DebugEventInfo info;
        int32_t numThread = 0;
        for(;;)
        {
            result = nn::svc::GetDebugEvent(&info, targetProcess);
            ASSERT_RESULT_SUCCESS(result);
            if (info.event == nn::svc::DebugEvent_CreateThread)
            {
                numThread++;
            }
            else if ((info.event == nn::svc::DebugEvent_Exception
                        && info.info.exception.exceptionCode == nn::svc::DebugException_AttachBreak))
            {
                result = nn::svc::ContinueDebugEvent(
                        targetProcess,
                        nn::svc::ContinueFlag_ExceptionHandled |
                        nn::svc::ContinueFlag_EnableExceptionEvent |
                        nn::svc::ContinueFlag_ContinueAll, 0);
                ASSERT_RESULT_SUCCESS(result);
                break;
            }
        }

        nn::svc::ResetSignal(targetProcess);

        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0000, 0, 0, 0, 0, 0, 0, 0));
        result = nn::svc::ReplyAndReceive(&index, handles, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        result = nn::svc::CloseHandle(serverSession);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &targetProcess, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::GetDebugEvent(&info, targetProcess);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(info.event == nn::svc::DebugEvent_Exception);

        nn::svc::DebugInfoException exceptInfo = info.info.exception;
        ASSERT_TRUE(exceptInfo.exceptionCode == nn::svc::DebugException_UserBreak);

#ifdef SUPPORT_BREAK_DETAIL_INFO
        nn::svc::DebugInfoExceptionUserBreak detailInfo = exceptInfo.detail.userBreak;
        int32_t data;
        ASSERT_TRUE(detailInfo.size == sizeof(int32_t));
        std::memcpy(&data, reinterpret_cast<void*>(detailInfo.data), detailInfo.size);

        switch(tag)
        {
            case BreakTag_Panic:
                ASSERT_TRUE(detailInfo.reason == nn::svc::BreakReason_Panic);
                ASSERT_TRUE(data == BreakTag_Panic);
                break;
            case BreakTag_Assert:
                ASSERT_TRUE(detailInfo.reason == nn::svc::BreakReason_Assert);
                ASSERT_TRUE(data == BreakTag_Assert);
                break;
            case BreakTag_User:
                ASSERT_TRUE(detailInfo.reason == nn::svc::BreakReason_User);
                ASSERT_TRUE(data == BreakTag_User);
                break;
            default: FAIL();
        }
#endif

        nn::svc::ResetSignal(targetProcess);

        result = nn::svc::TerminateDebugProcess(targetProcess);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::WaitSynchronization(&index, &targetProcess, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(targetProcess);
        ASSERT_RESULT_SUCCESS(result);
    }
} // NOLINT (readability/fn_size)

