﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_Common.h"
#include <nn/TargetConfigs/build_Base.h>
#if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
    #include "detail/sdmmc_ClockResetController.tegra.h"
#endif
#include "detail/sdmmc_IHostController.h"
#include "detail/sdmmc_IDeviceAccessor.h"
#if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    #include "detail/sdmmc_PortMmc0.h"
#endif
#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    #include "detail/sdmmc_PortSdCard0.h"
#endif
#if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
    #include "detail/sdmmc_PortGcAsic0.h"
#endif
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc1 {

namespace
{
    #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
        bool s_IsClockResetInitialized = false;
    #endif

    detail::IHostController* GetHostController(Port port) NN_NOEXCEPT
    {
        detail::IHostController* pHostController = nullptr;
        switch (port)
        {
        #if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
            case Port_Mmc0:
                pHostController = detail::GetHostControllerOfPortMmc0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
            case Port_SdCard0:
                pHostController = detail::GetHostControllerOfPortSdCard0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
            case Port_GcAsic0:
                pHostController = detail::GetHostControllerOfPortGcAsic0();
                break;
        #endif
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT_UNLESS_NOT_NULL(pHostController);
        return pHostController;
    }

    detail::IDeviceAccessor* GetDeviceAccessor(Port port) NN_NOEXCEPT
    {
        detail::IDeviceAccessor* pDeviceAccessor = nullptr;
        switch (port)
        {
        #if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
            case Port_Mmc0:
                pDeviceAccessor = detail::GetDeviceAccessorOfPortMmc0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
            case Port_SdCard0:
                pDeviceAccessor = detail::GetDeviceAccessorOfPortSdCard0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
            case Port_GcAsic0:
                pDeviceAccessor = detail::GetDeviceAccessorOfPortGcAsic0();
                break;
        #endif
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT_UNLESS_NOT_NULL(pDeviceAccessor);
        return pDeviceAccessor;
    }
}

void Initialize(Port port, const BufferInfo* pBufferInfos, int numBufferInfos) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        detail::IHostController* pHostController = GetHostController(port);
        pHostController->PreSetBufferInfos(pBufferInfos, numBufferInfos);
    #else
        NN_UNUSED(pBufferInfos);
        NN_UNUSED(numBufferInfos);
        NN_ABORT("NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED isn't defined.\n");
    #endif

    Initialize(port);
}

void Initialize(Port port) NN_NOEXCEPT
{
    #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
        // TODO: 暫定実装
        if (!s_IsClockResetInitialized)
        {
            detail::ClockResetController::Initialize();
            s_IsClockResetInitialized = true;
        }
    #endif

    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    pDeviceAccessor->Initialize();
}

void Finalize(Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    pDeviceAccessor->Finalize();
}

#if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
    void SetWorkBufferForHostController(Port port, void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        detail::IHostController* pHostController = GetHostController(port);
        pHostController->SetWorkBuffer(pWorkBuffer, workBufferSize);
    }
#endif

void Deactivate(Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    pDeviceAccessor->Deactivate();
}

Result Read(void* pOutDataBuffer, size_t dataBufferSize, Port port, uint32_t sectorIndex, uint32_t numSectors) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->ReadWrite(sectorIndex, numSectors, pOutDataBuffer, dataBufferSize, true);
}

Result Write(Port port, uint32_t sectorIndex, uint32_t numSectors, const void* pDataBuffer, size_t dataBufferSize) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->ReadWrite(sectorIndex, numSectors, const_cast<void*>(pDataBuffer), dataBufferSize, false);
}

void GetHostBusStatus(HostBusStatus* pOutHostBusStatus, Port port) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHostBusStatus);
    detail::IHostController* pHostController = GetHostController(port);
    pOutHostBusStatus->busPower = pHostController->GetBusPower();
    pOutHostBusStatus->busWidth = pHostController->GetBusWidth();
    pOutHostBusStatus->deviceClockFrequencyKHz = pHostController->GetDeviceClockFrequencyKHz();;
}

Result GetDeviceMemoryCapacity(uint32_t* pOutNumSectors, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetMemoryCapacity(pOutNumSectors);
}

Result GetDeviceStatus(uint32_t* pOutDeviceStatus, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetDeviceStatus(pOutDeviceStatus);
}

Result GetDeviceOcr(uint32_t* pOutOcr, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetOcr(pOutOcr);
}

Result GetDeviceRca(uint16_t* pOutRca, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetRca(pOutRca);
}

Result GetDeviceCid(void* pOutCid, size_t cidSize, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetCid(pOutCid, cidSize);
}

Result GetDeviceCsd(void* pOutCsd, size_t csdSize, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetCsd(pOutCsd, csdSize);
}

void SetAbortCommandForDebug(Port port, uint32_t commandIndex, uint32_t commandArgument, int64_t abortTimingNanoSeconds) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        detail::IHostController* pHostController = GetHostController(port);
        return pHostController->SetAbortCommandForDebug(commandIndex, commandArgument, abortTimingNanoSeconds);
    #else
        NN_UNUSED(port);
        NN_UNUSED(commandIndex);
        NN_UNUSED(commandArgument);
        NN_UNUSED(abortTimingNanoSeconds);
        NN_ABORT("NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG isn't defined.\n");
    #endif
}

void ClearAbortCommandForDebug(Port port) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        detail::IHostController* pHostController = GetHostController(port);
        return pHostController->ClearAbortCommandForDebug();
    #else
        NN_UNUSED(port);
        NN_ABORT("NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG isn't defined.\n");
    #endif
}

Result IssueCommandForDebug(uint32_t* pOutResponse, Port port, uint32_t commandIndex, uint32_t commandArgument,
    DataTransfer* pDataTransfer, bool isBusy) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ISSUE_COMMAND_FOR_DEBUG))
        detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
        return pDeviceAccessor->IssueCommandForDebug(pOutResponse, commandIndex, commandArgument, pDataTransfer, isBusy);
    #else
        NN_UNUSED(pOutResponse);
        NN_UNUSED(port);
        NN_UNUSED(commandIndex);
        NN_UNUSED(commandArgument);
        NN_UNUSED(pDataTransfer);
        NN_UNUSED(isBusy);
        NN_ABORT("NN_DETAIL_SDMMC_ISSUE_COMMAND_FOR_DEBUG isn't defined.\n");
        return ResultNotSupported();
    #endif
}

}} // namespace nn { namespace sdmmc1 {
