﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_SdmmcController.tegra-x1.h"
#include <nn/nn_Abort.h>
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc1 {
namespace detail {

namespace
{
    // Tuning Command 完了タイムアウト (ms), TODO: 正式なタイムアウト時間の決定
    const uint32_t TuningCommandSwTimeout = 100;

    // レジスタログ
    #define NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(basePointer, memberVariable) NN_DETAIL_SDMMC_DEBUG_LOG(#memberVariable "(0x%03x) 0x%08X\n",\
        reinterpret_cast<uintptr_t>(&(basePointer->memberVariable)) - reinterpret_cast<uintptr_t>(basePointer), (basePointer->memberVariable))

    // 03Eh Host Control 2 Register ビット定義
    const uint16_t REG_HOST_CONTROL_2__UHS_MODE_SELECT__HS200   = REG_HOST_CONTROL_2__UHS_MODE_SELECT__SDR104;
    //const uint16_t REG_HOST_CONTROL_2__UHS_MODE_SELECT__DDR50   = 0x4U << REG_HOST_CONTROL_2__UHS_MODE_SELECT__SHIFT;
    const uint16_t REG_HOST_CONTROL_2__UHS_MODE_SELECT__HS400   = 0x5U << REG_HOST_CONTROL_2__UHS_MODE_SELECT__SHIFT;

    // 0x100 VENDOR_CLOCK_CNTRL_0 ビット定義
    const uint8_t REG_VENDOR_CLOCK_CNTRL_0__TRIM_VAL__SHIFT     = 24;
    const uint32_t REG_VENDOR_CLOCK_CNTRL_0__TRIM_VAL__MASK     = 0x1FU << REG_VENDOR_CLOCK_CNTRL_0__TRIM_VAL__SHIFT;
    const uint8_t REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__SHIFT      = 16;
    const uint32_t REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__MASK      = 0xFFU << REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__SHIFT;

    // 0x10c VENDOR_CAP_OVERRIDES_0 ビット定義
    const uint8_t REG_VENDOR_CAP_OVERRIDES_0__DQS_TRIM_VAL__SHIFT   = 8;
    const uint32_t REG_VENDOR_CAP_OVERRIDES_0__DQS_TRIM_VAL__MASK   = 0x3FU << REG_VENDOR_CAP_OVERRIDES_0__DQS_TRIM_VAL__SHIFT;
    const uint32_t REG_VENDOR_CAP_OVERRIDES_0__DQS_TRIM_VAL__HS400  = 40U << REG_VENDOR_CAP_OVERRIDES_0__DQS_TRIM_VAL__SHIFT;

    // 0x1ac VENDOR_IO_TRIM_CNTRL_0 ビット定義
    const uint8_t REG_VENDOR_IO_TRIM_CNTRL_0__SEL_VREG          = 0x01U << 2;

    // 0x1b0 VENDOR_DLLCAL_CFG_0 ビット定義
    const uint32_t REG_VENDOR_DLLCAL_CFG_0__CALIBRATE           = 0x1U << 31;

    // 0x1bc VENDOR_DLLCAL_CFG_STA_0 ビット定義
    const uint32_t REG_VENDOR_DLLCAL_CFG_STA_0__DLL_CAL_ACTIVE  = 0x1U << 31;
    #if (defined(NN_DETAIL_SDMMC_DLL_CALIBLATION_LOG_ENABLE))
        const uint8_t REG_VENDOR_DLLCAL_CFG_STA_0__TX_DLY_CODE_ADJ__SHIFT   = 16;
        const uint32_t REG_VENDOR_DLLCAL_CFG_STA_0__TX_DLY_CODE_ADJ__MASK   = 0x7FU << REG_VENDOR_DLLCAL_CFG_STA_0__TX_DLY_CODE_ADJ__SHIFT;
        const uint8_t REG_VENDOR_DLLCAL_CFG_STA_0__RX_DLY_CODE_ADJ__SHIFT   = 8;
        const uint32_t REG_VENDOR_DLLCAL_CFG_STA_0__RX_DLY_CODE_ADJ__MASK   = 0x7FU << REG_VENDOR_DLLCAL_CFG_STA_0__RX_DLY_CODE_ADJ__SHIFT;
    #endif

    // 0x1c0 VENDOR_TUNING_CNTRL0_0 ビット定義
    const uint32_t REG_VENDOR_TUNING_CNTRL0_0__TAP_VAL_UPDATED_BY_HW                = 0x1U << 17;
    const uint8_t REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__SHIFT         = 13;
    const uint32_t REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__MASK         = 0x7U << REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__SHIFT;
    //const uint32_t REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__TRIES_40     = 0x0U << REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__SHIFT;
    //const uint32_t REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__TRIES_64     = 0x1U << REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__SHIFT;
    const uint32_t REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__TRIES_128    = 0x2U << REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__SHIFT;
    //const uint32_t REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__TRIES_192    = 0x3U << REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__SHIFT;
    const uint32_t REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__TRIES_256    = 0x4U << REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__SHIFT;
    const uint8_t REG_VENDOR_TUNING_CNTRL0_0__MUL_M__SHIFT                          = 6;
    const uint32_t REG_VENDOR_TUNING_CNTRL0_0__MUL_M__MASK                          = 0x7FU << REG_VENDOR_TUNING_CNTRL0_0__MUL_M__SHIFT;
    #if (defined(NN_DETAIL_SDMMC_TUNING_LOG_ENABLE))
        const uint8_t REG_VENDOR_TUNING_CNTRL0_0__TUNING_WORD_SEL__SHIFT            = 0;
        const uint32_t REG_VENDOR_TUNING_CNTRL0_0__TUNING_WORD_SEL__MASK            = 0x7U << REG_VENDOR_TUNING_CNTRL0_0__TUNING_WORD_SEL__SHIFT;
    #endif

    // 0x1e0 SDMEMCOMPPADCTRL_0 ビット定義
    const uint32_t REG_SDMEMCOMPPADCTRL_0__PAD_E_INPUT_OR_E_PWRD    = 0x1U << 31;
    const uint8_t REG_SDMEMCOMPPADCTRL_0__SDMMC2TMC_CFG_SDMEMCOMP_VREF_SEL__SHIFT   = 0;
    const uint32_t REG_SDMEMCOMPPADCTRL_0__SDMMC2TMC_CFG_SDMEMCOMP_VREF_SEL__MASK   = 0xFU << REG_SDMEMCOMPPADCTRL_0__SDMMC2TMC_CFG_SDMEMCOMP_VREF_SEL__SHIFT;
    const uint32_t REG_SDMEMCOMPPADCTRL_0__SDMMC2TMC_CFG_SDMEMCOMP_VREF_SEL__DEFAULT = 0x7U << REG_SDMEMCOMPPADCTRL_0__SDMMC2TMC_CFG_SDMEMCOMP_VREF_SEL__SHIFT;

    // 0x1e4 AUTO_CAL_CONFIG_0 ビット定義
    const uint32_t REG_AUTO_CAL_CONFIG_0__AUTO_CAL_START        = 0x1U << 31;
    const uint32_t REG_AUTO_CAL_CONFIG_0__AUTO_CAL_ENABLE       = 0x1U << 29;
    const uint8_t REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PD_OFFSET__SHIFT = 8;
    const uint32_t REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PD_OFFSET__MASK = 0x7FU << REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PD_OFFSET__SHIFT;
    const uint8_t REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PU_OFFSET__SHIFT = 0;
    const uint32_t REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PU_OFFSET__MASK = 0x7FU << REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PU_OFFSET__SHIFT;

    // 0x1ec AUTO_CAL_STATUS_0 ビット定義
    const uint32_t REG_AUTO_CAL_STATUS_0__AUTO_CAL_ACTIVE       = 0x1U << 31;
    #if (defined(NN_DETAIL_SDMMC_AUTO_CALIBLATION_LOG_ENABLE))
        const uint8_t REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLDOWN__SHIFT = 8;
        const uint32_t REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLDOWN__MASK = 0x7FU << REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLDOWN__SHIFT;
        const uint8_t REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLUP__SHIFT = 0;
        const uint32_t REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLUP__MASK = 0x7FU << REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLUP__SHIFT;
    #endif

    // 0x1f0 IO_SPARE_0 ビット定義
    const uint32_t REG_IO_SPARE_0__SPARE_OUT_3  = 0x1U << 19;

    void GetDividerSetting(uint32_t* pOutTargetSourceClockFrequencyKHz, uint16_t* pOutX, SpeedMode speedMode) NN_NOEXCEPT
    {
        // Divided by "X = 1 / (2 * N)", 1 分周より大きいものは 2 の倍数
        switch (speedMode)
        {
        case SpeedMode_Identification:
            // 400kHz 以下、25MHz 以下を 64 分周
            *pOutTargetSourceClockFrequencyKHz = 25000;
            *pOutX = 64;
            return;
        case SpeedMode_SdCardSdr12:
            // 25MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 25000;
            *pOutX = 1;
            return;
        case SpeedMode_SdCardDefaultSpeed:
            // 25MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 25000;
            *pOutX = 1;
            return;
        case SpeedMode_MmcLegacySpeed:
            // 26MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 26000;
            *pOutX = 1;
            return;
        case SpeedMode_SdCardHighSpeed:
            // 50MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 50000;
            *pOutX = 1;
            return;
        case SpeedMode_MmcHighSpeed:
            // 52MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 52000;
            *pOutX = 1;
            return;
        case SpeedMode_GcAsicFpgaSpeed:
            // 40.8MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 40800;
            *pOutX = 1;
            return;
        case SpeedMode_SdCardSdr50:
            // 100MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 100000;
            *pOutX = 1;
            return;
        case SpeedMode_GcAsicSpeed:
            // 200MHz 以下を 2 分周（HS200 と同じクロックソースを選択）
            *pOutTargetSourceClockFrequencyKHz = 200000;
            *pOutX = 2;
            return;
        case SpeedMode_MmcHs200:
            // 200MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 200000;
            *pOutX = 1;
            return;
        case SpeedMode_MmcHs400:
            // 200MHz 以下を 1 分周（HS200 と同じにする必要がある）
            *pOutTargetSourceClockFrequencyKHz = 200000;
            *pOutX = 1;
            return;
        case SpeedMode_SdCardSdr104:
            // 208MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 208000;
            *pOutX = 1;
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void SdmmcController::SetClockTrimmer(SpeedMode speedMode) NN_NOEXCEPT
{
    if (speedMode == SpeedMode_MmcHs400)
    {
        // DQS trimmer tap 設定
        m_pRegisters->vendorCapOverrides0 = (m_pRegisters->vendorCapOverrides0 & (~REG_VENDOR_CAP_OVERRIDES_0__DQS_TRIM_VAL__MASK)) | REG_VENDOR_CAP_OVERRIDES_0__DQS_TRIM_VAL__HS400;
    }

    // TAP_VAL を SW によって設定する
    m_pRegisters->vendorTuningCntrl00 &= (~REG_VENDOR_TUNING_CNTRL0_0__TAP_VAL_UPDATED_BY_HW);

    // Inbound tap 設定
    uint8_t tapValue;
    if (speedMode == SpeedMode_MmcHs400)
    {
        NN_ABORT_UNLESS(m_IsValidTapValueForHs400);
        tapValue = m_TapValueForHs400;
    }
    else
    {
        tapValue = GetDefaultInboundTapValue();
    }
    m_pRegisters->vendorClockCntrl0 = (m_pRegisters->vendorClockCntrl0 & (~REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__MASK)) | (tapValue << REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__SHIFT);
}

Result SdmmcController::CalibrateDll() NN_NOEXCEPT
{
    bool isClockDisabled = false;
    if ((m_pRegisters->sdHostStandardRegisters.clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) == 0)
    {
        isClockDisabled = true;
        m_pRegisters->sdHostStandardRegisters.clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
    }

    // calibration 開始
    Result result = ResultSuccess();
    m_pRegisters->vendorDllcalCfg0 |= REG_VENDOR_DLLCAL_CFG_0__CALIBRATE;

    ManualTimer timer1(5);
    while (true)
    {
        // SoC 内部の処理のため、デバイスの抜けチェックは行わない

        if ((m_pRegisters->vendorDllcalCfg0 & REG_VENDOR_DLLCAL_CFG_0__CALIBRATE) == 0)
        {
            break;  // calibration 完了
        }
        if (!(timer1.Update()))
        {
            // calibration が完了しないまま、タイムアウトを過ぎている
            NN_DETAIL_SDMMC_ERROR_LOG("DLL Calibration is timeout\n");
            result = ResultSdmmcDllCalibrationSwTimetut();
            break;
        }
        // us オーダで完了する処理のはずなので Sleep はしない
    }
    if (result.IsSuccess())
    {
        ManualTimer timer2(10);
        while (true)
        {
            // SoC 内部の処理のため、デバイスの抜けチェックは行わない

            if ((m_pRegisters->vendorDllcalCfgSta0 & REG_VENDOR_DLLCAL_CFG_STA_0__DLL_CAL_ACTIVE) == 0)
            {
                // delay code 反映完了
                #if (defined(NN_DETAIL_SDMMC_DLL_CALIBLATION_LOG_ENABLE))
                    uint32_t value = (m_pRegisters->vendorDllcalCfgSta0 & REG_VENDOR_DLLCAL_CFG_STA_0__TX_DLY_CODE_ADJ__MASK) >> REG_VENDOR_DLLCAL_CFG_STA_0__TX_DLY_CODE_ADJ__SHIFT;
                    NN_DETAIL_SDMMC_LOG("[sdmmc] TX_DLY_CODE_ADJ: 0x%02X\n", value);
                    value = (m_pRegisters->vendorDllcalCfgSta0 & REG_VENDOR_DLLCAL_CFG_STA_0__RX_DLY_CODE_ADJ__MASK) >> REG_VENDOR_DLLCAL_CFG_STA_0__RX_DLY_CODE_ADJ__SHIFT;
                    NN_DETAIL_SDMMC_LOG("[sdmmc] RX_DLY_CODE_ADJ: 0x%02X\n", value);
                #endif
                break;
            }
            if (!(timer2.Update()))
            {
                // delay code が反映されないまま、タイムアウトを過ぎている
                NN_DETAIL_SDMMC_ERROR_LOG("DLL Application is timeout\n");
                result = ResultSdmmcDllApplicationSwTimetut();
                break;
            }
            // us オーダで完了する処理のはずなので Sleep はしない
        }
    }

    if (isClockDisabled)
    {
        m_pRegisters->sdHostStandardRegisters.clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
    }

    return result;
}

Result SdmmcController::IssueTuningCommand(uint32_t commandIndex) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!SdHostStandardController::IsPowerSavingEnable());

    Result result = SdHostStandardController::WaitWhileCommandInhibit(true);
    if (result.IsFailure())
    {
        (void)SdHostStandardController::AbortTransaction();
        return result;
    }

    // データ転送設定（コマンド発行前に設定する必要がある）
    SdHostStandardController::SetTransferForTuning();

    // ステータスレジスタ許可、その後にクリア
    m_pRegisters->sdHostStandardRegisters.normalInterruptStatusEnable |= REG_NORMAL_INTERRUPT__BUFFER_READ_READY;
    m_pRegisters->sdHostStandardRegisters.normalInterruptStatus = m_pRegisters->sdHostStandardRegisters.normalInterruptStatus;

    // Eratta MMC-1 WAR: グリッジ防止のため、クロック停止
    m_pRegisters->sdHostStandardRegisters.clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);

    // コマンド発行
    SdHostStandardController::SetCommandForTuning(commandIndex);

    // Eratta MMC-1 WAR: Wait 1us 後に CMD, DAT 回路をリセットする
    WaitMicroseconds(1);
    (void)SdHostStandardController::AbortTransaction();

    // Eratta MMC-1 WAR: クロック再開
    m_pRegisters->sdHostStandardRegisters.clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;

    ManualTimer timer(TuningCommandSwTimeout);
    while (true)
    {
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            if (SdHostStandardController::IsRemoved())
            {
                result = ResultDeviceRemoved();
                break;
            }
        #endif

        if ((m_pRegisters->sdHostStandardRegisters.normalInterruptStatus & REG_NORMAL_INTERRUPT__BUFFER_READ_READY) != 0)
        {
            // Buffer Read Ready だけクリア（1 を立てたビットだけがクリアされる）
            m_pRegisters->sdHostStandardRegisters.normalInterruptStatus = REG_NORMAL_INTERRUPT__BUFFER_READ_READY;
            break;
        }
        if (!(timer.Update()))
        {
            // Tuning Command の通信完了を認識しないまま、タイムアウトを過ぎている
            (void)SdHostStandardController::AbortTransaction();
            NN_DETAIL_SDMMC_ERROR_LOG("ResultIssueTuningCommandTimeout\n");
            result = ResultIssueTuningCommandSwTimeout();
            break;
        }
        // us から ms オーダの待ちが想定されるが、パフォーマンスに影響するため Sleep はしない
    }

    // ステータスレジスタ禁止
    m_pRegisters->sdHostStandardRegisters.normalInterruptStatusEnable &= (~REG_NORMAL_INTERRUPT__BUFFER_READ_READY);

    // 念のため、デバイスが tran-state に戻る時間を想定し、次のコマンド発行まで 8 clock 保証する
    WaitClocks(8, SdHostStandardController::GetDeviceClockFrequencyKHz());

    return result;
}

void SdmmcController::SetDriveCodeOffsets(BusPower busPower) NN_NOEXCEPT
{
    uint8_t autoCalPdOffset;
    uint8_t autoCalPuOffset;
    GetAutoCalOffsets(&autoCalPdOffset, &autoCalPuOffset, busPower);
    uint32_t value = m_pRegisters->autoCalConfig0;
    value = (value & (~REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PD_OFFSET__MASK)) | (autoCalPdOffset << REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PD_OFFSET__SHIFT);
    value = (value & (~REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PU_OFFSET__MASK)) | (autoCalPuOffset << REG_AUTO_CAL_CONFIG_0__AUTO_CAL_PU_OFFSET__SHIFT);
    m_pRegisters->autoCalConfig0 = value;
}

void SdmmcController::CalibrateDriveStrength(BusPower busPower) NN_NOEXCEPT
{
    bool isClockEnabled = false;

    if ((m_pRegisters->sdHostStandardRegisters.clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) != 0)
    {
        isClockEnabled = true;

        // Abnormal CLK 回避のため、Calibration 中はクロックを停止する
        m_pRegisters->sdHostStandardRegisters.clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
    }

    if ((m_pRegisters->sdmemcomppadctrl0 & REG_SDMEMCOMPPADCTRL_0__PAD_E_INPUT_OR_E_PWRD) == 0)
    {
        m_pRegisters->sdmemcomppadctrl0 |= REG_SDMEMCOMPPADCTRL_0__PAD_E_INPUT_OR_E_PWRD;
        WaitMicroseconds(1);
    }

    m_pRegisters->autoCalConfig0 |= (REG_AUTO_CAL_CONFIG_0__AUTO_CAL_START | REG_AUTO_CAL_CONFIG_0__AUTO_CAL_ENABLE);
    WaitMicroseconds(1);

    ManualTimer timer(10);
    while (true)
    {
        // SoC 内部の処理のため、デバイスの抜けチェックは行わない

        if ((m_pRegisters->autoCalStatus0 & REG_AUTO_CAL_STATUS_0__AUTO_CAL_ACTIVE) == 0)
        {
            #if (defined(NN_DETAIL_SDMMC_AUTO_CALIBLATION_LOG_ENABLE))
                uint32_t value = (m_pRegisters->autoCalStatus0 & REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLDOWN__MASK) >> REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLDOWN__SHIFT;
                NN_DETAIL_SDMMC_LOG("[sdmmc] AUTO_CAL_PULLDOWN: %u\n", value);
                value = (m_pRegisters->autoCalStatus0 & REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLUP__MASK) >> REG_AUTO_CAL_STATUS_0__AUTO_CAL_PULLUP__SHIFT;
                NN_DETAIL_SDMMC_LOG("[sdmmc] AUTO_CAL_PULLUP: %u\n", value);
            #endif
            break;
        }
        if (!(timer.Update()))
        {
            // calibration が完了しないまま、タイムアウトを過ぎている
            NN_DETAIL_SDMMC_ERROR_LOG("Auto Calibration is timeout\n");
            SetDriveStrengthToDefaultValues(busPower);
            m_pRegisters->autoCalConfig0 &= (~REG_AUTO_CAL_CONFIG_0__AUTO_CAL_ENABLE);
            break;
        }
        // us オーダで完了する処理のはずなので Sleep はしない
    }

    m_pRegisters->sdmemcomppadctrl0 &= (~REG_SDMEMCOMPPADCTRL_0__PAD_E_INPUT_OR_E_PWRD);

    if (isClockEnabled)
    {
        m_pRegisters->sdHostStandardRegisters.clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
    }
}

void SdmmcController::LogRegisters() NN_NOEXCEPT
{
    SdHostStandardController::LogRegisters();

    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorClockCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorSysSwCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorErrIntrStatus0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorCapOverrides0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorBootCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorBootActTimeout0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, bootDatTimeout0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorDebounceCount0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorMiscCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, maxCurrentOverride0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, maxCurrentOverrideHi0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorIoTrimCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorDllcalCfg0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorDllCtrl00);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorDllCtrl10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorDllcalCfgSta0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorTuningCntrl00);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorTuningCntrl10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorTuningStatus00);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorTuningStatus10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorClkGateHysteresisCount0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorPresetVal00);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorPresetVal10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, vendorPresetVal20);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, sdmemcomppadctrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, autoCalConfig0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, autoCalInterval0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, autoCalStatus0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, ioSpare0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, mmcifFifoctrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pRegisters, timeoutWcalSdmmc0);
}

Result SdmmcController::Startup(BusPower busPower, BusWidth busWidth, SpeedMode speedMode, bool isPowerSavingEnable) NN_NOEXCEPT
{
    detail::ClockResetController::Module module = GetClockResetModule();
    if (detail::ClockResetController::IsAvailable(module))
    {
        // クロックソース変更によるグリッジを防ぐためにクロック供給を止める
        SdHostStandardController::DisableDeviceClock();
    }
    detail::ClockResetController::Reset(module);
    //LogRegisters();

    // ここまで実行したら Shutdown() が可能
    m_IsShutdowned = false;

    Result result = PowerOn(busPower);
    if (result.IsFailure())
    {
        return result;
    }

    SetSchmittTrigger(busPower);

    // one cycle delayed version を選択
    m_pRegisters->ioSpare0 |= REG_IO_SPARE_0__SPARE_OUT_3;

    // trimmer power supply として VREG を選択
    m_pRegisters->vendorIoTrimCntrl0 &= (~REG_VENDOR_IO_TRIM_CNTRL_0__SEL_VREG);

    // Outbound tap 設定
    m_pRegisters->vendorClockCntrl0 = (m_pRegisters->vendorClockCntrl0 & (~REG_VENDOR_CLOCK_CNTRL_0__TRIM_VAL__MASK)) | (GetOutboundTapValue() << REG_VENDOR_CLOCK_CNTRL_0__TRIM_VAL__SHIFT);

    // Calibration pad VSEL 設定
    m_pRegisters->sdmemcomppadctrl0 = (m_pRegisters->sdmemcomppadctrl0 & REG_SDMEMCOMPPADCTRL_0__SDMMC2TMC_CFG_SDMEMCOMP_VREF_SEL__MASK) | REG_SDMEMCOMPPADCTRL_0__SDMMC2TMC_CFG_SDMEMCOMP_VREF_SEL__DEFAULT;

    // 最初の Drive Strength の Calibration 実行
    SetDriveCodeOffsets(busPower);
    CalibrateDriveStrength(busPower);

    result = SdHostStandardController::EnableInternalClock();
    if (result.IsFailure())
    {
        return result;
    }

    SdHostStandardController::SetBusWidth(busWidth);
    SdHostStandardController::SetBusPower(busPower);
    result = SetSpeedMode(speedMode);
    if (result.IsFailure())
    {
        return result;
    }
    SetPowerSaving(isPowerSavingEnable);
    SdHostStandardController::EnableDeviceClock();

    return ResultSuccess();
}

void SdmmcController::Shutdown() NN_NOEXCEPT
{
    if (m_IsShutdowned)
    {
        return;
    }
    SdHostStandardController::DisableDeviceClock();
    SdHostStandardController::SetBusPower(BusPower_Off);
    PowerOff();
    m_IsShutdowned = true;
}

Result SdmmcController::SwitchToSdr12() NN_NOEXCEPT
{
    // クロック停止
    m_pRegisters->sdHostStandardRegisters.clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);

    // SD カードが電圧変更シーケンスを始める状態になっているか確認
    if ((m_pRegisters->sdHostStandardRegisters.presentState & REG_PRESENT_STATE__DAT3_0_LINE_SIGNAL_LEVEL) != 0)
    {
        return ResultSdCardNotReadyToVoltageSwitch();
    }

    // 1.8V モードに入る（REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE 設定を含む）
    SetSpeedMode(SpeedMode_SdCardSdr12);

    // バス電圧元を 1.8V に変更する
    Result result = LowerBusPower();
    if (result.IsFailure())
    {
        return result;
    }
    SetSchmittTrigger(BusPower_1_8V);

    // 最初の Drive Strength の Calibration 実行
    SetDriveCodeOffsets(BusPower_1_8V);
    CalibrateDriveStrength(BusPower_1_8V);

    SdHostStandardController::SetBusPower(BusPower_1_8V);

    // クロック再開まで最小 5ms 空ける
    WaitMicroseconds(5000);

    // HostController が 1.8V モードに入ったか確認
    if ((m_pRegisters->sdHostStandardRegisters.hostControl2 | REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE) == 0)
    {
        ResultSdHostStandarFailSwitchTo18V();
    }

    // クロック再開し、最大 1ms 待つ
    m_pRegisters->sdHostStandardRegisters.clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
    WaitMicroseconds(1000);

    // SD カードが電圧変更を完了したか確認
    if ((m_pRegisters->sdHostStandardRegisters.presentState & REG_PRESENT_STATE__DAT3_0_LINE_SIGNAL_LEVEL) != REG_PRESENT_STATE__DAT3_0_LINE_SIGNAL_LEVEL)
    {
        return ResultSdCardNotCompleteVoltageSwitch();
    }

    return ResultSuccess();
}

Result SdmmcController::SetSpeedMode(SpeedMode speedMode) NN_NOEXCEPT
{
    bool isClockEnabled = false;

    if ((m_pRegisters->sdHostStandardRegisters.clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) != 0)
    {
        isClockEnabled = true;

        // クロックソースを変更するため、一時的にクロックを停止する
        m_pRegisters->sdHostStandardRegisters.clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
    }

    SetClockTrimmer(speedMode);

    switch (speedMode)
    {
    case SpeedMode_Identification:
        m_pRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        m_pRegisters->sdHostStandardRegisters.hostControl2 &= (~REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE);
        break;
    case SpeedMode_MmcLegacySpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        m_pRegisters->sdHostStandardRegisters.hostControl2 &= (~REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE);
        break;
    case SpeedMode_MmcHighSpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 |= REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE;
        m_pRegisters->sdHostStandardRegisters.hostControl2 &= (~REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE);
        break;
    case SpeedMode_MmcHs200:
        m_pRegisters->sdHostStandardRegisters.hostControl2 = (m_pRegisters->sdHostStandardRegisters.hostControl2 & (~REG_HOST_CONTROL_2__UHS_MODE_SELECT__MASK)) | REG_HOST_CONTROL_2__UHS_MODE_SELECT__HS200;
        m_pRegisters->sdHostStandardRegisters.hostControl2 |= REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE;
        break;
    case SpeedMode_MmcHs400:
        m_pRegisters->sdHostStandardRegisters.hostControl2 = (m_pRegisters->sdHostStandardRegisters.hostControl2 & (~REG_HOST_CONTROL_2__UHS_MODE_SELECT__MASK)) | REG_HOST_CONTROL_2__UHS_MODE_SELECT__HS400;
        m_pRegisters->sdHostStandardRegisters.hostControl2 |= REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE;
        break;
    case SpeedMode_SdCardDefaultSpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        m_pRegisters->sdHostStandardRegisters.hostControl2 &= (~REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE);
        break;
    case SpeedMode_SdCardHighSpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl1 |= REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE;
        m_pRegisters->sdHostStandardRegisters.hostControl2 &= (~REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE);
        break;
    case SpeedMode_SdCardSdr12:
        m_pRegisters->sdHostStandardRegisters.hostControl2 = (m_pRegisters->sdHostStandardRegisters.hostControl2 & (~REG_HOST_CONTROL_2__UHS_MODE_SELECT__MASK)) | REG_HOST_CONTROL_2__UHS_MODE_SELECT__SDR12;
        m_pRegisters->sdHostStandardRegisters.hostControl2 |= REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE;
        break;
    case SpeedMode_SdCardSdr50:
        m_pRegisters->sdHostStandardRegisters.hostControl2 = (m_pRegisters->sdHostStandardRegisters.hostControl2 & (~REG_HOST_CONTROL_2__UHS_MODE_SELECT__MASK)) | REG_HOST_CONTROL_2__UHS_MODE_SELECT__SDR50;
        m_pRegisters->sdHostStandardRegisters.hostControl2 |= REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE;
        break;
    case SpeedMode_SdCardSdr104:
        m_pRegisters->sdHostStandardRegisters.hostControl2 = (m_pRegisters->sdHostStandardRegisters.hostControl2 & (~REG_HOST_CONTROL_2__UHS_MODE_SELECT__MASK)) | REG_HOST_CONTROL_2__UHS_MODE_SELECT__SDR104;
        m_pRegisters->sdHostStandardRegisters.hostControl2 |= REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE;
        break;
    case SpeedMode_GcAsicFpgaSpeed:
        NN_FALL_THROUGH;
    case SpeedMode_GcAsicSpeed:
        m_pRegisters->sdHostStandardRegisters.hostControl2 = (m_pRegisters->sdHostStandardRegisters.hostControl2 & (~REG_HOST_CONTROL_2__UHS_MODE_SELECT__MASK)) | REG_HOST_CONTROL_2__UHS_MODE_SELECT__HS200;
        m_pRegisters->sdHostStandardRegisters.hostControl2 |= REG_HOST_CONTROL_2__1_8V_SIGNALING_ENABLE;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 分周設定の取得
    uint32_t targetSourceClockFrequencyKHz;
    uint16_t x; // Divided by "X = 1 / (2 * N)", 1 分周より大きいものは 2 の倍数
    GetDividerSetting(&targetSourceClockFrequencyKHz, &x, speedMode);

    // ソースクロックの設定
    uint32_t actualSourceClockFrequencyKHz;
    detail::ClockResetController::SetClockFrequencyKHz(&actualSourceClockFrequencyKHz, GetClockResetModule(), targetSourceClockFrequencyKHz);

    // デバイスクロックの算出（Wait 時間算出に使用するため、繰り上げ計算）
    uint32_t actualDeviceClockFrequencyKHz = (actualSourceClockFrequencyKHz + (x - 1)) / x;
    SdHostStandardController::SetDeviceClockFrequencyKHz(actualDeviceClockFrequencyKHz);

    // SDMMC での分周設定
    NN_ABORT_UNLESS((x == 1) || ((x & 0x1) == 0));
    uint16_t n = x / 2;     // N = 1 / (2 * X), 0: Base Clock
    uint16_t upper_n = 0;
    if (n > 0xFF)
    {
        upper_n = n >> 8;
    }
    uint16_t mask = REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__MASK | REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__MASK;
    m_pRegisters->sdHostStandardRegisters.clockControl = (m_pRegisters->sdHostStandardRegisters.clockControl & (~mask)) | (n << REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__SHIFT) | (upper_n << REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__SHIFT);

    if (isClockEnabled)
    {
        m_pRegisters->sdHostStandardRegisters.clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
    }

    if (speedMode == SpeedMode_MmcHs400)
    {
        Result result = CalibrateDll();
        if (result.IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}

Result SdmmcController::Tuning(SpeedMode speedMode, uint32_t commandIndex) NN_NOEXCEPT
{
    // Eratta MMC-1 WAR: STEP_SIZE を 1 に設定する
    m_pRegisters->vendorTuningCntrl10 = 0;

    // Eratta MMC-1 WAR: 1.5UI ほどのステップ数設定 (1TAP 70ps - 500ps)
    int numSteps;
    uint32_t numTuniingIterations;
    switch (speedMode)
    {
    case SpeedMode_GcAsicFpgaSpeed:
        NN_FALL_THROUGH;
    case SpeedMode_GcAsicSpeed:
        NN_FALL_THROUGH;
    case SpeedMode_SdCardSdr50:
        numSteps = 256;
        numTuniingIterations = REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__TRIES_256;
        break;
    case SpeedMode_SdCardSdr104:
        NN_FALL_THROUGH;
    case SpeedMode_MmcHs200:
        NN_FALL_THROUGH;
    case SpeedMode_MmcHs400:
        numSteps = 128;
        numTuniingIterations = REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__TRIES_128;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_pRegisters->vendorTuningCntrl00 = (m_pRegisters->vendorTuningCntrl00 & (~REG_VENDOR_TUNING_CNTRL0_0__NUM_TUNING_INTERATIONS__MASK)) | numTuniingIterations;

    // Window の中央を Ideal Value とする
    // MUL_M = 1, DIV_N = 2(Reset Value), Q = (MUL_M + 1) / (2^DIV_N) = 50%
    m_pRegisters->vendorTuningCntrl00 = (m_pRegisters->vendorTuningCntrl00 & (~REG_VENDOR_TUNING_CNTRL0_0__MUL_M__MASK)) | (0x1U << REG_VENDOR_TUNING_CNTRL0_0__MUL_M__SHIFT);

    // TAP_VAL を HW によって更新する
    m_pRegisters->vendorTuningCntrl00 |= REG_VENDOR_TUNING_CNTRL0_0__TAP_VAL_UPDATED_BY_HW;

    // Tuning 開始
    m_pRegisters->sdHostStandardRegisters.hostControl2 |= REG_HOST_CONTROL_2__EXECUTE_TUNING;

    int i = 0;
    while (true)
    {
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            if (SdHostStandardController::IsRemoved())
            {
                return ResultDeviceRemoved();
            }
        #endif

        NN_DETAIL_SDMMC_DEBUG_LOG("No.%d Issue Tuning Command\n", i);
        (void)IssueTuningCommand(commandIndex);     // 実行結果は無視する
        if ((m_pRegisters->sdHostStandardRegisters.hostControl2 & REG_HOST_CONTROL_2__EXECUTE_TUNING) == 0)
        {
            break;
        }
        if (i >= numSteps)
        {
            // 想定外(想定 + 1)の回数のため、あきらめる
            NN_DETAIL_SDMMC_DEBUG_LOG("Unexpected loop of Tuning\n");
            break;
        }
        i++;
    }

    #if (defined(NN_DETAIL_SDMMC_TUNING_LOG_ENABLE))
        NN_DETAIL_SDMMC_LOG("[sdmmc] Tuning Status:\n");
        for (int tuningWordSel = 7; tuningWordSel >= 0; tuningWordSel--)
        {
            m_pRegisters->vendorTuningCntrl00 = (m_pRegisters->vendorTuningCntrl00 & (~REG_VENDOR_TUNING_CNTRL0_0__TUNING_WORD_SEL__MASK)) | (tuningWordSel << REG_VENDOR_TUNING_CNTRL0_0__TUNING_WORD_SEL__SHIFT);
            NN_DETAIL_SDMMC_LOG("%08X ", m_pRegisters->vendorTuningStatus00);
        }
        uint32_t tapVal = (m_pRegisters->vendorClockCntrl0 & REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__MASK) >> REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__SHIFT;
        NN_DETAIL_SDMMC_LOG("\n[sdmmc] TAP_VAL: %u\n", tapVal);
    #endif

    if ((m_pRegisters->sdHostStandardRegisters.hostControl2 & REG_HOST_CONTROL_2__SAMPLING_CLOCK_SELECT) != 0)
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Tuning succeed\n");
        return ResultSuccess();
    }
    else
    {
        NN_DETAIL_SDMMC_ERROR_LOG("ResultTuningFailed\n");
        return ResultTuningFailed();
    }
}

void SdmmcController::SaveTuningStatusForHs400() NN_NOEXCEPT
{
    uint32_t tapVal = (m_pRegisters->vendorClockCntrl0 & REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__MASK) >> REG_VENDOR_CLOCK_CNTRL_0__TAP_VAL__SHIFT;
    m_TapValueForHs400 = static_cast<uint8_t>(tapVal);
    m_IsValidTapValueForHs400 = true;
}

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InterruptEventType Sdmmc1Controller::m_InterruptEvent;
    #endif
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InterruptEventType Sdmmc2Controller::m_InterruptEvent;
    #endif
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InterruptEventType Sdmmc4Controller::m_InterruptEvent;
    #endif
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    Result Sdmmc1Controller::ControlVddioSdmmc1(BusPower busPower) NN_NOEXCEPT
    {
        // TODO: pcv への移動
        uint8_t data[2];
        data[0] = 0x27; // CNFG1_LDO2
        switch (busPower)
        {
        case BusPower_Off:
            data[1] = 0x00;                 // [7:6] 0b00 : Output Disabled
            break;
        case BusPower_1_8V:
            data[1] = (0x3U << 6) | 0x14U;  // [7:6] 0b11 : Normal Mode, [5:0] 0x14 : 1.8V
            break;
        case BusPower_3_3V:
            data[1] = (0x3U << 6) | 0x32U;  // [7:6] 0b11 : Normal Mode, [5:0] 0x32 : 3.3V
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        Result result = nn::i2c::driver::Send(m_I2cMax77620PmicSession, data, sizeof(data),
            static_cast<nn::i2c::TransactionOption>(nn::i2c::TransactionOption_StartCondition | nn::i2c::TransactionOption_StopCondition));

        return result;
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    void Sdmmc1Controller::SetSdmmc1IoMode(bool isBusPower_3_3V) NN_NOEXCEPT
    {
        // TODO: pcv への移動
        // PMC
        uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(0x7000e000ull, 0x1000);
        // APBDEV_PMC_PWR_DET_VAL_0
        uint32_t value = *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x4e4);
        if (isBusPower_3_3V)
        {
            value |= (0x1U << 12);
        }
        else
        {
            value &= (~(0x1U << 12));
        }
        *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x4e4) = value;
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    void Sdmmc1Controller::ControlRailSdmmc1Io(bool isPowerOn) NN_NOEXCEPT
    {
        // TODO: pcv への移動
        // PMC
        uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(0x7000e000ull, 0x1000);
        // APBDEV_PMC_NO_IOPOWER_0
        if (isPowerOn)
        {
            *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x444) &= (~(0x1U << 12));
        }
        else
        {
            *reinterpret_cast<volatile uint32_t*>(registersAddress + 0x444) |= (0x1U << 12);
        }
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    Result Sdmmc1Controller::PowerOn(BusPower busPower) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(busPower == BusPower_3_3V); // 電源 ON では 3.3V しか想定しない

        ControlRailSdmmc1Io(true);

        // SD カード電源 3.3V

#if (defined(NN_DETAIL_SDMMC_SET_GPIO_DIRECTLY))
        uintptr_t gpioRegistersAddress = nn::dd::QueryIoMappingAddress(0x6000d000ull, 0x1000);
        #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2))
            // GPIO_OUT PZ4  出力 HIGH
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x6A4) = 0x1010;
        #else   // NN_BUILD_CONFIG_HARDWARE_NX
            // GPIO_OUT PE4 出力 HIGH
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x1A0) = 0x1010;
        #endif
#else
        nn::gpio::SetValue(&m_GpioPadPowSdEnSession, nn::gpio::GpioValue_High);
#endif
        WaitMicroseconds(18);   // 安定待ち

        // IO 電源 3.3V
        SetSdmmc1IoMode(true);
        Result result = ControlVddioSdmmc1(BusPower_3_3V);
        if (result.IsFailure())
        {
            return result;
        }
        WaitMicroseconds(40);   // 安定待ち
        m_IsBusPowerLowered = false;

        return ResultSuccess();
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    void Sdmmc1Controller::PowerOff() NN_NOEXCEPT
    {
        if (!m_IsBusPowerLowered)
        {
            // WAR: IO 電源 1.8V モードを経由して、0V に落とす
            (void)ControlVddioSdmmc1(BusPower_1_8V);
            WaitMicroseconds(18);   // 安定待ち
            SetSdmmc1IoMode(false);
            m_IsBusPowerLowered = true;
        }

#if (defined(NN_DETAIL_SDMMC_SET_GPIO_DIRECTLY))
        // TODO: pinmux, gpio 経由
        // PINMUX
        uintptr_t pinmuxRegistersAddress = nn::dd::QueryIoMappingAddress(0x70000000ull, 0x4000);
        // GPIO
        uintptr_t gpioRegistersAddress = nn::dd::QueryIoMappingAddress(0x6000d000ull, 0x1000);

        // WAR: 逆流回避のため、1.8V モードで HIGH 出力状態で、電源を落とす
        // Remove CLK pull-down
        *reinterpret_cast<volatile uint32_t*>(pinmuxRegistersAddress + 0x3000) &= (~(0x3 << 2));
        // Set GPIO_CNF PM0-5(CLK, CMD, DAT0-3) to GPIO mode
        *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x380) = 0x3F3F;
        // Set GPIO_OUT PM0-5(CLK, CMD, DAT0-3) to HIGH
        *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x3A0) = 0x3F3F;
        // Set GPIO_OE PM0-5(CLK, CMD, DAT0-3) to Output
        *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x390) = 0x3F3F;
#else
        // WAR: 逆流回避のため、1.8V モードで HIGH 出力状態で、電源を落とす

        nn::pinmux::SetPinAssignment(&m_PinmuxSession, nn::pinmux::PinAssignment_Sdmmc1OutputHigh);
#endif

        // IO 電源 0V
        (void)ControlVddioSdmmc1(BusPower_Off);
        WaitMicroseconds(22);   // 安定待ち

        // SD カード電源 0V

#if (defined(NN_DETAIL_SDMMC_SET_GPIO_DIRECTLY))
        #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2))
            // GPIO_OUT PZ4  出力 HIGH
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x6A4) = 0x1000;
        #else   // NN_BUILD_CONFIG_HARDWARE_NX
            // GPIO_OUT PE4 出力 HIGH
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x1A0) = 0x1000;
        #endif
#else
        nn::gpio::SetValue(&m_GpioPadPowSdEnSession, nn::gpio::GpioValue_Low);
#endif
        WaitMicroseconds(128 * 1000);   // 安定待ち

        ControlRailSdmmc1Io(false);

        // WAR: ダメージ回避のため、3.3V モードにし、元の端子設定に戻す
        SetSdmmc1IoMode(true);

#if (defined(NN_DETAIL_SDMMC_SET_GPIO_DIRECTLY))
        // Set GPIO_CNF PM0-5(CLK, CMD, DAT0-3) to SPIO mode
        *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x380) = 0x3F00;
        // Set GPIO_OE PM0-5(CLK, CMD, DAT0-3) to Input
        *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x390) = 0x3F00;
        // Set GPIO_OUT PM0-5(CLK, CMD, DAT0-3) to LOW
        *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x3A0) = 0x3F00;
        // Set CLK to pull-down
        *reinterpret_cast<volatile uint32_t*>(pinmuxRegistersAddress + 0x3000) = (*reinterpret_cast<volatile uint32_t*>(pinmuxRegistersAddress + 0x3000) & (~(0x3 << 2))) | (1 << 2);
#else
        nn::pinmux::SetPinAssignment(&m_PinmuxSession, nn::pinmux::PinAssignment_Sdmmc1ResetState);
#endif
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    Result Sdmmc1Controller::LowerBusPower() NN_NOEXCEPT
    {
        Result result = ControlVddioSdmmc1(BusPower_1_8V);
        if (result.IsFailure())
        {
            return result;
        }
        WaitMicroseconds(18);   // 安定待ち
        SetSdmmc1IoMode(false);

        m_IsBusPowerLowered = true;

        return ResultSuccess();
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    void Sdmmc1Controller::Initialize() NN_NOEXCEPT
    {

#if (defined(NN_DETAIL_SDMMC_SET_GPIO_DIRECTLY))
        uintptr_t gpioRegistersAddress = nn::dd::QueryIoMappingAddress(0x6000d000ull, 0x1000);
        #if (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2))
            // GPIO_OUT PZ4  出力 Enable, GPIO モード
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x694) = 0x1010;
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x684) = 0x1010;
        #else   // NN_BUILD_CONFIG_HARDWARE_NX
            // GPIO_OUT PE4 出力 Enable, GPIO モード
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x190) = 0x1010;
            *reinterpret_cast<volatile uint16_t*>(gpioRegistersAddress + 0x180) = 0x1010;
        #endif
#else
        nn::pinmux::Initialize();
        nn::pinmux::OpenSession(&m_PinmuxSession, nn::pinmux::AssignablePinGroupName_Sdmmc1);

        nn::gpio::Initialize();
        #if (defined(NN_BUILD_CONFIG_SPEC_NX))
            nn::gpio::OpenSession(&m_GpioPadPowSdEnSession, nn::gpio::GpioPadName_PowSdEn);
        #else
            nn::gpio::OpenSession(&m_GpioPadPowSdEnSession, nn::gpio::GpioPadName_EnablePowerToTheSdCard);
        #endif

        // 出力 LOW （電源 OFF）状態から開始する
        nn::gpio::SetValue(&m_GpioPadPowSdEnSession, nn::gpio::GpioValue_Low);
        nn::gpio::SetDirection(&m_GpioPadPowSdEnSession, nn::gpio::Direction_Output);
#endif

        nn::i2c::driver::Initialize();
        nn::i2c::driver::OpenSession(&m_I2cMax77620PmicSession, nn::i2c::I2cDevice_Max77620Pmic);

        SdmmcController::Initialize();
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    void Sdmmc1Controller::Finalize() NN_NOEXCEPT
    {
        SdmmcController::Finalize();

        nn::i2c::driver::CloseSession(m_I2cMax77620PmicSession);
        nn::i2c::driver::Finalize();

#if (!defined(NN_DETAIL_SDMMC_SET_GPIO_DIRECTLY))
        nn::gpio::CloseSession(&m_GpioPadPowSdEnSession);
        nn::gpio::Finalize();

        nn::pinmux::CloseSession(&m_PinmuxSession);
        nn::pinmux::Finalize();
#endif
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    void Sdmmc1Controller::SetPowerSaving(bool isPowerSavingEnable) NN_NOEXCEPT
    {
        if (!isPowerSavingEnable)
        {
            if (SdHostStandardController::IsDeviceClockEnable())
            {
                // クロックを常時供給にする前に Calibration を実行
                SdmmcController::CalibrateDriveStrength(SdHostStandardController::GetBusPower());
            }
        }
        SdHostStandardController::SetPowerSaving(isPowerSavingEnable);
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    void Sdmmc1Controller::EnableDeviceClock() NN_NOEXCEPT
    {
        if (!SdHostStandardController::IsPowerSavingEnable())
        {
            // クロックを常時供給にする前に Calibration を実行
            SdmmcController::CalibrateDriveStrength(SdHostStandardController::GetBusPower());
        }
        SdHostStandardController::EnableDeviceClock();
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    Result Sdmmc1Controller::IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT
    {
        if (SdHostStandardController::IsPowerSavingEnable())
        {
            // 省電力中はコマンド発行前に Calibration を実行
            SdmmcController::CalibrateDriveStrength(SdHostStandardController::GetBusPower());
        }
        return SdHostStandardController::IssueCommand(pCommand, pTransferData, pOutTransferredNumBlocks);
    }
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    Result Sdmmc1Controller::IssueStopTransmissionCommand(uint32_t* pOutResponse, bool isForced) NN_NOEXCEPT
    {
        if (SdHostStandardController::IsPowerSavingEnable())
        {
            // 省電力中はコマンド発行前に Calibration を実行
            SdmmcController::CalibrateDriveStrength(SdHostStandardController::GetBusPower());
        }
        return SdHostStandardController::IssueStopTransmissionCommand(pOutResponse, isForced);
    }
#endif

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
