﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_MmcDeviceAccessor.h"
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif

namespace nn { namespace sdmmc1 {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_MmcDevice.m_DeviceMutex)
#else
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD
#endif

namespace
{
    // OCR Register ビット定義
    const uint32_t OcrCardPowerUpStatus = 0x1U << 31;
    const uint32_t OcrAccessModeShift = 29;
    const uint32_t OcrAccessModeMask = 0x3U << OcrAccessModeShift;
    const uint32_t OcrAccessModeSectorMode = 0x2U << OcrAccessModeShift;

    // Extended CSD Register DEVICE_TYPE ビット定義
    const uint8_t DeviceTypeHs400__200MHz_1_8V = 0x1U << 6;
    const uint8_t DeviceTypeHs200__200MHz_1_8V = 0x1U << 4;
    const uint8_t DeviceTypeHighSpeed_52MHz = 0x1U << 1;

    bool IsToshibaMmc(uint8_t* pCid) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pCid);
        // pCid: CID[127:8] が格納されている
        // pCid[14]: CID[127:120]
        // CID[127:120]: MID
        uint8_t mid = pCid[14];
        NN_DETAIL_SDMMC_DEBUG_LOG("MID of CID: 0x%02X\n", mid);
        if (mid == 0x11)    // 東芝製
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("Toshiba eMMC\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("NOT Toshiba eMMC\n");
            return false;
        }
    }

    bool IsLessThanStandardVersion(uint8_t* pCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pCsd);
        // pCsd: CSD[127:8] が格納されている
        // pCsd[14]: CSD[127:120]
        // CSD[125:122]: SPEC_VERS
        uint8_t specVers = (pCsd[14] >> 2) & 0xF;
        NN_DETAIL_SDMMC_DEBUG_LOG("SPEC_VERS of CSD: %d\n", specVers);
        if (specVers < 4)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SPEC_VERS indicates non standerd version\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SPEC_VERS indicates version 4.0 or higher\n");
            return false;
        }
    }

    bool IsBkopsAutoEnable(uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[163] = BKOPS_EN
        uint8_t bkopsEn = pExtendedCsd[163];
        NN_DETAIL_SDMMC_DEBUG_LOG("BKOPS_EN of Extended CSD: 0x%02X\n", bkopsEn);
        if ((bkopsEn & (0x1U << 1)) != 0)   // AUTO_EN bit
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("AUTO_EN is enabled\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("AUTO_EN is disabled\n");
            return false;
        }
    }

    uint8_t GetMmcDeviceType(uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[196] = DEVICE_TYPE
        uint8_t mmcDevcieType = pExtendedCsd[196];
        NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE of Extended CSD: 0x%02X\n", mmcDevcieType);
        return mmcDevcieType;
    }

    bool IsSupportedHighSpeedMode(uint8_t mmcDevcieType) NN_NOEXCEPT
    {
        if (mmcDevcieType & DeviceTypeHighSpeed_52MHz)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that High-Speed 52MHz is supported\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that High-Speed 52MHz is NOT supported\n");
            return false;
        }
    }

    bool IsSupportedHs200Mode(uint8_t mmcDevcieType) NN_NOEXCEPT
    {
        if (mmcDevcieType & DeviceTypeHs200__200MHz_1_8V)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS200 - 1.8V is supported\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS200 - 1.8V is NOT supported\n");
            return false;
        }
    }

    bool IsSupportedHs400Mode(uint8_t mmcDevcieType) NN_NOEXCEPT
    {
        if (mmcDevcieType & DeviceTypeHs400__200MHz_1_8V)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS400 - 1.8V is supported\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS400 - 1.8V is NOT supported\n");
            return false;
        }
    }

    uint32_t GetMemoryCapacity(uint32_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // pExtendedCsd[53]: Extended CSD[215:212]
        // Extended CSD[215:212]: SEC_COUNT
        return pExtendedCsd[53];
    }

    uint32_t GetBootPartitionCapacity(uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[226]: BOOT_SIZE_MULT
        return pExtendedCsd[226] * ((128 * 1024) / SectorSize);
    }

    Result GetCurrentSpeedMode(SpeedMode* pOutSpeedMode, uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[185]: HS_TIMING,
        switch (pExtendedCsd[185] & 0xF)    // Bit3:0 Timing Interface
        {
        case 0x0:   // Selecting backwards compatibility interfae timing
            *pOutSpeedMode = SpeedMode_MmcLegacySpeed;
            return ResultSuccess();
        case 0x1:   // High Speed
            *pOutSpeedMode = SpeedMode_MmcHighSpeed;
            return ResultSuccess();
        case 0x2:   // HS200
            *pOutSpeedMode = SpeedMode_MmcHs200;
            return ResultSuccess();
        case 0x3:   // HS400
            *pOutSpeedMode = SpeedMode_MmcHs400;
            return ResultSuccess();
        default:
            NN_DETAIL_SDMMC_DEBUG_LOG("Unexpected Extended CSD[185]: 0x%02X\n", pExtendedCsd[185]);
            return ResultUnexpectedMmcExtendedCsdValue();
        }
    }
}

void MmcDevice::SetOcrAndHighCapacity(uint32_t ocr) NN_NOEXCEPT
{
    BaseDevice::SetOcr(ocr);
    if ((ocr & OcrAccessModeMask) == OcrAccessModeSectorMode)
    {
        BaseDevice::SetHighCapacity(true);
    }
    else
    {
        BaseDevice::SetHighCapacity(false);
    }
}

Result MmcDeviceAccessor::IssueCommandSendOpCond(uint32_t* pOutOcr, BusPower busPower) const NN_NOEXCEPT
{
    // argument[31]    reserved
    // argument[30:29] Host support access mode, 10b : sector mode
    // argument[28:24] reserved
    // argument[23:15] 111 1111b : 2.7-3.6V
    // argument[14: 8] 1 1111 1111b : 2.0-2,6V
    // argument[7]     1b : 1.70-1.95V
    // argument[6:0]   reserved
    uint32_t argument = OcrAccessModeSectorMode;
    switch (busPower)
    {
    case BusPower_1_8V:
        argument = argument | (0x1U << 7);
        break;
    case BusPower_3_3V:
        argument = argument | (0x7FU << 15);
        break;
    case BusPower_Off:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    ResponseType responseType = ResponseType_R3;
    Command command(1, argument, responseType, false);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    Result result = pHostController->IssueCommand(&command);
    if (result.IsFailure())
    {
        return result;
    }

    pHostController->GetLastResponse(pOutOcr, sizeof(uint32_t), responseType);

    return ResultSuccess();
}

Result MmcDeviceAccessor::IssueCommandSetRelativeAddr() const NN_NOEXCEPT
{
    uint32_t rca = m_MmcDevice.GetRca();
    NN_ABORT_UNLESS(rca > 0);

    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    // R1, このコマンドで状態遷移するため、state チェックは除外する
    uint32_t argument = static_cast<uint32_t>(rca) << 16;
    return BaseDeviceAccessor::IssueCommandAndCheckR1(3, argument, false, DeviceState_Unknown);
}

Result MmcDeviceAccessor::IssueCommandSwitch(CommandSwitch commandSwitch) const NN_NOEXCEPT
{
    // argument[31:26] Set to 0
    // argument[25:24] Access
    // argument[23:16] Index of Extended CSD
    // argument[15: 8] Value
    // argument[ 7: 3] Set to 0
    // argument[ 2: 0] Cmd Set
    uint32_t argument;
    switch (commandSwitch)
    {
    case CommandSwitch_SetBitsBkopsEnAutoEn:
        argument = 0x01A30200;
        break;
    case CommandSwitch_WriteBusWidth1Bit:
        argument = 0x03B70000;
        break;
    case CommandSwitch_WriteBusWidth4Bit:
        argument = 0x03B70100;
        break;
    case CommandSwitch_WriteBusWidth8Bit:
        argument = 0x03B70200;
        break;
    case CommandSwitch_WriteBusWidth8BitDdr:
        argument = 0x03B70600;
        break;
    case CommandSwitch_WriteHsTimingLegacySpeed:
        argument = 0x03B90000;
        break;
    case CommandSwitch_WriteHsTimingHighSpeed:
        argument = 0x03B90100;
        break;
    case CommandSwitch_WriteHsTimingHs200:
        argument = 0x03B90200;
        break;
    case CommandSwitch_WriteHsTimingHs400:
        argument = 0x03B90300;
        break;
    case CommandSwitch_WritePartitionAccessDefault:
        argument = 0x03B30000;
        break;
    case CommandSwitch_WritePartitionAccessRwBootPartition1:
        argument = 0x03B30100;
        break;
    case CommandSwitch_WritePartitionAccessRwBootPartition2:
        argument = 0x03B30200;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return BaseDeviceAccessor::IssueCommandAndCheckR1(6, argument, true, DeviceState_Unknown);
}

Result MmcDeviceAccessor::IssueCommandSendExtCsd(void* pOutExtendedCsdBuffer, size_t extendedCsdBufferSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutExtendedCsdBuffer);
    NN_ABORT_UNLESS(extendedCsdBufferSize >= MmcExtendedCsdSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // argument[31:0] stuff bits
    ResponseType responseType = ResponseType_R1;
    Command command(8, 0, responseType, false);
    TransferData transferData(pOutExtendedCsdBuffer, MmcExtendedCsdSize, 1, TransferDirection_ReadFromDevice);
    Result result = pHostController->IssueCommand(&command, &transferData);
    if (result.IsFailure())
    {
        return result;
    }

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    result = m_MmcDevice.CheckDeviceStatus(response);
    if (result.IsFailure())
    {
        return result;
    }

    NN_DETAIL_SDMMC_DEBUG_LOG("Extended CSD:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutExtendedCsdBuffer, MmcExtendedCsdSize);

    return ResultSuccess();
}

Result MmcDeviceAccessor::ChangeToReadyState(BusPower busPower) NN_NOEXCEPT
{
    // ready state へ遷移、High Capacity 有無取得
    ManualTimer timer(1500);  // 規格上は 1 秒だが、余裕を見て 1.5 秒とする
    while (true)
    {
        uint32_t ocr;
        Result result = IssueCommandSendOpCond(&ocr, busPower);
        if (result.IsFailure())
        {
            return result;
        }
        if ((ocr & OcrCardPowerUpStatus) != 0)
        {
            m_MmcDevice.SetOcrAndHighCapacity(ocr);
            break;
        }
        if (!(timer.Update()))
        {
            // power up routine が完了しないまま、タイムアウトを過ぎている
            return ResultMmcInitializationSwTimeout();
        }
        // 最大 1 秒待つ処理に対して 1ms 間隔でチェック
        WaitMicroseconds(1000);
    }

    return ResultSuccess();
}

Result MmcDeviceAccessor::ExtendBusWidth() NN_NOEXCEPT
{
    BusWidth busWidth = BusWidth_1Bit;
    CommandSwitch commandSwitch;
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    if (pHostController->IsSupportedBusWidth(BusWidth_8Bit))
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Set 8bit Bus\n");
        busWidth = BusWidth_8Bit;
        commandSwitch = CommandSwitch_WriteBusWidth8Bit;
    }
    else if (pHostController->IsSupportedBusWidth(BusWidth_4Bit))
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Set 4bit Bus\n");
        busWidth = BusWidth_4Bit;
        commandSwitch = CommandSwitch_WriteBusWidth4Bit;
    }
    else
    {
        // 1bit バス幅のまま
        NN_DETAIL_SDMMC_DEBUG_LOG("Bus Witdh is 1 bit (Host Controller supports only 1 bit)\n");
        return ResultSuccess();
    }
    Result result = IssueCommandSwitch(commandSwitch);
    if (result.IsFailure())
    {
        return result;
    }
    result = BaseDeviceAccessor::IssueCommandSendStatus();
    if (result.IsFailure())
    {
        return result;
    }
    pHostController->SetBusWidth(busWidth);
    return ResultSuccess();
}

Result MmcDeviceAccessor::EnableBkopsAuto() NN_NOEXCEPT
{
    Result result = IssueCommandSwitch(CommandSwitch_SetBitsBkopsEnAutoEn);
    if (result.IsFailure())
    {
        return result;
    }
    result = BaseDeviceAccessor::IssueCommandSendStatus();
    if (result.IsFailure())
    {
        return result;
    }
    return ResultSuccess();
}

Result MmcDeviceAccessor::ChangeToHighSpeed(bool isClockup) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEBUG_LOG("Set MMC to High Speed\n");
    Result result = IssueCommandSwitch(CommandSwitch_WriteHsTimingHighSpeed);
    if (result.IsFailure())
    {
        return result;
    }
    if (isClockup)
    {
        // SWITCH_ERROR が起こっていないか確認してから、クロックを上げる
    result = BaseDeviceAccessor::IssueCommandSendStatus();
    if (result.IsFailure())
    {
        return result;
    }
    }
    NN_DETAIL_SDMMC_DEBUG_LOG("Set HostController to MMC High Speed\n");
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    result = pHostController->SetSpeedMode(SpeedMode_MmcHighSpeed);
    if (result.IsFailure())
    {
        return result;
    }
    if (!isClockup)
    {
        // クロックを下げてから、SWITCH_ERROR が起こっていないか確認する
        result = BaseDeviceAccessor::IssueCommandSendStatus();
        if (result.IsFailure())
        {
            return result;
        }
    }
    return ResultSuccess();
}

Result MmcDeviceAccessor::ChangeToHs200() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEBUG_LOG("Set MMC to HS200\n");
    Result result = IssueCommandSwitch(CommandSwitch_WriteHsTimingHs200);
    if (result.IsFailure())
    {
        return result;
    }
    NN_DETAIL_SDMMC_DEBUG_LOG("Set HostController to HS200\n");
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    result = pHostController->SetSpeedMode(SpeedMode_MmcHs200);
    if (result.IsFailure())
    {
        return result;
    }
    result = pHostController->Tuning(SpeedMode_MmcHs200, 21);
    if (result.IsFailure())
    {
        return result;
    }
    // MMC とのコマンド通信確認
    // (レスポンス取得にも Tuning が必要なため、SWITCH_ERROR をチェックするタイミングはない)
    result = BaseDeviceAccessor::IssueCommandSendStatus();
    if (result.IsFailure())
    {
        return result;
    }
    return ResultSuccess();
}

Result MmcDeviceAccessor::ChangeToHs400() NN_NOEXCEPT
{
    // HS400 のレスポンス受信のために Tuning を行う
    Result result = ChangeToHs200();
    if (result.IsFailure())
    {
        return result;
    }
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->SaveTuningStatusForHs400();

    // HS400 に設定するコマンド通信のために High Speed に下げる
    result = ChangeToHighSpeed(false);
    if (result.IsFailure())
    {
        return result;
    }

    NN_DETAIL_SDMMC_DEBUG_LOG("Set MMC to HS400\n");
    result = IssueCommandSwitch(CommandSwitch_WriteBusWidth8BitDdr);
    if (result.IsFailure())
    {
        return result;
    }
    result = IssueCommandSwitch(CommandSwitch_WriteHsTimingHs400);
    if (result.IsFailure())
    {
        return result;
    }
    NN_DETAIL_SDMMC_DEBUG_LOG("Set HostController to HS400\n");
    result = pHostController->SetSpeedMode(SpeedMode_MmcHs400);
    if (result.IsFailure())
    {
        return result;
    }
    // レスポンス取得タイミングを合わせてから、SWITCH_ERROR が起こっていないか確認する
    result = BaseDeviceAccessor::IssueCommandSendStatus();
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

Result MmcDeviceAccessor::ExtendBusSpeed(uint8_t mmcDevcieType, SpeedMode maxSpeedMode) NN_NOEXCEPT
{
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    if (pHostController->IsSupportedTuning()
        && (pHostController->GetBusPower() == BusPower_1_8V)
        && (pHostController->GetBusWidth() != BusWidth_1Bit))
    {
        if (IsSupportedHs400Mode(mmcDevcieType) && (maxSpeedMode == SpeedMode_MmcHs400))
        {
            return ChangeToHs400();
        }
        else if (IsSupportedHs200Mode(mmcDevcieType) && ((maxSpeedMode == SpeedMode_MmcHs400) || (maxSpeedMode == SpeedMode_MmcHs200)))
        {
            return ChangeToHs200();
        }
        else
        {
            // 以下で High Speed にトライする
        }
    }

    if (IsSupportedHighSpeedMode(mmcDevcieType))
    {
        // Default Speed から High Speed に上げる
        return ChangeToHighSpeed(true);
    }

    // Default Speed のまま
    NN_DETAIL_SDMMC_DEBUG_LOG("Speed Mode is Default (MMC doesn't support High Speed)\n");
    return ResultSuccess();
}

Result MmcDeviceAccessor::StartupMmcDevice(SpeedMode maxSpeedMode, void* pMmcWorkBuffer, size_t mmcWorkBufferSize) NN_NOEXCEPT
{
    // HostController 起動
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    BusPower busPower = BusPower_1_8V;  // 1.8V 優先
    if (!(pHostController->IsSupportedBusPower(BusPower_1_8V)))
    {
        busPower = BusPower_3_3V;
    }
    Result result = pHostController->Startup(busPower, BusWidth_1Bit, SpeedMode_Identification, false);
    if (result.IsFailure())
    {
        return result;
    }

    // 1ms + 74 clock cycles 待ち
    WaitMicroseconds(1000);
    WaitClocks(74, pHostController->GetDeviceClockFrequencyKHz());

    // idle state へ遷移
    result = BaseDeviceAccessor::IssueCommandGoIdleState();
    if (result.IsFailure())
    {
        return result;
    }

    // ready state へ遷移、High Capacity 有無取得
    result = ChangeToReadyState(busPower);
    if (result.IsFailure())
    {
        return result;
    }

    // ident state へ遷移、CID 取得
    result = BaseDeviceAccessor::IssueCommandAllSendCid(pMmcWorkBuffer, mmcWorkBufferSize);
    if (result.IsFailure())
    {
        return result;
    }
    m_MmcDevice.SetCid(pMmcWorkBuffer, mmcWorkBufferSize);
    bool isToshibaMmc = IsToshibaMmc(reinterpret_cast<uint8_t*>(pMmcWorkBuffer));

    // RCA 設定、stby state へ遷移
    result = IssueCommandSetRelativeAddr();
    if (result.IsFailure())
    {
        return result;
    }

    // CSD 取得
    result = BaseDeviceAccessor::IssueCommandSendCsd(pMmcWorkBuffer, mmcWorkBufferSize);
    if (result.IsFailure())
    {
        return result;
    }
    m_MmcDevice.SetCsd(pMmcWorkBuffer, mmcWorkBufferSize);
    bool isLessThanStandardVersion = IsLessThanStandardVersion(reinterpret_cast<uint8_t*>(pMmcWorkBuffer));

    // Legacy Speed にクロックアップ
    result = pHostController->SetSpeedMode(SpeedMode_MmcLegacySpeed);
    if (result.IsFailure())
    {
        return result;
    }

    // tran state へ遷移
    result = BaseDeviceAccessor::IssueCommandSelectCard();
    if (result.IsFailure())
    {
        return result;
    }

    // ブロックサイズ設定
    result = BaseDeviceAccessor::IssueCommandSetBlockLenToSectorSize();
    if (result.IsFailure())
    {
        return result;
    }

    if (isLessThanStandardVersion)
    {
        // 1bit, Default Speed のまま終了
        NN_DETAIL_SDMMC_DEBUG_LOG("1bit, Default Speed (MMC is less than Standard Version)\n");

        result = m_MmcDevice.SetLegacyMemoryCapacity();
        if (result.IsFailure())
        {
            return result;
        }

        m_MmcDevice.SetActive();
        return ResultSuccess();
    }

    // バス幅拡張
    result = ExtendBusWidth();
    if (result.IsFailure())
    {
        return result;
    }

    // Extended CSD 取得
    result = IssueCommandSendExtCsd(pMmcWorkBuffer, mmcWorkBufferSize);
    if (result.IsFailure())
    {
        return result;
    }
    NN_ABORT_UNLESS((reinterpret_cast<uintptr_t>(pMmcWorkBuffer) % sizeof(uint32_t)) == 0);
    m_MmcDevice.SetMemoryCapacity(::nn::sdmmc1::detail::GetMemoryCapacity(reinterpret_cast<uint32_t*>(pMmcWorkBuffer)));
    if (isToshibaMmc)
    {
        if (!IsBkopsAutoEnable(reinterpret_cast<uint8_t*>(pMmcWorkBuffer)))
        {
            result = EnableBkopsAuto();
            if (result.IsFailure())
            {
                NN_DETAIL_SDMMC_DEBUG_LOG("EnableBkopsAuto() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                // エラーにはせず、以降の処理を続ける
            }
        }
    }
    uint8_t mmcDevcieType = GetMmcDeviceType(reinterpret_cast<uint8_t*>(pMmcWorkBuffer));

    // バススピード拡張
    result = ExtendBusSpeed(mmcDevcieType, maxSpeedMode);
    if (result.IsFailure())
    {
        return result;
    }

    // 以降、クロック供給を抑える
    pHostController->SetPowerSaving(true);

    return ResultSuccess();
}

void MmcDeviceAccessor::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;

    if (m_IsInitialized)
    {
        return;
    }

    BaseDeviceAccessor::SetDevice(&m_MmcDevice);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Initialize();

    m_IsInitialized = true;
}

void MmcDeviceAccessor::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;

    if (!m_IsInitialized)
    {
        return;
    }
    m_IsInitialized = false;

    BaseDeviceAccessor::Deactivate();

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Finalize();
}

Result MmcDeviceAccessor::ActivateMmc(void* pMmcWorkBuffer, size_t mmcWorkBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;

    if (m_MmcDevice.IsActive())
    {
        // Activate 済みならば何もしない
        return ResultSuccess();
    }

    #if (defined(NN_DETAIL_SDMMC_MMC_HS400_ENABLE))
        SpeedMode maxSpeedMode = SpeedMode_MmcHs400;
    #elif (defined(NN_DETAIL_SDMMC_MMC_HS200_ENABLE))
        SpeedMode maxSpeedMode = SpeedMode_MmcHs200;
    #else
        SpeedMode maxSpeedMode = SpeedMode_MmcHighSpeed;
    #endif
    Result result = StartupMmcDevice(maxSpeedMode, pMmcWorkBuffer, mmcWorkBufferSize);
    if (result.IsFailure())
    {
        // 一旦 HostController を落とす
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        pHostController->Shutdown();

        // リトライ
        result = StartupMmcDevice(SpeedMode_MmcHighSpeed, pMmcWorkBuffer, mmcWorkBufferSize);
        if (result.IsFailure())
        {
            // HostController を落としておく
            pHostController->Shutdown();
            return result;
        }
    }

    m_MmcDevice.SetActive();
    return ResultSuccess();
}

Result MmcDeviceAccessor::GetMmcSpeedMode(SpeedMode* pOutSpeedMode, void* pMmcWorkBuffer, size_t mmcWorkBufferSize) const NN_NOEXCEPT
{
    Result result = GetMmcExtendedCsd(pMmcWorkBuffer, mmcWorkBufferSize);
    if (result.IsFailure())
    {
        return result;
    }
    NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);
    result = GetCurrentSpeedMode(pOutSpeedMode, reinterpret_cast<uint8_t*>(pMmcWorkBuffer));
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

Result MmcDeviceAccessor::SelectMmcPartition(MmcPartition mmcPartition) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_MmcDevice.CheckAccessible());

    CommandSwitch commandSwitch;
    switch (mmcPartition)
    {
    case MmcPartition_UserData:
        commandSwitch = CommandSwitch_WritePartitionAccessDefault;
        break;
    case MmcPartition_BootPartition1:
        commandSwitch = CommandSwitch_WritePartitionAccessRwBootPartition1;
        break;
    case MmcPartition_BootPartition2:
        commandSwitch = CommandSwitch_WritePartitionAccessRwBootPartition2;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    Result result = IssueCommandSwitch(commandSwitch);
    if (result.IsFailure())
    {
        return result;
    }
    result = BaseDeviceAccessor::IssueCommandSendStatus();
    if (result.IsFailure())
    {
        return result;
    }
    return ResultSuccess();
}

Result MmcDeviceAccessor::GetMmcBootPartitionCapacity(uint32_t* pOutNumSectors, void* pMmcWorkBuffer, size_t mmcWorkBufferSize) const NN_NOEXCEPT
{
    Result result = GetMmcExtendedCsd(pMmcWorkBuffer, mmcWorkBufferSize);
    if (result.IsFailure())
    {
        return result;
    }
    NN_ABORT_UNLESS_NOT_NULL(pOutNumSectors);
    *pOutNumSectors = GetBootPartitionCapacity(reinterpret_cast<uint8_t*>(pMmcWorkBuffer));
    return ResultSuccess();
}

Result MmcDeviceAccessor::GetMmcExtendedCsd(void* pOutExtendedCsdBuffer, size_t extendedCsdBufferSize) const NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_MmcDevice.CheckAccessible());

    uint8_t csd[DeviceCsdSize];
    m_MmcDevice.GetCsd(csd, sizeof(csd));
    bool isLessThanStandardVersion = IsLessThanStandardVersion(csd);
    if (isLessThanStandardVersion)
    {
        // Extended CSD を保持しない
        return ResultMmcNotSupportExtendedCsd();
    }

    Result result = IssueCommandSendExtCsd(pOutExtendedCsdBuffer, extendedCsdBufferSize);
    if (result.IsFailure())
    {
        return result;
    }
    return ResultSuccess();
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
