﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_DeviceDetector.h"
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc1 {
namespace detail {

bool DeviceDetector::IsInserted(bool isNeedToCheckReady) NN_NOEXCEPT
{
    if (isNeedToCheckReady)
    {
        // 最初の gpio の設定が終わるまで待つ
        nn::os::WaitEvent(&m_ReadyDeviceStatusEvent);
    }

    nn::gpio::GpioValue currentGpioValue = nn::gpio::GetValue(&m_GpioPadSession);
    if (currentGpioValue == m_GpioValueInserted)
    {
        return true;
    }
    else
    {
        return false;
    }
}

void DeviceDetector::DetectorThread() NN_NOEXCEPT
{
    // gpioEvent が挿抜でシグナルされるよう設定
    nn::gpio::Initialize();
    nn::gpio::OpenSession(&m_GpioPadSession, m_GpioPadName);
    nn::gpio::SetDirection(&m_GpioPadSession, nn::gpio::Direction_Input);
    nn::gpio::SetDebounceTime(&m_GpioPadSession, m_GpioDebounceMicroSeconds);
    nn::gpio::SetDebounceEnabled(&m_GpioPadSession, true);
    nn::gpio::SetInterruptMode(&m_GpioPadSession, nn::gpio::InterruptMode_AnyEdge);
    nn::os::SystemEventType gpioEvent;
    Result result = nn::gpio::BindInterrupt(&gpioEvent, &m_GpioPadSession);
    NN_ABORT_UNLESS(result.IsSuccess());

    // waiter に gpioEvent とスレッド終了イベントを登録
    nn::os::MultiWaitType waiter;
    nn::os::MultiWaitHolderType gpioEventHolder;
    nn::os::MultiWaitHolderType detectorThreadEndEventHolder;
    nn::os::InitializeMultiWait(&waiter);
    nn::os::InitializeMultiWaitHolder(&gpioEventHolder, &gpioEvent);
    nn::os::LinkMultiWaitHolder(&waiter, &gpioEventHolder);
    nn::os::InitializeMultiWaitHolder(&detectorThreadEndEventHolder, &m_DetectorThreadEndEvent);
    nn::os::LinkMultiWaitHolder(&waiter, &detectorThreadEndEventHolder);

    // 最初の状態を確定
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_GpioDebounceMicroSeconds));    // 最初の状態が定まる時間待つ
    bool isCurrentInserted = IsInserted(false);
    NN_DETAIL_SDMMC_DETECTOR_LOG("%s\n", (isCurrentInserted ? "Inserted" : "Removed"));

    // 以降は状態取得可能
    nn::os::SignalEvent(&m_ReadyDeviceStatusEvent);

    // 割り込み許可
    nn::gpio::SetInterruptEnable(&m_GpioPadSession, true);

    while (true)
    {
        nn::os::MultiWaitHolderType* signaledHolder = nn::os::WaitAny(&waiter);
        if (signaledHolder == &detectorThreadEndEventHolder)
        {
            break;
        }

        if (m_DeviceDetectionEventCallback != nullptr)
        {
            // 上位レイヤに挿抜イベントがあったことを通知
            m_DeviceDetectionEventCallback(m_pDeviceDetectionEventCallbackParameter);
        }

        // イベントのクリア
        nn::os::ClearSystemEvent(&gpioEvent);

        // 割り込みステータスをクリア
        nn::gpio::ClearInterruptStatus(&m_GpioPadSession);

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&m_GpioPadSession, true);

        bool isLastInserted = isCurrentInserted;
        isCurrentInserted = IsInserted(false);
        if ((!isLastInserted) && (!isCurrentInserted))
        {
            // 抜 -> ... -> 抜 の場合は何もしない
            NN_DETAIL_SDMMC_DETECTOR_LOG("Insert (LOST)\n");
            NN_DETAIL_SDMMC_DETECTOR_LOG("Remove (ignore)\n");
        }
        else if ((!isLastInserted) && isCurrentInserted)
        {
            // 抜 -> ... -> 挿 の場合
            NN_DETAIL_SDMMC_DETECTOR_LOG("Insert\n");
            if (m_InsertedCallback != nullptr)
            {
                m_InsertedCallback(m_pInsertedCallbackParameter);
            }
        }
        else if (isLastInserted && (!isCurrentInserted))
        {
            // 挿 -> ... -> 抜 の場合
            NN_DETAIL_SDMMC_DETECTOR_LOG("Remove\n");
            if (m_RemovedCallback != nullptr)
            {
                m_RemovedCallback(m_pRemovedCallbackParameter);
            }
        }
        else    // (isLastInserted && m_IsInserted)
        {
            // 挿 -> ... -> 挿 の場合
            NN_DETAIL_SDMMC_DETECTOR_LOG("Remove (LOST)\n");
            if (m_RemovedCallback != nullptr)
            {
                m_RemovedCallback(m_pRemovedCallbackParameter);
            }
            NN_DETAIL_SDMMC_DETECTOR_LOG("Insert\n");
            if (m_InsertedCallback != nullptr)
            {
                m_InsertedCallback(m_pInsertedCallbackParameter);
            }
        }
    }

    // waiter の後片付け
    nn::os::UnlinkMultiWaitHolder(&detectorThreadEndEventHolder);
    nn::os::FinalizeMultiWaitHolder(&detectorThreadEndEventHolder);
    nn::os::UnlinkMultiWaitHolder(&gpioEventHolder);
    nn::os::FinalizeMultiWaitHolder(&gpioEventHolder);
    nn::os::FinalizeMultiWait(&waiter);

    // gpio の後片付け
    nn::gpio::UnbindInterrupt(&m_GpioPadSession);
    nn::gpio::CloseSession(&m_GpioPadSession);
    nn::gpio::Finalize();
}

void DeviceDetector::Initialize(CallbackInfos* pCallbackInfos) NN_NOEXCEPT
{
    m_InsertedCallback = pCallbackInfos->insertedCallback;
    m_pInsertedCallbackParameter = pCallbackInfos->pInsertedCallbackParameter;
    m_RemovedCallback = pCallbackInfos->removedCallback;
    m_pRemovedCallbackParameter = pCallbackInfos->pRemovedCallbackParameter;

    // 最初の挿抜状態が確定するまでロック
    nn::os::InitializeEvent(&m_ReadyDeviceStatusEvent, false, nn::os::EventClearMode_ManualClear);

    // 挿抜監視スレッドを起動
    nn::os::InitializeEvent(&m_DetectorThreadEndEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::CreateThread(&m_DetectorThread, DetectorThreadEntry, this,
        m_DetectorThreadStack, sizeof(m_DetectorThreadStack), DetectorThreadPriority);
    nn::os::StartThread(&m_DetectorThread);
}

void DeviceDetector::Finalize() NN_NOEXCEPT
{
    // 挿抜監視スレッドを停止
    nn::os::SignalEvent(&m_DetectorThreadEndEvent);
    nn::os::WaitThread(&m_DetectorThread);
    nn::os::DestroyThread(&m_DetectorThread);
    nn::os::FinalizeEvent(&m_DetectorThreadEndEvent);

    nn::os::FinalizeEvent(&m_ReadyDeviceStatusEvent);
}

bool DeviceDetector::IsInserted() NN_NOEXCEPT
{
    bool isInserted = IsInserted(true);
    NN_DETAIL_SDMMC_DETECTOR_LOG("return %s\n", (isInserted ? "Inserted" : "Removed"));
    return isInserted;
}

void DeviceDetector::RegisterDetectionEventCallback(DeviceDetectionEventCallback callback, void* pParameter) NN_NOEXCEPT
{
    m_pDeviceDetectionEventCallbackParameter = pParameter;
    m_DeviceDetectionEventCallback = callback;
}

void DeviceDetector::UnregisterDetectionEventCallback() NN_NOEXCEPT
{
    m_DeviceDetectionEventCallback = nullptr;
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
