﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   製品環境では利用できない開発目的の I2C ドライバのバスアクセス機能 API を宣言するヘッダ
 */

#pragma once

#include <nn/nn_Common.h>
#include "../i2c_Type.h"
#include "../i2c_Result.h"
#include "i2c_Bus.h"

namespace nn {
namespace i2c {
namespace driver {

/**
 * @brief 指定したデバイスがボード上に存在するかどうかを返します。
 *
 * @param[in]   busIdx                         バス番号
 * @param[in]   slaveAddress                   スレーブアドレス
 * @param[in]   addressingMode                 アドレスモード
 * @param[in]   speedMode                      動作速度
 * @return      指定したデバイスが存在するかどうかを返します。
 * @retval      true        指定したデバイスが存在します。
 * @retval      false       指定したデバイスは存在しないか、または使用できません。
 * @pre         ライブラリは初期化済の状態である必要があります。
 *              引数に指定する speedMode は製品環境相当の固定の値を指定してください。
 *              同一の busIdx に対して異なる speedMode を指定した呼出しはサポートされません。
 * @details    　HasDevice() とは異なり、種々のパラメータを柔軟に設定できますが、
 *              開発目的での利用を想定しており、本関数は製品環境では利用できません。
 *              指定した設定情報を元にデバイスとの通信を行い、
 *              その結果からデバイスの有無を判定します。
 *              製品環境では、操作対象のデバイス向けの I2cDevice 識別子の割り当てを行ったうえで HasDevice() をご利用ください。
 *              なお、 I2cDevice 識別子の割り当ての要望は担当窓口までお問い合わせください。
 */
bool HasDeviceForDev( int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode ) NN_NOEXCEPT;

/**
 * @brief 指定したスレーブデバイスとのセッションをオープンします。すなわちスレーブデバイスとの通信を可能な状態にします。
 *
 * @param[out]  pOutSession                    セッション用パラメータ
 * @param[in]   busIdx                         バス番号
 * @param[in]   slaveAddress                   スレーブアドレス
 * @param[in]   addressingMode                 アドレスモード
 * @param[in]   speedMode                      動作速度
 * @pre         ライブラリは初期化済の状態である必要があります。
 *              引数に指定したパラメータがボード上でサポートされている必要があります。
 * @post        指定した設定パラメータが反映され、デバイスが使用可能な状態になります。
 * @details     OpenSession() とは異なり、種々のパラメータを柔軟に設定できますが、
 *              開発目的での利用を想定しており、本関数は製品環境では利用できません。
 *              セッションのオープンに成功した場合、有効な　I2cSession　が設定されます。
 *              この関数は指定したスレーブアドレスのデバイスの接続状態の確認は行いません。
 *              既に利用中のバスに対して異なる動作速度で本関数が呼ばれた場合は NN_ABORT を呼び出します。
 *              製品環境では、操作対象のデバイス向けの I2cDevice 識別子の割り当てを行ったうえで OpenSession() をご利用ください。
 *              なお、 I2cDevice 識別子の割り当ての要望は担当窓口までお問い合わせください。
 */
void OpenSessionForDev(I2cSession* pOutSession, int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode ) NN_NOEXCEPT;

} // driver
} // i2c
} // nn


