﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   I2C ドライバのバスアクセサクラスの実装部(Tegra ファミリー向け)
 * @details Jetson TK1/2 で使用可能な I2Cバスのみを対象とします。
 *          参考文献：
 *            - [Ref1] Tegra K1 Technical Reference Manual, Revision v02p, Jul 15 2014
 */

#include <climits>
#include <mutex>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include <nn/os.h>

#include "../../../include/nn/i2c/driver/i2c.h"
#include <nn/result/result_HandlingUtility.h>

#include "i2c_BusAccessor.tegra.h"

#include "../i2c_Util.h"
#include "../i2c_DdUtil.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

/**
 * @brief バスのオープン処理を行う関数
 */
void BusAccessor::Open(int busIdx, SpeedMode speedMode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lockUserCount( m_UserCountMutex );
    NN_SDK_ASSERT( m_UserCount < INT_MAX ); // 値域チェック
    m_UserCount++;

    NN_DETAIL_I2C_LOG("%s(%d): BusAccessor::Open() is called at busIdx=%d, userCount=%d.\n", __FUNCTION__, __LINE__, busIdx, m_UserCount );

    if( m_UserCount > 1 )
    {
        NN_DETAIL_I2C_LOG("%s(%d): Bus Already Opened.\n", __FUNCTION__, __LINE__ );

        if( m_SpeedMode != speedMode )
        {
            // 初回 Open 時に設定された SpeedMode と異なる場合は ABORT します
            NN_ABORT("SpeedMode should be %d[bps]\n", m_SpeedMode );
        }
        return;
    }

    // 以下、初回 Open 時のみ実行
    std::lock_guard<nn::os::Mutex> lock( m_RegisterMutex );

    // 仮想アドレスをセットする。
    SetBaseAddress(busIdx);

    // Pinmux の設定
    SetPinmux(busIdx);

    // I2C コントローラのリセット
    ResetController();

    // クロック関連レジスタのセット
    SetClockRegisters(speedMode);

    // Packet mode への初期設定
    SetPacketModeRegisters();

    // FIFO の Flush
    FlushFifos();

    // 現在の設定を保持
    m_SpeedMode = speedMode;

    // DumpRegisters(__LINE__);
}

/**
 * @brief バスのクローズ処理を行う関数
 */
void BusAccessor::Close() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_UserCountMutex );

    NN_SDK_ASSERT( m_UserCount > INT_MIN ); // 値域チェック
    m_UserCount--;

    // ユーザー数が 0 に達していない場合は何も行わない
    if ( m_UserCount > 0 )
    {
        return;
    }

    NN_DETAIL_I2C_LOG("%s(%d) Bus closed. userCount=%d\n", __FUNCTION__, __LINE__, m_UserCount );
}

/**
 * @brief バスの状態をチェックする関数
 */
bool BusAccessor::CheckBusy() const NN_NOEXCEPT
{
    uint32_t BusWaitIntervalMicroSeconds = m_TimeoutMicroSeconds / m_CheckRetryTimes;
    NN_SDK_ASSERT( BusWaitIntervalMicroSeconds > 0 );

    for ( uint32_t i = 0; i < m_CheckRetryTimes; i++ )
    {
        if ( !IsBusy() )
        {
            return false; // Bus が Free 状態
        }
        if( i == m_CheckRetryTimes - 1 )
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(BusWaitIntervalMicroSeconds));
    }
    return true; // Bus が Busy 状態
}

/**
 * @brief I2Cプロトコル仕様に基づく StartCondition を発行する関数
 */
void BusAccessor::StartCondition() const NN_NOEXCEPT
{
    // Start condition に相当する処理が無いので実装不要。
}

/**
 * @brief I2Cプロトコル仕様に基づく StopCondition を発行する関数
 */
void BusAccessor::StopCondition() const NN_NOEXCEPT
{
    // Stop condition に相当する処理が無いので実装不要
}

/**
 * @brief トランザクション開始時に呼ばれる関数
 */
Result BusAccessor::StartTransaction( Bit16 slaveAddress, AddressingMode addressingMode, Command command ) NN_NOEXCEPT
{
    // トランザクション開始時に必要な処理が無いので実装不要
    return ResultSuccess();
}

void   BusAccessor::WriteHeader( Xfer xfer, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    uint32_t genericHeaderWord0     = 0;
    uint32_t genericHeaderWord1     = 0;
    uint32_t protocolSpecificHeader = 0;
    uint32_t slaveAddr              = ( static_cast<uint32_t>( slaveAddress & 0x7F ) << 1 ) | ( ( xfer == Xfer_Read ) ? 1 : 0 );
    const uint32_t ProtocolI2c      = 1; // [Ref1] 31.2.2 I/O Packet Format

    // [Ref1] Table 133
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_ProtHdrSz::Pos,           I2cPacket::Generic0_ProtHdrSz::Mask );
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_PktId::Pos,               I2cPacket::Generic0_PktId::Mask );
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_ControllerId::Pos,        I2cPacket::Generic0_ControllerId::Mask );
    Modify32 ( &genericHeaderWord0,  ProtocolI2c << I2cPacket::Generic0_Protocol::Pos,  I2cPacket::Generic0_Protocol::Mask );
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_PktType::Pos,             I2cPacket::Generic0_PktType::Mask );

    // [Ref1] Table 134
    Modify32 ( &genericHeaderWord1, static_cast<uint32_t>( dataBytes - 1 ) << I2cPacket::Generic1_PayloadSize::Pos,  I2cPacket::Generic1_PayloadSize::Mask );

    // [Ref1] Table 135
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_RespPktFreq::Pos,              I2cPacket::Transmit_RespPktFreq::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_RespPktEnable::Pos,            I2cPacket::Transmit_RespPktEnable::Mask );
    Modify32 ( &protocolSpecificHeader,
        ( m_SpeedMode == SpeedMode_HighSpeed ) << I2cPacket::Transmit_HsMode::Pos,
        I2cPacket::Transmit_HsMode::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_ContinueOnNack::Pos,           I2cPacket::Transmit_ContinueOnNack::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_SendStartByte::Pos,            I2cPacket::Transmit_SendStartByte::Mask );
    Modify32 ( &protocolSpecificHeader,
        ( ( xfer == Xfer_Read ) ? 1 : 0 ) << I2cPacket::Transmit_ReadWrite::Pos,
        I2cPacket::Transmit_ReadWrite::Mask );
    Modify32 ( &protocolSpecificHeader,
        !( addressingMode == AddressingMode_BitWidth7 ) << I2cPacket::Transmit_AddressMode::Pos,
        I2cPacket::Transmit_AddressMode::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_Ie::Pos,                       I2cPacket::Transmit_Ie::Mask );
    Modify32 ( &protocolSpecificHeader,
        !( inOption & TransactionOption_StopCondition ) << I2cPacket::Transmit_RepeatStartStop::Pos,
        I2cPacket::Transmit_RepeatStartStop::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_ContinueXfer::Pos,             I2cPacket::Transmit_ContinueXfer::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_HsMasterAddr::Pos,             I2cPacket::Transmit_HsMasterAddr::Mask );
    Modify32 ( &protocolSpecificHeader, slaveAddr << I2cPacket::Transmit_SlaveAddr::Pos,        I2cPacket::Transmit_SlaveAddr::Mask );

    // Clear interrupt status register
    Write32  ( &(m_pRegI2c->interruptStatusRegister), 0 );

    FlushFifos();

    // TX FIFO に書き込み
    Write32  ( &(m_pRegI2c->txPacketFifo), genericHeaderWord0 );
    Write32  ( &(m_pRegI2c->txPacketFifo), genericHeaderWord1 );
    Write32  ( &(m_pRegI2c->txPacketFifo), protocolSpecificHeader );
}

/**
 * @brief  送受信の完了を待つ関数
 */
Result BusAccessor::WaitXferCompletion( Xfer xfer ) NN_NOEXCEPT
{
    const int NumberOfTimes = 10;
    int       currentTimes  = 0;

    auto mask = ( xfer == Xfer_Read ) ? I2cRegister::InterruptStatus_RfifoDataReq::Mask : I2cRegister::InterruptStatus_TfifoDataReq::Mask;
    while( !( Read32(&(m_pRegI2c->interruptStatusRegister)) & mask ) )
    {
        if( ++currentTimes >= NumberOfTimes)
        {
            return ResultBusBusy();
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1) );
    }
    return ResultSuccess();
}


/**
 * @brief Send 関数
 */
Result   BusAccessor::Send( const Bit8* pInData,  size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_RegisterMutex );

    size_t      remainingBytes = dataBytes;
    const Bit8* pData          = pInData;

    WriteHeader( Xfer_Write, dataBytes, inOption, slaveAddress, addressingMode );

    while (remainingBytes > 0)
    {
        const size_t MaxSingleDataBytes = 4; // 1 word
        size_t nextBytes = (remainingBytes < MaxSingleDataBytes) ? remainingBytes : MaxSingleDataBytes;

        // データレジスタに書きこむ値のセットアップ
        uint32_t data = 0;
        for(size_t i = 0; i < nextBytes; i++ )
        {
            data |= pData[i] << (8 * i);
        }
        Write32  ( &(m_pRegI2c->txPacketFifo), data );

        remainingBytes -= nextBytes;
        pData += nextBytes;
    }

    NN_RESULT_DO(WaitXferCompletion(Xfer_Write));

    auto result = GetTransactionResult();
    HandleTransactionError(result);
    return result;
}

/**
 * @brief Receive 関数
 */
Result    BusAccessor::Receive( Bit8 *pOutData, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_RegisterMutex );

    size_t     remainingBytes = dataBytes;
    Bit8*      pData          = pOutData;

    WriteHeader( Xfer_Read, dataBytes, inOption, slaveAddress, addressingMode );

    NN_RESULT_DO(WaitXferCompletion(Xfer_Read));

    while (remainingBytes > 0)
    {
        const size_t MaxSingleDataBytes = 4; // 1 word
        size_t nextBytes = (remainingBytes < MaxSingleDataBytes) ? remainingBytes : MaxSingleDataBytes;

        // データレジスタからバッファに読み出し
        uint32_t data  = Read32 ( &(m_pRegI2c->rxFifo) );

        for(size_t i = 0; i < nextBytes; i++ )
        {
            pData[i] = (data >> (8 * i)) & 0xff;
        }
        remainingBytes -= nextBytes;
        pData += nextBytes;
    }
    auto result = GetTransactionResult();
    HandleTransactionError(result);
    return result;
}

/**
 * @brief 仮想アドレスをセットする関数
 */
void BusAccessor::SetBaseAddress(int busIdx) NN_NOEXCEPT
{
    // 使用する I2C コンローラーに対応するレジスタをセットする
    m_pRegI2c = GetI2cReg(busIdx);
    m_RegClkRst.Set(busIdx);
    m_RegPinmux.Set(busIdx);
}

/**
 * @brief リセットレジスタを使ってバスをリセットする。
 *        クロックソースは PLLP_OUT0 (408MHz) 一択。
 */
void BusAccessor::ResetController() const NN_NOEXCEPT
{
    // [5.7.1] Precautions
    // 1. Assert reset signal
    SetBitOn32  (m_RegClkRst.pReset, m_RegClkRst.resetBitMask);

    // 2. Enable clock
    // [Ref1] p.2189 : The clock enable must also be given to the I2C controller, before any of registers are written.
    SetBitOn32  (m_RegClkRst.pClockEnable, m_RegClkRst.clockEnableBitMask);

    // 3. Change Clock divisor
    Modify32  ( m_RegClkRst.pClockSource, (0x04 << 0),   (0xFF) );      // [5.7.68] I2C#_CLK_DIVISOR = 0x04

    // 4. Wait 1 us
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(1) );

    // 5. Change the clock source
    Modify32  ( m_RegClkRst.pClockSource, (0 << 29),     (0x7 << 29) ); // [5.7.68] CLK_M = PLLP_OUT0

    // 6. Wait 2 us
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(2) );

    // 7. De-assert reset signal
    SetBitOff32 (m_RegClkRst.pReset, m_RegClkRst.resetBitMask);
}

/**
 * @brief [Ref1] 31.1 バスクリア用関数。Arbitration lost が起きた時に暗黙的に呼ばれる
 */
void BusAccessor::BusClear() const NN_NOEXCEPT
{
    // [Ref1] 31.1.1 Bus Clear Master Operation
    // 番号は Bus Clear Programming Procedure の記載内容に準拠しています

    // 1
    // 2
    uint32_t  clockPulses = 0x9;
    Write32   ( &(m_pRegI2c->busClearConfig), ( clockPulses << I2cRegister::BusClearConfig_BcSclkThreshold::Pos ) & I2cRegister::BusClearConfig_BcSclkThreshold::Mask);

    // 3
    SetBitOn32( &(m_pRegI2c->busClearConfig), I2cRegister::BusClearConfig_BcStopCond::Mask);

    // 4
    SetBitOn32( &(m_pRegI2c->busClearConfig), I2cRegister::BusClearConfig_BcTerminate::Mask);

    // 5
    SetBitOn32( &(m_pRegI2c->configLoad),     I2cRegister::ConfigLoad_MstrConfigLoad::Mask);
    while ( Read32(&(m_pRegI2c->configLoad)) & (I2cRegister::ConfigLoad_MstrConfigLoad::Mask) )
    {
    }

    // 6
    SetBitOn32( &(m_pRegI2c->busClearConfig), I2cRegister::BusClearConfig_BcEnable::Mask);
    while ( Read32(&(m_pRegI2c->busClearConfig)) & (I2cRegister::BusClearConfig_BcEnable::Mask) )
    {
    }

    // 7
    // 8
    // 9
    while ( !( Read32(&(m_pRegI2c->busClearStatus)) & (I2cRegister::BusClearStatus_BcStatus::Mask) ) )
    {
    }
}

/**
 * @brief Pinmux の設定。Pinmuxライブラリが提供されたら実装変更が必要になる想定。
 */
void BusAccessor::SetPinmux(int busIdx) const NN_NOEXCEPT
{
    // PINMUX の設定。所定のバス番号に対応する SDA/SCL をアクティベートします。
    // Mask が 0 の場合は操作不要と見做して設定処理を飛ばします。
    if( m_RegPinmux.sclBitMask != 0 )
    {
        NN_DETAIL_I2C_LOG("%s(%d): Set PinmuxScl @ busIdx=%d.\n", __FUNCTION__, __LINE__, busIdx );
        Modify32  ( m_RegPinmux.pScl, m_RegPinmux.sclBitValue, m_RegPinmux.sclBitMask );
    }

    if( m_RegPinmux.sdaBitMask != 0 )
    {
        NN_DETAIL_I2C_LOG("%s(%d): Set PinmuxSda @ busIdx=%d.\n", __FUNCTION__, __LINE__, busIdx );
        Modify32  ( m_RegPinmux.pSda, m_RegPinmux.sdaBitValue, m_RegPinmux.sdaBitMask );
    }
}

/**
 * @brief Clock関連のレジスタ設定を行う関数
 */
void BusAccessor::SetClockRegisters( SpeedMode speedMode ) NN_NOEXCEPT
{
    uint32_t tHigh;
    uint32_t tLow;
    uint32_t clkDivisor;
    uint32_t debounce;
    uint32_t offset;
    bool     isHighSpeed = false;

    // [Ref1] Table 137 @ p.2189
    switch(speedMode)
    {
    case SpeedMode_Standard  :
        tHigh      = 4;
        tLow       = 4;
        clkDivisor = 0x19;
        debounce   = 2;
        break;
    case SpeedMode_Fast      :
        tHigh      = 4;
        tLow       = 4;
        clkDivisor = 0x19;
        debounce   = 2;
        break;
    case SpeedMode_FastPlus  :
        tHigh      = 4;
        tLow       = 4;
        clkDivisor = 0x10;
        debounce   = 0;
        break;
    case SpeedMode_HighSpeed :
        tHigh      = 6; // where tHigh + tLow = 13 (TORIAEZU)
        tLow       = 7; // where tHigh + tLow = 13 (TORIAEZU)
        clkDivisor = 0x02; // HS Div
        debounce   = 0;
        isHighSpeed = true;
        break;
    default: NN_UNEXPECTED_DEFAULT;

    }

    // Frequency Divisor の設定
    if( !isHighSpeed )
    {
        // SM/FM/FM+ は処理が同一
        Write32( &(m_pRegI2c->interfaceTiming_0),  ( tHigh << I2cRegister::InterfaceTiming_0_Thigh::Pos |
                                                     tLow  << I2cRegister::InterfaceTiming_0_Tlow::Pos )
                                                     & I2cRegister::InterfaceTiming_0::Mask );
        Write32( &(m_pRegI2c->clkDivisorRegister), ( clkDivisor << I2cRegister::ClkDivisorRegister_StdFastMode::Pos )
                                                     & I2cRegister::ClkDivisorRegister_StdFastMode::Mask);
        offset = ( clkDivisor <= 3 ) ? 3 : 2;
    }
    else
    {
        // HS は書き込み先レジスタが上記3モードとは異なる
        Write32( &(m_pRegI2c->hsInterfaceTiming_0),  ( tHigh << I2cRegister::HsInterfaceTiming_0_Thigh::Pos |
                                                       tLow  << I2cRegister::HsInterfaceTiming_0_Tlow::Pos )
                                                       & I2cRegister::HsInterfaceTiming_0::Mask);
        Write32( &(m_pRegI2c->clkDivisorRegister),   ( clkDivisor << I2cRegister::ClkDivisorRegister_HsMode::Pos )
                                                       & I2cRegister::ClkDivisorRegister_HsMode::Mask);
        offset = ( clkDivisor <= 4 ) ? 4 : 2;
    }

    Write32( &(m_pRegI2c->cnfg), ( debounce << I2cRegister::Cnfg_Debounce::Pos ) & I2cRegister::Cnfg_Debounce::Mask);                  // Debounce  = 2

    // [5.7.68] I2C#_CLK_DIVISOR の変更
    Modify32( m_RegClkRst.pClockSource, ( CalcFrequencyDivisor( tHigh, tLow, clkDivisor, offset ) << 0), (0xFFFF) );
}

/**
 * @brief パケットベースモード向けレジスタの初期設定を行う
 */
void BusAccessor::SetPacketModeRegisters() NN_NOEXCEPT
{
    const uint32_t TxFifoTrigDefault = 7;
    const uint32_t RxFifoTrigDefault = 0;

    SetBitOn32  ( &(m_pRegI2c->cnfg),       I2cRegister::Cnfg_PacketModeEn::Mask);
    SetBitOn32  ( &(m_pRegI2c->configLoad), I2cRegister::ConfigLoad_MstrConfigLoad::Mask);

    // 割り込みマスク
    Write32  ( &(m_pRegI2c->interruptMaskRegister),
        I2cRegister::InterruptMask_BusClearDoneIntEn::Mask           |
        I2cRegister::InterruptMask_PacketXferCompleteIntEn::Mask     |
        I2cRegister::InterruptMask_AllPacketsXferCompleteIntEn::Mask |
        I2cRegister::InterruptMask_TfifoOvfIntEn::Mask               |
        I2cRegister::InterruptMask_RfifoUnfIntEn::Mask               |
        I2cRegister::InterruptMask_NoAckIntEn::Mask                  |
        I2cRegister::InterruptMask_ArbLostIntEn::Mask );

    // FIFO の Trigger level を設定
    Modify32 ( &(m_pRegI2c->fifoControl), TxFifoTrigDefault << I2cRegister::FifoControl_TxFifoTrig::Pos, I2cRegister::I2cRegister::FifoControl_TxFifoTrig::Mask );
    Modify32 ( &(m_pRegI2c->fifoControl), RxFifoTrigDefault << I2cRegister::FifoControl_RxFifoTrig::Pos, I2cRegister::I2cRegister::FifoControl_RxFifoTrig::Mask );
}


/**
 * @brief Bus の状態をチェックする関数
 */
bool BusAccessor::IsBusy() const NN_NOEXCEPT
{
    // Packet based な実装のときは busy を示すステータスレジスタが無いため false としておく
    return false;
}

/**
 * @brief Packet Transfer Status レジスタの状態を Result にマップしたものを取得します
 */
Result BusAccessor::GetTransactionResult() const NN_NOEXCEPT
{
    uint32_t status = Read32( &(m_pRegI2c->packetTransferStatus));

    if( status & I2cRegister::PacketTransferStatus_NoAckForAddr::Mask )
    {
        return ResultNoAck();
    }
    if( status & I2cRegister::PacketTransferStatus_NoAckForData::Mask )
    {
        return ResultNoAck();
    }
    if( status & I2cRegister::PacketTransferStatus_ArbLost::Mask )
    {
        //  [Ref1] 31.1 Arbitration lost 時の対処
        BusClear();
        // Arbitrationlost をユーザーに通知したところでハンドリングできないから Busy と見做す。
        return ResultBusBusy();
    }

    return  ResultSuccess();
}

/**
 * @brief [31.4.4] Error Handling に従ったエラーハンドリングを行う
 */
void BusAccessor::HandleTransactionError( Result result ) NN_NOEXCEPT
{
    if (result.IsFailure())
    {
        if ( ResultNoAck::Includes(result) ||
             ResultBusBusy::Includes(result) )
        {
            // I2C コントローラのリセット
            ResetController();

            // クロック関連レジスタのセット
            SetClockRegisters(m_SpeedMode);

            // Packet mode への初期設定
            SetPacketModeRegisters();

            // FIFO の Flush
            FlushFifos();
        }
        else
        {
            NN_ABORT("UNEXPECTED ERROR\n");
        }
    }
    else
    {
        // 成功時は何もしない
    }
}

/**
 * @brief [31.5.24] FrequencyDivisor の導出
 */
uint32_t  BusAccessor::CalcFrequencyDivisor( uint32_t tHigh, uint32_t tLow, uint32_t clkDivisor, uint32_t offset ) const NN_NOEXCEPT
{
    return ( m_ClockSource / ( m_SpeedMode * ( tHigh + tLow + offset ) * ( clkDivisor + 1 ) ) );
}

/**
 * @brief [31.5.20] FIFO を Flush する関数
 */
Result   BusAccessor::FlushFifos() NN_NOEXCEPT
{
    const int NumberOfMaxTries = 5;
    int       currentTries     = 0;

    SetBitOn32( &(m_pRegI2c->fifoControl), I2cRegister::FifoControl_RxFifoFlush::Mask);
    SetBitOn32( &(m_pRegI2c->fifoControl), I2cRegister::FifoControl_TxFifoFlush::Mask);

    while( Read32( &(m_pRegI2c->fifoControl)) & ( I2cRegister::FifoControl_RxFifoFlush::Mask | I2cRegister::FifoControl_TxFifoFlush::Mask ) )
    {
        if( ++currentTries >= NumberOfMaxTries )
        {
            NN_SDK_LOG("Failed to flush FIFO.\n");
            return ResultBusBusy();
        }
        NN_SDK_LOG("Flushing FIFO...\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }
    return ResultSuccess();
}

/**
 * @brief デバッグ用関数。関連レジスタの情報をダンプする。
 */
void BusAccessor::DumpRegisters(int line) const NN_NOEXCEPT
{
    NN_SDK_LOG("\t%s(%d)\n", __FUNCTION__, line );
    NN_SDK_LOG("\n\t*** I2C Register info ***\n\n");

    NN_SDK_LOG("\t[Register locations]---------------------\n");
    NN_SDK_LOG("\t  clockSource                  = 0x%08x @ 0x%p\n", *(m_RegClkRst.pClockSource), m_RegClkRst.pClockSource);
    NN_SDK_LOG("\t  clockEnable                  = 0x%08x @ 0x%p\n", *(m_RegClkRst.pClockEnable), m_RegClkRst.pClockEnable);
    NN_SDK_LOG("\t  clockEnableBitMask           = 0x%08x\n",        m_RegClkRst.clockEnableBitMask);

    NN_SDK_LOG("\t  reset                        = 0x%08x @ 0x%p\n", *(m_RegClkRst.pReset), m_RegClkRst.pReset);
    NN_SDK_LOG("\t  resetBitMask                 = 0x%08x\n",        m_RegClkRst.resetBitMask);

    NN_SDK_LOG("\t  pinmuxScl                    = 0x%08x @ 0x%p\n", *(m_RegPinmux.pScl), m_RegPinmux.pScl );
    NN_SDK_LOG("\t  pinmuxSclBitValue            = 0x%08x\n",        m_RegPinmux.sclBitValue );
    NN_SDK_LOG("\t  pinmuxSclBitMask             = 0x%08x\n",        m_RegPinmux.sclBitMask );
    NN_SDK_LOG("\t  pinmuxSda                    = 0x%08x @ 0x%p\n", *(m_RegPinmux.pSda), m_RegPinmux.pSda );
    NN_SDK_LOG("\t  pinmuxSdaBitValue            = 0x%08x\n",        m_RegPinmux.sdaBitValue );
    NN_SDK_LOG("\t  pinmuxSdaBitMask             = 0x%08x\n",        m_RegPinmux.sdaBitMask );

    NN_SDK_LOG("\t[Register values]---------------------\n");
    NN_SDK_LOG("\t  i2c->cnfg                    = 0x%08x\n", m_pRegI2c->cnfg );
    NN_SDK_LOG("\t  i2c->cmdAddr0                = 0x%08x\n", m_pRegI2c->cmdAddr0 );
    NN_SDK_LOG("\t  i2c->cmdAddr1                = 0x%08x\n", m_pRegI2c->cmdAddr1 );
    NN_SDK_LOG("\t  i2c->cmdData1                = 0x%08x\n", m_pRegI2c->cmdData1 );
    NN_SDK_LOG("\t  i2c->cmdData2                = 0x%08x\n", m_pRegI2c->cmdData2 );
    // NN_SDK_LOG("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    // NN_SDK_LOG("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_SDK_LOG("\t  i2c->status                  = 0x%08x\n", m_pRegI2c->status );
    NN_SDK_LOG("\t  i2c->slCnfg                  = 0x%08x\n", m_pRegI2c->slCnfg );
    NN_SDK_LOG("\t  i2c->slRcvd                  = 0x%08x\n", m_pRegI2c->slRcvd );
    NN_SDK_LOG("\t  i2c->slStatus                = 0x%08x\n", m_pRegI2c->slStatus );
    NN_SDK_LOG("\t  i2c->slAddr1                 = 0x%08x\n", m_pRegI2c->slAddr1 );
    NN_SDK_LOG("\t  i2c->slAddr2                 = 0x%08x\n", m_pRegI2c->slAddr2 );
    NN_SDK_LOG("\t  i2c->tlowSext                = 0x%08x\n", m_pRegI2c->tlowSext );
    // NN_SDK_LOG("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_SDK_LOG("\t  i2c->slDelayCount            = 0x%08x\n", m_pRegI2c->slDelayCount );
    NN_SDK_LOG("\t  i2c->slIntMask               = 0x%08x\n", m_pRegI2c->slIntMask );
    NN_SDK_LOG("\t  i2c->slIntSource             = 0x%08x\n", m_pRegI2c->slIntSource );
    NN_SDK_LOG("\t  i2c->slIntSet                = 0x%08x\n", m_pRegI2c->slIntSet );
    // NN_SDK_LOG("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_SDK_LOG("\t  i2c->txPacketFifo            = 0x%08x\n", m_pRegI2c->txPacketFifo );
    NN_SDK_LOG("\t  i2c->rxFifo                  = 0x%08x\n", m_pRegI2c->rxFifo );
    NN_SDK_LOG("\t  i2c->packetTransferStatus    = 0x%08x\n", m_pRegI2c->packetTransferStatus );
    NN_SDK_LOG("\t  i2c->fifoControl             = 0x%08x\n", m_pRegI2c->fifoControl );
    NN_SDK_LOG("\t  i2c->fifoStatus              = 0x%08x\n", m_pRegI2c->fifoStatus );
    NN_SDK_LOG("\t  i2c->interruptMaskRegister   = 0x%08x\n", m_pRegI2c->interruptMaskRegister );
    NN_SDK_LOG("\t  i2c->interruptStatusRegister = 0x%08x\n", m_pRegI2c->interruptStatusRegister );
    NN_SDK_LOG("\t  i2c->clkDivisorRegister      = 0x%08x\n", m_pRegI2c->clkDivisorRegister );
    NN_SDK_LOG("\t  i2c->interruptSourceRegister = 0x%08x\n", m_pRegI2c->interruptSourceRegister );
    NN_SDK_LOG("\t  i2c->interruptSetRegister    = 0x%08x\n", m_pRegI2c->interruptSetRegister );
    NN_SDK_LOG("\t  i2c->slvTxPacketFifo         = 0x%08x\n", m_pRegI2c->slvTxPacketFifo );
    NN_SDK_LOG("\t  i2c->slvRxFifo               = 0x%08x\n", m_pRegI2c->slvRxFifo );
    NN_SDK_LOG("\t  i2c->slvPacketStatus         = 0x%08x\n", m_pRegI2c->slvPacketStatus );
    NN_SDK_LOG("\t  i2c->busClearConfig          = 0x%08x\n", m_pRegI2c->busClearConfig );
    NN_SDK_LOG("\t  i2c->busClearStatus          = 0x%08x\n", m_pRegI2c->busClearStatus );
    NN_SDK_LOG("\t  i2c->configLoad              = 0x%08x\n", m_pRegI2c->configLoad );
    // NN_SDK_LOG("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_SDK_LOG("\t  i2c->interfaceTiming_0       = 0x%08x\n", m_pRegI2c->interfaceTiming_0 );
    NN_SDK_LOG("\t  i2c->interfaceTiming_1       = 0x%08x\n", m_pRegI2c->interfaceTiming_1 );
    NN_SDK_LOG("\t  i2c->hsInterfaceTiming_0     = 0x%08x\n", m_pRegI2c->hsInterfaceTiming_0 );
    NN_SDK_LOG("\t  i2c->hsInterfaceTiming_1     = 0x%08x\n", m_pRegI2c->hsInterfaceTiming_1 );
}

} // detail
} // driver
} // i2c
} // nn
