﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/srepo/detail/service/core/srepo_DurationManager.h>
#include <nn/srepo/detail/service/core/srepo_DurationReportGenerator.h>
#include <nn/srepo/detail/service/core/srepo_SomeValue.h>
#include <nn/srepo/detail/service/core/srepo_StateDuration.h>
#include <nn/srepo/detail/msgpack/srepo_MessagePack.h>
#include <nn/srepo/detail/srepo_ApiDetail.h>

//#define NNT_SREPO_DUMP_DURATION // stateの詳細ダンプマクロ

using namespace nn::srepo::detail::service::core;

namespace
{
    void DumpSystemStateDuration(
        const StateDuration values[SystemStateChangeManager::StateDurationCount]) NN_NOEXCEPT
    {
#ifdef NNT_SREPO_DUMP_DURATION
        NN_LOG("----------\n");
        for(int i = 0 ; i < SystemStateChangeManager::StateDurationCount ; i++)
        {
            const auto& value = values[i];
            SomeValueString holder(value.pStateValue);
            NN_LOG("[System %02d] state (%s) (duration:%lld sec)\n",
                i, holder.Get(), value.duration.GetSeconds());
        }
#else
        NN_UNUSED(values);
#endif
    };

    void DumpUserStateDuration(
        const StateDuration values[OneUserStateChangeManager::StateDurationCount]) NN_NOEXCEPT
    {
#ifdef NNT_SREPO_DUMP_DURATION
        NN_LOG("----------\n");
        for(int i = 0 ; i < OneUserStateChangeManager::StateDurationCount ; i++)
        {
            const auto& value = values[i];
            SomeValueString holder(value.pStateValue);
            NN_LOG("[User %02d] state (%s) (duration:%lld sec)\n",
                i, holder.Get(), value.duration.GetSeconds());
        }
#else
        NN_UNUSED(values);
#endif
    };

    template<typename StateValueType>
    bool IsEqual(
        const StateDuration& actualStateDuration,
        const StateValueType& expectStateValue,
        const nn::TimeSpanType& expectDuration) NN_NOEXCEPT
    {
        bool ret = true;

        if(actualStateDuration.pStateValue->GetTag() != SomeValue<StateValueType>::Tag)
        {
            NN_LOG("State tag not equal (expect:%d, actual:%d)\n",
                SomeValue<StateValueType>::Tag, actualStateDuration.pStateValue->GetTag());
            ret = false;
        }

        if(actualStateDuration.duration != expectDuration)
        {
            NN_LOG("Duration not equal (expect:%lld, actual:%lld)\n",
                nn::TimeSpan(expectDuration).GetSeconds(), nn::TimeSpan(actualStateDuration.duration).GetSeconds());
            ret = false;
        }

        auto expectState = CreateSomeValue(expectStateValue);
        SomeValueString expect(&expectState);
        SomeValueString actual(actualStateDuration.pStateValue);
        if(expect != actual)
        {
            NN_LOG("State value not equal (expect:%s, actual:%s)\n", expect.Get(), actual.Get());
            ret = false;
        }

        return ret;
    }
}

TEST(StateDuration, SomeValue)
{
    {
        auto value = CreateSomeValue(true);
        EXPECT_EQ(SomeValue<bool>::Tag, value.GetTag());
        EXPECT_EQ(SomeValue<bool>::Tag, static_cast<SomeValueBase*>(&value)->GetTag());
        EXPECT_TRUE(value.GetRef());

        value.Set(false);
        EXPECT_FALSE(value.GetRef());
    }
    {
        auto value = CreateSomeValue(false);
        EXPECT_EQ(SomeValue<bool>::Tag, value.GetTag());
        EXPECT_EQ(SomeValue<bool>::Tag, static_cast<SomeValueBase*>(&value)->GetTag());
        EXPECT_FALSE(value.GetRef());

        value.Set(true);
        EXPECT_TRUE(value.GetRef());
    }
    {
        auto value = CreateSomeValue(123);
        EXPECT_EQ(SomeValue<int32_t>::Tag, value.GetTag());
        EXPECT_EQ(SomeValue<int32_t>::Tag, static_cast<SomeValueBase*>(&value)->GetTag());
        EXPECT_EQ(123, value.GetRef());

        value.Set(321);
        EXPECT_EQ(321, value.GetRef());
    }
    {
        auto value = CreateSomeValue(String16::Create("TEST"));
        EXPECT_EQ(SomeValue<String16>::Tag, value.GetTag());
        EXPECT_EQ(SomeValue<String16>::Tag, static_cast<SomeValueBase*>(&value)->GetTag());
        EXPECT_STREQ("TEST", value.GetRef().value);

        value.Set(String16::Create("hoge"));
        EXPECT_STREQ("hoge", value.GetRef().value);
    }
    {
        nn::ncm::ProgramId programId = {0x0123456789abcdef};
        auto value = CreateSomeValue(programId);
        EXPECT_EQ(SomeValue<nn::ncm::ProgramId>::Tag, value.GetTag());
        EXPECT_EQ(SomeValue<nn::ncm::ProgramId>::Tag, static_cast<SomeValueBase*>(&value)->GetTag());
        EXPECT_EQ(programId, value.GetRef());

        nn::ncm::ProgramId programId2 = {0xfedcba9876543210};
        value.Set(programId2);
        EXPECT_EQ(programId2, value.GetRef());
    }
}

// シーケンス番号
TEST(StateDuration, MultiUserDurationManager_SeqNo)
{
    const nn::account::Uid User1 = {0x00, 0x01};
    const nn::account::Uid User2 = {0x00, 0x02};
    const nn::account::Uid User3 = {0x00, 0x03};
    const nn::account::Uid User4 = {0x00, 0x04};
    const nn::account::Uid User5 = {0x00, 0x05};
    const nn::account::Uid User6 = {0x00, 0x06};
    const nn::account::Uid User7 = {0x00, 0x07};
    const nn::account::Uid User8 = {0x00, 0x08};

    MultiUserDurationManager manager;
    nn::TimeSpan current = nn::TimeSpan::FromSeconds(10);

    manager.Register(User1, current);
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User1));
    EXPECT_EQ(2LLU, manager.GetSeqNoAndIncrease(User1));

    manager.Register(User2, current);
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User2));
    EXPECT_EQ(2LLU, manager.GetSeqNoAndIncrease(User2));

    EXPECT_EQ(3LLU, manager.GetSeqNoAndIncrease(User1));
    EXPECT_EQ(4LLU, manager.GetSeqNoAndIncrease(User1));

    manager.Unregister(User2);

    EXPECT_EQ(5LLU, manager.GetSeqNoAndIncrease(User1));
    EXPECT_EQ(6LLU, manager.GetSeqNoAndIncrease(User1));

    manager.Register(User2, current);
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User2));
    EXPECT_EQ(2LLU, manager.GetSeqNoAndIncrease(User2));

    manager.Register(User3, current);
    manager.Register(User4, current);
    manager.Register(User5, current);
    manager.Register(User6, current);
    manager.Register(User7, current);
    manager.Register(User8, current);

    EXPECT_EQ(7LLU, manager.GetSeqNoAndIncrease(User1));
    EXPECT_EQ(3LLU, manager.GetSeqNoAndIncrease(User2));
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User3));
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User4));
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User5));
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User6));
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User7));
    EXPECT_EQ(1LLU, manager.GetSeqNoAndIncrease(User8));
}

// ユーザー登録できるか
TEST(StateDuration, MultiUserDurationManager1)
{
    const nn::account::Uid User1 = {0x00, 0x01};
    const nn::account::Uid User2 = {0x00, 0x02};
    const nn::account::Uid User3 = {0x00, 0x03};
    const nn::account::Uid User4 = {0x00, 0x04};
    const nn::account::Uid User5 = {0x00, 0x05};
    const nn::account::Uid User6 = {0x00, 0x06};
    const nn::account::Uid User7 = {0x00, 0x07};
    const nn::account::Uid User8 = {0x00, 0x08};

    MultiUserDurationManager manager;
    nn::TimeSpan current = nn::TimeSpan::FromSeconds(10);

    manager.Register(User1, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(1, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
    }

    manager.Register(User2, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(2, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
        EXPECT_EQ(User2, uids[1]);
    }

    manager.Register(User3, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(3, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
    }

    manager.Register(User4, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(4, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
        EXPECT_EQ(User4, uids[3]);
    }

    manager.Register(User5, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(5, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
        EXPECT_EQ(User4, uids[3]);
        EXPECT_EQ(User5, uids[4]);
    }

    manager.Register(User6, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(6, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
        EXPECT_EQ(User4, uids[3]);
        EXPECT_EQ(User5, uids[4]);
        EXPECT_EQ(User6, uids[5]);
    }

    manager.Register(User7, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(7, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
        EXPECT_EQ(User4, uids[3]);
        EXPECT_EQ(User5, uids[4]);
        EXPECT_EQ(User6, uids[5]);
        EXPECT_EQ(User7, uids[6]);
    }

    manager.Register(User8, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(8, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User1, uids[0]);
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
        EXPECT_EQ(User4, uids[3]);
        EXPECT_EQ(User5, uids[4]);
        EXPECT_EQ(User6, uids[5]);
        EXPECT_EQ(User7, uids[6]);
        EXPECT_EQ(User8, uids[7]);
    }
} // NOLINT(impl/function_size)

// ユーザー削除後に追加できるか
TEST(StateDuration, MultiUserDurationManager2)
{
    const nn::account::Uid User1 = {0x00, 0x01};
    const nn::account::Uid User2 = {0x00, 0x02};
    const nn::account::Uid User3 = {0x00, 0x03};
    const nn::account::Uid User4 = {0x00, 0x04};
    const nn::account::Uid User5 = {0x00, 0x05};
    const nn::account::Uid User6 = {0x00, 0x06};
    const nn::account::Uid User7 = {0x00, 0x07};
    const nn::account::Uid User8 = {0x00, 0x08};

    MultiUserDurationManager manager;
    nn::TimeSpan current = nn::TimeSpan::FromSeconds(10);

    manager.Register(User1, current);
    manager.Register(User2, current);
    manager.Register(User3, current);
    manager.Register(User4, current);
    manager.Register(User5, current);
    manager.Register(User6, current);
    manager.Register(User7, current);
    manager.Register(User8, current);

    manager.Unregister(User1);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(7, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User2, uids[0]);
        EXPECT_EQ(User3, uids[1]);
        EXPECT_EQ(User4, uids[2]);
        EXPECT_EQ(User5, uids[3]);
        EXPECT_EQ(User6, uids[4]);
        EXPECT_EQ(User7, uids[5]);
        EXPECT_EQ(User8, uids[6]);
    }

    manager.Unregister(User5);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(6, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User2, uids[0]);
        EXPECT_EQ(User3, uids[1]);
        EXPECT_EQ(User4, uids[2]);
        EXPECT_EQ(User6, uids[3]);
        EXPECT_EQ(User7, uids[4]);
        EXPECT_EQ(User8, uids[5]);
    }

    const nn::account::Uid User9 = {0x00, 0x09};
    manager.Register(User9, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(7, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User9, uids[0]); // <- 空いているとこに登録される
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
        EXPECT_EQ(User4, uids[3]);
        EXPECT_EQ(User6, uids[4]);
        EXPECT_EQ(User7, uids[5]);
        EXPECT_EQ(User8, uids[6]);
    }

    const nn::account::Uid User10 = {0x00, 0x10};
    manager.Register(User10, current);
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        ASSERT_EQ(8, manager.GetRegisteredUserList(uids));
        EXPECT_EQ(User9, uids[0]);
        EXPECT_EQ(User2, uids[1]);
        EXPECT_EQ(User3, uids[2]);
        EXPECT_EQ(User4, uids[3]);
        EXPECT_EQ(User10, uids[4]); // <- 空いているとこに登録される
        EXPECT_EQ(User6, uids[5]);
        EXPECT_EQ(User7, uids[6]);
        EXPECT_EQ(User8, uids[7]);
    }
} // NOLINT(impl/function_size)

// 継続時間が正しいか
TEST(StateDuration, MultiUserDurationManager3)
{
    const nn::account::Uid User1 = {0x00, 0x01};
    const nn::account::Uid User2 = {0x00, 0x02};
    const nn::account::Uid User3 = {0x00, 0x03};
    const nn::account::Uid User4 = {0x00, 0x04};

    MultiUserDurationManager manager;
    StateDuration stateDurations[OneUserStateChangeManager::StateDurationCount];

    nn::TimeSpan current = nn::TimeSpan::FromSeconds(10);

    manager.Register(User1, current);
    manager.Register(User2, current);

    current += nn::TimeSpan::FromSeconds(10);
    {
        //User1
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(10)));

        //User1 close->open
        ASSERT_TRUE(manager.IsStateChanged(User1, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(true))));
        manager.ChangeState(User1, current, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(true)));

        //User2
        manager.GenerateStateDurationArray(stateDurations, User2, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(10)));
    }

    current += nn::TimeSpan::FromSeconds(15);
    {
        //User1
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(true), nn::TimeSpan::FromSeconds(15)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(25)));

        //User1 open->close
        ASSERT_TRUE(manager.IsStateChanged(User1, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(false))));
        manager.ChangeState(User1, current, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(false)));

        //User2
        manager.GenerateStateDurationArray(stateDurations, User2, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(25)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(25)));
    }

    current += nn::TimeSpan::FromSeconds(20);
    {
        //User1
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(20)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(45)));

        //User2
        manager.GenerateStateDurationArray(stateDurations, User2, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(45)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(45)));
    }

    // 遅れて登録
    manager.Register(User3, current);
    current += nn::TimeSpan::FromSeconds(10);
    {
        //User1
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(30)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(55)));

        //User2
        manager.GenerateStateDurationArray(stateDurations, User2, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(55)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(55)));

        //User3
        manager.GenerateStateDurationArray(stateDurations, User3, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(10)));
    }

    current += nn::TimeSpan::FromSeconds(10);
    {
        //User1
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(40)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(65)));

        //User2
        manager.GenerateStateDurationArray(stateDurations, User2, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(65)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(65)));

        //User3
        manager.GenerateStateDurationArray(stateDurations, User3, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(20)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(20)));

        //User3 close->open
        ASSERT_TRUE(manager.IsStateChanged(User1, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(true))));
        manager.ChangeState(User3, current, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(true)));
    }

    // User2 消す
    manager.Unregister(User2);

    // User4 追加
    manager.Register(User4, current);
    current += nn::TimeSpan::FromSeconds(10);
    {
        //User1
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(50)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(75)));

        //User3
        manager.GenerateStateDurationArray(stateDurations, User3, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(true), nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(30)));

        //User3 open->close
        ASSERT_TRUE(manager.IsStateChanged(User3, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(false))));
        manager.ChangeState(User3, current, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(false)));

        //User4
        manager.GenerateStateDurationArray(stateDurations, User4, current);
        DumpUserStateDuration(stateDurations);
        EXPECT_TRUE(IsEqual(stateDurations[0], AccountStatus::Create(false), nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(stateDurations[1], nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(10)));
    }

    // 変化なしのため失敗
    ASSERT_FALSE(manager.IsStateChanged(User1, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(false))));
} // NOLINT(impl/function_size)

// 複数の継続時間を扱えるか
TEST(StateDuration, MultiUserDurationManager4)
{
    const nn::account::Uid User1 = {0x00, 0x01};

    MultiUserDurationManager manager;
    StateDuration stateDurations[OneUserStateChangeManager::StateDurationCount];

    nn::TimeSpan current = nn::TimeSpan::FromSeconds(10);
    manager.Register(User1, current);

    current += nn::TimeSpan::FromSeconds(10);
    {
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(
                stateDurations[static_cast<int>(UserDurationKind::AccountStatus)],
                    AccountStatus::Create(false), nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(
                stateDurations[static_cast<int>(UserDurationKind::FriendPresence)],
                    nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(10)));

        // Account close->open
        ASSERT_TRUE(manager.IsStateChanged(User1, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(true))));
        manager.ChangeState(User1, current, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(true)));
    }

    current += nn::TimeSpan::FromSeconds(10);
    {
        manager.GenerateStateDurationArray(stateDurations, User1, current);

        DumpUserStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(
                stateDurations[static_cast<int>(UserDurationKind::AccountStatus)],
                    AccountStatus::Create(true), nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(
                stateDurations[static_cast<int>(UserDurationKind::FriendPresence)],
                    nn::srepo::FriendPresence::Offline, nn::TimeSpan::FromSeconds(20)));

        // Friend Presence offline->online
        ASSERT_TRUE(manager.IsStateChanged(User1, UserDurationKind::FriendPresence, CreateSomeValue(nn::srepo::FriendPresence::Online)));
        manager.ChangeState(User1, current, UserDurationKind::FriendPresence, CreateSomeValue(nn::srepo::FriendPresence::Online));
    }

    current += nn::TimeSpan::FromSeconds(10);
    {
        manager.GenerateStateDurationArray(stateDurations, User1, current);
        DumpUserStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(
                stateDurations[static_cast<int>(UserDurationKind::AccountStatus)],
                    AccountStatus::Create(true), nn::TimeSpan::FromSeconds(20)));
        EXPECT_TRUE(IsEqual(
                stateDurations[static_cast<int>(UserDurationKind::FriendPresence)],
                    nn::srepo::FriendPresence::Online, nn::TimeSpan::FromSeconds(10)));

        // Friend Presence online->offline
        ASSERT_TRUE(manager.IsStateChanged(User1, UserDurationKind::FriendPresence, CreateSomeValue(nn::srepo::FriendPresence::Offline)));
        manager.ChangeState(User1, current, UserDurationKind::FriendPresence, CreateSomeValue(nn::srepo::FriendPresence::Offline));
    }

    // 変化なしのため失敗
    ASSERT_FALSE(manager.IsStateChanged(User1, UserDurationKind::AccountStatus, CreateSomeValue(AccountStatus::Create(true))));
    ASSERT_FALSE(manager.IsStateChanged(User1, UserDurationKind::FriendPresence, CreateSomeValue(nn::srepo::FriendPresence::Offline)));
} // NOLINT(impl/function_size)

TEST(StateDuration, SystemStateChangeManager)
{
    SystemStateChangeManager manager;
    StateDuration stateDurations[SystemStateChangeManager::StateDurationCount];

    const auto& GetDuration = [&stateDurations](SystemDurationKind kind) NN_NOEXCEPT
    {
        return stateDurations[static_cast<int>(kind)];
    };

    nn::TimeSpan current = nn::TimeSpan::FromSeconds(10);
    manager.Initialize(current);

    current += nn::TimeSpan::FromSeconds(5);
    {
        manager.GenerateStateDurationArray(stateDurations, current);
        DumpSystemStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::CompletedNetworkRequestType), nn::srepo::CompletedNetworkRequestType::None, nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ConnectionOfNotification), false, nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ForegroundProgramId), nn::ncm::ProgramId::GetInvalidId(), nn::TimeSpan::FromSeconds(5)));
    }

    current += nn::TimeSpan::FromSeconds(5);
    {
        manager.GenerateStateDurationArray(stateDurations, current);
        DumpSystemStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::CompletedNetworkRequestType), nn::srepo::CompletedNetworkRequestType::None, nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ConnectionOfNotification), false, nn::TimeSpan::FromSeconds(10)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ForegroundProgramId), nn::ncm::ProgramId::GetInvalidId(), nn::TimeSpan::FromSeconds(10)));
    }

    current += nn::TimeSpan::FromSeconds(5);
    {
        manager.GenerateStateDurationArray(stateDurations, current);
        DumpSystemStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::CompletedNetworkRequestType), nn::srepo::CompletedNetworkRequestType::None, nn::TimeSpan::FromSeconds(15)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ConnectionOfNotification), false, nn::TimeSpan::FromSeconds(15)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ForegroundProgramId), nn::ncm::ProgramId::GetInvalidId(), nn::TimeSpan::FromSeconds(15)));

        // InternetConnection offline->online
        ASSERT_TRUE(manager.IsStateChanged(SystemDurationKind::CompletedNetworkRequestType, CreateSomeValue(nn::srepo::CompletedNetworkRequestType::WideAreaNetwork)));
        manager.ChangeState(current, SystemDurationKind::CompletedNetworkRequestType, CreateSomeValue(nn::srepo::CompletedNetworkRequestType::WideAreaNetwork));
    }

    current += nn::TimeSpan::FromSeconds(5);
    {
        manager.GenerateStateDurationArray(stateDurations, current);
        DumpSystemStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::CompletedNetworkRequestType), nn::srepo::CompletedNetworkRequestType::WideAreaNetwork, nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ConnectionOfNotification), false, nn::TimeSpan::FromSeconds(20)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ForegroundProgramId), nn::ncm::ProgramId::GetInvalidId(), nn::TimeSpan::FromSeconds(20)));

        // InternetConnection online->offline
        ASSERT_TRUE(manager.IsStateChanged(SystemDurationKind::CompletedNetworkRequestType, CreateSomeValue(nn::srepo::CompletedNetworkRequestType::None)));
        manager.ChangeState(current, SystemDurationKind::CompletedNetworkRequestType, CreateSomeValue(nn::srepo::CompletedNetworkRequestType::None));
    }

    current += nn::TimeSpan::FromSeconds(5);
    {
        manager.GenerateStateDurationArray(stateDurations, current);
        DumpSystemStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::CompletedNetworkRequestType), nn::srepo::CompletedNetworkRequestType::None, nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ConnectionOfNotification), false, nn::TimeSpan::FromSeconds(25)));
        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ForegroundProgramId), nn::ncm::ProgramId::GetInvalidId(), nn::TimeSpan::FromSeconds(25)));

        // ConnectionOfNotification not connected->connected
        ASSERT_TRUE(manager.IsStateChanged(SystemDurationKind::ConnectionOfNotification, CreateSomeValue(true)));
        manager.ChangeState(current, SystemDurationKind::ConnectionOfNotification, CreateSomeValue(true));
    }

    current += nn::TimeSpan::FromSeconds(5);
    // 変化なしのため失敗
    ASSERT_FALSE(manager.IsStateChanged(SystemDurationKind::CompletedNetworkRequestType, CreateSomeValue(nn::srepo::CompletedNetworkRequestType::None)));

    // ProgramIdを扱えるか
    nn::ncm::ProgramId programId = {0x0123456789abcdef};
    {
        manager.GenerateStateDurationArray(stateDurations, current);
        DumpSystemStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ForegroundProgramId), nn::ncm::ProgramId::GetInvalidId(), nn::TimeSpan::FromSeconds(30)));

        // ForegroundProgramId change
        ASSERT_TRUE(manager.IsStateChanged(SystemDurationKind::ForegroundProgramId, CreateSomeValue(programId)));
        manager.ChangeState(current, SystemDurationKind::ForegroundProgramId, CreateSomeValue(programId));
    }
    current += nn::TimeSpan::FromSeconds(5);
    {
        manager.GenerateStateDurationArray(stateDurations, current);
        DumpSystemStateDuration(stateDurations);

        EXPECT_TRUE(IsEqual(GetDuration(SystemDurationKind::ForegroundProgramId), programId, nn::TimeSpan::FromSeconds(5)));
    }

    // 変化なしのため失敗
    ASSERT_FALSE(manager.IsStateChanged(SystemDurationKind::ForegroundProgramId, CreateSomeValue(programId)));
} // NOLINT(impl/function_size)

namespace
{
using namespace nn::srepo::detail;

namespace detail {
    void CheckValue(const int64_t& expect, const msgpack::AnyData& any, const nn::Bit8*, size_t) NN_NOEXCEPT
    {
        EXPECT_EQ(expect, any.si);
    }

    void CheckValue(const char* expect, const msgpack::AnyData& any, const nn::Bit8* pBuffer, size_t) NN_NOEXCEPT
    {
        char value[128] = {};
        std::memcpy(value, pBuffer + any.string.position, any.string.length);
        EXPECT_STREQ(expect, value);
    }
}

// TODO レポートが固まって順序変更などなくなったら Tests/Prepo/Sources/Tests/Common/testPrepo_Common.h のAPIを使うように変更
template <typename T>
void VerifyKeyValue(const char* expectKey, const T& expectValue, const nn::Bit8* pBuffer, size_t size) NN_NOEXCEPT
{
    msgpack::InputStreamParam stream = {pBuffer, size, 0};

    msgpack::AnyData any;
    msgpack::ReadCurrent(&any, &stream);

    ASSERT_EQ(any.type, msgpack::AnyDataType_Map);

    // 所望のkeyがでるまで ReadCurrent を進める
    while(stream.GetRemainSize() > 0)
    {
        // key
        ASSERT_TRUE(msgpack::ReadCurrent(&any, &stream));
        ASSERT_EQ(any.type, msgpack::AnyDataType_String);

        char key[128] = {};
        std::memcpy(key, pBuffer + any.string.position, any.string.length);

        if(nn::util::Strncmp(expectKey, key, sizeof(key)) != 0)
        {
            // value読み飛ばし
            ASSERT_TRUE(msgpack::ReadCurrent(&any, &stream));
            continue;
        }

        // value読み込み
        ASSERT_TRUE(msgpack::ReadCurrent(&any, &stream));

        ASSERT_NO_FATAL_FAILURE(detail::CheckValue(expectValue, any, pBuffer, size));
        return;
    }

    ASSERT_TRUE(false) << "key:" << expectKey << " not found.";
}

};

TEST(StateDuration, DurationReportGenerator)
{
    using namespace nn::srepo::detail;

    const nn::account::Uid User1 = {0x00, 0x01};
    const nn::account::Uid User2 = {0x00, 0x02};

    NN_FUNCTION_LOCAL_STATIC(DurationReportGenerator::ReportOutput, output, );

    DurationReportGenerator generator;

    nn::TimeSpan current = nn::TimeSpan::FromSeconds(10);
    generator.Initialize(current);

    // システム状態の変更
    //
    //

    current += nn::TimeSpan::FromSeconds(10);
    {
        // ユーザーなし, 変更あり

        NNT_ASSERT_RESULT_SUCCESS(generator.GenerateSystemTriggerReport(
            &output,
            current,
            SystemDurationKind::CompletedNetworkRequestType,
            nn::srepo::CompletedNetworkRequestType::WideAreaNetwork));

        ASSERT_EQ(output.count, 0);
    }

    generator.RegisterUser(User1, current);

    current += nn::TimeSpan::FromSeconds(10);
    {
        // ユーザー1人

        NNT_ASSERT_RESULT_SUCCESS(generator.GenerateSystemTriggerReport(
            &output,
            current,
            SystemDurationKind::CompletedNetworkRequestType,
            nn::srepo::CompletedNetworkRequestType::None));

        ASSERT_EQ(output.count, 1);
        EXPECT_EQ(User1, output.values[0].uid);

        ASSERT_TRUE(nn::srepo::detail::VerifyReport(output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_type", "network", output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_old", "WideAreaNetwork", output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_new", "None", output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("up_time_sec", static_cast<int64_t>(30LL), output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("stable_duration_sec", static_cast<int64_t>(10LL), output.values[0].buffer, output.values[0].actualBufferSize));

        //TODO 詳細確認
    }

    //current += nn::TimeSpan::FromSeconds(10);
    {
        // ユーザー1人

        NNT_ASSERT_RESULT_SUCCESS(generator.GenerateSystemTriggerReport(
            &output,
            current,
            SystemDurationKind::ControllerStatus,
            ControllerStatus::Create(1, 1)));

        ASSERT_EQ(output.count, 1);
        EXPECT_EQ(User1, output.values[0].uid);

        ASSERT_TRUE(nn::srepo::detail::VerifyReport(output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_type", "controller(total,rail)", output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_old", "0,0", output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_new", "1,1", output.values[0].buffer, output.values[0].actualBufferSize));

        //TODO 詳細確認
    }
    //current += nn::TimeSpan::FromSeconds(10);
    {
        // ユーザー1人

        NNT_ASSERT_RESULT_SUCCESS(generator.GenerateSystemTriggerReport(
            &output,
            current,
            SystemDurationKind::SystemPowerState,
            nn::srepo::SystemPowerState::MinimumAwake));

        ASSERT_EQ(output.count, 1);
        EXPECT_EQ(User1, output.values[0].uid);

        ASSERT_TRUE(nn::srepo::detail::VerifyReport(output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_type", "power", output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_old", "FullAwake", output.values[0].buffer, output.values[0].actualBufferSize));
        ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_new", "MinimumAwake", output.values[0].buffer, output.values[0].actualBufferSize));

        //TODO 詳細確認
    }

    generator.RegisterUser(User2, current);

    current += nn::TimeSpan::FromSeconds(10);
    {
        // ユーザー2人

        NNT_ASSERT_RESULT_SUCCESS(generator.GenerateSystemTriggerReport(
            &output,
            current,
            SystemDurationKind::CompletedNetworkRequestType,
            nn::srepo::CompletedNetworkRequestType::LocalAreaNetwork));

        ASSERT_EQ(output.count, 2);

        {
            EXPECT_EQ(User1, output.values[0].uid);
            ASSERT_TRUE(nn::srepo::detail::VerifyReport(output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_type", "network", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_old", "None", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_new", "LocalAreaNetwork", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("up_time_sec", static_cast<int64_t>(40LL), output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("stable_duration_sec", static_cast<int64_t>(10LL), output.values[0].buffer, output.values[0].actualBufferSize));

            //TODO 詳細確認
        }

        {
            EXPECT_EQ(User2, output.values[1].uid);
            ASSERT_TRUE(nn::srepo::detail::VerifyReport(output.values[1].buffer, output.values[1].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_type", "network", output.values[1].buffer, output.values[1].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_old", "None", output.values[1].buffer, output.values[1].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_new", "LocalAreaNetwork", output.values[1].buffer, output.values[1].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("up_time_sec", static_cast<int64_t>(40LL), output.values[1].buffer, output.values[1].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("stable_duration_sec", static_cast<int64_t>(10LL), output.values[0].buffer, output.values[0].actualBufferSize));

            //TODO 詳細確認
        }
    }

    // ユーザー状態の変更
    //
    //

    current += nn::TimeSpan::FromSeconds(20);
    {
        // ユーザー1変更
        NNT_ASSERT_RESULT_SUCCESS(generator.GenerateUserTriggerReport(
            &output,
            User1,
            current,
            UserDurationKind::AccountStatus,
            AccountStatus::Create(true)));

        ASSERT_EQ(output.count, 1);

        {
            EXPECT_EQ(User1, output.values[0].uid);
            ASSERT_TRUE(nn::srepo::detail::VerifyReport(output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_type", "account", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_old", "close", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_new", "open", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("up_time_sec", static_cast<int64_t>(60LL), output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("stable_duration_sec", static_cast<int64_t>(20LL), output.values[0].buffer, output.values[0].actualBufferSize));

            //TODO 詳細確認
        }
    }

    current += nn::TimeSpan::FromSeconds(30);
    {
        // ユーザー2変更
        NNT_ASSERT_RESULT_SUCCESS(generator.GenerateUserTriggerReport(
            &output,
            User2,
            current,
            UserDurationKind::AccountStatus,
            AccountStatus::Create(true)));

        ASSERT_EQ(output.count, 1);

        {
            EXPECT_EQ(User2, output.values[0].uid);
            ASSERT_TRUE(nn::srepo::detail::VerifyReport(output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_type", "account", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_old", "close", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("trigger_state_new", "open", output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("up_time_sec", static_cast<int64_t>(90LL), output.values[0].buffer, output.values[0].actualBufferSize));
            ASSERT_NO_FATAL_FAILURE(VerifyKeyValue("stable_duration_sec", static_cast<int64_t>(30LL), output.values[0].buffer, output.values[0].actualBufferSize));

            //TODO 詳細確認
        }
    }
} // NOLINT(impl/function_size)
