﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_StringUtil.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/srepo.h>
#include <nn/srepo/srepo_ApiAdmin.h>
#include <nn/srepo/detail/msgpack/srepo_MessagePack.h>

namespace
{
    const size_t ThreadStackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);

    std::atomic<int64_t> g_Count(0);

    const char TestEventId[] = "test_match_pomp";

    nn::os::SystemEvent g_PushEvent;
}

void PopAndPushSystemReport(void* argument) NN_NOEXCEPT
{
    const auto expectedUid = *static_cast<nn::account::Uid*>(argument);

    int64_t count = 0;

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitAny(g_StopEvent.GetBase(), g_PushEvent.GetBase());

        if (g_StopEvent.TryWait())
        {
            g_StopEvent.Clear();
            return;
        }
        if (g_PushEvent.TryWait())
        {
            g_PushEvent.Clear();
        }

        nn::srepo::ReportCategory category;
        nn::account::Uid uid;
        char eventId[nn::srepo::EventIdLengthMax + 1];
        nn::ApplicationId applicationId;
        static nn::Bit8 s_Data[nn::srepo::SystemReport::BufferSizeMax];
        size_t dataSize;

        nn::Result result;
        do
        {
            result = nn::srepo::PopReport(&category, &uid, eventId, &applicationId, &dataSize, s_Data, sizeof (eventId), sizeof (s_Data));
            ASSERT_TRUE(result.IsSuccess() || nn::srepo::ResultBufferEmpty::Includes(result));

            if (result.IsSuccess())
            {
                ASSERT_EQ(nn::srepo::ReportCategory_Normal, category);

                if (!(nn::util::Strncmp(eventId, TestEventId, sizeof (TestEventId)) == 0))
                {
                    continue;
                }

                ASSERT_EQ(expectedUid, uid);

                ASSERT_EQ(0x0123456789abcdef, applicationId.value);

                nn::srepo::detail::msgpack::AnyData any = {};
                nn::srepo::detail::msgpack::InputStreamParam stream = {s_Data, dataSize};

                ASSERT_TRUE(nn::srepo::detail::msgpack::ReadCurrent(&any, &stream));
                ASSERT_EQ(nn::srepo::detail::msgpack::AnyDataType_Map, any.type);
                ASSERT_EQ(1u, any.num);

                ASSERT_TRUE(nn::srepo::detail::msgpack::ReadCurrent(&any, &stream));
                ASSERT_EQ(nn::srepo::detail::msgpack::AnyDataType_String, any.type);
                const char* pKey = reinterpret_cast<const char*>(&stream.data[any.string.position]);
                const size_t keyLength = any.string.position;
                ASSERT_EQ(0, std::memcmp("count", pKey, keyLength));

                ASSERT_TRUE(nn::srepo::detail::msgpack::ReadCurrent(&any, &stream));
                ASSERT_EQ(nn::srepo::detail::msgpack::AnyDataType_SInteger, any.type);
                ASSERT_EQ(count++, any.si);

                ASSERT_EQ(0u, stream.GetRemainSize());

                g_Count--;
            }
        } while (result.IsSuccess());
    }
}

void ExecuteTest(nn::account::Uid user)
{
    g_Count = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, PopAndPushSystemReport, &user, g_ThreadStack, ThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()) - 1));
    nn::os::StartThread(&g_Thread);

    for (int64_t count = 0; count < 100000; count++)
    {
        nn::srepo::SystemReport report(TestEventId);

        nn::ApplicationId applicationId = {0x0123456789abcdef};
        NNT_ASSERT_RESULT_SUCCESS(report.SetApplicationId(applicationId));

        char buffer[1024];
        report.SetBuffer(buffer, sizeof (buffer));

        NNT_ASSERT_RESULT_SUCCESS(report.Add("count", count));

        nn::Result result = user == nn::account::InvalidUid
            ? report.Save()
            : report.Save(user);
        ASSERT_TRUE(result.IsSuccess() || nn::srepo::ResultOutOfResource::Includes(result));

        if (result.IsSuccess())
        {
            g_Count++;
        }
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    g_StopEvent.Signal();

    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);

    ASSERT_EQ(0, g_Count.load());
}

TEST(RegisterPushEvent, Normal)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::srepo::RegisterPushEvent(&g_PushEvent, nn::os::EventClearMode_ManualClear));
}

TEST(Save, WithoutUser)
{
    ExecuteTest(nn::account::InvalidUid);
}

TEST(Save, WithUser)
{
    int userCount;
    nn::account::Uid users[nn::account::UserCountMax] = {};

    nn::account::InitializeForSystemService();

    NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&userCount,
        users, nn::account::UserCountMax));
    if (!(userCount > 0))
    {
        NN_LOG("Skip the test because it needs one or more users.\n");
        return;
    }

    ExecuteTest(users[0]);
}
