﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/settings/factory/settings_DeviceKey.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

namespace
{
    const int GenerationMax = nn::spl::LatestKeyGeneration;
}

class SplKeyGenerationTest : public ::testing::Test, public ::testing::WithParamInterface<int>
{
};

INSTANTIATE_TEST_CASE_P(
    SplKeyGenerationTest,
    SplKeyGenerationTest,
    ::testing::Range(0, GenerationMax + 1));

TEST_P(SplKeyGenerationTest, DecryptAesKey)
{
    struct TestSet
    {
        uint8_t key[16];
    };
    const TestSet testSet[] =
    {
        {
            {
                0x1D,0x91,0xCD,0xB6,0x0D,0x4E,0x7C,0x8C,0x22,0x51,0x2F,0x82,0x88,0x5A,0x40,0xB5,
            },
        },
        {
            {
                0x1D,0x91,0xCD,0xB6,0x0D,0x4E,0x7C,0x8C,0x22,0x51,0x2F,0x82,0x88,0x5A,0x40,0xB5,
            },
        },
        {
            {
                0x04,0x72,0x3a,0xb1,0xba,0xa6,0x19,0x32,0xa1,0x32,0xb3,0x3d,0x43,0xb1,0x32,0xc3,
            },
        },
        {
            {
                0xa9,0xf3,0x20,0x60,0x00,0xbe,0xb9,0x76,0x49,0xd4,0x79,0x51,0xcb,0xd2,0x25,0xf9,
            },
        },
        {
            {
                0x07,0x28,0x33,0x5a,0x97,0x8e,0x78,0xc0,0xa1,0x97,0xa4,0xcd,0xff,0x2e,0xcc,0x2e,
            },
        },
        {
            {
                0x43,0x66,0x22,0xfc,0x7e,0x1f,0x5c,0x72,0xe4,0xdd,0x96,0x13,0xcf,0xcd,0x4d,0x9e,
            },
        },
    };
    NN_STATIC_ASSERT(sizeof(testSet) / sizeof(testSet[0]) == GenerationMax + 1);

    auto generation = GetParam();
    const auto& key = testSet[generation].key;

    const uint8_t encryptedKey[] =
    {
        0x9A,0x95,0x48,0x16,0x6B,0x37,0x61,0x95,0xD4,0x58,0x20,0xFF,0x7C,0x19,0xD0,0x86,
    };

    ::nn::spl::InitializeForCrypto();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    uint8_t result[16];
    auto option = 0;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::DecryptAesKey(
            result, sizeof(result),
            encryptedKey, sizeof(encryptedKey),
            generation,
            option));

    EXPECT_EQ(::std::memcmp(result, key, sizeof(key)), 0);
}

TEST_P(SplKeyGenerationTest, PrepareCommonEsTitleKey)
{
    struct TestSet
    {
        uint8_t cipher[16];
        uint8_t iv[16];
        uint8_t plain[16];
    };
    const TestSet testSet[] =
    {
        {
            {
                0x75,0x82,0x50,0xd8,0xa4,0xf8,0x16,0x86,0x96,0x92,0xad,0x02,0x70,0xb7,0x9b,0x38,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
        },
        {
            {
                0x75,0x82,0x50,0xd8,0xa4,0xf8,0x16,0x86,0x96,0x92,0xad,0x02,0x70,0xb7,0x9b,0x38,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
        },
        {
            {
                0x16,0xb6,0x6b,0xff,0xf1,0x8a,0x42,0xf7,0x8f,0x3e,0x42,0x0c,0x8f,0xcb,0xdb,0x0d,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
        },
        {
            {
                0xff,0x4d,0x32,0x32,0x09,0x7e,0xd1,0x49,0x63,0xc8,0x89,0xd7,0xc5,0x11,0xb6,0x1b,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
        },
        {
            {
                0x89,0x73,0x49,0x7c,0xc3,0x51,0x2f,0x55,0x71,0x77,0x15,0x16,0xd9,0x61,0x0f,0xf8,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
        },
        {
            {
                0x64,0xd4,0x7e,0x25,0x97,0xac,0x8a,0xfd,0x8f,0x71,0x47,0xe8,0x08,0x91,0xe8,0x2d,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
            {
                0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            },
        },
    };
    NN_STATIC_ASSERT(sizeof(testSet) / sizeof(testSet[0]) == GenerationMax + 1);

    auto generation = GetParam();
    const auto& cipher = testSet[generation].cipher;
    const auto& iv     = testSet[generation].iv;
    const auto& plain  = testSet[generation].plain;

    const uint8_t encryptedTitleKey[] =
    {
        0x27,0x76,0x21,0x28,0x70,0x8d,0x61,0x8b,0x3e,0x20,0x36,0xa0,0x44,0xd5,0x73,0x05
    };

    nn::spl::AccessKey accessKey;
    {
        ::nn::spl::InitializeForEs();
        NN_UTIL_SCOPE_EXIT
        {
            ::nn::spl::Finalize();
        };

        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::PrepareCommonEsTitleKey(
                &accessKey,
                encryptedTitleKey, sizeof(encryptedTitleKey),
                generation));
    }
    {
        ::nn::spl::InitializeForFs();
        NN_UTIL_SCOPE_EXIT
        {
            ::nn::spl::Finalize();
        };

        int slotIndex;
        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::AllocateAesKeySlot(
                &slotIndex));
        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::LoadPreparedAesKey(
                slotIndex,
                accessKey));

        uint8_t result[16];
        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::ComputeCtr(
                result, sizeof(result),
                slotIndex,
                cipher, sizeof(cipher),
                iv, sizeof(iv)));

        EXPECT_EQ(::std::memcmp(result, plain, sizeof(plain)), 0);
    }
} // NOLINT(impl/function_size)
