﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using Microsoft.VSSDK.Tools.VsIdeTesting;
using Nintendo.NintendoSdkVsExtension.Base;
using Nintendo.NintendoSdkVsExtension.VcAccessors;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml.Linq;

namespace Nintendo.NintendoSdkVsExtension.UnitTest.VcAccessorsTest
{
    [TestClass]
    [DeploymentItem(InIdeTestConstants.VsVersionSpecificTestDataSourceDirectory + @"OldProject.Generic", @"Data\OldProject.Generic")]
    public class VcProjectAccessorReadTestClass
    {
        public TestContext TestContext { get; set; }
        private static XDocument projectXml;

        [ClassInitialize]
        public static void ClassInitialize(TestContext context)
        {
            projectXml = XDocument.Load(@"Data\OldProject.Generic\OldProject\OldProject.vcxproj");
        }

        [ClassCleanup]
        public static void ClassCleanup()
        {
            projectXml = null;
        }

        [TestInitialize]
        public void TestInitialize()
        {
            VsTestUtil.EnsureOpenSolution(@"Data\OldProject.Generic\OldProject.sln");
        }

        private static VcProjectAccessor GetTargetProject()
        {
            return VsTestUtil.GetVcProjectOfName(@"OldProject\OldProject.vcxproj");
        }

        [TestMethod]
        [HostType("VS IDE")]
        public void TestProjectAccessor()
        {
            VcProjectAccessor project = GetTargetProject();
            var configuraionPairsFromXml = VsTestUtil.EnumerateProjectConfigurationFromXml(projectXml.Root).ToArray();

            Assert.AreEqual(Path.GetFullPath(@"Data\OldProject.Generic\OldProject\OldProject.vcxproj"), project.ProjectFilePath);

            CollectionAssert.AreEquivalent(
                configuraionPairsFromXml,
                project.Configurations.Select(x => x.ConfigurationPair).ToArray());

            foreach (var configurationPair in configuraionPairsFromXml)
            {
                var c = project.GetConfigurationAccessor(configurationPair);
                Assert.AreEqual(configurationPair, c.ConfigurationPair);
            }
        }

        [TestMethod]
        [HostType("VS IDE")]
        public void TestConfigurationAccessor()
        {
            VcProjectAccessor project = GetTargetProject();

            foreach (var c in project.Configurations)
            {
                Assert.AreEqual(c.ConfigurationPair, new ConfigurationPair(c.ConfigurationName, c.PlatformName));
                var propertySheets = c.PropertySheets;
                Assert.IsNotNull(propertySheets);

                Assert.AreEqual("Generic", c.GetEvaluatedPropertyValue("NintendoSdkSpec"));

                Assert.AreEqual(@"$(NINTENDO_SDK_ROOT)\", c.GetUnevaluatedPropertyValue("NintendoSdkRoot"));
                Assert.AreEqual(c.GetEvaluatedPropertyValue("NINTENDO_SDK_ROOT") + @"\", c.GetEvaluatedPropertyValue("NintendoSdkRoot"));
            }
        }

        [TestMethod]
        [HostType("VS IDE")]
        public void TestPropertySheetAccessor()
        {
            var project = VsTestUtil.GetVcProjectOfName(@"OldProject\OldProject.vcxproj");
            var projectRoot = Path.GetFullPath(@"Data\OldProject.Generic\OldProject\");
            foreach (var c in project.Configurations)
            {
                var propertySheets = c.PropertySheets;

                {
                    var propertySheetNames = propertySheets.Select(x => x.Name).ToArray();
                    CollectionAssert.Contains(propertySheetNames, "NintendoSdkVcProjectSettings");
                    CollectionAssert.Contains(propertySheetNames, "NintendoSdkSpec_Generic");
                    CollectionAssert.Contains(propertySheetNames, "NintendoSdkBuildType_" + c.ConfigurationName);
                }

                {
                    var propertySheetFiles = propertySheets.Select(x => x.PropertySheetFile).ToArray();
                    CollectionAssert.Contains(propertySheetFiles, Path.Combine(projectRoot, "NintendoSdkVcProjectSettings.props"));
                    CollectionAssert.Contains(propertySheetFiles, Path.Combine(projectRoot, "NintendoSdkSpec_Generic.props"));
                    CollectionAssert.Contains(propertySheetFiles, Path.Combine(projectRoot, "NintendoSdkBuildType_" + c.ConfigurationName + ".props"));
                }
            }
        }
    }
}
