﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <sstream>

#include <nn/mem/mem_StandardAllocator.h>

#include <nn/os/os_Tick.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>

// 文字列変換クラス
class Convert
{
public:
    // 任意の数値型から文字列型へ
    template <typename T>
    static std::string ToString(T inVal) NN_NOEXCEPT
    {
        std::ostringstream oss;
        oss << inVal;
        return oss.str();
    }

    // 文字列型からUInt32数値型へ
    static uint32_t ToUInt32(const std::string& inNumStr) NN_NOEXCEPT;

    // 文字列型からUInt64数値型へ
    static uint64_t ToUInt64(const std::string& inNumStr) NN_NOEXCEPT;

    // 文字列型からUINT8数値型へ
    static uint8_t ToUInt8(const std::string& inNumStr) NN_NOEXCEPT;

    // 数値を指定した桁の文字列にする(満たない桁は0埋めする)
    static std::string ToStringFillZero(int value, int digit) NN_NOEXCEPT;
};

// バージョン値のユーティリティクラス
struct VersionHolder
{
    uint16_t releaseVer;
    uint16_t privateVer;
    uint32_t unifiedVer;

    VersionHolder() : releaseVer(0), privateVer(0), unifiedVer(0) {}

    void Convert(const std::string& inRV, const std::string& inPV,
                 const std::string& inUV) NN_NOEXCEPT;
};

// ストップウォッチクラス
class Stopwatch
{
public:
    Stopwatch() : m_IsStarted(false)
    {
    }

    ~Stopwatch()
    {
    }

    void Start()
    {
        if (m_IsStarted == false)
        {
            m_IsStarted = true;
            m_Start = nn::os::GetSystemTick().ToTimeSpan();
        }
    }

    void Stop()
    {
        if (m_IsStarted == true)
        {
            m_Stop = nn::os::GetSystemTick().ToTimeSpan();
            m_IsStarted = false;
        }
    }

    int64_t GetMilliSecond()
    {
        if (m_IsStarted == true)
        {
            return -1;
        }

        return (m_Stop - m_Start).GetMilliSeconds();
    }

private:
    bool m_IsStarted;
    nn::TimeSpan m_Start;
    nn::TimeSpan m_Stop;
};

class TimeStamp
{
public:
    TimeStamp()
    {
    }

    ~TimeStamp()
    {
    }

    // カレンダータイムを取得する
    static nn::time::CalendarTime GetCalendarTime() NN_NOEXCEPT;
};

// 実質、ソフトウェアキーボード用のメモリ確保用クラス
class ApplicationHeap
{
    NN_DISALLOW_COPY(ApplicationHeap);
    NN_DISALLOW_MOVE(ApplicationHeap);

public:
    explicit ApplicationHeap(size_t size) NN_NOEXCEPT;
    virtual ~ApplicationHeap() NN_NOEXCEPT;

    void* Allocate(size_t size, size_t alignment) NN_NOEXCEPT;
    void* Allocate(size_t size) NN_NOEXCEPT;

    void Deallocate(void* ptr) NN_NOEXCEPT;

private:
    void Initialize(size_t size) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

private:
    void* m_pMemoryHeap;
    nn::mem::StandardAllocator m_ApplicationHeapHandle;
};
