﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/nifm/nifm_ApiNetworkConnection.h>

#include "TestAppSimple_NetworkScene.h"
#include "TestAppSimple_FsUtilities.h"
#include "TestAppSimple_EcServerAccessor.h"

namespace {
    const char* const RequestFlagFilePath = "Contents:/NetworkRequest.txt";

    const char* const SubmitStr = "A: Submit Network Request";
    const char* const CancelStr = "A: Cancel Network Request";
}

NetworkScene::NetworkScene() NN_NOEXCEPT
    : m_State(State_None), m_NetworkStatus(NetworkStatus_NotAvailable)
{
}

NetworkScene::~NetworkScene() NN_NOEXCEPT
{
}

void NetworkScene::InternalSetup() NN_NOEXCEPT
{
    {
        m_NetworkRequestRange.pos.x = 70.0f;
        m_NetworkRequestRange.pos.y = 480.0f;
        m_NetworkRequestRange.labelStr = SubmitStr;
    }

    if (fsutil::IsExistPath(RequestFlagFilePath) == true)
    {
        // 起動時にネットワーク接続要求を実施するフラグ用ファイルが存在すれば、
        // ネットワーク接続要求を出す (ひとまずファイルの存在有無のみを発動条件とする)
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_ConnectNetwork);
        this->UpdateIsNetworkAvailable();
    }
}

void NetworkScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_ProcessRequest;
        }
    }
}

void NetworkScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (m_NetworkRequestRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_ProcessRequest;
        }
    }
}

void NetworkScene::DoNetworkRequest() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    switch (m_NetworkStatus)
    {
    case NetworkStatus_NotAvailable:
        {
            EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_ConnectNetwork);
        }
        break;

    case NetworkStatus_Verifying:
    case NetworkStatus_Available:
        {
            nn::nifm::CancelNetworkRequest();
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    switch (m_NetworkStatus)
    {
    case NetworkStatus_NotAvailable:
        {
            m_NetworkStatus = NetworkStatus_Available;
            m_NetworkRequestRange.labelStr = SubmitStr;
            m_NetworkRequestRange.range.isSetting = false;
        }
        break;

    case NetworkStatus_Verifying:
    case NetworkStatus_Available:
        {
            m_NetworkStatus = NetworkStatus_NotAvailable;
            m_NetworkRequestRange.labelStr = CancelStr;
            m_NetworkRequestRange.range.isSetting = false;
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

void NetworkScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_ProcessRequest)
    {
        this->DoNetworkRequest();
    }
    this->UpdateIsNetworkAvailable();
    m_State = State_None;
}

void NetworkScene::UpdateIsNetworkAvailable() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    const int64_t UpdateIntervalMilliSeconds = 100;
    auto now = nn::os::GetSystemTick();
    auto elapsed = (now - m_LastUpdateTick).ToTimeSpan().GetMilliSeconds();
    if (elapsed > UpdateIntervalMilliSeconds)
    {
        if (nn::nifm::IsNetworkRequestOnHold())
        {
            if (m_NetworkStatus != NetworkStatus_Verifying)
            {
                m_NetworkStatus = NetworkStatus_Verifying;
                m_NetworkRequestRange.labelStr = CancelStr;
                m_NetworkRequestRange.range.isSetting = false;
            }
        }
        else if (nn::nifm::IsNetworkAvailable())
        {
            if (m_NetworkStatus != NetworkStatus_Available)
            {
                m_NetworkStatus = NetworkStatus_Available;
                m_NetworkRequestRange.labelStr = CancelStr;
                m_NetworkRequestRange.range.isSetting = false;
                EcServerAccessor::GetInstance().ClearNetworkResult();
            }
        }
        else
        {
            if (m_NetworkStatus != NetworkStatus_NotAvailable)
            {
                m_NetworkStatus = NetworkStatus_NotAvailable;
                m_NetworkRequestRange.labelStr = SubmitStr;
                m_NetworkRequestRange.range.isSetting = false;
            }
        }
        m_LastUpdateTick = now;
    }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

void NetworkScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[Network]");

    writer->SetScale(1.8f, 1.8f);

    writer->SetCursor(60.0f, 120.0f);
    writer->Print("NetworkStatus : ");
    switch (m_NetworkStatus)
    {
    case NetworkStatus_NotAvailable:
        {
            writer->SetTextColor(Red);
            auto result = EcServerAccessor::GetInstance().GetNetworkResult();
            if (result.IsFailure())
            {
                writer->Print("Not Available ( 0x%08x )", result.GetInnerValueForDebug());
            }
            else
            {
                writer->Print("Not Available");
            }
        }
        break;

    case NetworkStatus_Verifying:
        {
            writer->SetTextColor(Yellow);
            writer->Print("Verifying ...");
        }
        break;

    case NetworkStatus_Available:
        {
            writer->SetTextColor(Green);
            writer->Print("Available");
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // 目立つように少し大きめに描画しておく
    writer->SetScale(2.5f, 2.5f);
    this->WriteTouchRange(writer, &m_NetworkRequestRange);
}

