﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_Rom.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/util/util_ScopeExit.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include "TestAppSimple_GraphicSystem.h"
#include "TestAppSimple_Application.h"

namespace {
    const size_t MemoryHeapSize = ApplicationHeapSize * 2;
    const size_t DefaultHeapSize = MemoryHeapSize;

    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return std::malloc(size);
}

void Deallocate(void* addr, size_t) NN_NOEXCEPT
{
    std::free(addr);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // no namespace

extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(MemoryHeapSize));
    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, DefaultHeapSize));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), DefaultHeapSize);
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <filesystem>
#include <nn/nn_Windows.h>
#endif

extern "C" void nnMain()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    const char* ROM_ROOT_NAME = "Contents";
    // nn::fs の初期化
    size_t mountRomCacheBufferSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    void* mountRomCacheBuffer = Allocate(mountRomCacheBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(mountRomCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        Deallocate(mountRomCacheBuffer, mountRomCacheBufferSize);
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom(ROM_ROOT_NAME, mountRomCacheBuffer, mountRomCacheBufferSize));

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::InitializeGraphics(std::malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    Application application;
    application.Initialize();

    application.Run();

    application.Finalize();

    nn::fs::Unmount(ROM_ROOT_NAME);

    return;
}
