﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx.h>
#include <nn/vi.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Matrix.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <filesystem>
#include <nn/nn_Windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/init.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#include "TestAppSimple_GraphicSystem.h"

static bool g_Initialized = false;

//------------------------------------------------------------------------------
//
// ビットマップフォントをディスクから読み込んで、表示する最も基本的なデモです。
//
// フォントモジュールには、
//   サイズスケール、行間隔指定、文字間隔指定、カラー変更、
//   カラーグラデーション、斜体描画、カゲ描画、
//   フチ付フォントのフチ表示ON/OFF
//   ...など多彩な機能があります。
//
//   詳しくは、API リファレンスをごらんください。
//   また、レイアウトライブラリのソースコードもフォントライブラリのコードサンプルとして
//   参考にできます（font::で grep すると該当コードにたどり着けます）。
//
// なお、ビットマップフォントの生成には、専用のコンバータを利用します。
// (Tools/Graphics/FontConverter 以下)
//

//------------------------------------------------------------------------------
//   前方宣言
//------------------------------------------------------------------------------

static void Finalize_();

//------------------------------------------------------------------------------
//   定数定義
//------------------------------------------------------------------------------

static const uint32_t CharCount = 1023;
static const size_t MemoryPoolSize          = 32 * 1024 * 1024;  // 64k の倍数である必要がある
static const size_t MemoryPoolSizeInvisible = 16 * 1024 * 1024;  // 64k の倍数である必要がある
static const size_t CommandBufferControlMemorySize = 256 * 1024;

//------------------------------------------------------------------------------
//   変数定義
//------------------------------------------------------------------------------

// メモリヒープ
static nn::util::BytePtr g_pMemoryHeap(NULL);
static nn::lmem::HeapHandle g_ApplicationHeapHandle;
// コマンドバッファコントロール用メモリ
static void* g_pCommandBufferControlMemory = NULL;
// メモリプール
static nn::gfx::MemoryPool             g_MemoryPool;
static nn::util::BytePtr               g_pMemoryPool( NULL );
static nn::util::BytePtr               g_pMemoryPoolCurrent( NULL );
static nn::util::BytePtr               g_pMemoryPoolCommandBufferStart( NULL );
static nn::util::BytePtr               g_pMemoryPoolCommandBufferEnd( NULL );

static nn::gfx::MemoryPool             g_MemoryPoolInvisible;
static nn::util::BytePtr               g_pMemoryPoolInvisible( NULL );
static nn::util::BytePtr               g_pMemoryPoolInvisibleCurrent( NULL );

// gfx デバイス
static nn::gfx::Device g_Device;

static nn::vi::Display* g_pDisplay;
static nn::vi::Layer* g_pLayer;

static int g_ScreenWidth;
static int g_ScreenHeight;

// gfx スワップチェーン
static nn::gfx::SwapChain g_SwapChain;

// gfx キュー
static nn::gfx::Queue g_Queue;

// gfx フェンス
static nn::gfx::Fence g_Fence;

// gfx コマンドバッファ
static nn::gfx::CommandBuffer g_CommandBuffer;

// gfx カラーバッファ用テクスチャ
static nn::gfx::Texture g_ColorBuffer;

// gfx カラーターゲットビュー
static nn::gfx::ColorTargetView g_ColorBufferView;

// gfx ビューポートシザー
static nn::gfx::ViewportScissorState g_ViewportScissorState;

static int g_TextureDescriptorBaseIndex = 0;
static int g_SamplerDescriptorBaseIndex = 0;


//------------------------------------------------------------------------------
// DescriptorPool 関連
const  int                      DescriptorPoolStartSlot = 256;
nn::gfx::DescriptorPool         g_TextureDescriptorPool;
nn::gfx::DescriptorPool         g_SamplerDescriptorPool;
int                             g_SamplerSlotCount = DescriptorPoolStartSlot;
int                             g_TextureSlotCount = DescriptorPoolStartSlot;

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する(アライン指定あり)
//------------------------------------------------------------------------------
static void* AllocateFromApplicationHeap(size_t size, size_t alignment)
{
    return nn::lmem::AllocateFromExpHeap(g_ApplicationHeapHandle, size, static_cast<int>(alignment));
}

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する(アライン指定あり)
//------------------------------------------------------------------------------
static void FreeApplicationHeap(void* pAddr)
{
    nn::lmem::FreeToExpHeap(g_ApplicationHeapHandle, pAddr);
}

//------------------------------------------------------------------------------

void InitializeMemoryPool()
{
    // 単純さのため全体で 2種類のメモリプールを使います
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
        info.SetPoolMemory( g_pMemoryPool.Get(), MemoryPoolSize );

        g_MemoryPool.Initialize( &g_Device, info );
    }

    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);
        info.SetPoolMemory( g_pMemoryPoolInvisible.Get(), MemoryPoolSizeInvisible );

        g_MemoryPoolInvisible.Initialize( &g_Device, info );
    }
}

//-----------------------------------------------------------------------------

static void InitializeGfxDescriptorPool_()
{
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
        info.SetSlotCount(g_TextureDescriptorBaseIndex + 1);

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );

        g_TextureDescriptorPool.Initialize( &g_Device, info, &g_MemoryPool, g_pMemoryPool.Distance( g_pMemoryPoolCurrent.Get() ), size );
        g_pMemoryPoolCurrent.Advance( static_cast< ptrdiff_t >( size ) );
    }

    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler);
        info.SetSlotCount(g_SamplerDescriptorBaseIndex + 1);

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );

        g_SamplerDescriptorPool.Initialize( &g_Device, info, &g_MemoryPool, g_pMemoryPool.Distance( g_pMemoryPoolCurrent.Get() ), size );
        g_pMemoryPoolCurrent.Advance( static_cast< ptrdiff_t >( size ) );
    }
}

//------------------------------------------------------------------------------

bool AllocateSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(pUserData);

    const int slot = g_TextureSlotCount;
    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView( slot, &textureView);
        g_TextureSlotCount++;
    }
    g_TextureDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.GetDescriptorSlot(pDstSlot, slot);

    return true;
}

//------------------------------------------------------------------------------

bool AllocateSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(pUserData);

    const int slot = g_SamplerSlotCount;
    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler( slot, &sampler);
        g_SamplerSlotCount++;
    }
    g_SamplerDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.GetDescriptorSlot(pDstSlot, slot);

    return true;
}

//------------------------------------------------------------------------------

void FreeSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(pUserData);
    NN_UNUSED(pDstSlot);
    NN_UNUSED(textureView);
}

//------------------------------------------------------------------------------

void FreeSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(pUserData);
    NN_UNUSED(pDstSlot);
    NN_UNUSED(sampler);
}

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
//------------------------------------------------------------------------------
static void InitializeGfxWin_(int width, int height)
{
    // 2015/4/1現在、gfxではウインドウサイズが設定出来ないようなので
    // Win32APIを呼び出して強制的にウインドウサイズを設定
    // 将来的には削除予定

    nn::vi::NativeWindowHandle wndHandle = nullptr;
    auto result = nn::vi::GetNativeWindow(&wndHandle, g_pLayer);
    if (result.IsSuccess() && wndHandle != nullptr)
    {
        HWND hwnd = reinterpret_cast<HWND>(wndHandle);
        RECT rw, rc;
        ::GetWindowRect(hwnd, &rw);
        ::GetClientRect(hwnd, &rc);

        int new_width = (rw.right - rw.left) - (rc.right - rc.left) + width;
        int new_height = (rw.bottom - rw.top) - (rc.bottom - rc.top) + height;

        SetWindowPos(hwnd, nullptr, 0, 0, new_width, new_height, SWP_NOMOVE | SWP_NOZORDER);
        SetWindowText(hwnd, L"TestAppSimple");
    }
}
#endif

//-----------------------------------------------------------------------------
// コマンドバッファーのコールバック処理
//-----------------------------------------------------------------------------
void OutOfCommandMemoryEventCallback( nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& arg )
{
    const size_t AddedBufferSize = 1024;
    NN_ASSERT( AddedBufferSize >= arg.minRequiredSize );
    NN_UNUSED( arg );
    g_pMemoryPoolCommandBufferEnd.AlignUp( nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device) );
    pCommandBuffer->AddCommandMemory( &g_MemoryPool, g_pMemoryPool.Distance( g_pMemoryPoolCommandBufferEnd.Get() ), AddedBufferSize );
    g_pMemoryPoolCommandBufferEnd.Advance( AddedBufferSize );
}

void OutOfControlMemoryEventCallback( nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& arg )
{
    const size_t AddedBufferSize = 256;
    NN_ASSERT( AddedBufferSize >= arg.minRequiredSize );
    NN_UNUSED( arg );
    pCommandBuffer->AddControlMemory(AllocateFromApplicationHeap(AddedBufferSize, nn::gfx::CommandBuffer::GetControlMemoryAlignment(&g_Device)), AddedBufferSize);
}

//------------------------------------------------------------------------------
// デモ共有の初期化
//------------------------------------------------------------------------------
static void
InitDemoCommon_()
{
    // デバイスを初期化
    {
        nn::gfx::Device::InfoType info;
        info.SetDefault();
        g_Device.Initialize( info );
    }

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(g_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex);
#endif

    // メモリプールの初期化
    {
        nn::gfx::MemoryPoolInfo info;
        info.SetDefault();

        const size_t MemoryPoolAligment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
        g_pMemoryPool.Reset(AllocateFromApplicationHeap(MemoryPoolSize, MemoryPoolAligment));
        NN_SDK_ASSERT(g_pMemoryPool.Get());
        g_pMemoryPoolCurrent = g_pMemoryPool;

        g_pMemoryPoolInvisible.Reset(AllocateFromApplicationHeap(MemoryPoolSizeInvisible, MemoryPoolAligment));
        NN_SDK_ASSERT(g_pMemoryPoolInvisible.Get());
        g_pMemoryPoolInvisibleCurrent = g_pMemoryPoolInvisible;

        InitializeMemoryPool();
    }

    size_t memoryPoolInvisibleOffset = 0;

    // ディスプレイを取得
    {
        auto result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
        NN_ASSERT(result.IsSuccess());

        // (SIGLO-47445) 非推奨API呼び出しを削除
        // とりあえず固定値を設定しておく
        g_ScreenWidth = 1280;
        g_ScreenHeight = 720;

        result = nn::vi::CreateLayer(&g_pLayer, g_pDisplay);
        NN_ASSERT(result.IsSuccess());

        result = nn::vi::SetLayerScalingMode(g_pLayer, nn::vi::ScalingMode_FitToLayer);
        NN_ASSERT(result.IsSuccess());
    }

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    {
        InitializeGfxWin_(g_ScreenWidth, g_ScreenHeight);
    }
#endif

    // スワップチェーンを初期化
    {
        nn::gfx::SwapChain::InfoType info;
        info.SetDefault();
        info.SetLayer( g_pLayer );
        info.SetWidth( g_ScreenWidth );
        info.SetHeight( g_ScreenHeight );
        info.SetBufferCount(2);

        if( NN_STATIC_CONDITION( nn::gfx::SwapChain::IsMemoryPoolRequired ) )
        {
            memoryPoolInvisibleOffset = nn::util::align_up(memoryPoolInvisibleOffset, nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info));

            size_t size = g_SwapChain.CalculateScanBufferSize( &g_Device, info );
            g_pMemoryPoolInvisibleCurrent.AlignUp( nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info) );
            g_SwapChain.Initialize( &g_Device, info, &g_MemoryPoolInvisible, memoryPoolInvisibleOffset, size );
            g_pMemoryPoolInvisibleCurrent.Advance( static_cast< ptrdiff_t >( size ) );

            memoryPoolInvisibleOffset += size;
        }
        else
        {
            g_SwapChain.Initialize( &g_Device, info, NULL, 0, 0 );
        }
    }

    // キューを初期化
    {
        nn::gfx::Queue::InfoType info;
        info.SetDefault();
        info.SetCapability( nn::gfx::QueueCapability_Graphics );
        g_Queue.Initialize( &g_Device, info );
    }

    // フェンスを初期化
    {
        nn::gfx::Fence::InfoType info;
        info.SetDefault();
        g_Fence.Initialize( &g_Device, info );
    }

    // カラーバッファ用テクスチャを初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth( g_ScreenWidth );
        info.SetHeight( g_ScreenHeight );
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_ColorBuffer );
        info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
        info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
        info.SetMipCount(1);
        info.SetDepth(1);

        if( NN_STATIC_CONDITION( nn::gfx::Texture::IsMemoryPoolRequired ) )
        {
            memoryPoolInvisibleOffset = nn::util::align_up(memoryPoolInvisibleOffset, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));

            g_pMemoryPoolInvisibleCurrent.AlignUp( nn::gfx::Texture::CalculateMipDataAlignment( &g_Device, info ) );

            size_t size = nn::gfx::Texture::CalculateMipDataSize( &g_Device, info );
            g_ColorBuffer.Initialize( &g_Device, info, &g_MemoryPoolInvisible, memoryPoolInvisibleOffset, size );

            g_pMemoryPoolInvisibleCurrent.Advance( static_cast< ptrdiff_t >( size ) );

            memoryPoolInvisibleOffset += size;
        }
        else
        {
            g_ColorBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
        }
    }

    // カラーバッファビューを初期化
    {
        nn::gfx::ColorTargetView::InfoType info;
        info.SetDefault();
        info.SetImageDimension( nn::gfx::ImageDimension_2d );
        info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
        info.SetTexturePtr( &g_ColorBuffer );
        g_ColorBufferView.Initialize( &g_Device, info );
    }

    // ディスクリプションプールを初期化
    InitializeGfxDescriptorPool_();

    // コマンドバッファを初期化
    {
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        g_CommandBuffer.Initialize( &g_Device, info );
        g_CommandBuffer.SetOutOfCommandMemoryEventCallback( OutOfCommandMemoryEventCallback );
        g_CommandBuffer.SetOutOfControlMemoryEventCallback( OutOfControlMemoryEventCallback );

        // バッファ（メモリプールより確保）
        g_pMemoryPoolCurrent.AlignUp( nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
        g_pMemoryPoolCommandBufferStart = g_pMemoryPoolCurrent;
        g_pMemoryPoolCommandBufferEnd = g_pMemoryPoolCommandBufferStart;

        // コントロールメモリ（通常メモリより確保）
        g_pCommandBufferControlMemory = AllocateFromApplicationHeap(CommandBufferControlMemorySize, 1024 * 4);
    }

    // ビューポートシザーを初期化
    {
        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled(true);
        nn::gfx::ViewportStateInfo viewportInfo;
        {
            viewportInfo.SetDefault();
            viewportInfo.SetWidth(static_cast< float >(g_ScreenWidth));
            viewportInfo.SetHeight(static_cast< float >(g_ScreenHeight));
        }
        nn::gfx::ScissorStateInfo scissorInfo;
        {
            scissorInfo.SetDefault();
            scissorInfo.SetWidth(g_ScreenWidth);
            scissorInfo.SetHeight(g_ScreenHeight);
        }
        info.SetViewportStateInfoArray(&viewportInfo, 1);
        info.SetScissorStateInfoArray(&scissorInfo, 1);
        g_ViewportScissorState.Initialize(&g_Device, info);
    }

} // NOLINT (readability/fn_size)

//------------------------------------------------------------------------------
// 初期化処理の呼び出し
//------------------------------------------------------------------------------
static void
Initialize_()
{
    NN_SDK_ASSERT(! g_Initialized);

    // メモリヒープの初期化
    g_pMemoryHeap.Reset(new uint8_t[ApplicationHeapSize]);
    NN_SDK_ASSERT(g_pMemoryHeap.Get());
    g_ApplicationHeapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(g_pMemoryHeap.Get()), ApplicationHeapSize, nn::lmem::CreationOption_NoOption);

    // デモシステム共有の初期化
    InitDemoCommon_();

    g_Initialized = true;
}

//------------------------------------------------------------------------------
// デモ共有の終了処理
//------------------------------------------------------------------------------
static void
FinarizeDemoCommon_()
{
    // 各オブジェクトを破棄
    g_ViewportScissorState.Finalize( &g_Device );
    g_ColorBufferView.Finalize( &g_Device );
    g_ColorBuffer.Finalize( &g_Device );
    g_CommandBuffer.Finalize( &g_Device );
    g_Fence.Finalize( &g_Device );
    g_Queue.Finalize( &g_Device );
    g_SwapChain.Finalize( &g_Device );

    g_TextureDescriptorPool.Finalize( &g_Device );
    g_SamplerDescriptorPool.Finalize( &g_Device );

    g_MemoryPool.Finalize(&g_Device);
    FreeApplicationHeap(g_pMemoryPool.Get());
    g_MemoryPoolInvisible.Finalize(&g_Device);
    FreeApplicationHeap(g_pMemoryPoolInvisible.Get());

    g_Device.Finalize();

    nn::vi::DestroyLayer(g_pLayer);
    nn::vi::CloseDisplay(g_pDisplay);
}

//------------------------------------------------------------------------------
// 終了処理の呼び出し
//------------------------------------------------------------------------------
static void
Finalize_()
{
    NN_SDK_ASSERT(g_Initialized);

    // デモ共有の終了処理
    FinarizeDemoCommon_();

    // アロケータの終了処理
    nn::lmem::DestroyExpHeap(g_ApplicationHeapHandle);

    delete[] reinterpret_cast<uint8_t*>(g_pMemoryHeap.Get());

    g_Initialized = false;
}

bool IsGraphicSystemInitialized() NN_NOEXCEPT
{
    return g_Initialized;
}

void InitializeGraphicSystem() NN_NOEXCEPT
{
    nn::gfx::Initialize();
    nn::vi::Initialize();

    Initialize_();
}

void FinalizeGraphicSystem() NN_NOEXCEPT
{
    Finalize_();

    nn::vi::Finalize();
    nn::gfx::Finalize();
}

nn::util::BytePtr g_DebugFontHeap(nullptr);

void InitializeDebugFontWriter(nn::gfx::util::DebugFontTextWriter* inWriter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(inWriter);

    // デバッグフォント初期化
    const uint32_t charCountMax = CharCount + 1;
    nn::gfx::util::DebugFontTextWriterInfo debugWriterInfo;
    debugWriterInfo.SetDefault();
    debugWriterInfo.SetCharCountMax(charCountMax);
    debugWriterInfo.SetUserMemoryPoolEnabled(false);

    const size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
        &g_Device,
        debugWriterInfo
        );

    g_DebugFontHeap.Reset(new uint8_t[debugFontHeapSize]);

    inWriter->Initialize(
        &g_Device,
        debugWriterInfo,
        g_DebugFontHeap.Get(),
        debugFontHeapSize,
        nullptr,
        0,
        0
        );

    inWriter->SetDisplayWidth(g_ScreenWidth);
    inWriter->SetDisplayHeight(g_ScreenHeight);
    inWriter->SetTextureDescriptor(&g_TextureDescriptorPool, g_TextureDescriptorBaseIndex);
    inWriter->SetSamplerDescriptor(&g_SamplerDescriptorPool, g_SamplerDescriptorBaseIndex);
}

void FinalizeDebugFontWriter(nn::gfx::util::DebugFontTextWriter* inWriter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(inWriter);

    // デバッグフォント終了
    inWriter->Finalize();
    delete[] reinterpret_cast<uint8_t*>(g_DebugFontHeap.Get());
    g_DebugFontHeap.Reset(nullptr);
}

void ProcFrame(nn::gfx::util::DebugFontTextWriter* inWriter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(inWriter);

    // コマンドを生成
    g_CommandBuffer.Reset();

    // コマンドバッファ用のメモリを設定します
    {
        g_pMemoryPoolCommandBufferEnd = g_pMemoryPoolCommandBufferStart;

        const size_t BufferSize = 1024 * 1024 * 16;

        g_pMemoryPoolCommandBufferEnd.AlignUp(nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
        g_CommandBuffer.AddCommandMemory(&g_MemoryPool, g_pMemoryPool.Distance(g_pMemoryPoolCommandBufferEnd.Get()), BufferSize);
        g_pMemoryPoolCommandBufferEnd.Advance(BufferSize);

        g_CommandBuffer.AddControlMemory(g_pCommandBufferControlMemory, CommandBufferControlMemorySize);
    }

    g_CommandBuffer.Begin();
    {
        g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);

        // ディスクリプタプールをセットする。
        g_TextureSlotCount = DescriptorPoolStartSlot;
        g_SamplerSlotCount = DescriptorPoolStartSlot;
        g_CommandBuffer.SetDescriptorPool(&g_TextureDescriptorPool);
        g_CommandBuffer.SetDescriptorPool(&g_SamplerDescriptorPool);

        // 描画ステートの設定
        //SetupDrawState_();
        g_CommandBuffer.SetViewportScissorState(&g_ViewportScissorState);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();
#else
        nn::gfx::ColorTargetView* pTarget = &g_ColorBufferView;
#endif
        g_CommandBuffer.ClearColor(pTarget, 0.1f, 0.1f, 0.1f, 1.0f, NULL);
        g_CommandBuffer.SetRenderTargets(1, &pTarget, NULL);

        // 描画
        inWriter->Draw(&g_CommandBuffer);
    }
    g_CommandBuffer.End();

    // コマンドの実行
    g_Queue.ExecuteCommand(&g_CommandBuffer, &g_Fence);

    // 結果の表示
#if !defined(NN_BUILD_CONFIG_OS_HORIZON)
    g_Queue.CopyToScanBuffer(&g_SwapChain, &g_ColorBufferView);
#endif
    g_Queue.Flush();
    g_Fence.Sync(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / 60));

    g_Queue.Present(&g_SwapChain, 1 /*presentInterval*/);
}

