﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <string>
#include <unordered_map>
#include <vector>

#include <nn/ec/ec_ConsumableServiceItemTypes.h>

#include "TestAppSimple_EcServiceConfig.h"

struct EcOcsiRightInfo
{
    bool isSelected;
    bool isConsumed;
    uint32_t rightHash;
    nn::ec::ConsumableServiceItemRightData rightData;
};

static const int EcOcsiRightCountMax = 40;
NN_STATIC_ASSERT(EcOcsiRightCountMax > nn::ec::ConsumableServiceItemRightDataCountMax);

class EcOcsiRightList
{
    NN_DISALLOW_COPY( EcOcsiRightList );
    NN_DISALLOW_MOVE( EcOcsiRightList );
public:
    EcOcsiRightInfo infos[EcOcsiRightCountMax];
    int count;

    EcOcsiRightList() NN_NOEXCEPT {}
    ~EcOcsiRightList() NN_NOEXCEPT {}

    void Cleanup() NN_NOEXCEPT;
    void CheckAll(bool value) NN_NOEXCEPT;
    bool Switch(int index) NN_NOEXCEPT;

    bool IsValid() const NN_NOEXCEPT;
    int GetSelectedCount() const NN_NOEXCEPT;

    void PrintLog() const NN_NOEXCEPT;
};

class EcConsumableItemRegularPrice
{
    NN_DISALLOW_COPY( EcConsumableItemRegularPrice );
    NN_DISALLOW_MOVE( EcConsumableItemRegularPrice );
public:
    std::string currency;
    std::string formattedValue;
    std::string rawValue;

    EcConsumableItemRegularPrice() NN_NOEXCEPT {}
    ~EcConsumableItemRegularPrice() NN_NOEXCEPT
    {
        TESTAPP_EC_DEBUG_LOG("%s {%s, %s, %s}\n", __FUNCTION__, currency.c_str(), formattedValue.c_str(), rawValue.c_str());
    }
};

class EcConsumableItemDiscountPrice : public EcConsumableItemRegularPrice
{
    NN_DISALLOW_COPY( EcConsumableItemDiscountPrice );
    NN_DISALLOW_MOVE( EcConsumableItemDiscountPrice );
public:
    std::string startDatetime;
    std::string endDatetime;

    EcConsumableItemDiscountPrice() NN_NOEXCEPT {}
    ~EcConsumableItemDiscountPrice() NN_NOEXCEPT
    {
        TESTAPP_EC_DEBUG_LOG("%s {%s, %s}\n", __FUNCTION__, startDatetime.c_str(), endDatetime.c_str());
    }
};

class EcConsumableItemPrice
{
    NN_DISALLOW_COPY( EcConsumableItemPrice );
    NN_DISALLOW_MOVE( EcConsumableItemPrice );
public:
    std::string salesStatus;
    std::unique_ptr<EcConsumableItemRegularPrice> regularPrice;
    std::unique_ptr<EcConsumableItemDiscountPrice> discountPrice;

    EcConsumableItemPrice() NN_NOEXCEPT {}
    ~EcConsumableItemPrice() NN_NOEXCEPT
    {
        TESTAPP_EC_DEBUG_LOG("%s {%s}\n", __FUNCTION__, salesStatus.c_str());
    }
};

class EcConsumableItem
{
    NN_DISALLOW_COPY( EcConsumableItem );
    NN_DISALLOW_MOVE( EcConsumableItem );
public:
    std::string itemId;
    std::string name;
    int64_t nsUid;
    std::unique_ptr<EcConsumableItemPrice> price;

    EcConsumableItem() NN_NOEXCEPT : nsUid(0) {}
    ~EcConsumableItem() NN_NOEXCEPT
    {
        TESTAPP_EC_DEBUG_LOG("%s {%s, %s, 0x%016llx}\n", __FUNCTION__, itemId.c_str(), name.c_str(), nsUid);
    }
};

class EcConsumableItemResponse
{
    NN_DISALLOW_COPY( EcConsumableItemResponse );
    NN_DISALLOW_MOVE( EcConsumableItemResponse );
public:
    std::string language;
    int64_t length;
    int64_t offset;
    int64_t total;
    std::vector<std::unique_ptr<EcConsumableItem>> consumableItems;

    EcConsumableItemResponse() NN_NOEXCEPT : length(0), offset(0), total(0) {}
    ~EcConsumableItemResponse() NN_NOEXCEPT
    {
        TESTAPP_EC_DEBUG_LOG("%s {%s, %lld, %lld, %lld}\n", __FUNCTION__, language.c_str(), length, offset, total);
    }
};

class EcConsumable
{
    NN_DISALLOW_COPY( EcConsumable );
    NN_DISALLOW_MOVE( EcConsumable );
public:
    std::string consumableId;
    std::string name;
    std::string description;
    std::string disclaimer;

    EcConsumable() NN_NOEXCEPT {}
    ~EcConsumable() NN_NOEXCEPT
    {
        TESTAPP_EC_DEBUG_LOG("%s {%s, %s}\n", __FUNCTION__, consumableId.c_str(), name.c_str());
    }
};

class EcConsumableResponse
{
    NN_DISALLOW_COPY( EcConsumableResponse );
    NN_DISALLOW_MOVE( EcConsumableResponse );
public:
    std::string language;
    std::vector<std::unique_ptr<EcConsumable>> consumables;

    EcConsumableResponse() NN_NOEXCEPT {}
    ~EcConsumableResponse() NN_NOEXCEPT
    {
        TESTAPP_EC_DEBUG_LOG("%s {%s}\n", __FUNCTION__, language.c_str());
    }
};

class EcCatalog
{
    NN_DISALLOW_COPY( EcCatalog );
    NN_DISALLOW_MOVE( EcCatalog );
public:
    std::unique_ptr<EcConsumableResponse> list;
    std::unordered_map<std::string, std::unique_ptr<EcConsumableItemResponse>> map;
    int selectedConsumableIndex;
    int selectedConsumableItemIndex;

    EcCatalog() NN_NOEXCEPT : selectedConsumableIndex(-1), selectedConsumableItemIndex(-1) {}
    ~EcCatalog() NN_NOEXCEPT {}

    void Cleanup() NN_NOEXCEPT;
    bool SelectConsumable(int index) NN_NOEXCEPT;
    bool IsConsumableSelected() const NN_NOEXCEPT;
    bool SelectConsumableItem(int index) NN_NOEXCEPT;
    bool IsConsumableItemSelected() const NN_NOEXCEPT;

    int GetConsumableCount() const NN_NOEXCEPT;
    bool HasConsumable(int index) const NN_NOEXCEPT;
    const EcConsumable* GetConsumable(int index) const NN_NOEXCEPT;
    const EcConsumable* GetSelectedConsumable() const NN_NOEXCEPT;

    bool IsConsumableItemInquired() const NN_NOEXCEPT;
    int GetConsumableItemCount() const NN_NOEXCEPT;
    bool HasConsumableItem(int index) const NN_NOEXCEPT;
    const EcConsumableItem* GetConsumableItem(int index) const NN_NOEXCEPT;
    const EcConsumableItem* GetSelectedConsumableItem() const NN_NOEXCEPT;

    const EcConsumableItemResponse* FindConsumableItemResponse() const NN_NOEXCEPT;
};
