﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>

#include <nn/nn_Assert.h>
#include <nn/util/util_StringUtil.h>

#include "TestAppSimple_EcServiceSceneCommon.h"
#include "TestAppSimple_EcServerAccessor.h"

namespace {

    // ラベル書式
    const char NameFormat[] = "Name: \"%s\"";
    const char UidFormat[] = "UID: %08x_%08x_%08x_%08x";
    const char NsaIdFormat[] = "NSAID: %08x_%08x";

    // ラベルバッファ
    char g_NameLabel[48];   // 接頭部(6) + ニックネーム最大長(32) + ダブルクォーテーション(2) + 終端(1)
    char g_UidLabel[48];    // 接頭部(5) + UID(32) + 区切り(3) + 終端(1)
    char g_NsaIdLabel[32];  // 接頭部(7) + NSAID(16) + 区切り(1) + 終端(1)
}

void EcServiceSceneCommon::UpdaetAccount() NN_NOEXCEPT
{
    if (EcServerAccessor::GetInstance().IsUserValid())
    {
        nn::Bit32 value[4];

        auto nickname =  EcServerAccessor::GetInstance().GetNickname();
        NN_ABORT_UNLESS(nn::util::SNPrintf(g_NameLabel, sizeof(g_NameLabel), NameFormat, nickname.name) < sizeof(g_NameLabel));

        auto uid =  EcServerAccessor::GetInstance().GetUid();
        value[0] = static_cast<nn::Bit32>(uid._data[0] >> 32);
        value[1] = static_cast<nn::Bit32>(uid._data[0]);
        value[2] = static_cast<nn::Bit32>(uid._data[1] >> 32);
        value[3] = static_cast<nn::Bit32>(uid._data[1]);
        NN_ABORT_UNLESS(nn::util::SNPrintf(g_UidLabel, sizeof(g_UidLabel), UidFormat, value[0], value[1], value[2], value[3]) < sizeof(g_UidLabel));

        auto nsaid = EcServerAccessor::GetInstance().GetNsaId();
        value[0] = static_cast<nn::Bit32>(nsaid.id >> 32);
        value[1] = static_cast<nn::Bit32>(nsaid.id);
        NN_ABORT_UNLESS(nn::util::SNPrintf(g_NsaIdLabel, sizeof(g_NsaIdLabel), NsaIdFormat, value[0], value[1]) < sizeof(g_NsaIdLabel));
    }
    else
    {
        g_NameLabel[0] = '\0';
        g_UidLabel[0] = '\0';
        g_NsaIdLabel[0] = '\0';
    }
}

EcServiceSceneCommon::EcServiceSceneCommon() NN_NOEXCEPT
{
}

EcServiceSceneCommon::~EcServiceSceneCommon() NN_NOEXCEPT
{
}

void EcServiceSceneCommon::SetEcServiceHeader(Position* line, const char* name) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(line);
    NN_SDK_ASSERT_NOT_NULL(name);
    *line = BaseLine;

    char buff[32];
    NN_ABORT_UNLESS(nn::util::SNPrintf(buff, sizeof(buff), "[%s]", name) < sizeof(buff));
    m_PageLabel.base = *line;
    m_PageLabel.prefix = buff;
    line->y += NewLine; // [PageName]

    m_AccountLabel.base = *line;
    m_AccountLabel.prefix = "Account";
    m_AccountLabel.range.labelStr = "Y: Select";
    line->y += NewLine; // Account
    line->y += NewLine; // - Name
    line->y += NewLine; // - UID,NSAID
}

void EcServiceSceneCommon::DrawEcServiceHeader(Position* line, nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);
    NN_ABORT_UNLESS_NOT_NULL(line);

    *line = BaseLine;
    SetDefaultScale(writer);

    // 状態
    WriteLabel(writer, &m_PageLabel);
    if (EcServerAccessor::GetInstance().IsProcessing())
    {
        writer->SetTextColor(Green);
        writer->SetCursor(m_PageLabel.range.endPos.x + Margin, line->y);
        writer->Print("Processing: %s...", EcServerAccessor::GetInstance().GetCurrentStatus());
        writer->SetTextColor(White);
    }
    else if (EcServerAccessor::GetInstance().IsFailed())
    {
        writer->SetTextColor(Red);
        writer->SetCursor(m_PageLabel.range.endPos.x + Margin, line->y);
        writer->Print(EcServerAccessor::GetInstance().GetErrorMessage());
        writer->SetTextColor(White);

        if (EcServerAccessor::GetInstance().IsErrorViewable())
        {
            writer->SetCursor(920.0f + Margin, line->y);
            writer->Print("StickR Push: ShowError");
        }
    }
    line->y += NewLine;

    // アカウント
    WriteLabel(writer, &m_AccountLabel);
    line->y += NewLine;
    if (EcServerAccessor::GetInstance().IsUserValid())
    {
        writer->SetCursor(line->x + Indent, line->y);
        writer->Print(g_NameLabel);
        line->y += NewLine;

        Position pos(line->x + Indent, line->y);
        writer->SetScale(1.0f, 1.0f);

        writer->SetCursor(pos.x, pos.y);
        writer->Print(g_UidLabel);

        writer->SetCursor(pos.x + 450.0f, line->y);
        writer->Print(g_NsaIdLabel);

        SetDefaultScale(writer);
        line->y += NewLine;
    }
    else
    {
        writer->SetTextColor(Red);
        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("Unselected");

        writer->SetTextColor(White);
        line->y += NewLine;
        line->y += NewLine;
    }
}

void EcServiceSceneCommon::WriteLabel(nn::gfx::util::DebugFontTextWriter* writer, Label* label) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);
    NN_ABORT_UNLESS_NOT_NULL(label);

    // 範囲設定（※一度だけ設定）
    if (!label->range.isSetting)
    {
        label->range.isSetting = true;

        // label->base を基点に位置を確定していく
        float x = label->base.x;
        auto width = writer->CalculateStringWidth(label->prefix.c_str());

        label->range.isTouchable = (label->range.labelStr.length() > 0);
        if (label->range.isTouchable)
        {
            // タッチフィールドがある場合
            x += (width > 0) ? (width + Margin) : 0.0f;

            // 左括弧 x 位置
            label->xBracketL = x;
            x += writer->CalculateStringWidth(BracketLeft);

            // ラベル x 位置
            label->xRangeLabel = x;
            x += writer->CalculateStringWidth(label->range.labelStr.c_str());

            // 右括弧 x 位置
            label->xBracketR = x;
            x += writer->CalculateStringWidth(BracketRight);

            // タッチフィールド始点
            label->range.beginPos = Position(label->xBracketL, label->base.y - 2.0f);

            // タッチフィールド終点
            auto height = writer->CalculateStringHeight(label->range.labelStr.c_str());
            label->range.endPos = Position(x, label->base.y + height + 2.0f);
        }
        else
        {
            // タッチフィールドがない場合、接頭辞の左端を設定
            x += width;
            label->xBracketL = x;
            label->xRangeLabel = x;
            label->xBracketR = x;
            label->range.beginPos = Position(x, label->base.y);
            label->range.endPos = Position(x, label->base.y);
        }
    }

    // 接頭辞
    writer->SetTextColor(White);
    writer->SetCursor(label->base.x, label->base.y);
    writer->Print(label->prefix.c_str());

    // タッチフィールド
    if (label->range.isTouchable)
    {
        writer->SetCursor(label->xBracketL, label->base.y);
        writer->Print(BracketLeft);

        writer->SetCursor(label->xRangeLabel, label->base.y);
        writer->SetTextColor(LinkBlue);
        writer->Print(label->range.labelStr.c_str());

        writer->SetCursor(label->xBracketR, label->base.y);
        writer->SetTextColor(White);
        writer->Print(BracketRight);
    }
}
