﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/bcat.h>

#include "TestAppSimple_MetaInfo.h"
#include "TestAppSimple_BcatScene.h"

// トレースログ出力を有効にしたい場合はこの define を有効にする
//#define ENABLE_TRACE_BCAT_SCENE_LOG
#if defined( ENABLE_TRACE_BCAT_SCENE_LOG )
#define TRACE_BCAT_SCENE_LOG(...)     NN_LOG(__VA_ARGS__)
#else
#define TRACE_BCAT_SCENE_LOG(...)
#endif // defined( ENABLE_TRACE_BCAT_SCENE_LOG )

namespace {
    const size_t BcatProcessThreadStackSize = 8192; // スレッドのスタックサイズ
    NN_OS_ALIGNAS_THREAD_STACK char g_BcatProcessThreadStack[BcatProcessThreadStackSize]; // BCAT読み込み処理用のスレッドスタック
}

BcatScene::BcatScene() NN_NOEXCEPT
    : m_State(State_None), m_CurrentSelectPos(0), m_CurrentMaxPosNum(1), m_CurrentPage(1), m_LastPage(1),
    m_ReadingCalcCount(0), m_SelectedDirectoryPtr(nullptr), m_IsReadingBcatFiles(false), m_IsStopReadingBcatFiles(true),
    m_IsSyncingDeliveryCache(false), m_IsStopSyncingDeliveryCache(true)
{
}

BcatScene::~BcatScene() NN_NOEXCEPT
{
    // 念のためスレッドの終了処理を呼んでおく
    this->StopBcatRequestSyncDeliveryCacheAsync();
    this->StopReadBcatFilesListAsync();
}

void BcatScene::InternalSetup() NN_NOEXCEPT
{
    // 読み込みボタンの定義
    {
        m_UpdateFilesRange.pos.x = 350.0f;
        m_UpdateFilesRange.pos.y = 75.0f;
        m_UpdateFilesRange.labelStr = "Y: UpdateFiles";
    }

    // 同期要求ボタンの定義
    {
        m_RequestSyncRange.pos.x = 600.0f;
        m_RequestSyncRange.pos.y = 75.0f;
        m_RequestSyncRange.labelStr = "X: RequestSync";
    }

    // ページスクロールボタンの定義
    {
        m_BackPageRange.pos.x = 50.0f;
        m_BackPageRange.pos.y = 640.0f;
        m_BackPageRange.labelStr = "<- : BackPage";
    }
    {
        m_NextPageRange.pos.x = 1050.0f;
        m_NextPageRange.pos.y = 640.0f;
        m_NextPageRange.labelStr = "-> : NextPage";
    }

    m_LastUpdateProcessResult = nn::ResultSuccess();
    m_LastRequestProcessResult = nn::ResultSuccess();

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // Generic 版では動作確認を円滑に行うため、BCATの有効・無効をインスタントに切り替えれるようにしておく
    IsBcatSettingEnabled = true;
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)
}

void BcatScene::RequestDeliveryCacheSync() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    nn::bcat::DeliveryCacheProgress progress;
    auto result = nn::bcat::RequestSyncDeliveryCache(&progress);
    if (result.IsFailure())
    {
        TRACE_BCAT_SCENE_LOG("[Error] nn::bcat::RequestSyncDeliveryCache() failed. e = %08x\n", result.GetInnerValueForDebug());
        m_LastRequestProcessResult = result;
        m_IsSyncingDeliveryCache = false;
        return;
    }

    this->WaitForDeliveryCacheSyncDone(&progress);
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    m_IsSyncingDeliveryCache = false;
}

void BcatScene::WaitForDeliveryCacheSyncDone(nn::bcat::DeliveryCacheProgress* inPrg) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(inPrg);

    TRACE_BCAT_SCENE_LOG("[Trace] BcatDeliveryCacheSyncStart!\n");

    m_GetCuurentDirectoryName = "---";
    m_GetCuurentFileName = "---";
    m_GetCurrentDownloaded = 0;
    m_GetCurrentTotal = 0;
    m_GetWholeDownloaded = 0;
    m_GetWholeTotal = 0;

    // 完了するまで待機します。
    while (inPrg->GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
    {
        if (m_IsStopSyncingDeliveryCache == true)
        {
            break;
        }

        inPrg->Update();
        switch (inPrg->GetStatus())
        {
        case nn::bcat::DeliveryCacheProgressStatus_Queued:
            m_CurrentProgressStatus = "Queued";
            TRACE_BCAT_SCENE_LOG("Progress: status = queued\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Connect:
            m_CurrentProgressStatus = "Connect";
            TRACE_BCAT_SCENE_LOG("Progress: status = connect\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
            m_CurrentProgressStatus = "ProcessList";
            TRACE_BCAT_SCENE_LOG("Progress: status = process list\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Download:
            m_CurrentProgressStatus = "Download";
            {
                m_GetCuurentDirectoryName = inPrg->GetCurrentDirectoryName().value;
                m_GetCuurentFileName = inPrg->GetCurrentFileName().value;
                m_GetCurrentDownloaded = inPrg->GetCurrentDownloaded();
                m_GetCurrentTotal = inPrg->GetCurrentTotal();
                m_GetWholeDownloaded = inPrg->GetWholeDownloaded();
                m_GetWholeTotal = inPrg->GetWholeTotal();
            }
            TRACE_BCAT_SCENE_LOG("Progress: status = download, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
                m_GetCuurentDirectoryName.c_str(), m_GetCuurentFileName.c_str(),
                m_GetCurrentDownloaded, m_GetCurrentTotal,
                m_GetWholeDownloaded, m_GetWholeTotal);
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Commit:
            m_CurrentProgressStatus = "Commit";
            {
                m_GetCuurentDirectoryName = inPrg->GetCurrentDirectoryName().value;
                m_GetWholeDownloaded = inPrg->GetWholeDownloaded();
                m_GetWholeTotal = inPrg->GetWholeTotal();
            }
            TRACE_BCAT_SCENE_LOG("Progress: status = commit, current = (%s), whole = (%lld/%lld)\n",
                m_GetCuurentDirectoryName.c_str(),
                m_GetWholeDownloaded, m_GetWholeTotal);
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Done:
            m_CurrentProgressStatus = "Done";
            {
                m_GetWholeDownloaded = inPrg->GetWholeDownloaded();
                m_GetWholeTotal = inPrg->GetWholeTotal();
            }
            TRACE_BCAT_SCENE_LOG("Progress: status = done, whole = (%lld/%lld), e = %08x\n",
                m_GetWholeDownloaded, m_GetWholeTotal,
                inPrg->GetResult().GetInnerValueForDebug());
            break;
        default:
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(250));
    }

    m_LastRequestProcessResult = inPrg->GetResult();

    TRACE_BCAT_SCENE_LOG("[Trace] BcatDeliveryCacheSyncDone!\n");
}

void BcatScene::ReadBcatFiles() NN_NOEXCEPT
{
    // キャッシュストレージをマウント
    auto result = this->MountDeliveryCacheStorage();
    if (result.IsFailure())
    {
        TRACE_BCAT_SCENE_LOG("[Error] BcatScene::ReadBcatFiles() Failed. e = %08x\n", result.GetInnerValueForDebug());
        m_LastUpdateProcessResult = result;
        m_IsReadingBcatFiles = false;
        return;
    }

    // ディレクトリ・ファイル一覧を全て読み込む
    this->EnumerateAllFiles();

    // キャッシュストレージをアンマウント
    nn::bcat::UnmountDeliveryCacheStorage();

    {
        // ディレクトリ一覧のページ数がこの段階で確定するはずなので計算しておく
        m_LastPage = (static_cast<int>(m_BcatDirList.size()) / MaxDirectoryItemCount) + 1;
        if (static_cast<int>(m_BcatDirList.size()) % MaxDirectoryItemCount == 0)
        {
            --m_LastPage;
        }

        m_CurrentPage = 1;
        m_CurrentSelectPos = 0;
    }

    m_IsReadingBcatFiles = false;
}

// データ配信キャッシュストレージをマウントします。
nn::Result BcatScene::MountDeliveryCacheStorage() NN_NOEXCEPT
{
    auto result = nn::bcat::MountDeliveryCacheStorage();

    // ファイルダウンロード中にアプリを起動した場合、ダウンロードが完了するまでアプリがマウントすることはできません。
    while (nn::bcat::ResultLocked::Includes(result))
    {
        if (m_IsStopReadingBcatFiles == true)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        // 一定期間待機した後、再度マウントを試みます。
        result = nn::bcat::MountDeliveryCacheStorage();
    }

    if (nn::bcat::ResultServiceUnavailable::Includes(result))
    {
        TRACE_BCAT_SCENE_LOG("[Warning] Please link Nintendo account or register Network service account.\n");
    }

    return result;
}

// 受信したファイルをすべて列挙します。
void BcatScene::EnumerateAllFiles() NN_NOEXCEPT
{
    m_BcatDirList.clear();

    static nn::bcat::DirectoryName dirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int dirCount = 0;

    auto result = nn::bcat::EnumerateDeliveryCacheDirectory(&dirCount, dirNames, NN_ARRAY_SIZE(dirNames));
    if (result.IsFailure())
    {
        TRACE_BCAT_SCENE_LOG("[Error] nn::bcat::EnumerateDeliveryCacheDirectory() Failed. e = %08x\n", result.GetInnerValueForDebug());
        m_LastUpdateProcessResult = result;
        return;
    }

    TRACE_BCAT_SCENE_LOG("----------------------------------------------------------------------------------------------------\n");
    TRACE_BCAT_SCENE_LOG("DirectoryCount = %d\n", dirCount);

    for (int i = 0; i < dirCount; ++i)
    {
        if (m_IsStopReadingBcatFiles == true)
        {
            break;
        }

        TRACE_BCAT_SCENE_LOG("----------------------------------------------------------------------------------------------------\n");
        TRACE_BCAT_SCENE_LOG("Directory[%3d] = %s\n", i, dirNames[i].value);

        nn::bcat::DeliveryCacheDirectory directory;
        BcatDirectory bDir(dirNames[i].value);
        bDir.name = bDir.name + " / ";

        result = directory.Open(dirNames[i]);
        if (result.IsFailure())
        {
            TRACE_BCAT_SCENE_LOG("[Error] nn::bcat::DeliveryCacheDirectory::Open() Failed. e = %08x\n", result.GetInnerValueForDebug());
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
            m_LastUpdateProcessResult = result;
            return;
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        }

        static nn::bcat::DeliveryCacheDirectoryEntry entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
        int entryCount = 0;

        result = directory.Read(&entryCount, entries, NN_ARRAY_SIZE(entries));
        if (result.IsFailure())
        {
            TRACE_BCAT_SCENE_LOG("[Error] nn::bcat::DeliveryCacheDirectory::Read() Failed. e = %08x\n", result.GetInnerValueForDebug());
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
            m_LastUpdateProcessResult = result;
            return;
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        }

        TRACE_BCAT_SCENE_LOG("    EntryCount = %d\n", entryCount);

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        // 大量ファイルの動作確認用コード
        for (int k = 0; k < 50; ++k)
        {
            std::string testName = "WWWWWWWWWWWWWWWWWWWWWWWWWWWWWWW";
            BcatFile bFile(testName);
            bFile.digest[0] = 0x123456790abcdef1;
            bDir.fileList.push_back(bFile);
        }
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)

        for (int j = 0; j < entryCount; ++j)
        {
            if (m_IsStopReadingBcatFiles == true)
            {
                break;
            }

            TRACE_BCAT_SCENE_LOG("    Entry[%3d] = %32s, %016llx%016llx, %lld\n", j,
                entries[j].name.value, entries[j].digest.value[0], entries[j].digest.value[1], entries[j].size);

            nn::bcat::DeliveryCacheFile file;
            BcatFile bFile(entries[j].name.value);
            bFile.digest[0] = entries[j].digest.value[0];
            bFile.digest[1] = entries[j].digest.value[1];

            bDir.fileList.push_back(bFile);
        }

        m_BcatDirList.push_back(bDir);
    }

    TRACE_BCAT_SCENE_LOG("----------------------------------------------------------------------------------------------------\n");
}

void BcatScene::ProcessBcatFunctionStatic(void* inPtr) NN_NOEXCEPT
{
    BcatScene* ptr = reinterpret_cast<BcatScene*>(inPtr);
    if (ptr != nullptr)
    {
        switch (ptr->m_State)
        {
        case State_ReadingFiles:
            ptr->ReadBcatFiles();
            break;
        case State_SyncingDeliveryCache:
            ptr->RequestDeliveryCacheSync();
            break;
        default:
            TRACE_BCAT_SCENE_LOG("[Warning] BcatScene::ProcessBcatFunctionStatic() Unexpected State : state = %d.\n", ptr->m_State);
            break;
        }
    }
}

int BcatScene::StartReadBcatFilesListAsync() NN_NOEXCEPT
{
    // スレッドの作成
    auto result = nn::os::CreateThread(&m_BcatProcessThread, ProcessBcatFunctionStatic, this,
                                g_BcatProcessThreadStack, BcatProcessThreadStackSize, nn::os::DefaultThreadPriority);
    if (result.IsFailure())
    {
        // 作成に失敗した場合はすぐに抜ける
        return -1;
    }

    // フラグ変数の初期値設定
    m_IsReadingBcatFiles = true;
    m_IsStopReadingBcatFiles = false;

    m_State = State_ReadingFiles;

    // スレッドの実行を開始する
    nn::os::StartThread(&m_BcatProcessThread);

    return 0;
}

void BcatScene::StopReadBcatFilesListAsync() NN_NOEXCEPT
{
    if (m_IsStopReadingBcatFiles == false)
    {
        m_IsStopReadingBcatFiles = true;

        // スレッドが終了するのを待つ
        nn::os::WaitThread(&m_BcatProcessThread);
        // スレッドを破棄する
        nn::os::DestroyThread(&m_BcatProcessThread);
    }
}

int BcatScene::StartBcatRequestSyncDeliveryCacheAsync() NN_NOEXCEPT
{
    // スレッドの作成
    auto result = nn::os::CreateThread(&m_BcatProcessThread, ProcessBcatFunctionStatic, this,
                                g_BcatProcessThreadStack, BcatProcessThreadStackSize, nn::os::DefaultThreadPriority);
    if (result.IsFailure())
    {
        // 作成に失敗した場合はすぐに抜ける
        return -1;
    }

    // フラグ変数の初期値設定
    m_IsSyncingDeliveryCache = true;
    m_IsStopSyncingDeliveryCache = false;

    m_State = State_SyncingDeliveryCache;

    // スレッドの実行を開始する
    nn::os::StartThread(&m_BcatProcessThread);

    return 0;
}

void BcatScene::StopBcatRequestSyncDeliveryCacheAsync() NN_NOEXCEPT
{
    if (m_IsStopSyncingDeliveryCache == false)
    {
        m_IsStopSyncingDeliveryCache = true;

        // スレッドが終了するのを待つ
        nn::os::WaitThread(&m_BcatProcessThread);
        // スレッドを破棄する
        nn::os::DestroyThread(&m_BcatProcessThread);
    }
}

void BcatScene::ResetBcatDirRangeList() NN_NOEXCEPT
{
    for (auto& range : m_BcatDirRangeList)
    {
        range.isSetting = false;
        range.dataPtr = nullptr;
    }
    m_CurrentSelectPos = 0;
}

void BcatScene::EnterDirectoryHandle() NN_NOEXCEPT
{
    m_State = State_ShowFiles;

    m_BackDirectoryRange.isSetting = false;
    m_CurrentDirPage = m_CurrentPage;
    m_LastDirPage = m_LastPage;

    {
        m_LastPage = (static_cast<int>(m_SelectedDirectoryPtr->fileList.size()) / MaxFileItemCount) + 1;
        if (static_cast<int>(m_SelectedDirectoryPtr->fileList.size()) % MaxFileItemCount == 0)
        {
            --m_LastPage;
        }
    }
    m_CurrentPage = 1;
}

void BcatScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (IsBcatSettingEnabled == false)
    {
        // そもそも BCAT が有効でないアプリの場合は何もできないようにする
        return;
    }

    if (m_State == State_None || m_State == State_ShowDirectories || m_State == State_ShowFiles)
    {

        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
        {
            m_State = State_StartReadFiles;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            m_State = State_StartRequestSync;
        }
    }

    if (m_State == State_ShowDirectories)
    {
        if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask))
            && m_CurrentPage < m_LastPage)
        {
            ++m_CurrentPage;
            this->ResetBcatDirRangeList();
        }
        else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask))
            && m_CurrentPage > 1)
        {
            --m_CurrentPage;
            this->ResetBcatDirRangeList();
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
        {
            if (m_CurrentSelectPos < m_CurrentMaxPosNum)
            {
                ++m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = 0;
            }
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
        {
            if (m_CurrentSelectPos > 0)
            {
                --m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = m_CurrentMaxPosNum;
            }
        }

        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            auto ptr = reinterpret_cast<BcatDirectory*>(m_BcatDirRangeList[m_CurrentSelectPos].dataPtr);
            if (ptr != nullptr)
            {
                m_SelectedDirectoryPtr = ptr;
                this->EnterDirectoryHandle();
            }
        }
    }
    else if (m_State == State_ShowFiles)
    {
        if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask))
            && m_CurrentPage < m_LastPage)
        {
            ++m_CurrentPage;
        }
        else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask))
            && m_CurrentPage > 1)
        {
            --m_CurrentPage;
        }

        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            m_State = State_ShowDirectories;
            m_LastPage = m_LastDirPage;
            m_CurrentPage = m_CurrentDirPage;
        }
    }
}

void BcatScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (IsBcatSettingEnabled == false)
    {
        // そもそも BCAT が有効でないアプリの場合は何もできないようにする
        return;
    }

    if (m_State == State_None || m_State == State_ShowDirectories || m_State == State_ShowFiles)
    {
        if (m_UpdateFilesRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_StartReadFiles;
        }
        else if (m_RequestSyncRange.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_StartRequestSync;
        }
    }

    if (m_State == State_ShowDirectories)
    {
        int count = 0;
        for (auto& range : m_BcatDirRangeList)
        {
            if (range.isSetting == false)
            {
                break;
            }

            if (range.IsInRange(m_PreviousTouch) == true)
            {
                m_SelectedDirectoryPtr = reinterpret_cast<BcatDirectory*>(range.dataPtr);
                m_CurrentSelectPos = count;

                this->EnterDirectoryHandle();

                break;
            }

            ++count;
        }

        if (m_NextPageRange.range.IsInRange(m_PreviousTouch) == true && m_CurrentPage < m_LastPage)
        {
            ++m_CurrentPage;
            this->ResetBcatDirRangeList();
        }
        else if (m_BackPageRange.range.IsInRange(m_PreviousTouch) == true && m_CurrentPage > 1)
        {
            --m_CurrentPage;
            this->ResetBcatDirRangeList();
        }
    }
    else if (m_State == State_ShowFiles)
    {
        if (m_BackDirectoryRange.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_ShowDirectories;
            m_LastPage = m_LastDirPage;
            m_CurrentPage = m_CurrentDirPage;
        }
        else if (m_NextPageRange.range.IsInRange(m_PreviousTouch) == true && m_CurrentPage < m_LastPage)
        {
            ++m_CurrentPage;
        }
        else if (m_BackPageRange.range.IsInRange(m_PreviousTouch) == true && m_CurrentPage > 1)
        {
            --m_CurrentPage;
        }
    }
}

void BcatScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_StartReadFiles)
    {
        // ハッシュ計算処理の開始
        if (this->StartReadBcatFilesListAsync() < 0)
        {
            // 起こらないと思うが、計算処理のスレッド生成失敗
            TRACE_BCAT_SCENE_LOG("[Error] BcatScene::InternalProcess() StartReadBcatFilesListAsync Failed\n");
            m_State = State_None;
        }
        else
        {
            // スレッド生成成功 (計算中に遷移しておく)
            m_State = State_ReadingFiles;
        }
    }
    else if (m_State == State_ReadingFiles)
    {
        if (m_IsReadingBcatFiles == false)
        {
            // 後始末としてスレッドの終了処理を呼び出しておく
            this->StopReadBcatFilesListAsync();
            m_State = State_ShowDirectories;
        }
    }
    else if (m_State == State_StartRequestSync)
    {
        // ハッシュ計算処理の開始
        if (this->StartBcatRequestSyncDeliveryCacheAsync() < 0)
        {
            // 起こらないと思うが、計算処理のスレッド生成失敗
            TRACE_BCAT_SCENE_LOG("[Error] BcatScene::InternalProcess() StartBcatRequestSyncDeliveryCacheAsync Failed\n");
            m_State = State_None;
        }
        else
        {
            // スレッド生成成功
            m_State = State_SyncingDeliveryCache;
        }
    }
    else if (m_State == State_SyncingDeliveryCache)
    {
        if (m_IsSyncingDeliveryCache == false)
        {
            // 後始末としてスレッドの終了処理を呼び出しておく
            this->StopBcatRequestSyncDeliveryCacheAsync();
            // ファイル一覧を自動で表示するために読み込み処理に自動遷移する
            m_State = State_StartReadFiles;
        }
    }
}

void BcatScene::DrawAnimationDots(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    // 処理中を表現するちょっとしたアニメーション表現
    if (m_ReadingCalcCount > 100)
    {
        m_ReadingCalcCount = 0;
    }

    ++m_ReadingCalcCount;

    if (m_ReadingCalcCount < 25)
    {
        writer->Print(".   ");
    }
    else if (m_ReadingCalcCount < 50)
    {
        writer->Print("..  ");
    }
    else if (m_ReadingCalcCount < 75)
    {
        writer->Print("... ");
    }
    else
    {
        writer->Print("....");
    }
}

void BcatScene::DrawReadingInfo(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetScale(2.5f, 2.5f);
    writer->SetCursor(400.0f, 280.0f);
    writer->Print("Reading Files ");

    this->DrawAnimationDots(writer);
}

void BcatScene::DrawSyncingDeliveryCache(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetScale(1.6f, 1.6f);
    const float xPos = 50.0f;
    const float xPos2 = 80.0f;
    const float baseYPos = 200.0f;
    const float LineSize = 33.0f;
    int ypos = 0;

    writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
    writer->Print("Syncing DeliveryCache ");
    this->DrawAnimationDots(writer);
    ++ypos;
    ++ypos;

    writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
    writer->Print("- Status: %s", m_CurrentProgressStatus.c_str());
    ++ypos;
    ++ypos;

    writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
    writer->Print("- Current Downloading Directory / File: ");
    ++ypos;

    writer->SetCursor(xPos2, baseYPos + (LineSize * (ypos)));
    writer->Print("- %s / %s", m_GetCuurentDirectoryName.c_str(), m_GetCuurentFileName.c_str());
    ++ypos;

    writer->SetCursor(xPos2, baseYPos + (LineSize * (ypos)));
    writer->Print("- Current File [ Done / Total ] : %lld / %lld", m_GetCurrentDownloaded, m_GetCurrentTotal);
    ++ypos;
    ++ypos;

    writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
    writer->Print("- Whole [ Downloaded / Total ] : %lld / %lld", m_GetWholeDownloaded, m_GetWholeTotal);
}

void BcatScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);

    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[BCAT]");
    if (m_LastPage > 1)
    {
        writer->Print(" (%d / %d)", m_CurrentPage, m_LastPage);
    }

    if (IsBcatSettingEnabled == false)
    {
        writer->SetScale(2.25f, 2.25f);
        writer->SetTextColor(Coral);
        writer->SetCursor(340.0f, 280.0f + 33.0f);
        writer->Print("Non-BCAT-Compliant Application");
        // 以降の処理は書き出さない・・
        return;
    }

    if (m_State != State_ReadingFiles || m_State != State_SyncingDeliveryCache)
    {
        this->WriteTouchRange(writer, &m_UpdateFilesRange, true);
        this->WriteTouchRange(writer, &m_RequestSyncRange, true);

        writer->SetTextColor(White);
        writer->Print("   LastResult : ");

        // 位置を保持しておく
        auto resultXPos = writer->GetCursorX();
        auto resultYPos = writer->GetCursorY();
        if (m_LastUpdateProcessResult.IsFailure())
        {
            // エラーであれば赤色にしておく
            writer->SetTextColor(Red);
        }
        writer->Print("0x%08x", m_LastUpdateProcessResult.GetInnerValueForDebug());
        writer->SetTextColor(White);
        writer->Print(" (Y)");

        writer->SetCursor(resultXPos, resultYPos + 33.0f);
        if (m_LastRequestProcessResult.IsFailure())
        {
            // エラーであれば赤色にしておく
            writer->SetTextColor(Red);
        }
        writer->Print("0x%08x", m_LastRequestProcessResult.GetInnerValueForDebug());
        writer->SetTextColor(White);
        writer->Print(" (X)");
    }

    if (m_State == State_ShowDirectories)
    {
        this->DrawBcatDirectoryList(writer);
    }
    else if (m_State == State_ShowFiles)
    {
        this->DrawBcatFileList(writer);
    }
    else if (m_State == State_ReadingFiles)
    {
        this->DrawReadingInfo(writer);
    }
    else if (m_State == State_SyncingDeliveryCache)
    {
        this->DrawSyncingDeliveryCache(writer);
    }
}

void BcatScene::DrawBcatDirectoryList(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (m_BcatDirList.empty())
    {
        writer->SetScale(2.5f, 2.5f);
        writer->SetCursor(480.0f, 310.0f);
        writer->Print("No Directories.");
        return;
    }

    writer->SetCursor(73.0f, 120.0f + (60.0f * (m_CurrentSelectPos)));
    writer->Print("*");

    int count = 0;
    int yPointCount = 0;
    for (auto& dir : m_BcatDirList)
    {
        if (count < ((m_CurrentPage - 1) * MaxDirectoryItemCount))
        {
            ++count;
            continue;
        }

        if (count >= (m_CurrentPage * MaxDirectoryItemCount))
        {
            break;
        }

        auto& refRange = m_BcatDirRangeList[yPointCount];
        if (refRange.isSetting == false)
        {
           // BcatDirectory のポインタを保持させておく
            refRange.dataPtr = &dir;
        }

        // タッチ可能範囲を描画
        Position pos(100.0f, 120.0f + (60.0f * (yPointCount)));
        this->WriteTouchRange(writer, pos, dir.name.c_str(), &refRange);

        ++count;
        ++yPointCount;
    }

    m_CurrentMaxPosNum = (yPointCount - 1);

    writer->SetTextColor(White);
    writer->SetCursor(530.0f, 620.0f);
    writer->Print("A: Enter Directory");

    if (m_LastPage > 1)
    {
        if (m_CurrentPage > 1)
        {
            this->WriteTouchRange(writer, &m_BackPageRange);
        }
        if (m_CurrentPage < m_LastPage)
        {
            this->WriteTouchRange(writer, &m_NextPageRange);
        }
    }
}

void BcatScene::DrawBcatFileList(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    {
        Position pos(48.0f, 115.0f);
        const std::string label = "B: <- " + m_SelectedDirectoryPtr->name;
        this->WriteTouchRange(writer, pos, label.c_str(), &m_BackDirectoryRange);
    }

    writer->SetTextColor(White);

    if (m_SelectedDirectoryPtr->fileList.empty())
    {
        writer->SetScale(2.5f, 2.5f);
        writer->SetCursor(550.0f, 310.0f);
        writer->Print("No Files.");
        return;
    }

    writer->SetScale(1.4f, 1.4f);

    writer->SetCursor(50.0f, 152.0f);
    writer->Print("[ Name ]");

    writer->SetCursor(750.0f, writer->GetCursorY());
    writer->Print("[ Digest ]");

    this->SetDefaultScale(writer);

    int count = 0;
    int yPointCount = 0;
    for (auto& file : m_SelectedDirectoryPtr->fileList)
    {
        if (count < ((m_CurrentPage - 1) * MaxFileItemCount))
        {
            ++count;
            continue;
        }

        if (count >= (m_CurrentPage * MaxFileItemCount))
        {
            break;
        }

        const auto posY = 195.0f + (40.0f * (yPointCount));
        writer->SetCursor(57.0f, posY);

        writer->Print("%s", file.name.c_str());

        writer->SetCursor(780.0f, posY);
        // ひとまず Digest の前半部分のみを表示
        writer->Print("%016llx", file.digest[0]);

        ++count;
        ++yPointCount;
    }

    if (m_LastPage > 1)
    {
        if (m_CurrentPage > 1)
        {
            this->WriteTouchRange(writer, &m_BackPageRange);
        }
        if (m_CurrentPage < m_LastPage)
        {
            this->WriteTouchRange(writer, &m_NextPageRange);
        }
    }
}
