﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/aoc/aoc_PrivateApi.h>

#include "TestAppSimple_FsUtilities.h"
#include "TestAppSimple_AocInfoScene.h"

namespace {
    // 追加コンテンツインデックスを列挙するためのバッファ
    const int MaxListupCount = 200;
    nn::aoc::AddOnContentIndex g_AocListupBuffer[MaxListupCount];

    const size_t AocInfoListReadThreadStackSize = 8192; // スレッドのスタックサイズ
    NN_OS_ALIGNAS_THREAD_STACK char g_AocInfoListReadThreadStack[AocInfoListReadThreadStackSize]; // Aoc読み込み処理用のスレッドスタック

    const size_t PollingListChangedEventThreadStackSize = 8192; // スレッドのスタックサイズ
    NN_OS_ALIGNAS_THREAD_STACK char g_PollingListChangedEventThread[PollingListChangedEventThreadStackSize]; // イベント監視処理用のスレッドスタック

    const char* const MountName = "aoc";
    const char* const AocDataPath = "aoc:/AddOnContent.dat";
}

AocInfoScene::AocInfoScene() NN_NOEXCEPT
    : m_State(State_None), m_HasAoc(true),
      m_CurrentSelectPos(0), m_CurrentMaxPosNum(0), m_CurrentPage(1), m_LastPage(1),
      m_Hash(AocDataPath), m_CalcCount(0), m_Progress(0), m_CalcPropertyPtr(nullptr),
      m_IsReadingAocInfo(true), m_IsStopReading(true), m_TotalAocCount(0), m_ReadAocCount(0), m_ReadingCalcCount(0),
      m_IsBlinkReloadButton(false), m_ChangedEventCalcCount(0), m_IsStopPolling(true)
{
}

AocInfoScene::~AocInfoScene() NN_NOEXCEPT
{
    // 念のためスレッドの終了処理を呼んでおく
    this->StopReadAocInfoListAsync();
    this->StopPollListChangedEvent();
}

void AocInfoScene::InternalSetup() NN_NOEXCEPT
{
    // Reload ボタンの定義
    {
        m_ReloadRange.pos.x = 450.0f;
        m_ReloadRange.pos.y = 75.0f;
        m_ReloadRange.labelStr = "Y: Reload";
    }

    // 非同期で Aoc の読み込み処理を行う
    this->StartReadAocInfoListAsync();
    // リストの変更イベントを監視するスレッドを立ち上げる
    this->StartPollListChangedEvent();

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // Generic ではデバック用に初回読み込み時はボタンを点滅させる
    m_IsBlinkReloadButton = true;
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)
}

void AocInfoScene::ReadAocInfoList() NN_NOEXCEPT
{
    // Reload を考慮して読み込み前に Aoc 情報を保持するリストはクリアしておく
    m_AocInfoList.clear();
    m_ReadAocCount = 0;

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // インストールされている Aoc の総数を取得
    m_TotalAocCount = nn::aoc::CountAddOnContent();
    if (m_TotalAocCount <= 0)
    {
        m_HasAoc = false;
        // 読み込み処理中フラグを落とす
        m_IsReadingAocInfo = false;
        // 以下の処理は意味がないので返る
        return;
    }

    int offset = 0;
    while (offset < m_TotalAocCount)
    {
        if (m_IsStopReading == true)
        {
            // 中断フラグが立っていればすぐに抜ける
            break;
        }

        // Aoc のリストを取得
        const auto count = nn::aoc::ListAddOnContent(g_AocListupBuffer, offset, MaxListupCount);
        if (count == 0)
        {
            // 実質的には入らないパスだと思うが 0 が取得されたら抜ける
            break;
        }

        for (int i = 0; i < count; ++i)
        {
            if (m_IsStopReading == true)
            {
                // 中断フラグが立っていればすぐに抜ける
                break;
            }

            AocInfoProperty aip;
            aip.index = g_AocListupBuffer[i];

            {
                fsutil::ScopedMountAoc mountAoc(MountName, aip.index);

                auto result = mountAoc.GetLastResult();
                if (result.IsFailure())
                {
                    // マウントで失敗
                    NN_LOG("[Error] AocInfoScene::InternalSetup() MountProcess Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
                    // リストに詰めて、次の index へ
                    m_AocInfoList.push_back(aip);
                    continue;
                }

                // マウントされた追加コンテンツのファイルへのアクセス
                this->ReadAocPropertyInfo(&aip);
            }

            char formatBuf[64] = { 0 };
            // 表示する項目の文字列を作成
            if (aip.isDefaultData == true)
            {
                //nn::util::SNPrintf(formatBuf, 64, "Index %-4d[Ver.%s]:(", aip.index, aip.verVal.c_str());
                nn::util::SNPrintf(formatBuf, 64, "Index %-4d[Ver.", aip.index);
                aip.prefixStr = formatBuf;
                // ハッシュ値文字列の短縮形(先頭18桁)を保持する形にする
                nn::util::SNPrintf(formatBuf, 64, "%.18s", aip.hashVal.c_str());
                aip.displayStr = formatBuf;
            }
            else
            {
                nn::util::SNPrintf(formatBuf, 64, "Index %-4d[Ver. -------- ]:(", aip.index);
                aip.prefixStr = formatBuf;
                aip.displayStr = "Not DefaultContent";
            }

            m_AocInfoList.push_back(aip);
            ++m_ReadAocCount;
        }

        offset += count;
    }

    m_HasAoc = true;

    m_LastPage = (static_cast<int>(m_AocInfoList.size()) / MaxItemCount) + 1;
    if (static_cast<int>(m_AocInfoList.size()) % MaxItemCount == 0)
    {
        --m_LastPage;
    }
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Generic 環境ではAocは無いものとしておく
    m_HasAoc = false;

    // 読み込み処理中のエミュレーション(テストに合わせて m_TotalAocCount に設定する値を適宜変更する)
    m_TotalAocCount = 5;
    for (int i = 0; i < m_TotalAocCount; ++i)
    {
        m_ReadAocCount = i;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(75));

        if (m_IsStopReading == true)
        {
            // 中断フラグが立っていればすぐに抜ける
            break;
        }
    }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    // 読み込み処理中フラグを落とす
    m_IsReadingAocInfo = false;
}

void AocInfoScene::ProcessReadAocInfoListStatic(void* inPtr) NN_NOEXCEPT
{
    AocInfoScene* ptr = reinterpret_cast<AocInfoScene*>(inPtr);
    if (ptr != nullptr)
    {
        ptr->ReadAocInfoList();
    }
}

int AocInfoScene::StartReadAocInfoListAsync() NN_NOEXCEPT
{
    // スレッドの作成
    auto result = nn::os::CreateThread(&m_AocInfoReadThread, ProcessReadAocInfoListStatic, this,
                    g_AocInfoListReadThreadStack, AocInfoListReadThreadStackSize, nn::os::DefaultThreadPriority);
    if (result.IsFailure())
    {
        // 作成に失敗した場合はすぐに抜ける
        return -1;
    }

    // フラグ変数の初期値設定
    m_IsReadingAocInfo = true;
    m_IsStopReading = false;

    // スレッドの実行を開始する
    nn::os::StartThread(&m_AocInfoReadThread);

    return 0;
}

void AocInfoScene::StopReadAocInfoListAsync() NN_NOEXCEPT
{
    if (m_IsStopReading == false)
    {
        m_IsStopReading = true;

        // スレッドが終了するのを待つ
        nn::os::WaitThread(&m_AocInfoReadThread);
        // スレッドを破棄する
        nn::os::DestroyThread(&m_AocInfoReadThread);
    }
}

void AocInfoScene::ProcessPollListChangedEventStatic(void* inPtr) NN_NOEXCEPT
{
    AocInfoScene* ptr = reinterpret_cast<AocInfoScene*>(inPtr);
    if (ptr != nullptr)
    {
        ptr->PollListChangedEvent();
    }
}

int AocInfoScene::StartPollListChangedEvent() NN_NOEXCEPT
{
    // スレッドの作成
    auto result = nn::os::CreateThread(&m_PollingListChangedEventThread, ProcessPollListChangedEventStatic, this,
        g_PollingListChangedEventThread, PollingListChangedEventThreadStackSize, nn::os::DefaultThreadPriority);
    if (result.IsFailure())
    {
        // 作成に失敗した場合はすぐに抜ける
        return -1;
    }

    // フラグ変数の初期値設定
    m_IsPollingListChangedEvent = true;
    m_IsStopPolling = false;

    // スレッドの実行を開始する
    nn::os::StartThread(&m_PollingListChangedEventThread);

    return 0;
}

void AocInfoScene::StopPollListChangedEvent() NN_NOEXCEPT
{
    if (m_IsStopPolling == false)
    {
        m_IsStopPolling = true;

        // スレッドが終了するのを待つ
        nn::os::WaitThread(&m_PollingListChangedEventThread);
        // スレッドを破棄する
        nn::os::DestroyThread(&m_PollingListChangedEventThread);
    }
}

void AocInfoScene::PollListChangedEvent() NN_NOEXCEPT
{
    // Generic 版では Event 取得は実質不可なので、NX 版のみEvent監視処理を行う
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    nn::os::SystemEvent listChengedEvent;
    nn::aoc::GetAddOnContentListChangedEvent(&listChengedEvent);
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    while (m_IsStopPolling == false)
    {
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        if (listChengedEvent.TryWait())
        {
            m_IsBlinkReloadButton = true;
        }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }
}

void AocInfoScene::ReadAocPropertyInfo(AocInfoProperty* outProperty) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outProperty);

    static const char* hashFilePath = "aoc:/HashVal.txt";
    static const char* versionFilePath = "aoc:/_Version_.txt";

    // そもそも想定しているファイルが存在するかどうかの確認
    if (fsutil::IsExistPath(hashFilePath) == false ||
        fsutil::IsExistPath(AocDataPath) == false ||
        fsutil::IsExistPath(versionFilePath) == false)
    {
        // 存在しなければ早々と返る
        return;
    }

    // ハッシュ値を記録したファイルへのアクセス
    {
        const auto retString = this->ReadAocRecordFile(hashFilePath);
        if (retString == "")
        {
            NN_LOG("[Error] Read Failed : %s", hashFilePath);
            return;
        }
        // ハッシュ値の保持
        outProperty->hashVal = retString;
    }

    // バージョン値を記録したファイルへのアクセス
    {
        const auto retString = this->ReadAocRecordFile(versionFilePath);
        if (retString == "")
        {
            NN_LOG("[Error] Read Failed : %s", versionFilePath);
            return;
        }

        // バージョン値の保持
        outProperty->verVal = retString;

        // ReleaseVersion の抽出
        const auto verNum = Convert::ToUInt32(retString);
        const auto rvNum = verNum >> 16;

        // 後々の表示形式を考慮して現状は 16 進数で保持しておく
        char formatBuf[16] = { 0 };
        nn::util::SNPrintf(formatBuf, 16, "%04x", rvNum);
        outProperty->rVerVal = formatBuf;

        // PrivateVersion の抽出
        const auto pvNum = 0x0000FFFF & verNum;
        nn::util::SNPrintf(formatBuf, 16, "%04x", pvNum);
        outProperty->pVerVal = formatBuf;
    }

    // MakeTestApplication から作成されたデフォルトデータであるフラグを立てる
    outProperty->isDefaultData = true;
}

std::string AocInfoScene::ReadAocRecordFile(const char* inReadFilePath) NN_NOEXCEPT
{
    std::string retval;
    fsutil::File file;
    auto result = file.Open(inReadFilePath, nn::fs::OpenMode::OpenMode_Read);
    if (result.IsFailure())
    {
        // あり得ないと思うが、ファイルオープンに失敗
        NN_LOG("[Error] AocInfoScene::ReadAocRecordFile() versionFilePath Open Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return retval;
    }

    const auto fileSize = file.GetSize();
    if (fileSize <= 0)
    {
        // ファイルサイズが不正
        NN_LOG("[Error] AocInfoScene::ReadAocRecordFile() versionFilePath : fileSize = %lld\n", fileSize);
        return retval;
    }

    const size_t bufSize = static_cast<size_t>(fileSize - 3) + 1;
    std::unique_ptr<char[]> readBuffer(new char[bufSize]);
    *(readBuffer.get() + (bufSize - 1)) = '\0';

    size_t outSize = 0;
    size_t readSize = bufSize - 1;
    // UTF8のBOM付きファイルであることを想定(オフセットの3バイトはそのため)
    result = file.Read(&outSize, 3, readBuffer.get(), readSize);
    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗
        NN_LOG("[Error] AocInfoScene::ReadAocRecordFile() Read Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return retval;
    }

    return readBuffer.get();
}

void AocInfoScene::ReloadAocInfoList() NN_NOEXCEPT
{
    if (m_State == State_CalcProcessing)
    {
        // もしハッシュ計算中であれば、計算処理を強制停止させる
        m_Hash.StopSha1ProcessAsync();
        m_State = State_None;
    }

    // 一旦、終了処理を呼び出す
    this->StopReadAocInfoListAsync();

    // リロード時にいったんリセット (SIGLO-68235)
    this->ResetAocRangeList();

    // Aoc の再読み込み処理を実行
    this->StartReadAocInfoListAsync();
    // 点滅フラグは常に消しておく
    m_IsBlinkReloadButton = false;
}

void AocInfoScene::ResetAocRangeList() NN_NOEXCEPT
{
    for (auto& range : m_AocRangeList)
    {
        range.isSetting = false;
        range.isTouchable = true;
        range.dataPtr = nullptr;
    }
    m_CurrentSelectPos = 0;
}

void AocInfoScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_IsReadingAocInfo == true)
    {
        // AoC が読み込み中であれば以下の Hid 処理は受け付けないようにする
        return;
    }

    if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
    {
        // AoC 情報の再読み込み処理を実行する
        this->ReloadAocInfoList();
    }

    if (m_HasAoc == false)
    {
        // Aoc 情報が無ければ以下の処理は意味がないので処理をすぐに返す
        return;
    }

    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            auto ptr = reinterpret_cast<AocInfoProperty*>(m_AocRangeList[m_CurrentSelectPos].dataPtr);
            // デフォルトデータかつ未計算の場合のみ計算する状態に入る
            if (ptr != nullptr && ptr->isDefaultData == true && ptr->state == PropertyState_NotCalcHash)
            {
                m_State = State_CalcHash;
                m_CalcPropertyPtr = ptr;
                m_CalcPropertyPtr->state = PropertyState_CalcProcessing;
            }
        }
    }

    if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask))
        && m_CurrentPage < m_LastPage)
    {
        ++m_CurrentPage;
        this->ResetAocRangeList();
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
                || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask))
                && m_CurrentPage > 1)
    {
        --m_CurrentPage;
        this->ResetAocRangeList();
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
                || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
    {
        if (m_CurrentSelectPos < m_CurrentMaxPosNum)
        {
            ++m_CurrentSelectPos;
        }
        else
        {
            m_CurrentSelectPos = 0;
        }
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
                || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
    {
        if (m_CurrentSelectPos > 0)
        {
            --m_CurrentSelectPos;
        }
        else
        {
            m_CurrentSelectPos = m_CurrentMaxPosNum;
        }
    }
}

void AocInfoScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_IsReadingAocInfo == true)
    {
        // AoC が読み込み中であれば以下の Hid 処理は受け付けないようにする
        return;
    }

    if (m_ReloadRange.range.IsInRange(m_PreviousTouch) == true)
    {
        // AoC 情報の再読み込み処理を実行する
        this->ReloadAocInfoList();
    }

    if (m_HasAoc == false)
    {
        // Aoc 情報が無ければ以下の処理は意味がないので処理をすぐに返す
        return;
    }

    if (m_State == State_None)
    {
        int count = 0;
        for (auto& range : m_AocRangeList)
        {
            if (range.isSetting == false)
            {
                break;
            }

            if (range.IsInRange(m_PreviousTouch) == true)
            {
                if (range.dataPtr != nullptr)
                {
                    m_State = State_CalcHash;
                    m_CalcPropertyPtr = reinterpret_cast<AocInfoProperty*>(range.dataPtr);
                    m_CalcPropertyPtr->state = PropertyState_CalcProcessing;
                }
                m_CurrentSelectPos = count;
                break;
            }

            ++count;
        }
    }

    if (m_NextPageRange.IsInRange(m_PreviousTouch) == true && m_CurrentPage < m_LastPage)
    {
        ++m_CurrentPage;
        this->ResetAocRangeList();
    }
    else if (m_BackPageRange.IsInRange(m_PreviousTouch) == true && m_CurrentPage > 1)
    {
        --m_CurrentPage;
        this->ResetAocRangeList();
    }
}

void AocInfoScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_CalcHash)
    {
        // ハッシュ計算処理の開始
        if (m_Hash.StartSha1ProcessAocAsync(MountName, m_CalcPropertyPtr->index) < 0)
        {
            // 起こらないと思うが、計算処理のスレッド生成失敗
            NN_LOG("[Error] AocInfoScene::Process() StartSha1ProcessAocAsync Failed : index = %d\n", m_CalcPropertyPtr->index);
            m_CalcPropertyPtr->state = PropertyState_HashNGEnd;
            m_State = State_None;
        }
        else
        {
            // スレッド生成成功 (計算中に遷移しておく)
            m_State = State_CalcProcessing;
        }
    }
    else if (m_State == State_CalcProcessing)
    {
        if (m_Hash.IsProcessing() == false)
        {
            char HashBuf[Hash::Sha1MaxStrSize + 1] = { 0 };
            // 計算処理の終了を検知した場合は、計算結果を取得する
            if (m_Hash.GetSha1Data(HashBuf, (sizeof(HashBuf) / sizeof(HashBuf[0]))) == 0)
            {
                HashBuf[Hash::Sha1MaxStrSize] = '\0';
            }

            m_CalcPropertyPtr->state = PropertyState_HashNGEnd;
            // 期待値と比較する
            if (m_CalcPropertyPtr->hashVal == HashBuf)
            {
                m_CalcPropertyPtr->state = PropertyState_HashOKEnd;
            }

            m_State = State_None;
            // (念のための後始末として)スレッドの終了処理を呼び出しておく
            m_Hash.StopSha1ProcessAsync();
        }
        else
        {
            // 計算中はプログレス値を取得する
            m_Progress = m_Hash.GetProgressPercent();
        }
    }
}

void AocInfoScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);

    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[AocInfo : count = %d ]", m_TotalAocCount);
    if (m_LastPage > 1)
    {
        writer->Print(" (%d / %d)", m_CurrentPage, m_LastPage);
    }

    if (m_IsReadingAocInfo == true)
    {
        // Aoc の情報をまだ読み込んでいる場合は以下の進捗画面を表示させておく
        writer->SetScale(2.5f, 2.5f);
        writer->SetCursor(400.0f, 280.0f);
        writer->Print("Reading Aoc Information ");

        // 読み込み中処理のちょっとしたアニメーション表現
        if (m_ReadingCalcCount > 100)
        {
            m_ReadingCalcCount = 0;
        }

        ++m_ReadingCalcCount;

        if (m_ReadingCalcCount < 25)
        {
            writer->Print(".   ");
        }
        else if (m_ReadingCalcCount < 50)
        {
            writer->Print("..  ");
        }
        else if (m_ReadingCalcCount < 75)
        {
            writer->Print("... ");
        }
        else
        {
            writer->Print("....");
        }

        // プログレス表示 ( 読み込み済の数 / 読み込み総数 )
        writer->SetCursor(500.0f, 350.0f);
        writer->Print("( %4d / %4d )", m_ReadAocCount, m_TotalAocCount);

        // 以降の処理は書き出さない・・
        return;
    }

    // Reload ボタンを描画
    this->DrawReloadButton(writer);

    if (m_HasAoc == false)
    {
        writer->SetScale(2.25f, 2.25f);
        writer->SetTextColor(Coral);
        writer->SetCursor(450.0f, 280.0f + 33.0f);
        writer->Print("No AddOnContent.");
        // 以降の処理は書き出さない・・
        return;
    }

    this->DrawAocList(writer);

    if (m_LastPage > 1)
    {
        if (m_CurrentPage > 1)
        {
            Position backPagePos(50.0f, 640.0f);
            static const char* BackPageStr = "<- : BackPage";
            this->WriteTouchRange(writer, backPagePos, BackPageStr, &m_BackPageRange);
        }
        if (m_CurrentPage < m_LastPage)
        {
            Position nextPagePos(1050.0f, 640.0f);
            static const char* NextPageStr = "-> : NextPage";
            this->WriteTouchRange(writer, nextPagePos, NextPageStr, &m_NextPageRange);
        }
    }

    if (m_State == State_None)
    {
        writer->SetTextColor(White);
        writer->SetCursor(100.0f, 600.0f);
        writer->Print("A: Calculate SHA1");
    }
}

void AocInfoScene::DrawAocList(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    // 等幅描画設定にする
    writer->SetFixedWidthEnabled(true);
    writer->SetFixedWidth(18.0f);

    writer->SetTextColor(White);
    writer->SetCursor(28.0f, 120.0f + (60.0f * (m_CurrentSelectPos)));
    writer->Print("*");

    int count = 0;
    int yPointCount = 0;
    for (auto& aoc : m_AocInfoList)
    {
        if (count < ((m_CurrentPage - 1) * MaxItemCount))
        {
            ++count;
            continue;
        }

        if (count >= (m_CurrentPage * MaxItemCount))
        {
            break;
        }

        const auto posY = 120.0f + (60.0f * (yPointCount));
        writer->SetCursor(50.0f, posY);
        writer->Print(aoc.prefixStr.c_str());
        if (aoc.isDefaultData == true)
        {
            const auto posX1 = writer->GetCursorX();
            // Version値の描画
            writer->Print("%10s", aoc.verVal.c_str());
            const auto posX2 = writer->GetCursorX();
            {
                // 16進数も少し小さく併記しておく
                writer->SetCursor(posX1 - 20.0f, posY + 29.0f);
                writer->SetScale(1.35f, 1.10f);

                writer->Print("(0x%s", aoc.rVerVal.c_str());
                writer->SetTextColor(RightGray);
                // PrivateVersion の値は薄いグレーにして見分けやすくする
                writer->Print(aoc.pVerVal.c_str());
                writer->SetTextColor(White);
                writer->Print(")");

                writer->SetCursor(posX2, posY);
                this->SetDefaultScale(writer);
            }
            writer->Print("]:(");
        }

        this->DrawAocItemDetail(writer, &aoc, yPointCount);

        ++count;
        ++yPointCount;
    }

    m_CurrentMaxPosNum = (yPointCount - 1);

    writer->SetFixedWidthEnabled(false);
}

void AocInfoScene::DrawAocItemDetail(nn::gfx::util::DebugFontTextWriter* writer, AocInfoProperty* aoc, int yPoint) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);
    NN_ABORT_UNLESS_NOT_NULL(aoc);

    auto& refRange = m_AocRangeList[yPoint];
    refRange.isTouchable = false;
    if (aoc->state == PropertyState_NotCalcHash)
    {
        writer->SetTextColor(White);
        if (aoc->isDefaultData == true)
        {
            // 未計算かつデフォルトデータの場合のみタッチできるようにする
            refRange.isTouchable = true;
        }
    }
    else if (aoc->state == PropertyState_CalcProcessing)
    {
        writer->SetTextColor(Yellow);
    }
    else if (aoc->state == PropertyState_HashOKEnd)
    {
        writer->SetTextColor(Green);
    }
    else if (aoc->state == PropertyState_HashNGEnd)
    {
        writer->SetTextColor(Red);
    }

    if (refRange.isSetting == false)
    {
        // Aoc プロパティのポインタを保持させておく
        refRange.dataPtr = &aoc;
    }

    // タッチ可能範囲を描画 (タッチできない場合もあるが・・)
    Position pos(writer->GetCursorX(), 120.0f + (60.0f * (yPoint)));
    this->WriteTouchRange(writer, pos, aoc->displayStr.c_str(), &refRange);

    writer->SetTextColor(White);
    writer->Print(")");

    if (aoc->state == PropertyState_CalcProcessing)
    {
        writer->SetFixedWidth(12.0f);
        // ハッシュの計算中であることを表示
        writer->SetScale(1.20f, 1.20f);
        writer->SetTextColor(Yellow);
        writer->Print("  Calculating ");

        // ちょっとした計算中アニメーション表現
        {
            if (m_CalcCount > 100)
            {
                m_CalcCount = 0;
            }

            ++m_CalcCount;

            if (m_CalcCount < 25)
            {
                writer->Print(".   ");
            }
            else if (m_CalcCount < 50)
            {
                writer->Print("..  ");
            }
            else if (m_CalcCount < 75)
            {
                writer->Print("... ");
            }
            else
            {
                writer->Print("....");
            }
        }

        // プログレス(パーセント)表示
        writer->Print(" [ %3d % ]", m_Progress);
        // スケールを元に戻しておく
        this->SetDefaultScale(writer);
        writer->SetFixedWidth(18.0f);
        writer->SetTextColor(White);
    }
    else if (aoc->state == PropertyState_HashOKEnd)
    {
        // 検証がOKである旨を表示
        writer->Print(" OK");
    }
    else if (aoc->state == PropertyState_HashNGEnd)
    {
        // 検証がNGである旨を表示
        writer->Print(" NG");
    }
}

void AocInfoScene::DrawReloadButton(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (m_IsBlinkReloadButton == false)
    {
        // 点滅フラグが無効の場合、従来の Reload ボタンの描画処理
        this->WriteTouchRange(writer, &m_ReloadRange, true);
        return;
    }

    // 点滅フラグが有効の場合、ブラケットの部分を点滅させる (以下の処理を実施)
    const int blinkIntervalParam = 30;
    const bool isDrawBracket = (m_ChangedEventCalcCount < blinkIntervalParam) ? true : false;

    writer->SetCursorX(m_ReloadRange.pos.x);

    if (isDrawBracket)
    {
        writer->SetTextColor(Yellow);
        writer->Print("[ ");
    }

    // 元々のX軸の座標値を保持しておく
    const auto xPos = m_ReloadRange.pos.x;
    // ブラケットを描画する幅分だけずらしておく
    m_ReloadRange.pos.x += writer->CalculateStringWidth("[ ");
    // 共通のブラケット描画処理は使用せずに文字だけを描画
    this->WriteTouchRange(writer, &m_ReloadRange);
    // 元々のX軸の座標値に戻しておく
    m_ReloadRange.pos.x = xPos;

    if (isDrawBracket)
    {
        writer->SetTextColor(Yellow);
        writer->Print(" ] NEW");
        writer->SetTextColor(White);
    }

    // 点滅用カウンタ変数の処理
    ++m_ChangedEventCalcCount;
    if (m_ChangedEventCalcCount >= (blinkIntervalParam * 2))
    {
        m_ChangedEventCalcCount = 0;
    }
}
