﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/account.h>
#include <nn/account/account_Result.h>

#include <nn/err.h>
#include <nn/fs/fs_SaveData.h>

#include "TestAppSimple_AccountScene.h"
#include "TestAppSimple_AccountManager.h"

// トレースログ出力を有効にしたい場合はこの define を有効にする
//#define ENABLE_TRACE_ACCOUNT_LOG
#if defined( ENABLE_TRACE_ACCOUNT_LOG )
#define TRACE_ACCOUNT_LOG(...)     NN_LOG(__VA_ARGS__)
#else
#define TRACE_ACCOUNT_LOG(...)
#endif // defined( ENABLE_TRACE_ACCOUNT_LOG )

namespace {
    const uint32_t ResultSuccessValue = 0;
}

AccountScene::AccountScene() NN_NOEXCEPT
    : m_State(State::None), m_IsAlreadyEnsured(false), m_IsAlreadyEnsuredSaveData(false),
    m_EnsureNSAResultValue(ResultSuccessValue), m_EnsureSaveDataResultValue(ResultSuccessValue),
    m_CurrentSelectPos(0), m_CurrentMaxPosNum(0)
{
}

void AccountScene::InternalSetup() NN_NOEXCEPT
{
    // NSAの認証処理開始タッチ位置の定義
    {
        m_EnsureNSARange.pos.x = 90.0f;
        m_EnsureNSARange.pos.y = 580.0f;
        m_EnsureNSARange.labelStr = "A: Ensure NSA";
    }

    // SaveDataの作成処理開始タッチ位置の定義
    {
        m_EnsureSaveDataRange.pos.x = 800.0f;
        m_EnsureSaveDataRange.pos.y = 580.0f;
        m_EnsureSaveDataRange.labelStr = "Y: Ensure SaveData";
    }

    // (SIGLO-79976) OpenUser処理のタッチ位置の定義
    {
        m_OpenUserRange.pos.x = 650.0f;
        m_OpenUserRange.pos.y = 90.0f;
        m_OpenUserRange.labelStr = "X: Open User";
    }

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // (SIGLO-79976) マルチプログラムアプリケーション向けの処理、起動直後に Open 済のユーザーアカウントがあるかどうかの確認
    auto& am = AccountManager::GetInstance();
    if (am.PopOpenUsers())
    {
        // 存在する場合は画面表示用のリストへ追加する
        std::list<AccountManager::UserInfo> userList;
        am.GetOpenedUserList(&userList);

        DisplayUserInfo displayInfo;
        for (auto&& info : userList)
        {
            displayInfo.id = info.id;
            displayInfo.name = info.name;

            m_DisplayOpenedUserList.push_back(displayInfo);
            ++m_CurrentMaxPosNum;
        }
    }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}


void AccountScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State::None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            TRACE_ACCOUNT_LOG("[Trace] Press A\n");
            m_State = State::Ensuring;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
        {
            TRACE_ACCOUNT_LOG("[Trace] Press Y\n");
            m_State = State::EnsuringSaveData;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            TRACE_ACCOUNT_LOG("[Trace] Press X\n");
            m_State = State::OpenUser;
        }
        else if (m_CurrentMaxPosNum > 0)
        {
            if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
            {
                TRACE_ACCOUNT_LOG("[Trace] Press B\n");

                // 選択されている項目のイテレータを取得
                auto selectedItr = std::begin(m_DisplayOpenedUserList);
                std::advance(selectedItr, m_CurrentSelectPos);
                // 選択されている Uid を取得
                m_SelectedCloseUserId = selectedItr->id;

                m_State = State::CloseUser;
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
                || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
            {
                if (m_CurrentSelectPos < (m_CurrentMaxPosNum - 1))
                {
                    ++m_CurrentSelectPos;
                }
                else
                {
                    m_CurrentSelectPos = 0;
                }
            }
            else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
                || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
            {
                if (m_CurrentSelectPos > 0)
                {
                    --m_CurrentSelectPos;
                }
                else
                {
                    m_CurrentSelectPos = (m_CurrentMaxPosNum - 1);
                }
            }
        }
    }
}

void AccountScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State::None)
    {
        if (m_EnsureNSARange.range.IsInRange(m_PreviousTouch) == true)
        {
            TRACE_ACCOUNT_LOG("[Trace] Touch EnsureNSA\n");
            m_State = State::Ensuring;
        }
        else if (m_EnsureSaveDataRange.range.IsInRange(m_PreviousTouch) == true)
        {
            TRACE_ACCOUNT_LOG("[Trace] Touch EnsureSaveData\n");
            m_State = State::EnsuringSaveData;
        }
        else if (m_OpenUserRange.range.IsInRange(m_PreviousTouch) == true)
        {
            TRACE_ACCOUNT_LOG("[Trace] Touch OpenUser\n");
            m_State = State::OpenUser;
        }

        for (auto&& user : m_DisplayOpenedUserList)
        {
            if (user.closeButton.range.IsInRange(m_PreviousTouch) == true)
            {
                m_SelectedCloseUserId = user.id;
                m_State = State::CloseUser;
                break;
            }
        }
    }
}

void AccountScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State::Ensuring)
    {
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        this->EnsureNSAProcess();
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // Generic 版では常に成功する形にしておく
        m_UserName = "WWWWWWWWWWWWWWWWWWWWWWWWWWWWW";
        m_EnsureNSAResultString = "Success";
        m_EnsureNSAResultValue = ResultSuccessValue;
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

        if (m_IsAlreadyEnsured == false)
        {
            m_IsAlreadyEnsured = true;
        }
    }
    else if(m_State == State::EnsuringSaveData)
    {
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        this->EnsureSaveDataProcess();
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // Generic 版では常に成功する形にしておく
        m_UserNameForSaveData = "WWWWWWWWWWWWWWWWWWWWWWWWWWWWW";
        m_EnsureSaveDataResultString = "Success";
        m_EnsureSaveDataResultValue = ResultSuccessValue;
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

        if (m_IsAlreadyEnsuredSaveData == false)
        {
            m_IsAlreadyEnsuredSaveData = true;
        }
    }
    else if (m_State == State::OpenUser)
    {
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        this->OpenUser();
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    }
    else if (m_State == State::CloseUser)
    {
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        this->CloseUser();
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    }

    // 平常状態に戻しておく
    m_State = State::None;
}

int AccountScene::EnsureNSAProcess() NN_NOEXCEPT
{
    // 一応、0 (Success)で初期化しておく
    m_EnsureNSAResultValue = ResultSuccessValue;

    nn::account::UserHandle handle;
    this->GetUserHandle(&handle);
    if (m_EnsureNSAResultValue != ResultSuccessValue)
    {
        // エラーが発生していれば返る
        return -1;
    }

    // スコープを抜けるタイミングでハンドルを閉じる
    NN_UTIL_SCOPE_EXIT
    {
        if (m_IsTemporaryOpened)
        {
            // 一時的な Open 処理であれば Close する
            TRACE_ACCOUNT_LOG("[Trace] << Closing user\n");
            nn::account::CloseUser(handle);
        }
    };

    this->CheckNetworkConnection();
    if (m_EnsureNSAResultValue != ResultSuccessValue)
    {
        // エラーが発生していれば返る
        return -1;
    }

    this->EnsureNetworkServiceAccount(handle);
    if (m_EnsureNSAResultValue != ResultSuccessValue)
    {
        // エラーが発生していれば返る
        return -1;
    }

    // ここまで来たら成功(Success)としておく
    m_EnsureNSAResultString = "Success";

    return 0;
}

void AccountScene::GetUserHandle(nn::account::UserHandle* outHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outHandle);

    // ユーザーアカウントの選択画面の表示
    // - ユーザー操作によるキャンセル時には nn::account::ResultCancelledByUser が返る。
    nn::account::Uid uid;
    auto result = nn::account::ShowUserSelector(&uid);
    if (nn::account::ResultCancelledByUser::Includes(result))
    {
        TRACE_ACCOUNT_LOG("[Trace] - User selection cancelled\n");
        m_EnsureNSAResultString = "Cancelled By User";
        m_EnsureNSAResultValue = result.GetInnerValueForDebug();
        return;
    }

    // (SIGLO-79976) OpenUser操作で Open 済の場合を考慮
    m_IsTemporaryOpened = true; // 初期状態は常に true とする
    {
        auto& am = AccountManager::GetInstance();
        AccountManager::UserInfo info;
        const auto isOpened = am.GetOpenedUserInfo(uid, &info);
        if (isOpened)
        {
            // すでに Open 済のユーザーがあればそのハンドルを返す
            TRACE_ACCOUNT_LOG("[Trace] (GetUserHandle)- User Exist\n");
            m_UserName = info.name;
            outHandle = &info.handle;
            // Ensure 処理後に勝手に Close されないようにする
            m_IsTemporaryOpened = false;
            return;
        }
    }

    // セーブデータ領域の初期化
    result = nn::fs::EnsureSaveData(uid);
    if (result.IsFailure())
    {
        TRACE_ACCOUNT_LOG("[Trace] - EnsureSaveData() Failed\n");
        m_EnsureNSAResultString = "EnsureSaveData() Failed";
        m_EnsureNSAResultValue = result.GetInnerValueForDebug();
        nn::err::ShowError(result);
        return;
    }

    // 選択されたユーザーアカウントのニックネームの取得
    // - ユーザーアカウントのニックネームやアイコンは Open せずに参照できる。
    nn::account::Nickname name;
    result = nn::account::GetNickname(&name, uid);
    if (result.IsSuccess())
    {
        TRACE_ACCOUNT_LOG("[Trace] - User to open: \"%s\"\n", name.name);
        m_UserName = name.name;
    }
    else
    {
        // ほぼありえないと思うが、ニックネームの取得に失敗した場合
        TRACE_ACCOUNT_LOG("[Trace] - account::GetNickname() Failed\n");
        m_EnsureNSAResultString = "account::GetNickname() Failed";
        m_EnsureNSAResultValue = result.GetInnerValueForDebug();
        return;
    }

    // 選択されたユーザーを Open
    // - ユーザーが使用中のユーザーアカウントを Open することで、フレンドプレゼンスが更新されるなど、特定ユーザー向けのサービスが提供される。
    TRACE_ACCOUNT_LOG("[Trace] >> Opening user\n");
    result = nn::account::OpenUser(outHandle, uid);
    if (result.IsFailure())
    {
        TRACE_ACCOUNT_LOG("[Trace] - account::OpenUser() Failed\n");
        m_EnsureNSAResultString = "account::OpenUser() Failed";
        m_EnsureNSAResultValue = result.GetInnerValueForDebug();
        return;
    }
}

void AccountScene::CheckNetworkConnection() NN_NOEXCEPT
{
    // ネットワーク接続要求
    while (NN_STATIC_CONDITION(true))
    {
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            auto result = nn::nifm::HandleNetworkRequestResult();
            if (!result.IsSuccess())
            {
                if (!nn::nifm::ResultErrorHandlingCompleted::Includes(result))
                {
                    // エラーかつ、それを解消できなかった場合はシーンを終了する。
                    TRACE_ACCOUNT_LOG("[Trace] - Pass nn::nifm::IsNetworkAvailable()\n");
                    m_EnsureNSAResultString = "Network Disable";
                    m_EnsureNSAResultValue = result.GetInnerValueForDebug();
                    return;
                }
                // エラーかつ、それを解消できた場合はリトライする。
                continue;
            }
        }
        // ネットワーク接続を利用できる。
        break;
    }
    TRACE_ACCOUNT_LOG("[Trace] - Pass nn::nifm::IsNetworkAvailable()\n");
}

void AccountScene::EnsureNetworkServiceAccount(const nn::account::UserHandle& inHandle) NN_NOEXCEPT
{
    Stopwatch watch;
    while (NN_STATIC_CONDITION(true))
    {
        // ネットワークサービスアカウントの確定
        // - ユーザー操作によるキャンセル時には nn::account::ResultCancelledByUser が返る。
        auto result = nn::account::EnsureNetworkServiceAccountAvailable(inHandle);
        if (result.IsFailure())
        {
            if (nn::account::ResultCancelledByUser::Includes(result))
            {
                // ユーザーによるキャンセル処理
                TRACE_ACCOUNT_LOG("[Trace] - EnsureNetworkServiceAccountAvailable cancelled\n");
                m_EnsureNSAResultString = "Cancelled By User";
            }
            else
            {
                TRACE_ACCOUNT_LOG("[Trace] - EnsureNetworkServiceAccountAvailable() Failed\n");
                m_EnsureNSAResultString = "EnsureNetworkServiceAccountAvailable() Failed";
            }
            m_EnsureNSAResultValue = result.GetInnerValueForDebug();
            return;
        }

        // ネットワークサービスアカウントの情報
        // - ユーザーはネットワークサービスアカウントを付け替えることができるため、アプリケーション起動ごとに変わっている可能性がある。
        nn::account::NetworkServiceAccountId nsaId;
        result = nn::account::GetNetworkServiceAccountId(&nsaId, inHandle);
        if (result.IsFailure())
        {
            TRACE_ACCOUNT_LOG("[Trace] - GetNetworkServiceAccountId() Failed\n");
            m_EnsureNSAResultString = "account::GetNetworkServiceAccountId() Failed";
            m_EnsureNSAResultValue = result.GetInnerValueForDebug();
            return;
        }
        TRACE_ACCOUNT_LOG("[Trace] - Network Service Account ID: 0x%016llx\n", nsaId.id);

        // ネットワークサービスアカウントの ID トークンキャッシュを確定
        nn::account::AsyncContext ctx;
        {
            // このスコープの処理時間を測る
            watch.Start();
            NN_UTIL_SCOPE_EXIT
            {
                watch.Stop();
            };

            // ネットワークサービスアカウントへのログイン + ID トークンのキャッシュ API
            // - すでにキャッシュが存在する場合は即時完了する。
            // - ctx.Cancel() を呼ぶと、可能な限り早くイベントがシグナルされる。
            result = nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(&ctx, inHandle);
            if (result.IsFailure())
            {
                TRACE_ACCOUNT_LOG("[Trace] - EnsureNetworkServiceAccountIdTokenCacheAsync() Failed\n");
                m_EnsureNSAResultString = "EnsureNetworkServiceAccountIdTokenCacheAsync() Failed";
                m_EnsureNSAResultValue = result.GetInnerValueForDebug();
                return;
            }
            nn::os::SystemEvent e;
            NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
            TRACE_ACCOUNT_LOG("[Trace] - >> EnsureNetworkServiceAccountIdTokenCacheAsync : Wait In\n");
            e.Wait();
            TRACE_ACCOUNT_LOG("[Trace] - << EnsureNetworkServiceAccountIdTokenCacheAsync : Wait Out\n");
        }

        result = ctx.GetResult();
        TRACE_ACCOUNT_LOG("[Trace] - GetResult: 0x%08x\n", result.GetInnerValueForDebug());
        if (result.IsFailure())
        {
            if (nn::account::ResultNetworkServiceAccountUnavailable::Includes(result))
            {
                // ログイン時にネットワークサービスアカウントが利用できないことが発覚
                // - nn::account::EnsureNetworkServiceAccountAvailable() でエラーを解消できるため、再試行する。
                // - 再試行や本体システム UI の表示が困難な場合は nn::error::ShowError() に渡してもよい。
                TRACE_ACCOUNT_LOG("[Trace] - ResultNetworkServiceAccountUnavailable::Includes\n");
                continue;
            }
            else if (nn::account::ResultNetworkCommunicationError::Includes(result))
            {
                // ログイン時に通信関係のエラーが発生
                // - これにはアプリケーションの更新が必要な場合や、本体更新が必要な場合も含まれる。
                // - nn::error::ShowError() に渡し、詳細なエラー内容をユーザーに提示する必要がある。
                nn::err::ShowError(result);
                TRACE_ACCOUNT_LOG("[Trace] - ResultNetworkCommunicationError::Includes\n");
                m_EnsureNSAResultString = "Network Communication Error";
            }
            else
            {
                TRACE_ACCOUNT_LOG("[Trace] - EnsureNetworkServiceAccountIdTokenCacheAsync() Failed\n");
                m_EnsureNSAResultString = "EnsureNetworkServiceAccountIdTokenCacheAsync() Failed";
            }

            m_EnsureNSAResultValue = result.GetInnerValueForDebug();

            return;
        }
        break;
    }
    TRACE_ACCOUNT_LOG("[Trace] - Network Service Account login turnaround: %lld msec\n", watch.GetMilliSecond());
}

void AccountScene::EnsureSaveDataProcess() NN_NOEXCEPT
{
    // ユーザーアカウントの選択画面の表示
    // - ユーザー操作によるキャンセル時には nn::account::ResultCancelledByUser が返る。
    nn::account::Uid uid;
    auto result = nn::account::ShowUserSelector(&uid);
    if (nn::account::ResultCancelledByUser::Includes(result))
    {
        TRACE_ACCOUNT_LOG("[Trace] - User selection cancelled\n");
        m_EnsureSaveDataResultString = "Cancelled By User";
        m_EnsureSaveDataResultValue = result.GetInnerValueForDebug();
        return;
    }

    // 選択されたユーザーアカウントのニックネームの取得
    // - ユーザーアカウントのニックネームやアイコンは Open せずに参照できる。
    nn::account::Nickname name;
    result = nn::account::GetNickname(&name, uid);
    if (result.IsSuccess())
    {
        TRACE_ACCOUNT_LOG("[Trace] - User to open: \"%s\"\n", name.name);
        m_UserNameForSaveData = name.name;
    }
    else
    {
        // ほぼありえないと思うが、ニックネームの取得に失敗した場合
        TRACE_ACCOUNT_LOG("[Trace] - account::GetNickname() Failed\n");
        m_EnsureSaveDataResultString = "account::GetNickname() Failed";
        m_EnsureSaveDataResultValue = result.GetInnerValueForDebug();
        return;
    }

    // セーブデータ領域の初期化
    result = nn::fs::EnsureSaveData(uid);
    if (result.IsFailure())
    {
        TRACE_ACCOUNT_LOG("[Trace] - EnsureSaveData() Failed\n");
        m_EnsureSaveDataResultString = "EnsureSaveData() Failed";
        m_EnsureSaveDataResultValue = result.GetInnerValueForDebug();
        nn::err::ShowError(result);
        return;
    }

    // ここまで来たら成功(Success)としておく
    m_EnsureSaveDataResultString = "Success";
    m_EnsureSaveDataResultValue = ResultSuccessValue;
}

void AccountScene::OpenUser() NN_NOEXCEPT
{
    auto& am = AccountManager::GetInstance();

    AccountManager::UserInfo info;
    const auto isSuccess = am.OpenUserWithSelectorApplet(&info);
    if (isSuccess)
    {
        DisplayUserInfo displayInfo;
        displayInfo.id = info.id;
        displayInfo.name = info.name;

        m_DisplayOpenedUserList.push_back(displayInfo);
        ++m_CurrentMaxPosNum;
    }
}

void AccountScene::CloseUser() NN_NOEXCEPT
{
    // 念のためのガード処理
    if (m_CurrentMaxPosNum <= 0)
    {
        return;
    }

    auto& am = AccountManager::GetInstance();
    am.CloseUser(m_SelectedCloseUserId);

    bool isReset = false;
    auto itr = std::begin(m_DisplayOpenedUserList);
    while (itr != std::end(m_DisplayOpenedUserList))
    {
        if (isReset == true)
        {
            // 消されたユーザーの"後"のボタン設定をリセットする (ボタン配置が変わるため)
            itr->closeButton.range.isSetting = false;
        }

        if (isReset == false && m_SelectedCloseUserId == itr->id)
        {
            // 該当するデータを消す
            itr = m_DisplayOpenedUserList.erase(itr);
            isReset = true;
            continue;
        }

        ++itr;
    }

    --m_CurrentMaxPosNum;
    m_CurrentSelectPos = 0;
}

void AccountScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[Account]");

    const float xPos = 50.0f;
    const float xPos2 = 75.0f;
    const float baseYPos = 110.0f;
    const float LineSize = 33.0f;
    int ypos = 0;

    {
        writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
        writer->Print("- ApplicationId : %s", m_ApplicationInfo.id.c_str());
        ++ypos;
    }
    {
        writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
        writer->Print("- Version [32bit] : %d ( 0x%04x", m_ApplicationInfo.ver.unifiedVer, m_ApplicationInfo.ver.releaseVer);
        // PrivateVersion の部分は灰色にしておく
        writer->SetTextColor(RightGray);
        writer->Print("%04x", m_ApplicationInfo.ver.privateVer);
        writer->SetTextColor(White);
        writer->Print(" )");
        ++ypos;
    }

    ++ypos;

    {
        writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
        writer->Print("- Network Service Account (NSA)");
        ++ypos;
    }
    {
        writer->SetCursor(xPos2, baseYPos + (LineSize * (ypos)));
        this->DrawUserName(writer, m_IsAlreadyEnsured, m_UserName);
        ++ypos;
    }
    {
        writer->SetCursor(xPos2, baseYPos + (LineSize * (ypos)));
        this->DrawLastResult(writer, m_IsAlreadyEnsured, m_EnsureNSAResultString, m_EnsureNSAResultValue);
        ++ypos;
    }

    ++ypos;

    {
        writer->SetCursor(xPos, baseYPos + (LineSize * (ypos)));
        writer->Print("- User Account Save Data");
        ++ypos;
    }
    {
        writer->SetCursor(xPos2, baseYPos + (LineSize * (ypos)));
        this->DrawUserName(writer, m_IsAlreadyEnsuredSaveData, m_UserNameForSaveData);
        ++ypos;
    }
    {
        writer->SetCursor(xPos2, baseYPos + (LineSize * (ypos)));
        this->DrawLastResult(writer, m_IsAlreadyEnsuredSaveData, m_EnsureSaveDataResultString, m_EnsureSaveDataResultValue);
        ++ypos;
    }

    // (SIGLO-79976) Open 済ユーザーのリスト表示
    this->DrawOpenedUserList(writer);

    if (m_State == State::None)
    {
        // 目立つように少し大きめに描画しておく
        writer->SetScale(2.5f, 2.5f);
        this->WriteTouchRange(writer, &m_EnsureNSARange);
        this->WriteTouchRange(writer, &m_EnsureSaveDataRange);
    }
}

void AccountScene::DrawUserName(nn::gfx::util::DebugFontTextWriter* writer, bool inAlreadyFlag, const std::string& inUserName) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (inAlreadyFlag == false)
    {
        writer->Print("- User Name : ---");
    }
    else
    {
        writer->Print("- User Name : %s", inUserName.c_str());
    }
}

void AccountScene::DrawLastResult(nn::gfx::util::DebugFontTextWriter* writer, bool inAlreadyFlag, const std::string& inResultString, uint32_t inResultValue) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    if (inAlreadyFlag == false)
    {
        writer->Print("- Last Result : ---");
    }
    else
    {
        writer->Print("- Last Result : ");
        if (inResultValue == ResultSuccessValue)
        {
            writer->SetTextColor(Green);
        }
        writer->Print(inResultString.c_str());
        writer->SetTextColor(White);
        writer->Print(" (0x%08x)", inResultValue);
    }
}

void AccountScene::DrawOpenedUserList(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetScale(2.0f, 2.0f);
    this->WriteTouchRange(writer, &m_OpenUserRange, true);

    writer->SetTextColor(White);

    writer->SetCursorX(writer->GetCursorX() + 45.0f);
    if (m_DisplayOpenedUserList.empty() == false)
    {
        writer->Print("B: Close User");
    }

    this->SetDefaultScale(writer);
    auto posY = m_OpenUserRange.pos.y + 45.0f;
    writer->SetCursor(m_OpenUserRange.pos.x, posY);
    writer->Print("- Opened User List");

    constexpr auto spacingLength = 47.0f;
    if (m_DisplayOpenedUserList.empty() == false)
    {
        posY += spacingLength;
        writer->SetCursor(m_OpenUserRange.pos.x + 5.0f, posY + (m_CurrentSelectPos * spacingLength));
        writer->Print("*");
    }

    for (auto&& user : m_DisplayOpenedUserList)
    {
        writer->SetCursor(m_OpenUserRange.pos.x + 30.0f, posY);

        writer->Print("%s", user.name.c_str());

        auto& refRange = user.closeButton;
        if (refRange.range.isSetting == false)
        {
            refRange.labelStr = "B: Close";
            refRange.pos.y = posY;
            refRange.pos.x = 1000.0f;
        }

        this->WriteTouchRange(writer, &refRange, true);

        posY += spacingLength;
    }
}
