﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Text;
using System.Xml;
using Nintendo.Foundation.IO;
using System.Collections.Generic;

namespace MakeTestApplication
{
    internal class CommonParams
    {
        public const string WorkDirName = "_WorkTemporary_";
        public const string TempMetaFileName = "tempMeta.nmeta";
        public const string DefaultCodeSize = "4";
        public const string DefaultProgramId = "0x01004b9000490000";
        public const string DefaultVersion = "0";
        public const string DefaultRequiredVersion = "0";
        public const string DefaultIndex = "1";
        public const int    DefaultPallarelNum = 1;

        public const int AocMinIndexValue = 1;
        public const int AocMaxIndexValue = 2047;

        public const string ContentTypeApplication = "Application";
        public const string ContentTypeSystemProgram = "SystemProgram";
        public const string ContentTypeSystemData = "SystemData";
        public const string ContentTypeAddOnContent = "AddOnContent";
        public const string ContentTypePatch = "Patch";

        public static readonly HashSet<string> IconLanguageSet = new HashSet<string>()
        {
        "AmericanEnglish",
        "BritishEnglish",
        "Japanese",
        "French",
        "German",
        "LatinAmericanSpanish",
        "Spanish",
        "Italian",
        "Dutch",
        "CanadianFrench",
        "Portuguese",
        "Russian",
        "Korean",
        "SimplifiedChinese",
        "TraditionalChinese",
        };
        public const string DefaultIconLanguage = "AmericanEnglish";
        public const string NoDefaultIconSetString = "noicon";

        public const string NmetaRootName = "NintendoSdkMeta";

        public const string OriginalApplicationCopyFileName = "_original_app.nsp";

        private const string SigloRootMarkFileName = "SigloRootMark";
        private const string NintendoSdkRootMarkFileName = "NintendoSdkRootMark";

        /// <summary>
        /// Sigloのルートパスを取得
        /// </summary>
        /// <returns>Sigloルートパス文字列</returns>
        public static string GetSigloRoot()
        {
            // ルートパスを確定させるために "～RootMark" ファイルを検索する
            var rootPath = FindSigloRoot(Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location));
            if (rootPath == null)
            {
                // もし取得できない場合は、環境変数"NINTENDO_SDK_ROOT"を取得
                rootPath = Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");
                if (rootPath == null)
                {
                    throw new InvalidDataException("Rootパスを見つけられませんでした");
                }
            }

            if (rootPath.EndsWith(@"\") == false)
            {
                rootPath += @"\";
            }

            return rootPath;
        }

        private static string FindSigloRoot(string path)
        {
            string startPath = path;
            string currentDirectory = startPath;
            string rootDirectory = Path.GetPathRoot(currentDirectory);

            while (rootDirectory != currentDirectory)
            {
                // "SigloRootMark"の存在確認
                var sigloRootMarkFilePath = Path.Combine(currentDirectory, SigloRootMarkFileName);
                if (File.Exists(sigloRootMarkFilePath))
                {
                    return currentDirectory;
                }

                // "NintendoSdkRootMark"の存在確認
                var nintendoSdkRootMarkFilePath = Path.Combine(currentDirectory, NintendoSdkRootMarkFileName);
                if (File.Exists(nintendoSdkRootMarkFilePath))
                {
                    return currentDirectory;
                }

                currentDirectory = Path.GetFullPath(Path.Combine(currentDirectory, ".."));
            }

            return null;
        }
    }

    internal class CommandLineParams
    {
        [CommandLineOption("id",
            Description = "プログラム(アプリケーション)ID を指定します",
            ValueName = "プログラムID")]
        public string InputProgramId { get; set; }

        [CommandLineOption("program-index",
            Description = "プログラム Index を指定します",
            IsHidden = true)]
        public string InputProgramIndex { get; set; }

        [CommandLineOption("name",
            DefaultValue = null,
            Description = "プログラム名を指定します",
            ValueName = "プログラム名")]
        public string InputProgramName { get; set; }

        [CommandLineOption("ver",
            DefaultValue = null,
            Description = "(リリース)バージョン値を指定します",
            ValueName = "バージョン")]
        public string InputVersion { get; set; }

        [CommandLineOption("release-version",
            DefaultValue = null,
            Description = "リリースバージョン値を指定します",
            ValueName = "リリースバージョン",
            IsHidden = true)]
        public string InputReleaseVersion { get; set; }

        [CommandLineOption("private-version",
            DefaultValue = null,
            Description = "プライベートバージョン値を指定します",
            ValueName = "プライベートバージョン",
            IsHidden = true)]
        public string InputPrivateVersion { get; set; }

        [CommandLineOption("unified-version",
            DefaultValue = null,
            Description = "バージョン値(32ビット)を指定します",
            ValueName = "統合バージョン",
            IsHidden = true)]
        public string InputUnifiedVersion { get; set; }

        [CommandLineOption("required-version",
            DefaultValue = null,
            Description = "必須バージョン値を指定します",
            ValueName = "必須バージョン",
            IsHidden = true)]
        public string InputRequiredVersion { get; set; }

        [CommandLineOption("index",
            DefaultValue = null,
            Description = "インデックス番号を指定します(AddOnContentのみ有効)",
            ValueName = "インデックス")]
        public string[] InputIndex { get; set; }

        [CommandLineOption("tag",
            DefaultValue = null,
            Description = "タグ値を指定します(AddOnContentのみ有効)",
            ValueName = "タグ")]
        public string InputTag { get; set; }

        [CommandLineOption("size",
            DefaultValue = CommonParams.DefaultCodeSize,
            Description = "ダミーファイルのサイズ[Byte]を指定します(未指定の場合は4バイト固定)",
            ValueName = "サイズ")]
        public string OutputProgramSize { get; set; }

        [CommandLineOption("seed",
            DefaultValue = null,
            Description = "ランダムファイルのseed値を指定します",
            ValueName = "Seed",
            IsHidden = true)]
        public string SeedNumber { get; set; }

        [CommandLineOption("type",
            DefaultValue = CommonParams.ContentTypeApplication,
            Description = "コンテンツタイプを指定します(未指定の場合は\"Application\"を適用)",
            ValueName = "タイプ")]
        public string InputContentType { get; set; }

        [CommandLineOption("data-directory",
            DefaultValue = null,
            Description = "内部データに含めるディレクトリパスを指定します(SystemData, AddOnContentのみ有効)",
            ValueName = "dataディレクトリ")]
        public string DataDirPath { get; set; }

        [CommandLineOption('o', "out-directory",
            DefaultValue = null,
            Description = ".nsp を出力するディレクトリパスを指定します",
            ValueName = "出力先ディレクトリ")]
        public string OutputDirPath { get; set; }

        [CommandLineOption("output-file-name",
                    DefaultValue = null,
                    Description = "出力するファイル名を指定します",
                    ValueName = "出力ファイル名")]
        public string OutputFileName { get; set; }

        [CommandLineOption("output-file-path",
            DefaultValue = null,
            Description = "出力ファイルパスを指定します",
            ValueName = "出力ファイルパス")]
        public string OutputFilePath { get; set; }

        [CommandLineOption("work-directory",
            DefaultValue = null,
            Description = "作業用のディレクトリパスを指定します",
            ValueName = "作業ディレクトリ")]
        public string WorkDirPath { get; set; }

        [CommandLineOption("config-xml",
            DefaultValue = null,
            Description = "設定用XMLファイルパスを指定します",
            ValueName = "設定用XMLファイル")]
        public string ConfigXmlPath { get; set; }

        [CommandLineOption('j', "execute-parallel",
            DefaultValue = null,
            Description = "最大並列数を指定します(config-xml指定時のみ有効)",
            ValueName = "最大並列数")]
        public string MaxParallelNum { get; set; }

        [CommandLineOption("meta",
            DefaultValue = null,
            Description = "metaファイルパスを指定します",
            ValueName = "metaファイル")]
        public string MetaFilePath { get; set; }

        [CommandLineOption("desc",
            DefaultValue = null,
            Description = "descファイルパスを指定します",
            ValueName = "descファイル")]
        public string DescFilePath { get; set; }

        [CommandLineOption("html-document",
            DefaultValue = null,
            Description = "遊び方ドキュメントのディレクトリパスを指定します",
            ValueName = "html-documentディレクトリ")]
        public string HtmlDocumentDirPath { get; set; }

        [CommandLineOption("accessible-urls",
            DefaultValue = null,
            Description = "接続可能なURL情報のファイルパスを指定します",
            ValueName = "accessible-urlsファイル")]
        public string AccessibleUrlsFilePath { get; set; }

        [CommandLineOption("legal-information",
            DefaultValue = null,
            Description = "ソフトリーガル情報のzipファイルパスを指定します",
            ValueName = "legal-information zipファイル")]
        public string LegalInformationZipFilePath { get; set; }

        [CommandLineOption("icon",
            DefaultValue = null,
            Description = "アイコン情報を指定します")]
        public string IconSetting { get; set; }

        [CommandLineOption("nx-icon",
            DefaultValue = null,
            Description = "NXで使用するアイコン情報を指定します")]
        public string NxIconSetting { get; set; }

        [CommandLineOption('l', "default-icon-language",
            DefaultValue = null,
            Description = "デフォルトアイコンの設定言語を指定します",
            ValueName = "アイコン言語")]
        public string[] DefaultIconLanguageList { get; set; }

        [CommandLineOption("original-application",
            DefaultValue = null,
            Description = "初版アプリのファイルパスを指定します(Patch指定時のみ有効)",
            ValueName = "nspファイルパス")]
        public string OriginalApplicationFilePath { get; set; }

        [CommandLineOption("previous-patch",
            DefaultValue = null,
            Description = "直前パッチのファイルパスを指定します(Patch指定時のみ有効)",
            ValueName = "nspファイルパス")]
        public string PreviousPatchFilePath { get; set; }

        [CommandLineOption("original-application-for-creatensp",
            DefaultValue = null,
            Description = "初版アプリのファイルパスを指定します",
            IsHidden = true)]
        public string OriginalApplicationForCreateNspFilePath { get; set; }

        [CommandLineOption("keyindex",
            DefaultValue = null,
            Description = "keyindex値を指定",
            IsHidden = true)]
        public string KeyIndexNum { get; set; }

        [CommandLineOption("keyconfig",
            DefaultValue = null,
            Description = "keyconfigファイルのパスを指定",
            IsHidden = true)]
        public string KeyConfigFilePath { get; set; }

        [CommandLineOption("cardspec-size",
            DefaultValue = null,
            Description = "CardSpecのSize値を指定",
            IsHidden = true)]
        public string CardSpecSize { get; set; }

        [CommandLineOption("cardspec-clockrate",
            DefaultValue = null,
            Description = "CardSpecのClockRate値を指定",
            IsHidden = true)]
        public string CardSpecClockRate { get; set; }

        [CommandLineOption("savedata-size",
            DefaultValue = null,
            Description = "ユーザアカウントセーブデータのSize値を指定",
            IsHidden = true)]
        public string SaveDataSize { get; set; }

        [CommandLineOption("savedata-journal-size",
            DefaultValue = null,
            Description = "ユーザアカウントセーブデータのジャーナルSize値を指定",
            IsHidden = true)]
        public string SaveDataJournalSize { get; set; }

        [CommandLineOption("bcat-storage-size",
            DefaultValue = null,
            Description = "BCATデータの保存領域Size値を指定",
            IsHidden = true)]
        public string BcatStorageSize { get; set; }

        [CommandLineOption("bcat-passphrase",
            DefaultValue = null,
            Description = "BCATのパスフレーズを指定",
            IsHidden = true)]
        public string BcatPassphrase { get; set; }

        [CommandLineOption("cache-storage-size",
            DefaultValue = null,
            Description = "キャッシュストレージのSize値を指定",
            IsHidden = true)]
        public string CacheStorageSize { get; set; }

        [CommandLineOption("cache-storage-journal-size",
            DefaultValue = null,
            Description = "キャッシュストレージのジャーナルSize値を指定",
            IsHidden = true)]
        public string CacheStorageJournalSize { get; set; }

        [CommandLineOption("no-build",
            Description = "ビルドなし(旧)モードで動作します",
            IsHidden = true)]
        public bool NoBuildFlag { get; set; }

        [CommandLineOption("verbose",
            Description = "詳細ログ出力モードで動作します")]
        public bool VerboseFlag { get; set; }

        [CommandLineOption("build-app-code",
            Description = "テストアプリのコードを構築します",
            IsHidden = true)]
        public bool TestAppBuildFlag { get; set; }

        [CommandLineOption("small-code",
            Description = "小サイズ版テストアプリのコードを使用します",
            IsHidden = true)]
        public bool SmallCodeFlag { get; set; }

        [CommandLineOption("no-cleanup",
            Description = "処理終了時に作業ディレクトリを削除せず残します")]
        public bool NoCleanupFlag { get; set; }

        [CommandLineOption("ticket",
            Description = "外部鍵で暗号化します",
            IsHidden = true)]
        public bool TicketFlag { get; set; }

        [CommandLineOption("no-ticket",
            Description = "内部鍵で暗号化します",
            IsHidden = true)]
        public bool NoTicketFlag { get; set; }

        [CommandLineOption("code",
            DefaultValue = null,
            Description = "codeディレクトリを直接指定します",
            IsHidden = true)]
        public string CodeDirPath { get; set; }

        [CommandLineOption("romfs",
            DefaultValue = null,
            Description = "romfsディレクトリを直接指定します",
            IsHidden = true)]
        public string RomFsDirPath { get; set; }

        [CommandLineOption("debug-delta",
            Description = "パッチ間差分のデバック用オプションを付加します",
            IsHidden = true)]
        public bool DebugDeltaFlag { get; set; }

        [CommandLineOption("abort-result",
            DefaultValue = null,
            Description = "AbortさせるResult値を指定します(16進数8桁)",
            IsHidden = true)]
        public string AbortResultValue { get; set; }

        [CommandLineOption("network-request",
            Description = "ネットワーク接続要求を起動時に自動実行します",
            IsHidden = true)]
        public bool NetworkRequestFlag { get; set; }

        [CommandLineOption("required-startup-useraccount",
            Description = "起動時に本体アカウントを選択するようにします",
            IsHidden = true)]
        public bool RequiredStartupUserAccountFlag { get; set; }

        [CommandLineOption("runtime-install-aoc",
            Description = "AoCの動的コミット機能を有効にします",
            IsHidden = true)]
        public bool RuntimeInstallAocFlag { get; set; }

        [CommandLineOption("direct",
            DefaultValue = null,
            Description = "createnspのオプション文字列を直接指定します",
            IsHidden = true)]
        public string DirectString { get; set; }

        [CommandLineOption("direct-for-makepatch",
            DefaultValue = null,
            Description = "makepatchのオプション文字列を直接指定します",
            IsHidden = true)]
        public string DirectStringForMakePatch { get; set; }

        [CommandLineOption("multi-program",
            DefaultValue = null,
            Description = "マルチプログラムを生成します",
            IsHidden = true)]
        public string MultiProgram { get; set; }

        [CommandLineOption("check-valid-range-id",
            DefaultValue = false,
            Description = "プログラム(アプリケーション)ID が有効範囲内かチェックします")]
        public bool CheckValidRangeIdFlag { get; set; }
    }

    internal class SettingParams
    {
        internal class IconSettingPair
        {
            public string Language { get; set; }
            public string IconFilePath { get; set; }

            public IconSettingPair()
            {
            }

            public IconSettingPair(string inLanguage, string inFilePath)
            {
                Language = inLanguage;
                IconFilePath = inFilePath;
            }
        }

        internal class CardSpecParam
        {
            public string Size { get; set; }
            public string ClockRate { get; set; }

            public CardSpecParam()
            {
                // 初期値は null を入れておく
                Size = null;
                ClockRate = null;
            }

            // 何らかのデータが設定されているかどうかの判定関数
            public bool IsSetAnyData()
            {
                return (Size == null && ClockRate == null) ? false : true;
            }
        }

        internal class AocProperty
        {
            public string Index { get; set; }
            public string ApplicationId { get; set; }
            public string ReleaseVersion { get; set; }
            public string PrivateVersion { get; set; }
            public string Version { get; set; }
            public string RequiredApplicationReleaseVersion { get; set; }
            public string Tag { get; set; }
            public string DataPath { get; set; }
            public bool IsAlreadySetDataPath { get; set; }
        }

        public string ContentType { get; set; }
        public string MetaFilePath { get; set; }
        public string DescFilePath { get; set; }
        public string OutputFileName { get; set; }
        public string OutputFilePath { get; set; }

        public string HtmlDocumentDirPath { get; set; }
        public string AccessibleUrlsFilePath { get; set; }
        public string LegalInformationZipFilePath { get; set; }

        public string CodeSize { get; set; }
        public string SeedNumber { get; set; }
        public string DataDirPath { get; set; }

        public string CodeDirPath { get; set; }
        public string RomFsDirPath { get; set; }

        public string MetaName { get; set; }
        public string MetaInputId { get; set; } // 入力 ID 値をそのまま保持するイメージ
        public string MetaProgramIndex { get; set; }
        // 基本的には MetaInputId と同値となるが、ProgramIndex 設定時に異なる場合がありえる
        public string MetaProgramId { get; set; }
        public string MetaVersion { get; set; }
        public string MetaIndex { get; set; }
        public string MetaTag { get; set; }

        public string ReleaseVersion { get; set; }
        public string PrivateVersion { get; set; }
        public string Version { get; set; }
        public string RequiredVersion { get; set; }

        public bool IsSetName { get; set; }
        public bool IsSetMetaIcon { get; set; }
        public bool IsEditMetaFileForCommandArgs { get; set; }
        public bool IsSetOutputFileName { get; set; }

        public List<IconSettingPair> IconList { get; set; }
        public List<IconSettingPair> NxIconList { get; set; }
        public List<IconSettingPair> DefaultIconList { get; set; }
        public List<AocProperty> AocList { get; set; }

        public string KeyIndexNum { get; set; }
        public string KeyConfigFilePath { get; set; }

        public CardSpecParam CardSpec { get; set; }

        public string ChildWorkDirPath { get; set; }

        public bool TicketFlag { get; set; }
        public bool NoTicketFlag { get; set; }

        // 一括生成時に何番目のデータなのかを保持しておく
        public int SequenceNumber { get; set; }

        public string OriginalApplicationFilePath { get; set; }
        public string OriginalApplicationForCreateNspFilePath { get; set; }
        public string PreviousPatchFilePath { get; set; }
        public bool DebugDeltaFlag { get; set; }

        public string AbortResultValue { get; set; }

        public bool NetworkRequestFlag { get; set; }

        public string SaveDataSize { get; set; }
        public string SaveDataJournalSize { get; set; }
        public bool RequiredStartupUserAccountFlag { get; set; }

        public string BcatStorageSize { get; set; }
        public string BcatPassphrase { get; set; }

        public string CacheStorageSize { get; set; }
        public string CacheStorageJournalSize { get; set; }

        public bool RuntimeInstallAocFlag { get; set; }

        public string DirectString { get; set; }
        public string DirectStringForMakePatch { get; set; }

        public string MultiProgram { get; set; }

        public bool CheckValidRangeIdFlag { get; set; }

        private void InitParameter()
        {
            this.IconList = new List<IconSettingPair>();
            this.NxIconList = new List<IconSettingPair>();
            this.DefaultIconList = new List<IconSettingPair>();
            this.AocList = new List<AocProperty>();
            this.CardSpec = null;

            this.IsSetMetaIcon = false;
            this.IsEditMetaFileForCommandArgs = false;
        }

        public SettingParams()
        {
            this.InitParameter();
        }

        public SettingParams(CommandLineParams inParam)
        {
            this.InitParameter();
            this.SetMetaParam(inParam);
        }

        /// <summary>
        /// コマンドラインの入力情報からメタ情報を抽出し自身のメンバ変数に設定する
        /// </summary>
        /// <param name="inParam">コマンドラインの入力オプションパラメータ</param>
        public void SetMetaParam(CommandLineParams inParam)
        {
            // コードサイズはひとまずそのまま設定する
            this.CodeSize = inParam.OutputProgramSize;

            // プログラムIDの情報が入力されていない(nullの)場合はデフォルトのプログラムIDを適用する
            this.MetaInputId = inParam.InputProgramId ?? CommonParams.DefaultProgramId;

            this.IsSetName = false;
            if (inParam.InputProgramName != null && inParam.InputProgramName != string.Empty)
            {
                this.MetaName = inParam.InputProgramName;
                this.IsSetName = true;
            }

            this.IsSetOutputFileName = false;
            if (inParam.OutputFileName != null && inParam.OutputFileName != string.Empty)
            {
                this.OutputFileName = inParam.OutputFileName;
                this.IsSetOutputFileName = true;
            }

            // ReleaseVersion, PrivateVersion, Version(32bit) 値を設定
            this.SetVersionValue(inParam);
            // ツール的に MetaVersion は ReleaseVersion と同値にしておく
            this.MetaVersion = this.ReleaseVersion;

            // 必須バージョンが null の場合はデフォルト値( 0 )を適用する
            this.RequiredVersion = inParam.InputRequiredVersion ?? CommonParams.DefaultRequiredVersion;
            {
                // UInt32の範囲に値が設定されているかどうかの確認
                uint num = 0;
                var result = uint.TryParse(this.RequiredVersion, out num);
                if (result == false)
                {
                    // 不正な設定値の場合もデフォルト値( 0 )を適用する
                    this.RequiredVersion = CommonParams.DefaultRequiredVersion;
                }
            }

            // コンテンツタイプの設定 (デフォルト値はApplication)
            this.ContentType = CommonParams.ContentTypeApplication;
            switch (inParam.InputContentType)
            {
                case CommonParams.ContentTypeSystemProgram:
                case CommonParams.ContentTypeSystemData:
                case CommonParams.ContentTypeAddOnContent:
                case CommonParams.ContentTypePatch:
                    this.ContentType = inParam.InputContentType;
                    break;
                default:
                    // 指定外の文字列が設定された場合はデフォルト値(Application)を適用
                    break;
            }

            // コマンド入力情報向けの処理でコンテンツタイプが Application または Patch である場合は、
            // デフォルトタイトルの設定フラグを有効にしておく
            if (this.ContentType == CommonParams.ContentTypeApplication ||
                this.ContentType == CommonParams.ContentTypePatch ||
                this.ContentType == CommonParams.ContentTypeAddOnContent)
            {
                this.IsEditMetaFileForCommandArgs = true;
            }

            this.MetaIndex = CommonParams.DefaultIndex;
            if (inParam.InputIndex != null)
            {
                if (inParam.InputIndex.Length == 1)
                {
                    // 従来の処理
                    try
                    {
                        var indexNum = Convert.ToInt32(inParam.InputIndex[0]);
                        if (indexNum < CommonParams.AocMinIndexValue ||
                            indexNum > CommonParams.AocMaxIndexValue)
                        {
                            // 入力可能範囲外の値なので、強制的に1を指定することにする
                            inParam.InputIndex[0] = CommonParams.DefaultIndex;
                        }
                    }
                    catch (Exception err)
                    {
                        Console.Error.WriteLine(err.Message);
                        Console.Error.WriteLine("インデックス値の指定でエラーが発生。1が設定されたものとして処理します。");
                        inParam.InputIndex[0] = CommonParams.DefaultIndex;
                    }
                }
                else
                {
                    // --index が範囲設定された場合
                    bool isMultiSet = false;
                    uint num0 = 0;
                    uint num1 = 0;
                    {
                        // 1つ目の値を判定
                        var result = uint.TryParse(inParam.InputIndex[0], out num0);
                        if (result == true)
                        {
                            if (num0 >= CommonParams.AocMinIndexValue &&
                                num0 <= CommonParams.AocMaxIndexValue)
                            {
                                isMultiSet = true;
                            }
                        }

                        if (isMultiSet == true)
                        {
                            // 1つ目の値の判定が問題ない場合のみ、2つ目の値を判定する
                            isMultiSet = false;
                            result = uint.TryParse(inParam.InputIndex[1], out num1);
                            if (result == true)
                            {
                                if (num1 >= CommonParams.AocMinIndexValue &&
                                    num1 <= CommonParams.AocMaxIndexValue &&
                                    num0 < num1)
                                {
                                    // 従来の判定処理と (1つ目 < 2つ目) の場合のみ2つ目が正常だとみなす
                                    isMultiSet = true;
                                }
                            }
                        }
                        else
                        {
                            // 1つ目の値の判定が駄目だった場合はデフォルト値
                            inParam.InputIndex[0] = CommonParams.DefaultIndex;
                        }
                    }

                    if (isMultiSet == true)
                    {
                        // 1つ目と2つ目の値が問題ない場合は連番でAocの設定値を作成する
                        for (var i = num0; i <= num1; ++i)
                        {
                            var aoc = new AocProperty();
                            aoc.Index = i.ToString();
                            aoc.ApplicationId = this.MetaInputId;
                            aoc.ReleaseVersion = this.ReleaseVersion;
                            aoc.PrivateVersion = this.PrivateVersion;
                            aoc.Version = this.Version;
                            aoc.RequiredApplicationReleaseVersion = this.RequiredVersion;
                            aoc.Tag = string.Format("test_aoc_{0}", i);
                            aoc.IsAlreadySetDataPath = false;

                            this.AocList.Add(aoc);
                        }
                    }
                }
                // 入力値を設定する
                this.MetaIndex = inParam.InputIndex[0];
            }

            // (SIGLO-49375) Aoc の Tag 値を設定する
            if (this.AocList.Count == 0 && string.IsNullOrEmpty(inParam.InputTag) == false)
            {
                this.MetaTag = inParam.InputTag;
            }
            else
            {
                // 上記以外の条件ではデフォルト値を適用する
                this.MetaTag = string.Format("test_aoc_{0}", this.MetaIndex);
            }

            // CardSpec 関連の設定反映
            this.CardSpec = new CardSpecParam();
            if (inParam.CardSpecSize != null)
            {
                uint num = 0;
                var result = uint.TryParse(inParam.CardSpecSize, out num);
                if (result == true)
                {
                    // 明らかに不正な値でなければそのまま設定
                    CardSpec.Size = inParam.CardSpecSize;
                }
            }

            if (inParam.CardSpecClockRate != null)
            {
                uint num = 0;
                var result = uint.TryParse(inParam.CardSpecClockRate, out num);
                if (result == true)
                {
                    // 明らかに不正な値でなければそのまま設定
                    CardSpec.ClockRate = inParam.CardSpecClockRate;
                }
            }

            // (SIGLO-52833) SaveDataSize, SaveDataJournalSize, StartupUserAccountの設定
            // SaveDataSize, SaveDataJournalSize は入力値チェックを行わずそのまま設定する (チェックは AuthoringTool に任せる形)
            this.SaveDataSize = inParam.SaveDataSize;
            this.SaveDataJournalSize = inParam.SaveDataJournalSize;
            this.RequiredStartupUserAccountFlag = inParam.RequiredStartupUserAccountFlag;

            // (SIGLO-52835) BCAT 関連のパラメータ設定
            // 入力値チェックは実施せずにそのまま設定する (チェックは AuthoringTool に任せる形)
            this.BcatStorageSize = inParam.BcatStorageSize;
            this.BcatPassphrase = inParam.BcatPassphrase;

            // (SIGLO-67157) AoC の動的コミット機能を有効にする設定
            this.RuntimeInstallAocFlag = inParam.RuntimeInstallAocFlag;

            // (SIGLO-68600) CacheStorageSize, CacheStorageJournalSize の設定
            // 入力値チェックを行わずそのまま設定する (チェックは AuthoringTool に任せる形)
            this.CacheStorageSize = inParam.CacheStorageSize;
            this.CacheStorageJournalSize = inParam.CacheStorageJournalSize;

            // (SIGLO-78007) ProgramIndexのパラメータ設定 (入力値チェックは実施済)
            this.MetaProgramIndex = inParam.InputProgramIndex;
            if (this.MetaProgramIndex != null)
            {
                // ProgramIndex が設定されている場合は ProgramId も同時に生成しておく
                var numId = Convert.ToUInt64(this.MetaInputId, 16);
                if (numId > 0)
                {
                    ulong numIndex;
                    // MetaProgramIndex は入力チェック済なのでエラー処理は考慮しない
                    ulong.TryParse(this.MetaProgramIndex, out numIndex);
                    numId += numIndex;
                    // 16進数表記文字列に変換
                    this.MetaProgramId = "0x" + numId.ToString("x16");
                }
            }
            else
            {
                // ProgramIndex の指定がない場合は InputId の値を ProgramId に割り当てておく
                this.MetaProgramId = this.MetaInputId;
            }
        }

        public static List<string> ParseIconSettingArg(string inArgStr)
        {
            var retList = new List<string>();

            var parseList = inArgStr.Split(' ');

            // 2重引用符が入ってくる場合を想定して少々複雑だが自力でちょっとしたパース処理を行う。。
            bool isSkipMode = false;
            string addStr = string.Empty;
            for (int i = 0; i < parseList.Length; ++i)
            {
                var s = parseList[i];
                if (s.Length == 0)
                {
                    if (isSkipMode == true)
                    {
                        addStr += " ";
                    }
                    continue;
                }

                if (isSkipMode == false)
                {
                    if (s.IndexOf("\"") == 0)
                    {
                        if (s.Length > 1 && s.LastIndexOf("\"") == s.Length - 1)
                        {
                            addStr += s.Substring(1, (s.Length - 2));
                        }
                        else
                        {
                            addStr += s.Substring(1, (s.Length - 1));
                            isSkipMode = true;
                        }
                    }
                    else
                    {
                        addStr += s;
                    }
                }
                else
                {
                    if (s.Length > 0 && s.LastIndexOf("\"") == (s.Length - 1))
                    {
                        addStr += string.Format(" {0}", s.Substring(0, (s.Length - 1)));
                        isSkipMode = false;
                    }
                    else
                    {
                        addStr += string.Format(" {0}", s);
                    }
                }

                if (isSkipMode == false)
                {
                    retList.Add(addStr);
                    addStr = string.Empty;
                }
            }

            // (デバック用途)一応コメントとして残しておく
            //int count = 0;
            //foreach (var s in retList)
            //{
            //    Console.WriteLine("[Debug] icon[{0}] : \"{1}\"", count, s);
            //    ++count;
            //}

            return retList;
        }

        private void ConvertIconSettingList(string inArgs, List<IconSettingPair> outList)
        {
            if (inArgs != null)
            {
                var list = ParseIconSettingArg(inArgs);
                for (int i = 0; i < list.Count; i += 2)
                {
                    var setPair = new IconSettingPair(list[i], list[i + 1]);
                    outList.Add(setPair);
                }
            }
        }

        private void SetDefaultIconList(string[] inLanguageList)
        {
            // 念のためクリアしておく
            this.DefaultIconList.Clear();

            if (this.ContentType != CommonParams.ContentTypeApplication &&
                this.ContentType != CommonParams.ContentTypePatch)
            {
                // Application と Patch 以外はデフォルトアイコン設定を適用しない
                return;
            }

            var langSet = new HashSet<string>();
            if (inLanguageList != null)
            {
                foreach (var lang in inLanguageList)
                {
                    // 規定の言語文字列が設定されているかをチェック
                    if (CommonParams.IconLanguageSet.Contains(lang))
                    {
                        // 一旦、Setコンテナに格納する(重複設定があった場合を考慮)
                        langSet.Add(lang);
                    }
                    else if (CommonParams.NoDefaultIconSetString == lang.ToLower())
                    {
                        // 大文字・小文字(または混合)に関わらず noicon という文字列が設定されていればアイコン画像を設定しない
                        this.DefaultIconList.Clear();
                        return;
                    }
                }
            }

            // 無指定(null)または有効値が無い場合は、デフォルト値(AmericanEnglish)のみを設定する
            if (langSet.Count == 0)
            {
                langSet.Add(CommonParams.DefaultIconLanguage);
            }

            foreach (var lang in langSet)
            {
                // ファイルパスは一旦空文字で設定し、後から改めて設定する
                var setPair = new IconSettingPair(lang, string.Empty);
                this.DefaultIconList.Add(setPair);
            }
        }

        /// <summary>
        /// 一括設定XML時の DefaultIcon の Language 設定処理を行う
        /// </summary>
        /// <param name="inXmlLanguageList">DefaultIcon/LanguageのXML設定</param>
        public void SetDefaultIconList(XmlNodeList inXmlLanguageList)
        {
            if (inXmlLanguageList == null)
            {
                // 一括設定XMLで何も設定されていないパターンを考慮
                this.SetDefaultIconList((string[])null);
                return;
            }

            var langList = new List<string>();
            foreach (XmlNode node in inXmlLanguageList)
            {
                langList.Add(node.InnerText);
            }
            this.SetDefaultIconList(langList.ToArray());
        }

        /// <summary>
        /// コマンドラインの入力情報からmeta情報に依存しないオプション情報を設定する
        /// </summary>
        /// <param name="inParam">コマンドラインの入力オプションパラメータ</param>
        public void SetCommonSettingParam(CommandLineParams inParam)
        {
            // アプリケーションドキュメント関連情報はそのまま設定する
            this.HtmlDocumentDirPath = inParam.HtmlDocumentDirPath;
            this.AccessibleUrlsFilePath = inParam.AccessibleUrlsFilePath;
            this.LegalInformationZipFilePath = inParam.LegalInformationZipFilePath;

            // アイコンファイル関連の設定
            this.ConvertIconSettingList(inParam.IconSetting, this.IconList);
            this.ConvertIconSettingList(inParam.NxIconSetting, this.NxIconList);

            // アイコンオプションが明示指定されていない場合はデフォルトアイコンの指定言語を設定する
            if (this.IconList.Count == 0 && this.IsSetMetaIcon == false)
            {
                this.SetDefaultIconList(inParam.DefaultIconLanguageList);
            }

            // データディレクトリ情報はそのまま設定する
            this.DataDirPath = inParam.DataDirPath;

            // keyindex, keyconfig 情報はそのまま設定する
            this.KeyIndexNum = inParam.KeyIndexNum;
            this.KeyConfigFilePath = inParam.KeyConfigFilePath;

            // ticket オプションはそのまま設定する
            this.TicketFlag = inParam.TicketFlag;
            this.NoTicketFlag = inParam.NoTicketFlag;

            // code, romfs 情報はそのまま設定する
            this.CodeDirPath = inParam.CodeDirPath;
            this.RomFsDirPath = inParam.RomFsDirPath;

            // seed 値はそのまま設定しておく
            this.SeedNumber = inParam.SeedNumber;

            // Patch 関連の指定パスはそのまま設定する
            this.OriginalApplicationFilePath = inParam.OriginalApplicationFilePath;
            this.OriginalApplicationForCreateNspFilePath = inParam.OriginalApplicationForCreateNspFilePath;
            this.PreviousPatchFilePath = inParam.PreviousPatchFilePath;
            this.DebugDeltaFlag = inParam.DebugDeltaFlag;

            // 出力ファイルパス指定はそのまま設定する (特に内部的な判定処理は行わない)
            this.OutputFilePath = inParam.OutputFilePath;

            // (SIGLO-42685) Aobort させる Result 値を設定する
            this.AbortResultValue = inParam.AbortResultValue;

            // (SIGLO-52130) NetworkRequest フラグ値を設定する
            this.NetworkRequestFlag = inParam.NetworkRequestFlag;

            // (SIGLO-68043) 直接指定のオプション文字列を設定する
            this.DirectString = inParam.DirectString;
            // (SIGLO-73606) makepatch 用の直接指定オプションも追加する
            this.DirectStringForMakePatch = inParam.DirectStringForMakePatch;

            this.MultiProgram = inParam.MultiProgram;

            this.CheckValidRangeIdFlag = inParam.CheckValidRangeIdFlag;
        }

        /// <summary>
        /// 設定ファイルのメタ(XML)入力情報からメタ情報を抽出し自身のメンバ変数に設定する
        /// </summary>
        /// <param name="inXmlNode">メタ入力XMLオブジェクト</param>
        /// <param name="inContentType">コンテンツタイプ文字列</param>
        public void SetMetaParam(XmlNode inXmlNode, string inContentType)
        {
            if (inContentType == CommonParams.ContentTypeApplication
                || inContentType == CommonParams.ContentTypePatch)
            {
                var idNode = inXmlNode.SelectSingleNode("Core/ApplicationId");
                if (idNode != null)
                {
                    this.MetaInputId = idNode.InnerText;
                }
                else
                {
                    idNode = inXmlNode.SelectSingleNode("Core/ProgramId");
                    if (idNode != null)
                    {
                        this.MetaProgramId = idNode.InnerText;
                    }

                    var appIdNode = inXmlNode.SelectSingleNode("Application/ApplicationId");
                    if (appIdNode != null)
                    {
                        this.MetaInputId = appIdNode.InnerText;
                    }

                    var indexNode = inXmlNode.SelectSingleNode("Application/ProgramIndex");
                    if (indexNode != null)
                    {
                        this.MetaProgramIndex = indexNode.InnerText;
                    }
                }

                if (this.MetaInputId == null)
                {
                    // MetaInputId が未設定の場合は MetaProgramId と同等とする(旧仕様の互換も考慮)
                    // ID関連の情報が全く入力されていない場合はデフォルトのプログラムIDを適用する
                    this.MetaInputId = this.MetaProgramId ?? CommonParams.DefaultProgramId;
                }

                // アプリについては言語ごとに名前が変わる可能性があるため、ツール内部設定的には名前ナシとしておく
                this.IsSetName = false;

                // ReleaseVersion と PrivateVersion の値を設定
                SetVersionValue(inXmlNode.SelectSingleNode("Application"));

                // 明示的にアイコン画像の設定を行っているかどうかの確認
                var iconNodeList = inXmlNode.SelectNodes("Application/Icon");
                if (iconNodeList != null && iconNodeList.Count > 0)
                {
                    this.IsSetMetaIcon = true;
                }
            }
            else if (inContentType == CommonParams.ContentTypeSystemProgram)
            {
                var metaSysProgramPath = "SystemProgram";
                this.MetaInputId = CommonParams.DefaultProgramId;
                var idNode = inXmlNode.SelectSingleNode(metaSysProgramPath + "/Id");
                if (idNode != null)
                {
                    this.MetaInputId = idNode.InnerText;
                }
                else
                {
                    // 念のため古い仕様のID定義も確認する
                    var idNodeOld = inXmlNode.SelectSingleNode("Core/ProgramId");
                    if (idNodeOld != null)
                    {
                        this.MetaInputId = idNodeOld.InnerText;
                    }
                }

                this.IsSetName = false;
                var nameNode = inXmlNode.SelectSingleNode("Core/Name");
                if (nameNode != null && nameNode.InnerText != string.Empty)
                {
                    this.MetaName = nameNode.InnerText;
                    this.IsSetName = true;
                }

                // ReleaseVersion と PrivateVersion の値を設定
                SetVersionValue(inXmlNode.SelectSingleNode(metaSysProgramPath));
            }
            else if (inContentType == CommonParams.ContentTypeSystemData)
            {
                var metaSysDataPath = "SystemData";
                this.MetaInputId = CommonParams.DefaultProgramId;
                var idNode = inXmlNode.SelectSingleNode(metaSysDataPath + "/Id");
                if (idNode != null)
                {
                    this.MetaInputId = idNode.InnerText;
                }

                // ReleaseVersion と PrivateVersion の値を設定
                SetVersionValue(inXmlNode.SelectSingleNode(metaSysDataPath));
            }
            else if (inContentType == CommonParams.ContentTypeAddOnContent)
            {
                var aocNodeList = inXmlNode.SelectNodes("AddOnContent");
                if (aocNodeList != null && aocNodeList.Count == 1)
                {
                    // 複数個設定できるようになったので、1つだけ設定されている場合のみ旧来の動作を実施する
                    var metaAocPath = "AddOnContent";
                    this.MetaInputId = CommonParams.DefaultProgramId;
                    var idNode = inXmlNode.SelectSingleNode(metaAocPath + "/ApplicationId");
                    if (idNode != null)
                    {
                        this.MetaInputId = idNode.InnerText;
                    }

                    this.MetaIndex = CommonParams.DefaultIndex;
                    var indexNode = inXmlNode.SelectSingleNode(metaAocPath + "/Index");
                    if (indexNode != null)
                    {
                        try
                        {
                            var indexNum = Convert.ToInt32(indexNode.InnerText);
                            if (indexNum >= CommonParams.AocMinIndexValue &&
                                indexNum <= CommonParams.AocMaxIndexValue)
                            {
                                this.MetaIndex = indexNode.InnerText;
                            }
                        }
                        catch (Exception err)
                        {
                            Console.Error.WriteLine(err.Message);
                            Console.Error.WriteLine("インデックス値の指定でエラー発生。デフォルト値を設定します。");
                        }
                    }

                    // ReleaseVersion と PrivateVersion の値を設定
                    SetVersionValue(inXmlNode.SelectSingleNode(metaAocPath));
                }
                else if (aocNodeList != null && aocNodeList.Count > 1)
                {
                    foreach (XmlNode aocNode in aocNodeList)
                    {
                        var property = new AocProperty();

                        // 手続き的には不要かもしれないが一応XMLの設定値を読み込んでおく (ApplicatoinId, Index)
                        property.ApplicationId = CommonParams.DefaultProgramId;
                        var idNode = aocNode.SelectSingleNode("ApplicationId");
                        if (idNode != null)
                        {
                            property.ApplicationId = idNode.InnerText;
                        }

                        property.Index = CommonParams.DefaultIndex;
                        var indexNode = aocNode.SelectSingleNode("Index");
                        if (indexNode != null)
                        {
                            uint num = 0;
                            var result = uint.TryParse(indexNode.InnerText, out num);
                            if (result == true)
                            {
                                if (num >= CommonParams.AocMinIndexValue &&
                                    num <= CommonParams.AocMaxIndexValue)
                                {
                                    property.Index = indexNode.InnerText;
                                }
                            }
                        }

                        // ReleaseVersion と PrivateVersion と Version(32bit) の値を設定
                        SetVersionValue(property, aocNode);

                        // すでに DataPath が設定済であるかどうかの確認
                        property.IsAlreadySetDataPath = false;
                        var dataNode = aocNode.SelectSingleNode("DataPath");
                        if (dataNode != null)
                        {
                            property.IsAlreadySetDataPath = true;
                        }

                        AocList.Add(property);
                    }
                }
            }

            if (this.MetaProgramId == null)
            {
                // MetaProgramId が未設定である場合は MetaInputId と同値としておく
                this.MetaProgramId = this.MetaInputId;
            }

            this.ContentType = inContentType;
        }

        private void SetVersionValue(XmlNode inNode)
        {
            string rv;
            string pv;
            string v;
            this.SetVersionValueInternal(inNode, out rv, out pv, out v);

            this.ReleaseVersion = rv;
            this.PrivateVersion = pv;
            this.Version = v;
            //Console.Error.WriteLine("SetVersionValue() [Normal]: rv={0}, pv={1}, v={2}", rv, pv, v);

            // ツール的に MetaVersion は ReleaseVersion と同値にしておく
            this.MetaVersion = this.ReleaseVersion;
        }

        private void SetVersionValue(AocProperty inAp, XmlNode inNode)
        {
            string rv;
            string pv;
            string v;
            this.SetVersionValueInternal(inNode, out rv, out pv, out v);

            inAp.ReleaseVersion = rv;
            inAp.PrivateVersion = pv;
            inAp.Version = v;
            //Console.Error.WriteLine("SetVersionValue() [AocProperty]: rv={0}, pv={1}, v={2}", rv, pv, v);
        }

        private void SetVersionValueInternal(XmlNode inNode,
            out string releaseVersion, out string privateVersion, out string version)
        {
            // デフォルト値の設定
            releaseVersion = "0";
            privateVersion = "0";
            version = "0";

            if (inNode == null)
            {
                // ありえないと思うが大元のXMLノードが null だった場合はデフォルト値となる
                return;
            }

            var releaseVersionNode = inNode.SelectSingleNode("ReleaseVersion");
            var privateVersionNode = inNode.SelectSingleNode("PrivateVersion");
            if (releaseVersionNode == null && privateVersionNode == null)
            {
                // ReleaseVersion と PrivateVersion のどちらも設定されていない場合は、Version(無印) 設定値での設定を試みる
                var versionNode = inNode.SelectSingleNode("Version");
                if (versionNode != null && versionNode.InnerText != string.Empty)
                {
                    // Version(無印) の値が設定されている場合
                    uint num = 0;
                    var result = uint.TryParse(versionNode.InnerText, out num);
                    if (result == true)
                    {
                        version = versionNode.InnerText;
                    }

                    var rNum = num >> 16;
                    var pNum = 0x0000FFFF & num;

                    releaseVersion = rNum.ToString();
                    privateVersion = pNum.ToString();
                }
            }
            else
            {
                // ReleaseVersion と PrivateVersion のどちらかが設定されていればこちらの処理に入る
                if (releaseVersionNode != null)
                {
                    ushort num = 0;
                    var result = ushort.TryParse(releaseVersionNode.InnerText, out num);
                    if (result == true)
                    {
                        releaseVersion = releaseVersionNode.InnerText;
                    }
                }

                if (privateVersionNode != null)
                {
                    ushort num = 0;
                    var result = ushort.TryParse(privateVersionNode.InnerText, out num);
                    if (result == true)
                    {
                        privateVersion = privateVersionNode.InnerText;
                    }
                }

                var rvNum = Convert.ToUInt32(releaseVersion);
                var pvNum = Convert.ToUInt32(privateVersion);
                var verNum = (rvNum << 16) | pvNum;
                version = verNum.ToString();
            }
        }

        private void SetVersionValue(CommandLineParams inParam)
        {
            bool isCheckUnifiedVersion = true;

            // --ver オプションの入力値確認
            ushort releaseVersionNum = 0; // デフォルト値は 0
            if (inParam.InputVersion != null)
            {
                var num = releaseVersionNum;
                var result = ushort.TryParse(inParam.InputVersion, out num);
                if (result == true)
                {
                    releaseVersionNum = num;
                }
                // --ver の入力があれば --unified-version の値は実質的に無効とする
                isCheckUnifiedVersion = false;
            }

            // --release-version オプションの入力値確認
            // --ver のエイリアス的な動作とするが、仮に --ver と同時指定された場合は、--ver の値を優先させる。
            if (inParam.InputReleaseVersion != null && isCheckUnifiedVersion == true)
            {
                var num = releaseVersionNum;
                var result = ushort.TryParse(inParam.InputReleaseVersion, out num);
                if (result == true)
                {
                    releaseVersionNum = num;
                }
                // --release-version の入力があれば --unified-version の値は実質的に無効とする
                isCheckUnifiedVersion = false;
            }

            // --private-version オプションの入力値確認
            ushort privateVersionNum = 0; // デフォルト値は 0
            if (inParam.InputPrivateVersion != null)
            {
                var num = privateVersionNum;
                var result = ushort.TryParse(inParam.InputPrivateVersion, out num);
                if (result == true)
                {
                    privateVersionNum = num;
                }
                // --private-version の入力があれば --unified-version の値は実質的に無効とする
                isCheckUnifiedVersion = false;
            }

            if (isCheckUnifiedVersion == false)
            {
                // --ver (--release-version) または --private-version の入力値が有効とみなせるので、
                // これらの有効値から Version(32bit) の値を確定させる
                this.ReleaseVersion = releaseVersionNum.ToString();
                this.PrivateVersion = privateVersionNum.ToString();

                // ビット操作のため(念のため)型変換しておく
                uint rvNum = releaseVersionNum;
                uint pvNum = privateVersionNum;
                var verNum = (rvNum << 16) | pvNum;
                this.Version = verNum.ToString();

                // 以降の処理は実施しない
                return;
            }

            // --unified-version オプションの入力値確認
            uint unifiedVersionNum = 0; // デフォルト値は 0
            if (inParam.InputUnifiedVersion != null)
            {
                var num = unifiedVersionNum;
                var result = uint.TryParse(inParam.InputUnifiedVersion, out num);
                if (result == true)
                {
                    unifiedVersionNum = num;
                }
            }

            {
                this.Version = unifiedVersionNum.ToString();

                // Version(32bit) の値から ReleaseVersion, PrivateVersion を確定させる
                var rvNum = unifiedVersionNum >> 16;
                this.ReleaseVersion = rvNum.ToString();

                var pvNum = 0x0000FFFF & unifiedVersionNum;
                this.PrivateVersion = pvNum.ToString();
            }
        }
    }

    internal class TestAppSettingParam
    {
        public string SHA1HashString { get; set; }
        public List<string> OfflineHtmlPathList { get; set; }

        internal class SettingPair
        {
            public string Item { get; set; }
            public string Value { get; set; }

            public SettingPair()
            {
            }

            public SettingPair(string inItem, string inValue)
            {
                Item = inItem;
                Value = inValue;
            }
        }

        private List<SettingPair> metaSetList;

        public List<SettingPair> MetaSetList
        {
            get
            {
                return metaSetList;
            }
        }

        public TestAppSettingParam()
        {
            metaSetList = new List<SettingPair>();
        }

        public void DebugPrint()
        {
            // [デバック用] リスト内の設定内容をコンソールに一括出力
            Console.WriteLine(" ******* TestAppSettingParam Begin ******* ");
            foreach (var pr in metaSetList)
            {
                Console.WriteLine("{0} : {1}", pr.Item, pr.Value);
            }
            Console.WriteLine(" ******* TestAppSettingParam End ******* ");
        }
    }
}
