﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using System.Security.Cryptography;
using System.Text;

namespace MakeTestApplication
{
    internal class FsFacade
    {
        private string tempDirName = CommonParams.WorkDirName;
        private bool isDeleteTempDir = true;

        private bool isVerbose = false;
        public void SetVerboseMode()
        {
            isVerbose = true;
        }

        // 環境変数を含んだパスを置換するラムダ式
        private Func<string, string> replaceEnvVar = (path) =>
        {
            // %で囲んだ環境変数名をその値に置換します。
            // 置換対象の文字列が含まれていない場合は入力文字列をそのまま返します
            return Environment.ExpandEnvironmentVariables(path);
        };

        private XmlNode GetMetaRootNode(XmlDocument inMetaFileDoc)
        {
            // (SIGLO-50695) NintendoSdkMeta ディレクティブを確認して null だったら、旧形式の Meta ディレクティブを確認する
            return inMetaFileDoc.SelectSingleNode("/" + CommonParams.NmetaRootName) ?? inMetaFileDoc.SelectSingleNode("/Meta");
        }

        /// <summary>
        /// 作業フォルダの絶対パスを取得
        /// </summary>
        /// <returns>作業フォルダの絶対パス文字列</returns>
        public string GetWorkDirFullPath()
        {
            string fullPath = Path.GetFullPath(tempDirName);
            if (fullPath.EndsWith(@"\") == true)
            {
                return fullPath;
            }
            else
            {
                return fullPath + @"\";
            }
        }

        /// <summary>
        /// ファイルの移動処理
        /// 移動先に指定したファイルが存在した場合は削除する
        /// </summary>
        /// <param name="inSrcPath">移動元ファイルパス</param>
        /// <param name="inDstPath">移動先ファイルパス</param>
        public int MoveFile(string inSrcPath, string inDstPath)
        {
            try
            {
                if (File.Exists(inDstPath) == true)
                {
                    File.Delete(inDstPath);
                }
                File.Move(inSrcPath, inDstPath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.Message);
                Console.Error.WriteLine("MoveFile()に失敗：src={0}, dst={1}", inSrcPath, inDstPath);
                throw err;
            }
            return 0;
        }

        /// <summary>
        /// ディレクトリのコピー処理
        /// .Net のライブラリにディレクトリコピー処理は存在しない？模様
        /// </summary>
        /// <param name="inSrcDir">コピー元ディレクトリパス</param>
        /// <param name="inDstDir">コピー先ディレクトリパス</param>
        public static int CopyDirectory(string inSrcDir, string inDstDir)
        {
            try
            {
                var srcDir = new DirectoryInfo(inSrcDir);
                var dstDir = new DirectoryInfo(inDstDir);

                if (dstDir.Exists == false)
                {
                    dstDir.Create();
                    dstDir.Attributes = srcDir.Attributes;
                }

                foreach (var fileInfo in srcDir.GetFiles())
                {
                    fileInfo.CopyTo(dstDir.FullName + @"\" + fileInfo.Name, true);
                }

                foreach (var dirInfo in srcDir.GetDirectories())
                {
                    FsFacade.CopyDirectory(dirInfo.FullName, dstDir.FullName + @"\" + dirInfo.Name);
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.Message);
                Console.Error.WriteLine("CopyDirectory()に失敗：src={0}, dst={1}", inSrcDir, inDstDir);
                throw err;
            }
            return 0;
        }

        /// <summary>
        /// 入力ファイルの存在確認などのチェック処理
        /// </summary>
        /// <param name="inParam">コマンドラインの入力オプションパラメータ</param>
        public void CheckInputFilePath(CommandLineParams inParam)
        {
            if (inParam.ConfigXmlPath != null)
            {
                inParam.ConfigXmlPath = replaceEnvVar(inParam.ConfigXmlPath);
                if (File.Exists(inParam.ConfigXmlPath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--configXml=" + inParam.ConfigXmlPath + " は存在しません");
                }
            }

            if (inParam.MetaFilePath != null)
            {
                inParam.MetaFilePath = replaceEnvVar(inParam.MetaFilePath);
                if (File.Exists(inParam.MetaFilePath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--meta=" + inParam.MetaFilePath + " は存在しません");
                }
            }

            if (inParam.DescFilePath != null)
            {
                inParam.DescFilePath = replaceEnvVar(inParam.DescFilePath);
                if (File.Exists(inParam.DescFilePath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--desc=" + inParam.DescFilePath + " は存在しません");
                }
            }

            if (inParam.HtmlDocumentDirPath != null)
            {
                inParam.HtmlDocumentDirPath = replaceEnvVar(inParam.HtmlDocumentDirPath);
                if (Directory.Exists(inParam.HtmlDocumentDirPath) == false)
                {
                    // 指定ディレクトリパスが存在しない場合は例外を投げる
                    throw new Exception("--html-document=" + inParam.HtmlDocumentDirPath + " は存在しません");
                }
            }

            if (inParam.AccessibleUrlsFilePath != null)
            {
                inParam.AccessibleUrlsFilePath = replaceEnvVar(inParam.AccessibleUrlsFilePath);
                if (File.Exists(inParam.AccessibleUrlsFilePath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--accessible-urls=" + inParam.AccessibleUrlsFilePath + " は存在しません");
                }
            }

            if (inParam.LegalInformationZipFilePath != null)
            {
                inParam.LegalInformationZipFilePath = replaceEnvVar(inParam.LegalInformationZipFilePath);
                if (!(File.Exists(inParam.LegalInformationZipFilePath) &&
                       Path.GetExtension(inParam.LegalInformationZipFilePath).ToLower() == ".zip"))
                {
                    // 指定した zip ファイルが存在しない場合は例外を投げる
                    throw new Exception("--legal-Information=" + inParam.LegalInformationZipFilePath + " は存在しない、または拡張子が不正です");
                }
            }

            // アイコン情報については必要最低限のフォーマットチェックのみにしておく
            if (inParam.IconSetting != null)
            {
                inParam.IconSetting = replaceEnvVar(inParam.IconSetting);
                if (inParam.IconSetting == string.Empty)
                {
                    throw new Exception("--icon の設定が不正です");
                }

                var list = SettingParams.ParseIconSettingArg(inParam.IconSetting);
                if (list.Count % 2 != 0)
                {
                    throw new Exception(string.Format("--icon 項目内の設定数が不正です : {0}", list.Count));
                }
            }

            if (inParam.NxIconSetting != null)
            {
                inParam.NxIconSetting = replaceEnvVar(inParam.NxIconSetting);
                if (inParam.NxIconSetting == string.Empty)
                {
                    throw new Exception("--nx-icon の設定が不正です");
                }

                var list = SettingParams.ParseIconSettingArg(inParam.NxIconSetting);
                if (list.Count % 2 != 0)
                {
                    throw new Exception(string.Format("--nx-icon 項目内の設定数が不正です : {0}", list.Count));
                }
            }

            if (inParam.DefaultIconLanguageList != null)
            {
                foreach (var lang in inParam.DefaultIconLanguageList)
                {
                    // 言語文字列のチェック
                    if (CommonParams.IconLanguageSet.Contains(lang) == false && lang.ToLower() != CommonParams.NoDefaultIconSetString)
                    {
                        throw new Exception(string.Format("--default-icon-language (-l) が不正です : {0}", lang));
                    }
                }
            }

            if (inParam.DataDirPath != null)
            {
                inParam.DataDirPath = replaceEnvVar(inParam.DataDirPath);
                if (Directory.Exists(inParam.DataDirPath) == false)
                {
                    // 指定ディレクトリパスが存在しない場合は例外を投げる
                    throw new Exception("--data-directory=" + inParam.DataDirPath + " は存在しません");
                }
            }

            if (inParam.KeyConfigFilePath != null)
            {
                inParam.KeyConfigFilePath = replaceEnvVar(inParam.KeyConfigFilePath);
                if (File.Exists(inParam.KeyConfigFilePath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--keyconfig=" + inParam.KeyConfigFilePath + " は存在しません");
                }
            }

            if (inParam.CodeDirPath != null)
            {
                inParam.CodeDirPath = replaceEnvVar(inParam.CodeDirPath);
                if (Directory.Exists(inParam.CodeDirPath) == false)
                {
                    // 指定ディレクトリパスが存在しない場合は例外を投げる
                    throw new Exception("--code=" + inParam.CodeDirPath + " は存在しません");
                }
            }

            if (inParam.RomFsDirPath != null)
            {
                inParam.RomFsDirPath = replaceEnvVar(inParam.RomFsDirPath);
                if (Directory.Exists(inParam.RomFsDirPath) == false)
                {
                    // 指定ディレクトリパスが存在しない場合は例外を投げる
                    throw new Exception("--romfs=" + inParam.RomFsDirPath + " は存在しません");
                }
            }

            if (inParam.OriginalApplicationFilePath != null)
            {
                inParam.OriginalApplicationFilePath = replaceEnvVar(inParam.OriginalApplicationFilePath);
                if (File.Exists(inParam.OriginalApplicationFilePath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--original-application=" + inParam.OriginalApplicationFilePath + " は存在しません");
                }
            }

            if (inParam.OriginalApplicationForCreateNspFilePath != null)
            {
                inParam.OriginalApplicationForCreateNspFilePath = replaceEnvVar(inParam.OriginalApplicationForCreateNspFilePath);
                if (File.Exists(inParam.OriginalApplicationForCreateNspFilePath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--original-application-for-creatensp=" + inParam.OriginalApplicationForCreateNspFilePath + " は存在しません");
                }
            }

            if (inParam.PreviousPatchFilePath != null)
            {
                inParam.PreviousPatchFilePath = replaceEnvVar(inParam.PreviousPatchFilePath);
                if (File.Exists(inParam.PreviousPatchFilePath) == false)
                {
                    // 指定ファイルパスが存在しない場合は例外を投げる
                    throw new Exception("--previous-patch=" + inParam.PreviousPatchFilePath + " は存在しません");
                }
            }
        }

        /// <summary>
        /// 中間ファイル用のフォルダ(作業フォルダ)の初期化
        /// </summary>
        /// <param name="inParam">コマンドラインの入力オプションパラメータ</param>
        public void InitWorkDir(CommandLineParams inParam)
        {
            if (inParam.NoCleanupFlag == true)
            {
                // 処理終了時に作業ディレクトリを残す設定にする
                this.isDeleteTempDir = false;
            }

            if (inParam.WorkDirPath != null)
            {
                inParam.WorkDirPath = replaceEnvVar(inParam.WorkDirPath);
                if (Directory.Exists(inParam.WorkDirPath) == false)
                {
                    // 指定ディレクトリパスが存在しない場合は例外を投げる
                    throw new Exception("--work-directory=" + inParam.WorkDirPath + " は存在しません");
                }

                // 指定パスが設定されて存在する場合はそのまま作業用フォルダパスに設定
                this.tempDirName = inParam.WorkDirPath;
            }

            try
            {
                if (Directory.Exists(tempDirName) == false)
                {
                    // 作業ディレクトリが存在しない場合は作成する
                    Directory.CreateDirectory(tempDirName);
                    if (isVerbose == true)
                    {
                        Console.WriteLine("Create WorkDirectory : Path = {0}", tempDirName);
                    }
                }

                // もし作業フォルダ内にファイルやフォルダが存在する場合は全て削除しておく
                var workDirInfo = new DirectoryInfo(tempDirName);
                foreach (var file in workDirInfo.GetFiles())
                {
                    file.Delete();
                }
                foreach (var dir in workDirInfo.GetDirectories())
                {
                    dir.Delete(true);
                }
            }
            catch (IOException err)
            {
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        /// <summary>
        /// クラスが保持する作業フォルダを削除するかどうかの判断フラグを返す
        /// </summary>
        /// <returns>削除判断フラグ</returns>
        public bool IsDeleteTempDir()
        {
            return isDeleteTempDir;
        }

        /// <summary>
        /// 中間ファイル用のフォルダ(作業フォルダ)の後始末
        /// </summary>
        public void ClearWorkDir()
        {
            try
            {
                if (Directory.Exists(tempDirName) == true && this.isDeleteTempDir == true)
                {
                    // 作業フォルダが存在する場合はすべて削除する
                    Directory.Delete(tempDirName, true);
                }
            }
            catch (IOException err)
            {
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        /// <summary>
        /// 出力ファイル用のフォルダの初期化
        /// </summary>
        /// <param name="inParam">コマンドラインの入力オプションパラメータ</param>
        /// <returns>出力用フォルダパス</returns>
        public string InitOutputDir(CommandLineParams inParam)
        {
            // (SIGLO-48132) 場所的に微妙だが出力ファイルパスの環境変数置換処理をここで実施しておく
            if (string.IsNullOrEmpty(inParam.OutputFilePath) == false)
            {
                inParam.OutputFilePath = replaceEnvVar(inParam.OutputFilePath);
                // OutputFilePath が設定されている場合、以下の処理は特に不要なのでパスからディレクトリ名文字列を抽出して返しておく
                return Path.GetDirectoryName(inParam.OutputFilePath);
            }

            string outputDir = null;
            if (inParam.OutputDirPath != null)
            {
                inParam.OutputDirPath = replaceEnvVar(inParam.OutputDirPath);
                var workDirInfo = new DirectoryInfo(tempDirName);
                var outDirInfo = new DirectoryInfo(inParam.OutputDirPath);

                if (outDirInfo.Exists == true)
                {
                    // 指定パスが設定されて存在する場合はそのまま出力用フォルダパスに設定
                    outputDir = inParam.OutputDirPath;
                }
                else
                {
                    // 指定パスのディレクトリが存在していない場合は、例外を投げておく
                    throw new Exception("--output-directory=" + inParam.OutputDirPath + " は存在しません");
                }

                if (workDirInfo.FullName == outDirInfo.FullName)
                {
                    // 作業ディレクトリパス名と出力ディレクトリパス名が同一の場合、(最後の)フォルダ削除処理を行わない
                    this.isDeleteTempDir = false;
                }
            }

            if (outputDir == null)
            {
                var dt = DateTime.Now;
                var timeString = dt.ToString("yyyyMMdd_HHmmss");
                outputDir = timeString + @"_TestApp";
            }

            try
            {
                if (Directory.Exists(outputDir) == false)
                {
                    // フォルダが存在しない場合は作成する
                    Directory.CreateDirectory(outputDir);
                    if (isVerbose == true)
                    {
                        Console.WriteLine("Create OutputDirectory : Path = {0}", outputDir);
                    }
                }
            }
            catch (IOException err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }

            return outputDir;
        }

        private string MakeChildWorkDirectory(object inParam)
        {
            var childDirPath = Path.Combine(this.GetWorkDirFullPath(), string.Format("Item_{0}", inParam));
            if (Directory.Exists(childDirPath) == false)
            {
                Directory.CreateDirectory(childDirPath);
            }
            return childDirPath;
        }

        /// <summary>
        /// オーサリングツールに渡すためのmetaファイルを作成します
        /// </summary>
        /// <param name="settingData">メタ情報のパラメータ</param>
        /// <returns>作成したファイルパス</returns>
        public string CreateMetaFile(SettingParams settingData)
        {
            // MetaParamsのデータ内容に沿ったXMLファイルを作成する
            string metaXmlFilePath = null;
            try
            {
                // XMLファイルの中身は meta ファイルの書式に依存する
                var metaXmlDoc = new XmlDocument();
                var head = metaXmlDoc.CreateXmlDeclaration("1.0", "utf-8", null);
                var root = metaXmlDoc.CreateElement(CommonParams.NmetaRootName);

                metaXmlDoc.AppendChild(head);
                metaXmlDoc.AppendChild(root);

                // ラムダ式でバージョン設定処理を関数化しておく
                Action<XmlElement> addVersionAction = (XmlElement inAppendElement) =>
                {
                    if (settingData.ReleaseVersion != null)
                    {
                        // ReleaseVersion を追加する
                        var releaseVersionElement = metaXmlDoc.CreateElement("ReleaseVersion");
                        releaseVersionElement.InnerText = settingData.ReleaseVersion;
                        inAppendElement.AppendChild(releaseVersionElement);
                    }
                    if (settingData.PrivateVersion != null)
                    {
                        // PrivateVersion を追加する
                        var privateVersionElement = metaXmlDoc.CreateElement("PrivateVersion");
                        privateVersionElement.InnerText = settingData.PrivateVersion;
                        inAppendElement.AppendChild(privateVersionElement);
                    }
                };

                var type = settingData.ContentType;
                if ( type == CommonParams.ContentTypeApplication
                    || type == CommonParams.ContentTypePatch)
                {
                    // type に Application または Patch が指定された場合
                    var core = metaXmlDoc.CreateElement("Core");
                    root.AppendChild(core);

                    bool isSetProgramIndex = (settingData.MetaProgramIndex != null);
                    if (isSetProgramIndex)
                    {
                        // MetaProgramIndex が設定されてる場合は Core/ProgramId に MetaProgramId を設定
                        var idElementName = "ProgramId";
                        var idElement = metaXmlDoc.CreateElement(idElementName);
                        idElement.InnerText = settingData.MetaProgramId;
                        core.AppendChild(idElement);
                    }
                    else
                    {
                        // MetaProgramIndex が設定されていない場合は従来通りの処理 (Core/ApplicationIdに入力IDを設定)
                        var idElementName = "ApplicationId";
                        var idElement = metaXmlDoc.CreateElement(idElementName);
                        idElement.InnerText = settingData.MetaInputId;
                        core.AppendChild(idElement);
                    }

                    var app = metaXmlDoc.CreateElement("Application");
                    root.AppendChild(app);

                    // ラムダ式で Application へのエレメント追加処理を関数化しておく
                    Action<string, string> addAppElementAction = (string inElementName, string inSetValue) =>
                    {
                        if (inSetValue != null)
                        {
                            var element = metaXmlDoc.CreateElement(inElementName);
                            element.InnerText = inSetValue;
                            app.AppendChild(element);
                        }
                    };

                    if (isSetProgramIndex)
                    {
                        addAppElementAction("ApplicationId", settingData.MetaInputId);
                        addAppElementAction("ProgramIndex", settingData.MetaProgramIndex);
                    }

                    addVersionAction(app);

                    // 必須バージョン値は必ず設定されているはずなので <Application> タグ内に追加
                    {
                        // <RequiredSystemVersion> タグを追加する
                        var requiredElement = metaXmlDoc.CreateElement("RequiredSystemVersion");
                        requiredElement.InnerText = settingData.RequiredVersion;
                        app.AppendChild(requiredElement);
                    }

                    // (SIGLO-52833) StartupUserAccount の Required が指定されていた場合、nmeta に反映させる
                    if (settingData.RequiredStartupUserAccountFlag)
                    {
                        var startupUserAccountElement = metaXmlDoc.CreateElement("StartupUserAccount");
                        startupUserAccountElement.InnerText = "Required";
                        app.AppendChild(startupUserAccountElement);
                    }

                    // (SIGLO-52833) セーブデータサイズが指定されていた場合、nmeta に反映させる
                    addAppElementAction("UserAccountSaveDataSize", settingData.SaveDataSize);

                    // (SIGLO-52833) セーブデータジャーナルサイズが指定されていた場合、nmeta に反映させる
                    addAppElementAction("UserAccountSaveDataJournalSize", settingData.SaveDataJournalSize);

                    // (SIGLO-52835) BCATストレージサイズが指定されていた場合、nmeta に反映させる
                    addAppElementAction("BcatDeliveryCacheStorageSize", settingData.BcatStorageSize);

                    // (SIGLO-52835) BCATパスフレーズが指定されていた場合、nmeta に反映させる
                    addAppElementAction("BcatPassphrase", settingData.BcatPassphrase);

                    // (SIGLO-68600) キャッシュストレージサイズが指定されていた場合、nmeta に反映させる
                    addAppElementAction("CacheStorageSize", settingData.CacheStorageSize);

                    // (SIGLO-68600) キャッシュストレージジャーナルサイズが指定されていた場合、nmeta に反映させる
                    addAppElementAction("CacheStorageJournalSize", settingData.CacheStorageJournalSize);

                    // (SIGLO-67157) AoC の動的コミット機能フラグが有効に設定されていた場合、nmeta に反映させる
                    if (settingData.RuntimeInstallAocFlag)
                    {
                        addAppElementAction("RuntimeAddOnContentInstall", "AllowAppend");
                    }

                    var cp = settingData.CardSpec;
                    if (cp != null && cp.IsSetAnyData())
                    {
                        // <CardSpec> タグを追加する
                        var cardSpec = metaXmlDoc.CreateElement("CardSpec");
                        root.AppendChild(cardSpec);

                        if (cp.Size != null)
                        {
                            var sizeElement = metaXmlDoc.CreateElement("Size");
                            sizeElement.InnerText = cp.Size;
                            cardSpec.AppendChild(sizeElement);
                        }

                        if (cp.ClockRate != null)
                        {
                            var clockElement = metaXmlDoc.CreateElement("ClockRate");
                            clockElement.InnerText = cp.ClockRate;
                            cardSpec.AppendChild(clockElement);
                        }
                    }
                }
                else if (type == CommonParams.ContentTypeSystemProgram)
                {
                    // type に SystemProgram が指定された場合
                    // MakeMetaを通す関係で<Core>ディレクティブはひとまず必要な模様
                    var core = metaXmlDoc.CreateElement("Core");
                    root.AppendChild(core);

                    {
                        if (settingData.IsSetName == true)
                        {
                            var nameElement = metaXmlDoc.CreateElement("Name");
                            nameElement.InnerText = settingData.MetaName;
                            core.AppendChild(nameElement);
                        }

                        var idElementName = "ProgramId";
                        var idElement = metaXmlDoc.CreateElement(idElementName);
                        idElement.InnerText = settingData.MetaInputId;
                        core.AppendChild(idElement);
                    }

                    // 新たに<SystemProgram>ディレクティブを追加する
                    var sysProgram = metaXmlDoc.CreateElement(CommonParams.ContentTypeSystemProgram);
                    root.AppendChild(sysProgram);

                    var sysIdElement = metaXmlDoc.CreateElement("Id");
                    sysIdElement.InnerText = settingData.MetaInputId;
                    sysProgram.AppendChild(sysIdElement);

                    addVersionAction(sysProgram);
                }
                else if (type == CommonParams.ContentTypeSystemData)
                {
                    // type に SystemData が指定された場合
                    // 新たに<SystemData>ディレクティブを追加する
                    var sysData = metaXmlDoc.CreateElement(CommonParams.ContentTypeSystemData);
                    root.AppendChild(sysData);

                    var idElement = metaXmlDoc.CreateElement("Id");
                    idElement.InnerText = settingData.MetaInputId;
                    sysData.AppendChild(idElement);

                    addVersionAction(sysData);
                }
                else if (type == CommonParams.ContentTypeAddOnContent)
                {
                    if (settingData.AocList.Count == 0)
                    {
                        // type に AddOnContent が指定された場合 (--index が1つだけの場合)
                        var aoc = metaXmlDoc.CreateElement("AddOnContent");
                        root.AppendChild(aoc);

                        // Index は上位で必ず設定されているはずなので null のチェックは不要
                        var indexElement = metaXmlDoc.CreateElement("Index");
                        indexElement.InnerText = settingData.MetaIndex;
                        aoc.AppendChild(indexElement);

                        addVersionAction(aoc);

                        var idElement = metaXmlDoc.CreateElement("ApplicationId");
                        idElement.InnerText = settingData.MetaInputId;
                        aoc.AppendChild(idElement);

                        // 必須バージョン値は必ず設定されているはずなので <AddOnContent> タグ内に追加
                        {
                            // <RequiredApplicationReleaseVersion> タグを追加する
                            var requiredElement = metaXmlDoc.CreateElement("RequiredApplicationReleaseVersion");
                            requiredElement.InnerText = settingData.RequiredVersion;
                            aoc.AppendChild(requiredElement);
                        }

                        // (SIGLO-49375) Tag 値を設定
                        {
                            var tagElement = metaXmlDoc.CreateElement("Tag");
                            tagElement.InnerText = settingData.MetaTag;
                            aoc.AppendChild(tagElement);
                        }
                    }
                    else
                    {
                        // --index が範囲指定されたの場合
                        // AddOnContent の複数設定
                        foreach (var aoc in settingData.AocList)
                        {
                            var aocElement = metaXmlDoc.CreateElement("AddOnContent");
                            root.AppendChild(aocElement);

                            // <Index> を追加する
                            var indexElement = metaXmlDoc.CreateElement("Index");
                            indexElement.InnerText = aoc.Index;
                            aocElement.AppendChild(indexElement);

                            // <ReleaseVersion> を追加する
                            var releaseVersionElement = metaXmlDoc.CreateElement("ReleaseVersion");
                            releaseVersionElement.InnerText = aoc.ReleaseVersion;
                            aocElement.AppendChild(releaseVersionElement);

                            // <PrivateVersion> を追加する
                            var privateVersionElement = metaXmlDoc.CreateElement("PrivateVersion");
                            privateVersionElement.InnerText = aoc.PrivateVersion;
                            aocElement.AppendChild(privateVersionElement);

                            // <ApplicationId> を追加する
                            var idElement = metaXmlDoc.CreateElement("ApplicationId");
                            idElement.InnerText = aoc.ApplicationId;
                            aocElement.AppendChild(idElement);

                            // <RequiredApplicationReleaseVersion> を追加する
                            var requiredElement = metaXmlDoc.CreateElement("RequiredApplicationReleaseVersion");
                            requiredElement.InnerText = aoc.RequiredApplicationReleaseVersion;
                            aocElement.AppendChild(requiredElement);

                            // <Tag> を追加する
                            var tagElement = metaXmlDoc.CreateElement("Tag");
                            tagElement.InnerText = aoc.Tag;
                            aocElement.AppendChild(tagElement);
                        }
                    }
                }

                // 作業用の子ディレクトリを作成
                var childDirPath = this.MakeChildWorkDirectory(settingData.MetaProgramId);
                // ファイルに保存
                metaXmlFilePath = Path.Combine(childDirPath, CommonParams.TempMetaFileName);
                if (File.Exists(metaXmlFilePath) == true)
                {
                    File.Delete(metaXmlFilePath);
                }
                metaXmlDoc.Save(metaXmlFilePath);

                // 子ディレクトリの設定を保持させる
                settingData.ChildWorkDirPath = childDirPath;
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
            }

            return metaXmlFilePath;
        }

        private string CreateMetaFile(XmlNode inMetaAttributeNode, int inItemCount)
        {
            string metaXmlFilePath = null;
            try
            {
                var metaXmlDoc = new XmlDocument();

                var head = metaXmlDoc.CreateXmlDeclaration("1.0", "utf-8", null);
                var root = metaXmlDoc.CreateElement(CommonParams.NmetaRootName);

                metaXmlDoc.AppendChild(head);
                metaXmlDoc.AppendChild(root);

                // 引数のXMLの内容をそのままMeta要素に入れる
                root.InnerXml = inMetaAttributeNode.InnerXml;

                // 作業用の子ディレクトリを作成
                var childDirPath = this.MakeChildWorkDirectory(inItemCount);
                // ファイルに保存
                metaXmlFilePath = Path.Combine(childDirPath, CommonParams.TempMetaFileName);
                metaXmlDoc.Save(metaXmlFilePath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
            }

            return metaXmlFilePath;
        }

        /// <summary>
        /// デフォルトの設定情報をメタファイルに追記する
        /// デフォルト情報を追加しない設定である場合は何もしない
        /// </summary>
        /// <param name="inSettingParam">設定パラメータ</param>
        public void AddDefaultMetaParameter(SettingParams inSettingParam)
        {
            if (inSettingParam.ContentType != CommonParams.ContentTypeApplication &&
                inSettingParam.ContentType != CommonParams.ContentTypePatch)
            {
                // 本関数は Application または Patch のみが対象
                return;
            }

            try
            {
                var xmlDoc = new XmlDocument();
                // metaファイルの読み込み
                xmlDoc.Load(inSettingParam.MetaFilePath);

                var metaNode = this.GetMetaRootNode(xmlDoc);
                if (metaNode == null)
                {
                    // ありえないと思うが Meta ノードが存在しない場合は以降の処理は無意味なのですぐに抜ける
                    return;
                }

                var appNode = metaNode.SelectSingleNode("Application");
                if (appNode == null)
                {
                    // Application ノードが存在しない場合は作成して、以降の処理を行う
                    var appElement = xmlDoc.CreateElement("Application");
                    metaNode.AppendChild(appElement);
                    appNode = appElement;
                }

                var isSetDefaultTitle = true;
                var titleNodeList = appNode.SelectNodes("Title");
                if (titleNodeList != null && titleNodeList.Count > 0)
                {
                    // Title 情報が1つでも設定されていればデフォルト情報として追加しない
                    isSetDefaultTitle = false;
                }

                var isSetDefaultSupportedLang = true;
                var slNodeList = appNode.SelectNodes("SupportedLanguage");
                if (slNodeList != null && slNodeList.Count > 0)
                {
                    // SupportedLanguage 情報が1つでも設定されていればデフォルト情報として追加しない
                    isSetDefaultSupportedLang = false;
                }

                // ラムダ式な記述でタイトル設定処理を関数化しておく
                Action<List<SettingParams.IconSettingPair>> addTitleAction = (List<SettingParams.IconSettingPair> inList) =>
                {
                    if (isSetDefaultTitle)
                    {
                        foreach (var iconSetting in inList)
                        {
                            var titleNode = xmlDoc.CreateElement("Title");

                            var langNode = xmlDoc.CreateElement("Language");
                            langNode.InnerText = iconSetting.Language;
                            titleNode.AppendChild(langNode);

                            var titleNameText = inSettingParam.MetaName;
                            if (inSettingParam.IsSetName == false)
                            {
                                // 名前指定が無い場合はデフォルトタイトル名の設定
                                titleNameText = string.Format("{0}_v{1}_{2}",
                                    inSettingParam.MetaInputId, inSettingParam.ReleaseVersion, iconSetting.Language);
                            }
                            var nameNode = xmlDoc.CreateElement("Name");
                            nameNode.InnerText = titleNameText;
                            titleNode.AppendChild(nameNode);

                            // ひとまず Publisher は Nintendo 固定
                            var pubNode = xmlDoc.CreateElement("Publisher");
                            pubNode.InnerText = "Nintendo";
                            titleNode.AppendChild(pubNode);

                            appNode.AppendChild(titleNode);
                        }
                    }

                    if (isSetDefaultSupportedLang)
                    {
                        // 必須となる対応言語(SupportedLanguage)設定も合わせて設定しておく
                        foreach (var iconSetting in inList)
                        {
                            var supportedLanguageNode = xmlDoc.CreateElement("SupportedLanguage");
                            supportedLanguageNode.InnerText = iconSetting.Language;
                            appNode.AppendChild(supportedLanguageNode);
                        }
                    }
                };

                // アイコンの言語設定に合わせて Title 情報を追加する
                if (inSettingParam.IconList.Count > 0)
                {
                    addTitleAction(inSettingParam.IconList);
                }
                else if (inSettingParam.DefaultIconList.Count > 0)
                {
                    addTitleAction(inSettingParam.DefaultIconList);
                }

                var dvNode = appNode.SelectSingleNode("DisplayVersion");
                if (dvNode == null)
                {
                    // Meta/Application/DisplayVersion が存在しない場合、デフォルト情報として追加する
                    var dvElement = xmlDoc.CreateElement("DisplayVersion");
                    // 設定文字列のフォーマットは、"{ReleaseVersion}.{PrivateVersion}" とする
                    dvElement.InnerText = string.Format("{0}.{1}",
                        inSettingParam.ReleaseVersion, inSettingParam.PrivateVersion);
                    appNode.AppendChild(dvElement);
                }

                // metaファイルの上書き保存
                xmlDoc.Save(inSettingParam.MetaFilePath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        /// <summary>
        /// ファイルパス関連の設定をメタファイルに追記します
        /// スレッドセーフ化のため、作業ディレクトリにファイルまたはディレクトリをコピーする
        /// </summary>
        /// <param name="inSettingParam">設定パラメータ</param>
        public void AddFilePathMetaParameter(SettingParams inSettingParam)
        {
            if (inSettingParam.ContentType != CommonParams.ContentTypeApplication &&
                inSettingParam.ContentType != CommonParams.ContentTypePatch)
            {
                // 本関数は Application または Patch のみが対象
                return;
            }

            try
            {
                var xmlDoc = new XmlDocument();
                // metaファイルの読み込み
                xmlDoc.Load(inSettingParam.MetaFilePath);

                var metaNode = this.GetMetaRootNode(xmlDoc);
                if (metaNode == null)
                {
                    // ありえないと思うが Meta ノードが存在しない場合は以降の処理は無意味なのですぐに抜ける
                    return;
                }

                var appNode = metaNode.SelectSingleNode("Application");
                if (appNode == null)
                {
                    // Application ノードが存在しない場合は作成して、以降の処理を行う
                    var appElement = xmlDoc.CreateElement("Application");
                    metaNode.AppendChild(appElement);
                    appNode = appElement;
                }

                var htmlNode = appNode.SelectSingleNode("HtmlDocumentPath");
                if (htmlNode == null && inSettingParam.HtmlDocumentDirPath != null && inSettingParam.HtmlDocumentDirPath != string.Empty)
                {
                    // スレッドセーフ化のため作業ディレクトリにコピーする
                    var dstHtmlDir = Path.Combine(inSettingParam.ChildWorkDirPath, "HtmlDoc");
                    FsFacade.CopyDirectory(inSettingParam.HtmlDocumentDirPath, dstHtmlDir);
                    // 遊び方ドキュメントのディレクトリパスを指定
                    var htmlElement = xmlDoc.CreateElement("HtmlDocumentPath");
                    htmlElement.InnerText = dstHtmlDir;
                    appNode.AppendChild(htmlElement);
                }

                var urlsNode = appNode.SelectSingleNode("AccessibleUrlsFilePath");
                if (urlsNode == null && inSettingParam.AccessibleUrlsFilePath != null && inSettingParam.AccessibleUrlsFilePath != string.Empty)
                {
                    // スレッドセーフ化のため作業ディレクトリにコピーする
                    var dstUrlsFile = Path.Combine(inSettingParam.ChildWorkDirPath, "accessibleUrls.txt");
                    File.Copy(inSettingParam.AccessibleUrlsFilePath, dstUrlsFile);
                    // アプリケーションからどのサイトに接続できるかの情報のディレクトリパスを指定
                    var urlsElement = xmlDoc.CreateElement("AccessibleUrlsFilePath");
                    urlsElement.InnerText = dstUrlsFile;
                    appNode.AppendChild(urlsElement);
                }

                var legalNode = appNode.SelectSingleNode("LegalInformationFilePath");
                if (legalNode == null && inSettingParam.LegalInformationZipFilePath != null && inSettingParam.LegalInformationZipFilePath != string.Empty)
                {
                    // スレッドセーフ化のため作業ディレクトリにコピーする
                    var dstLegalInfoFile = Path.Combine(inSettingParam.ChildWorkDirPath, "legalInfo.zip");
                    File.Copy(inSettingParam.LegalInformationZipFilePath, dstLegalInfoFile);
                    // ソフトリーガル情報用のドキュメントの zip ファイルパスを指定
                    var legalElement = xmlDoc.CreateElement("LegalInformationFilePath");
                    legalElement.InnerText = dstLegalInfoFile;
                    appNode.AppendChild(legalElement);
                }

                // アイコン情報を指定
                var iconNodeList = appNode.SelectNodes("Icon");
                if (iconNodeList == null || iconNodeList.Count == 0)
                {
                    foreach (var iconSetting in inSettingParam.IconList)
                    {
                        var iconNode = xmlDoc.CreateElement("Icon");

                        // 言語文字列を指定
                        var langNode = xmlDoc.CreateElement("Language");
                        langNode.InnerText = iconSetting.Language;
                        iconNode.AppendChild(langNode);

                        // スレッドセーフ化のため作業ディレクトリにコピーする
                        var dstIconFileFile = Path.Combine(inSettingParam.ChildWorkDirPath,
                            string.Format("icon_{0}.bmp", iconSetting.Language));
                        File.Copy(iconSetting.IconFilePath, dstIconFileFile);
                        // Iconのファイルパスを設定
                        var pathNode = xmlDoc.CreateElement("IconPath");
                        pathNode.InnerText = dstIconFileFile;
                        iconNode.AppendChild(pathNode);

                        // 処理的にあまり効率的ではないが NxIconPath の設定処理もここで実施しておく
                        foreach (var nxIconSetting in inSettingParam.NxIconList)
                        {
                            if (nxIconSetting.Language == iconSetting.Language)
                            {
                                // スレッドセーフ化のため作業ディレクトリにコピーする
                                var dstNxIconFileFile = Path.Combine(inSettingParam.ChildWorkDirPath,
                                    string.Format("nxicon_{0}.jpg", nxIconSetting.Language));
                                File.Copy(nxIconSetting.IconFilePath, dstNxIconFileFile);
                                // NxIconのファイルパスを設定
                                var nxPathNode = xmlDoc.CreateElement("NxIconPath");
                                nxPathNode.InnerText = dstNxIconFileFile;
                                iconNode.AppendChild(nxPathNode);
                                break;
                            }
                        }

                        appNode.AppendChild(iconNode);
                    }

                    if (inSettingParam.IconList.Count == 0)
                    {
                        // デフォルト設定のアイコン画像用
                        // 作業ディレクトリに作成済なのでコピー処理は不要
                        // NxIcon も特に考慮する必要なし
                        foreach (var iconSetting in inSettingParam.DefaultIconList)
                        {
                            var iconNode = xmlDoc.CreateElement("Icon");

                            var langNode = xmlDoc.CreateElement("Language");
                            langNode.InnerText = iconSetting.Language;
                            iconNode.AppendChild(langNode);

                            var pathNode = xmlDoc.CreateElement("IconPath");
                            pathNode.InnerText = iconSetting.IconFilePath;
                            iconNode.AppendChild(pathNode);

                            appNode.AppendChild(iconNode);
                        }
                    }
                }

                // metaファイルの上書き保存
                xmlDoc.Save(inSettingParam.MetaFilePath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        /// <summary>
        /// ファイルパス関連の設定をメタファイルに追記します
        /// </summary>
        /// <param name="inSettingParam">設定パラメータ</param>
        public void AddAocDataFilePathMetaParameter(SettingParams inSettingParam, string inAddDirPath)
        {
            if (inSettingParam.ContentType != CommonParams.ContentTypeAddOnContent)
            {
                // 本関数は AddOnContent のみが対象
                return;
            }

            try
            {
                var xmlDoc = new XmlDocument();
                // metaファイルの読み込み
                xmlDoc.Load(inSettingParam.MetaFilePath);

                var metaNode = this.GetMetaRootNode(xmlDoc);
                if (metaNode == null)
                {
                    // ありえないと思うが Root ノードが存在しない場合は抜ける
                    Console.Error.WriteLine("AddAocDataFilePathMetaParameter() : <NintendoSdkMeta> or <Meta> is not found");
                    return;
                }

                var aocNodeList = metaNode.SelectNodes("AddOnContent");
                if (aocNodeList == null || aocNodeList.Count == 0)
                {
                    // ありえないと思うが AddOnContent ノードが存在しない場合は抜ける
                    Console.Error.WriteLine("AddAocDataFilePathMetaParameter() : <AddOnContent> is not found");
                    return;
                }

                const string DataPathTagName = "DataPath";

                // 念のため既存の処理への影響を考えて条件分岐しておく
                if (aocNodeList.Count == 1)
                {
                    // こちらは既存処理
                    var aocNode = aocNodeList[0];
                    var dataNode = aocNode.SelectSingleNode(DataPathTagName);
                    if (dataNode != null)
                    {
                        // DataPath タグがすでに設定済の場合は抜ける
                        return;
                    }

                    // コンテンツのディレクトリパスを指定
                    var dataPathNode = xmlDoc.CreateElement(DataPathTagName);
                    dataPathNode.InnerText = inAddDirPath;
                    aocNode.AppendChild(dataPathNode);
                }
                else
                {
                    // こちらは新たな複数設定用処理 (2個以上の場合を考慮)
                    for (int i = 0; i < aocNodeList.Count; ++i)
                    {
                        var dataNode = aocNodeList[i].SelectSingleNode(DataPathTagName);
                        if (dataNode != null)
                        {
                            // DataPath タグがすでに設定済の場合は次の要素へ
                            continue;
                        }

                        // コンテンツのディレクトリパスを新たに追加する
                        var dataPathNode = xmlDoc.CreateElement(DataPathTagName);
                        dataPathNode.InnerText = inSettingParam.AocList[i].DataPath;
                        aocNodeList[i].AppendChild(dataPathNode);
                    }
                }

                // metaファイルの上書き保存
                xmlDoc.Save(inSettingParam.MetaFilePath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        /// <summary>
        /// ファイルパス関連の設定をメタファイル設定をスレッドセーフのため
        /// 作業フォルダにコピーし、メタファイルを書き換える
        /// </summary>
        /// <param name="inSettingParam">設定パラメータ</param>
        public void ReplaceFilePathMetaParameter(SettingParams inSettingParam)
        {
            if (inSettingParam.IsEditMetaFileForCommandArgs == true)
            {
                // メタファイルを編集する設定では不要な処理なので、何もせずに返る
                return;
            }

            if (inSettingParam.ContentType == CommonParams.ContentTypeSystemProgram ||
                inSettingParam.ContentType == CommonParams.ContentTypeSystemData)
            {
                // システム系タイプのメタファイルでは関係しない処理なので、何もせずに返る
                return;
            }

            try
            {
                var xmlDoc = new XmlDocument();
                // metaファイルの読み込み
                xmlDoc.Load(inSettingParam.MetaFilePath);
                var metaNode = this.GetMetaRootNode(xmlDoc);
                if (metaNode == null)
                {
                    // Metaファイルの Root ノードが存在しない場合は以降の処理は無意味なのですぐに抜ける
                    return;
                }

                if (inSettingParam.ContentType == CommonParams.ContentTypeApplication ||
                    inSettingParam.ContentType == CommonParams.ContentTypePatch)
                {
                    var appNode = metaNode.SelectSingleNode("Application");
                    if (appNode == null)
                    {
                        // Application ノードが存在しない場合は以降の処理は無意味なのですぐに抜ける
                        return;
                    }

                    var htmlNode = appNode.SelectSingleNode("HtmlDocumentPath");
                    if (htmlNode != null)
                    {
                        // スレッドセーフ化のため作業ディレクトリにコピーする
                        var dstHtmlDir = Path.Combine(inSettingParam.ChildWorkDirPath, "HtmlDoc");
                        FsFacade.CopyDirectory(replaceEnvVar(htmlNode.InnerText), dstHtmlDir);
                        // 遊び方ドキュメントのディレクトリパスを置き換え
                        htmlNode.InnerText = dstHtmlDir;
                    }

                    var urlsNode = appNode.SelectSingleNode("AccessibleUrlsFilePath");
                    if (urlsNode != null)
                    {
                        // スレッドセーフ化のため作業ディレクトリにコピーする
                        var dstUrlsFile = Path.Combine(inSettingParam.ChildWorkDirPath, "accessibleUrls.txt");
                        File.Copy(replaceEnvVar(urlsNode.InnerText), dstUrlsFile);
                        // アプリケーションからどのサイトに接続できるかの情報のファイルパスを指定
                        urlsNode.InnerText = dstUrlsFile;
                    }

                    var legalNode = appNode.SelectSingleNode("LegalInformationFilePath");
                    if (legalNode != null)
                    {
                        // スレッドセーフ化のため作業ディレクトリにコピーする
                        var dstLegalInfoFile = Path.Combine(inSettingParam.ChildWorkDirPath, "legalInfo.zip");
                        File.Copy(replaceEnvVar(legalNode.InnerText), dstLegalInfoFile);
                        // ソフトリーガル情報用のドキュメントの zip ファイルパスを指定
                        legalNode.InnerText = dstLegalInfoFile;
                    }

                    // アイコン情報を指定
                    // 明示的に指定された場合
                    var iconNodeList = appNode.SelectNodes("Icon");
                    if (iconNodeList != null)
                    {
                        foreach (XmlNode iconNode in iconNodeList)
                        {
                            // 言語文字列を拝借
                            var langNode = iconNode.SelectSingleNode("Language");
                            if (langNode == null)
                            {
                                // ありえないと思うが Language ノードが存在しない場合は次の要素へ
                                continue;
                            }

                            var iconPathNode = iconNode.SelectSingleNode("IconPath");
                            if (iconPathNode != null)
                            {
                                // スレッドセーフ化のため作業ディレクトリにコピーする
                                var dstIconFileFile = Path.Combine(inSettingParam.ChildWorkDirPath,
                                    string.Format("icon_{0}.bmp", langNode.InnerText));
                                File.Copy(replaceEnvVar(iconPathNode.InnerText), dstIconFileFile);
                                // Iconのファイルパスを設定
                                iconPathNode.InnerText = dstIconFileFile;
                            }

                            var nxIconPathNode = iconNode.SelectSingleNode("NxIconPath");
                            if (nxIconPathNode != null)
                            {
                                // スレッドセーフ化のため作業ディレクトリにコピーする
                                var dstNxIconFileFile = Path.Combine(inSettingParam.ChildWorkDirPath,
                                    string.Format("nxicon_{0}.jpg", langNode.InnerText));
                                File.Copy(replaceEnvVar(nxIconPathNode.InnerText), dstNxIconFileFile);
                                // Iconのファイルパスを設定
                                nxIconPathNode.InnerText = dstNxIconFileFile;
                            }
                        }
                    }
                }
                else if (inSettingParam.ContentType == CommonParams.ContentTypeAddOnContent)
                {
                    var aocNodeList = metaNode.SelectNodes("AddOnContent");
                    if (aocNodeList == null || aocNodeList.Count == 0)
                    {
                        // AddOnContent ノードが存在しない場合は以降の処理は無意味なのですぐに抜ける
                        return;
                    }

                    for (int i = 0; i < aocNodeList.Count; ++i)
                    {
                        var aocNode = aocNodeList.Item(i);
                        var pathNode = aocNode.SelectSingleNode("DataPath");
                        if (pathNode != null)
                        {
                            // スレッドセーフ化のため作業ディレクトリにコピーする
                            var dstAocDir = Path.Combine(inSettingParam.ChildWorkDirPath, string.Format("AocDir_{0}", i));
                            FsFacade.CopyDirectory(replaceEnvVar(pathNode.InnerText), dstAocDir);
                            // データディレクトリパスを置き換え
                            pathNode.InnerText = dstAocDir;
                        }
                    }
                }

                // metaファイルの上書き保存
                xmlDoc.Save(inSettingParam.MetaFilePath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        /// <summary>
        /// オーサリングツールに渡すためのゼロ埋めのバイナリファイル(ダミーコード)を作成します
        /// </summary>
        /// <param name="settingData">メタ情報のパラメータ</param>
        /// <param name="setSize">作成するファイルの指定サイズ[Byte]</param>
        /// <returns>作成したファイルパス</returns>
        public string CreateZeroBinFile(SettingParams settingData, long setSize)
        {
            var binFilePath = Path.Combine(settingData.ChildWorkDirPath, "ZeroData.dat");
            using (var dFile = File.Create(binFilePath))
            {
                try
                {
                    dFile.Position = setSize - 1;
                    dFile.WriteByte(0x00);
                }
                catch (Exception err)
                {
                    Console.Error.WriteLine(err.StackTrace);
                    Console.Error.WriteLine(err.Message);
                }
            }

            return binFilePath;
        }

        /// <summary>
        /// 中身がランダムデータなバイナリファイル(ダミーデータ)を作成します
        /// </summary>
        /// <param name="settingData">メタ情報のパラメータ</param>
        /// <param name="setSize">作成するファイルの指定サイズ[Byte]</param>
        /// <returns>作成したファイルパス</returns>
        public string CreateRandomBinFile(SettingParams settingData, long setSize)
        {
            // こちらは従来の固定処理
            var binFilePath = Path.Combine(settingData.ChildWorkDirPath, "RandomData.dat");
            CreateRandomBinFile(binFilePath, setSize, settingData.SeedNumber);
            return binFilePath;
        }

        /// <summary>
        /// 中身がランダムデータなバイナリファイル(ダミーデータ)を作成します
        /// </summary>
        /// <param name="inFilePath">出力ファイルパス</param>
        /// <param name="inSettingSize">作成するファイルの指定サイズ[Byte]</param>
        /// <param name="inSeed">ランダムデータの指定Seed値</param>
        public void CreateRandomBinFile(string inFilePath, long inSettingSize, string inSeed)
        {
            // (SIGLO-46664) Seed値のチェック
            int seedNum = 0;
            var isSetSeed = int.TryParse(inSeed, out seedNum);

            // ファイルへの最大書き込み単位はひとまず 256 KB にしておく
            const long WritePerMaxSize = 256 * 1024;
            using (var dFile = File.Create(inFilePath))
            {
                var rnd = (isSetSeed) ? new Random(seedNum) : new Random();
                var bList = new byte[WritePerMaxSize];
                var bw = new BinaryWriter(dFile);
                try
                {
                    var resSize = inSettingSize;
                    while (resSize > 0)
                    {
                        rnd.NextBytes(bList);

                        // WritePerMaxSize が int 型の範囲内に収まる限り、int 型へキャストしても問題ないはず
                        var writeSize = (int)((resSize > WritePerMaxSize) ? WritePerMaxSize : resSize);

                        bw.Write(bList, 0, writeSize);
                        resSize -= writeSize;
                    }
                }
                catch (Exception err)
                {
                    Console.Error.WriteLine(err.StackTrace);
                    Console.Error.WriteLine(err.Message);
                }
            }
        }

        /// <summary>
        /// 引数文字列を書き込んだテキストファイルを作成します
        /// (UTF8のBOM付き形式で保存する)
        /// </summary>
        /// <param name="inFilePath">作成するファイルパス</param>
        /// <param name="inWriteValue">書き込む文字列</param>
        public static void CreateRecordFile(string inFilePath, string inWriteValue)
        {
            // 引数文字列をファイルに書き込み (UTF8のBOM付き形式で保存)
            using (var sw = new StreamWriter(inFilePath, false, Encoding.UTF8))
            {
                sw.Write(inWriteValue);
            }
        }

        /// <summary>
        /// ハッシュ値を書き込んだテキストファイル (HashVal.txt) を作成します
        /// </summary>
        /// <param name="inHashValue">書き込むハッシュ値文字列</param>
        /// <param name="inOutputDirPath">作成するディレクトリパス</param>
        /// <returns>作成したファイルパス</returns>
        public static string CreateHashValueFile(string inHashValue, string inOutputDirPath)
        {
            var filePath = Path.Combine(inOutputDirPath, @"HashVal.txt");
            CreateRecordFile(filePath, inHashValue);
            return filePath;
        }

        /// <summary>
        /// バージョン値を書き込んだテキストファイル (_Version_.txt) を作成します
        /// </summary>
        /// <param name="inVersionVal">書き込むバージョン値文字列</param>
        /// <param name="inOutputDirPath">作成するディレクトリパス</param>
        /// <returns>作成したファイルパス</returns>
        public static string CreateVersionValueFile(string inVersionValue, string inOutputDirPath)
        {
            var filePath = Path.Combine(inOutputDirPath, @"_Version_.txt");
            CreateRecordFile(filePath, inVersionValue);
            return filePath;
        }

        /// <summary>
        /// AbortさせるResult値を書き込んだテキストファイル (AbortResult.txt) を作成します
        /// </summary>
        /// <param name="inHashValue">書き込むResult値文字列</param>
        /// <param name="inOutputDirPath">作成するディレクトリパス</param>
        /// <returns>作成したファイルパス</returns>
        public static string CreateAbortResultValueFile(string inResultValue, string inOutputDirPath)
        {
            var filePath = Path.Combine(inOutputDirPath, @"AbortResult.txt");
            CreateRecordFile(filePath, inResultValue);
            return filePath;
        }

        /// <summary>
        /// 起動時にネットワーク接続要求を実行するかどうかの判断するためのファイル (NetworkRequest.txt) を作成します
        /// ファイルの中身は実質的に意味はなくファイルの有無で実行の可否を判断する予定
        /// </summary>
        /// <param name="inValue">書き込む文字列</param>
        /// <param name="inOutputDirPath">作成するディレクトリパス</param>
        /// <returns></returns>
        public static string CreateNetworkRequestFile(string inValue, string inOutputDirPath)
        {
            var filePath = Path.Combine(inOutputDirPath, @"NetworkRequest.txt");
            CreateRecordFile(filePath, inValue);
            return filePath;
        }

        /// <summary>
        /// 設定XMLファイルの内容を SettingParams のリストに変換する
        /// </summary>
        /// <param name="inXmlPath">設定XMLファイルのパス</param>
        /// <returns>SettingParamsリスト</returns>
        public List<SettingParams> ParseXmlSettingFile(string inXmlPath)
        {
            var settingList = new List<SettingParams>();
            var xmlDoc = new XmlDocument();

            try
            {
                xmlDoc.Load(inXmlPath);

                var appNodeList = xmlDoc.SelectNodes("/TestApplicationSetting/TestApplication");
                if (appNodeList == null)
                {
                    // 存在しない場合は無効である旨を表示して以降の処理を行わない
                    Console.Error.WriteLine("ParseXmlSettingFile()：XMLの設定ファイルフォーマットが不正です");
                    return settingList;
                }

                for (int i = 0; i < appNodeList.Count; ++i)
                {
                    var appNode = appNodeList.Item(i);

                    // コンテンツタイプのデフォルト値(Application)を設定しておく
                    string typeStr = CommonParams.ContentTypeApplication;
                    var node = appNode.SelectSingleNode("Type");
                    if (node != null)
                    {
                        switch (node.InnerText)
                        {
                            case CommonParams.ContentTypeSystemProgram:
                            case CommonParams.ContentTypeSystemData:
                            case CommonParams.ContentTypeAddOnContent:
                            case CommonParams.ContentTypePatch:
                                typeStr = node.InnerText;
                                break;
                            default:
                                // 指定外の文字列が設定された場合はデフォルト値(Application)を適用
                                break;
                        }
                    }

                    node = appNode.SelectSingleNode("MetaSetting");
                    if (this.IsValidMetaSetting(node, typeStr) == false)
                    {
                        // 明らかに不正な Meta 設定の場合は該当要素はスキップさせる
                        continue;
                    }

                    var param = new SettingParams();
                    param.MetaFilePath = this.CreateMetaFile(node, i);
                    if (param.MetaFilePath == null)
                    {
                        Console.Error.WriteLine("ParseXmlSettingFile()：(MetaSetting[{0}]) nmetaファイルの生成に失敗しました", i);
                        // 処理を続行してもしょうがないので次のApplication設定要素に進む
                        continue;
                    }
                    // 一括XML設定時の特殊な処理(作業用の子ディレクトリ設定)
                    param.ChildWorkDirPath = Path.GetDirectoryName(param.MetaFilePath);

                    param.SetMetaParam(node, typeStr);

                    // Size エレメントの確認
                    node = appNode.SelectSingleNode("Size");
                    if (node != null)
                    {
                        param.CodeSize = node.InnerText;
                    }

                    // Seed エレメントの確認
                    param.SeedNumber = null;
                    node = appNode.SelectSingleNode("Seed");
                    if (node != null)
                    {
                        param.SeedNumber = node.InnerText;
                    }

                    // Desc エレメントの確認
                    node = appNode.SelectSingleNode("Desc");
                    if (node != null)
                    {
                        var filePath = replaceEnvVar(node.InnerText);
                        // 念のため指定ファイルパスが存在するかどうかを確認
                        if (File.Exists(filePath) == true)
                        {
                            param.DescFilePath = filePath;
                        }
                    }

                    // HtmlDocument エレメントの確認
                    node = appNode.SelectSingleNode("HtmlDocument");
                    if (node != null)
                    {
                        var dirPath = replaceEnvVar(node.InnerText);
                        // 念のため指定ディレクトリパスが存在するかどうかを確認
                        if (Directory.Exists(dirPath) == true)
                        {
                            param.HtmlDocumentDirPath = dirPath;
                        }
                    }

                    // AccessibleUrls エレメントの確認
                    node = appNode.SelectSingleNode("AccessibleUrls");
                    if (node != null)
                    {
                        var filePath = replaceEnvVar(node.InnerText);
                        // 念のため指定ディレクトリパスが存在するかどうかを確認
                        if (File.Exists(filePath) == true)
                        {
                            param.AccessibleUrlsFilePath = filePath;
                        }
                    }

                    // LegalInformation エレメントの確認
                    node = appNode.SelectSingleNode("LegalInformation");
                    if (node != null)
                    {
                        var zipFilePath = replaceEnvVar(node.InnerText);
                        // 念のため指定ディレクトリパスが存在するかどうかを確認
                        if (File.Exists(zipFilePath) &&
                            Path.GetExtension(zipFilePath).ToLower() == ".zip")
                        {
                            param.LegalInformationZipFilePath = zipFilePath;
                        }
                    }

                    // FileName エレメントの確認
                    param.IsSetOutputFileName = false;
                    node = appNode.SelectSingleNode("FileName");
                    if (node != null)
                    {
                        param.OutputFileName = node.InnerText;
                        if (param.OutputFileName != null && param.OutputFileName != string.Empty)
                        {
                            param.IsSetOutputFileName = true;
                        }
                    }

                    // Icon 設定の確認
                    var iconNodeList = appNode.SelectNodes("Icon");
                    if (iconNodeList != null)
                    {
                        for (int j = 0; j < iconNodeList.Count; ++j)
                        {
                            var iconNode = iconNodeList.Item(j);
                            var lang = iconNode.SelectSingleNode("Language");
                            var path = iconNode.SelectSingleNode("FilePath");
                            if (lang != null && path != null)
                            {
                                var iconSetting = new SettingParams.IconSettingPair(
                                    lang.InnerText, replaceEnvVar(path.InnerText));
                                param.IconList.Add(iconSetting);
                            }
                        }
                    }

                    // NxIcon 設定の確認
                    var nxIconNodeList = appNode.SelectNodes("NxIcon");
                    if (nxIconNodeList != null)
                    {
                        for (int j = 0; j < nxIconNodeList.Count; ++j)
                        {
                            var iconNode = nxIconNodeList.Item(j);
                            var lang = iconNode.SelectSingleNode("Language");
                            var path = iconNode.SelectSingleNode("FilePath");
                            if (lang != null && path != null)
                            {
                                var iconSetting = new SettingParams.IconSettingPair(
                                    lang.InnerText, replaceEnvVar(path.InnerText));
                                param.NxIconList.Add(iconSetting);
                            }
                        }
                    }

                    // DefaultIcon/Language 設定の確認
                    if (param.IconList.Count == 0 && param.IsSetMetaIcon == false)
                    {
                        // Icon画像を明示指定されていない場合のみデフォルトアイコン設定を行う
                        param.SetDefaultIconList(appNode.SelectNodes("DefaultIcon/Language"));
                    }

                    // DataDirectory エレメントの確認
                    node = appNode.SelectSingleNode("DataDirectory");
                    if (node != null)
                    {
                        var dirPath = replaceEnvVar(node.InnerText);
                        // 念のため指定ディレクトリパスが存在するかどうかを確認
                        if (Directory.Exists(dirPath) == true)
                        {
                            param.DataDirPath = dirPath;
                        }
                    }

                    // KeyIndex エレメントの確認
                    node = appNode.SelectSingleNode("KeyIndex");
                    if (node != null)
                    {
                        var index = node.InnerText;
                        // 指定値が空文字でないかどうかを確認
                        if (index != string.Empty)
                        {
                            param.KeyIndexNum = index;
                        }
                    }

                    // KeyConfigPath エレメントの確認
                    node = appNode.SelectSingleNode("KeyConfigPath");
                    if (node != null)
                    {
                        var configFilePath = replaceEnvVar(node.InnerText);
                        // 念のため指定ファイルパスが存在するかどうかを確認
                        if (File.Exists(configFilePath) == true)
                        {
                            param.KeyConfigFilePath = configFilePath;
                        }
                    }

                    // Ticket エレメントの確認
                    node = appNode.SelectSingleNode("Ticket");
                    if (node != null)
                    {
                        // タグが書かれてさえいれば基本的には有効とする
                        param.TicketFlag = true;
                        // ただし false (大文字・小文字を考慮しない)という文字列が設定されていれば無効とする
                        var flag = node.InnerText;
                        if (flag != string.Empty && flag.ToLower() == "false")
                        {
                            param.TicketFlag = false;
                        }
                    }

                    // NoTicket エレメントの確認
                    node = appNode.SelectSingleNode("NoTicket");
                    if (node != null)
                    {
                        // タグが書かれてさえいれば基本的には有効とする
                        param.NoTicketFlag = true;
                        // ただし false (大文字・小文字を考慮しない)という文字列が設定されていれば無効とする
                        var flag = node.InnerText;
                        if (flag != string.Empty && flag.ToLower() == "false")
                        {
                            param.NoTicketFlag = false;
                        }
                    }

                    // CodePath エレメントの確認
                    node = appNode.SelectSingleNode("CodePath");
                    if (node != null)
                    {
                        var dirPath = replaceEnvVar(node.InnerText);
                        // 念のため指定ディレクトリパスが存在するかどうかを確認
                        if (Directory.Exists(dirPath) == true)
                        {
                            param.CodeDirPath = dirPath;
                        }
                    }

                    // RomFsPath エレメントの確認
                    node = appNode.SelectSingleNode("RomFsPath");
                    if (node != null)
                    {
                        var dirPath = replaceEnvVar(node.InnerText);
                        // 念のため指定ディレクトリパスが存在するかどうかを確認
                        if (Directory.Exists(dirPath) == true)
                        {
                            param.RomFsDirPath = dirPath;
                        }
                    }

                    // OriginalApplicationFilePath エレメントの確認
                    node = appNode.SelectSingleNode("OriginalApplicationFilePath");
                    if (node != null)
                    {
                        // 並列実行の場合(後から作成)を考慮してファイルパスが存在するかどうかの確認は行わない
                        param.OriginalApplicationFilePath = replaceEnvVar(node.InnerText);
                    }

                    // OriginalApplicationForCreateNspFilePath エレメントの確認
                    node = appNode.SelectSingleNode("OriginalApplicationForCreateNspFilePath");
                    if (node != null)
                    {
                        // 並列実行の場合(後から作成)を考慮してファイルパスが存在するかどうかの確認は行わない
                        param.OriginalApplicationForCreateNspFilePath = replaceEnvVar(node.InnerText);
                    }

                    // PreviousPatchFilePath エレメントの確認
                    node = appNode.SelectSingleNode("PreviousPatchFilePath");
                    if (node != null)
                    {
                        // 並列実行の場合(後から作成)を考慮してファイルパスが存在するかどうかの確認は行わない
                        param.PreviousPatchFilePath = replaceEnvVar(node.InnerText);
                    }

                    // (SIGLO-48132) OutputFilePath エレメントの確認
                    node = appNode.SelectSingleNode("OutputFilePath");
                    if (node != null)
                    {
                        // 出力先パス指定のため存在確認は実施しない
                        param.OutputFilePath = replaceEnvVar(node.InnerText);
                    }

                    // (SIGLO-42685) AbortResultValue エレメントの確認
                    node = appNode.SelectSingleNode("AbortResultValue");
                    if (node != null)
                    {
                        // ひとまずそのまま設定する(判定は後ほど実施する)
                        param.AbortResultValue = node.InnerText;
                    }

                    // (SIGLO-52130) NetworkRequest エレメントの確認
                    node = appNode.SelectSingleNode("NetworkRequest");
                    if (node != null)
                    {
                        // タグが書かれてさえいれば基本的には有効とする
                        param.NetworkRequestFlag = true;
                        // ただし false (大文字・小文字を考慮しない)という文字列が設定されていれば無効とする
                        var flag = node.InnerText;
                        if (flag != string.Empty && flag.ToLower() == "false")
                        {
                            param.NetworkRequestFlag = false;
                        }
                    }

                    // (SIGLO-68043) Direct エレメントの確認
                    node = appNode.SelectSingleNode("Direct");
                    if (node != null)
                    {
                        // そのまま設定する
                        param.DirectString = node.InnerText;
                    }

                    // (SIGLO-73606) DirectForMakePatch エレメントの確認
                    node = appNode.SelectSingleNode("DirectForMakePatch");
                    if (node != null)
                    {
                        // そのまま設定する
                        param.DirectStringForMakePatch = node.InnerText;
                    }

                    settingList.Add(param);
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
            }

            return settingList;
        }

        /// <summary>
        /// 入力metaファイルを作業用フォルダに取り込み、メタ属性情報を読み込む
        /// </summary>
        /// <param name="inCommandParams">コマンドラインの入力オプションパラメータ</param>
        /// <returns>読み込み済みSettingParams</returns>
        public SettingParams ImportInputMetaFile(CommandLineParams inCommandParams)
        {
            SettingParams retSetting = null;
            try
            {
                // 作業用の子ディレクトリを作成
                var childDirPath = this.MakeChildWorkDirectory(0);
                // 作業フォルダに指定されたmetaファイルをコピーする
                var tempMetaFilePath = Path.Combine(childDirPath, CommonParams.TempMetaFileName);
                File.Copy(inCommandParams.MetaFilePath, tempMetaFilePath, true);

                var xmlDoc = new XmlDocument();
                xmlDoc.Load(tempMetaFilePath);

                var metaNode = this.GetMetaRootNode(xmlDoc);
                var type = inCommandParams.InputContentType;
                // Meta設定の必須項目が存在しているかどうかの確認
                if (this.IsValidMetaSetting(metaNode, type) == false)
                {
                    throw new Exception("ImportMetaFile()：IsValidMetaSetting() で失敗しました");
                }

                retSetting = new SettingParams();
                // 子ディレクトリの設定を保持させる
                retSetting.ChildWorkDirPath = childDirPath;

                retSetting.SetMetaParam(metaNode, type);

                retSetting.MetaFilePath = tempMetaFilePath;

                retSetting.CodeSize = inCommandParams.OutputProgramSize;

                retSetting.IsSetOutputFileName = false;
                if (inCommandParams.OutputFileName != null && inCommandParams.OutputFileName != string.Empty)
                {
                    retSetting.OutputFileName = inCommandParams.OutputFileName;
                    retSetting.IsSetOutputFileName = true;
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }

            return retSetting;
        }

        private bool IsValidMetaSetting(XmlNode inMetaSetting, string inContentType)
        {
            if (inMetaSetting == null || inContentType == null)
            {
                // あり得ないと思うが引数が null の場合は失敗させる
                return false;
            }

            // <Meta> 以下の XML が inMetaSetting に入っているものとする
            // 以下のチェック処理はあくまでも最低限のチェックのみ実施し、厳密なチェックは MakeMeta や AuthoringTool に任せる
            if (inContentType == CommonParams.ContentTypeApplication
                || inContentType == CommonParams.ContentTypePatch)
            {
                // Core/ProgramId がそもそも存在しているかどうかの確認
                var idNode = inMetaSetting.SelectSingleNode("Core/ProgramId");
                var appIdNode = inMetaSetting.SelectSingleNode("Core/ApplicationId");
                if (idNode == null && appIdNode == null)
                {
                    // 存在しない場合は無効である旨を表示して以降の処理を行わない
                    //Console.Error.WriteLine("ImportMetaFile()：ProgramId が設定されていません");
                    Console.Error.WriteLine("IsValidMetaSetting()：ProgramId または ApplicationId が設定されていません");
                    return false;
                }
            }
            else if (inContentType == CommonParams.ContentTypeSystemProgram)
            {
                var idNodeOld = inMetaSetting.SelectSingleNode("Core/ProgramId");
                var idNode = inMetaSetting.SelectSingleNode("SystemProgram/Id");
                if (idNode == null && idNodeOld == null)
                {
                    Console.Error.WriteLine("IsValidMetaSetting()：SystemProgram/Id または ProgramId が設定されていません");
                    return false;
                }
            }
            else if (inContentType == CommonParams.ContentTypeSystemData)
            {
                var idNode = inMetaSetting.SelectSingleNode("SystemData/Id");
                if (idNode == null)
                {
                    Console.Error.WriteLine("IsValidMetaSetting()：SystemData/Id が設定されていません");
                    return false;
                }
            }
            else if (inContentType == CommonParams.ContentTypeAddOnContent)
            {
                var aocMetaPath = "AddOnContent/";
                var node = inMetaSetting.SelectSingleNode(aocMetaPath + "ApplicationId");
                if (node == null)
                {
                    Console.Error.WriteLine("IsValidMetaSetting()：AddOnContent/ApplicationId が設定されていません");
                    return false;
                }

                node = inMetaSetting.SelectSingleNode(aocMetaPath + "ReleaseVersion");
                if (node == null)
                {
                    Console.Error.WriteLine("IsValidMetaSetting()：AddOnContent/ReleaseVersion が設定されていません");
                    return false;
                }

                node = inMetaSetting.SelectSingleNode(aocMetaPath + "Index");
                if (node == null)
                {
                    Console.Error.WriteLine("IsValidMetaSetting()：AddOnContent/Index が設定されていません");
                    return false;
                }
            }
            else
            {
                Console.Error.WriteLine("IsValidMetaSetting()：不正な ContentType が指定されています");
                return false;
            }

            return true;
        }

        /// <summary>
        /// 指定ファイルパスのSHA1(ハッシュ)文字列を取得する
        /// </summary>
        /// <param name="inFilePath">ハッシュ取得対象のファイルパス文字列</param>
        /// <returns>SHA1の16進表記文字列</returns>
        public string GetSha1String(string inFilePath)
        {
            string sha1Str = string.Empty;
            try
            {
                using (var dFile = File.OpenRead(inFilePath))
                {
                    var byteHashList = SHA1Managed.Create().ComputeHash(dFile);
                    var hashStr = new StringBuilder();
                    foreach (var b in byteHashList)
                    {
                        hashStr.Append(b.ToString("X2"));
                    }
                    sha1Str = hashStr.ToString();
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
            }
            return sha1Str;
        }

        /// <summary>
        /// metaファイルの中身を解析し、その内容を TestAppSettingParam に格納する
        /// </summary>
        /// <param name="inMetaFilePath">解析対象のmetaファイルパス文字列</param>
        /// <returns>読み込み済みTestAppSettingParam</returns>
        public TestAppSettingParam ParseMetaFileSetting(string inMetaFilePath)
        {
            TestAppSettingParam param = null;
            var xmlDoc = new XmlDocument();

            try
            {
                xmlDoc.Load(inMetaFilePath);
                var metaNode = this.GetMetaRootNode(xmlDoc);
                if (metaNode == null)
                {
                    // Meta ファイルの Root ノードが存在しなければ意味がないので、処理を抜ける
                    return param;
                }

                param = new TestAppSettingParam();

                // ラムダ式な記述でメタ情報リストの抽出処理を関数化
                Action<XmlNode, string> addMetaList = (XmlNode inSetNode, string inItemName) =>
                {
                    foreach (XmlNode cNode in inSetNode.ChildNodes)
                    {
                        if ((cNode.HasChildNodes == true && cNode.FirstChild.NodeType == XmlNodeType.Element) ||
                            cNode.NodeType == XmlNodeType.Comment)
                        {
                            // 子持ちノード(エレメント)を持っている、またはコメントノードは対象外としておく
                            continue;
                        }

                        var setPair = new TestAppSettingParam.SettingPair();
                        setPair.Item = string.Format("{0}/{1}", inItemName, cNode.Name);
                        setPair.Value = cNode.InnerText;

                        param.MetaSetList.Add(setPair);
                    }
                };

                var coreNode = metaNode.SelectSingleNode("Core");
                if (coreNode != null)
                {
                    addMetaList(coreNode, "Core");
                }

                var appNode = metaNode.SelectSingleNode("Application");
                if (appNode != null)
                {
                    // Meta/Application ノードが存在した場合
                    int titleCount = 0;
                    int ratingCount = 0;
                    foreach (XmlNode acNode in appNode.ChildNodes)
                    {
                        if ((acNode.HasChildNodes == false && acNode.NodeType != XmlNodeType.Comment) ||
                            (acNode.HasChildNodes == true && acNode.FirstChild.NodeType == XmlNodeType.Text))
                        {
                            // 子持ちノード(エレメント)がない場合はCore属性とほぼ同等の処理
                            var setPair = new TestAppSettingParam.SettingPair();
                            setPair.Item = "App/" + acNode.Name;
                            setPair.Value = acNode.InnerText;
                            param.MetaSetList.Add(setPair);
                            // とりあえず次のノードへ・・
                            continue;
                        }

                        if (acNode.HasChildNodes == true)
                        { // 子持ちノード有りの処理 ここから
                            // <Title> か <Rating> のはず
                            if (acNode.Name == "Title")
                            {
                                var item = string.Format("App/Title[{0}]", titleCount);
                                addMetaList(acNode, item);
                                ++titleCount;
                            }
                            else if (acNode.Name == "Rating")
                            {
                                var item = string.Format("App/Rating[{0}]", ratingCount);
                                addMetaList(acNode, item);
                                ++ratingCount;
                            }
                            else if (acNode.Name == "UserAccountSaveDataOperation")
                            {
                                // (SIGLO-62504) UserAccountSaveDataOperation 情報も MetaInfo に表示するようにしておく
                                // 表示文字数の制約上、実際の画面表示は無理やり省略させる。。
                                addMetaList(acNode, "App/UASDOperation");
                            }
                        } // 子持ちノード有りの処理 ここまで
                    } // for
                } // if (appNodeList != null)

                var programNode = metaNode.SelectSingleNode("SystemProgram");
                if (programNode != null)
                {
                    addMetaList(programNode, "SystemProgram");
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
            }
            return param;
        }

        public List<string> GetHtmlFilePathList(string inHtmlDocPath, string rootDirName = "")
        {
            var pathList = new List<string>();
            try
            {
                var docDirInfo = new DirectoryInfo(inHtmlDocPath);
                // ".htdocs"の名前のディレクトリかどうかを確認
                if (rootDirName == string.Empty && System.Text.RegularExpressions.Regex.IsMatch(docDirInfo.Name, @".+\.htdocs"))
                {
                    // "*.html"のファイル一覧を取得し、リストに詰めておく
                    foreach (var fileInfo in docDirInfo.GetFiles("*.html"))
                    {
                        pathList.Add(docDirInfo.Name + @"\" + fileInfo.Name);
                    }

                    // 子ディレクトリ一覧を取得
                    foreach (var dirInfo in docDirInfo.GetDirectories())
                    {
                        // 再帰呼び出しで取得したリストを詰めておく
                        var childPathList = this.GetHtmlFilePathList(docDirInfo.FullName + @"\" + dirInfo.Name, docDirInfo.Name);
                        if (childPathList.Count > 0)
                        {
                            pathList.AddRange(childPathList);
                        }
                    }
                }
                else
                {
                    // すでにRootディレクトリが検索済かどうかを念のため確認
                    if (rootDirName != string.Empty)
                    {
                        // Rootディレクトリ内であれば、Rootディレクトリからのパス文字列を抽出
                        var pos = inHtmlDocPath.IndexOf(rootDirName);
                        var dirPath = inHtmlDocPath.Substring(pos);
                        // "*.html"のファイル一覧を取得し、いい感じにパス文字列を整形してリストに詰めておく
                        foreach (var fileInfo in docDirInfo.GetFiles("*.html"))
                        {
                            pathList.Add(dirPath + @"\" + fileInfo.Name);
                        }
                    }

                    // 子ディレクトリ一覧を取得
                    foreach (var dirInfo in docDirInfo.GetDirectories())
                    {
                        // 再帰呼び出しで取得したリストを詰めておく
                        var childPathList = this.GetHtmlFilePathList(docDirInfo.FullName + @"\" + dirInfo.Name, rootDirName);
                        if (childPathList.Count > 0)
                        {
                            pathList.AddRange(childPathList);
                        }
                    }
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.Message);
                Console.Error.WriteLine("GetHtmlFilePathList()に失敗：inHtmlDocPath={0}", inHtmlDocPath);
                throw err;
            }
            return pathList;
        }
    }
}
