﻿using Nintendo.Foundation.IO;
using System;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Threading;

namespace MakeMiniDump
{
    class CommandLineInterface
    {
        public static int Run(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

            try
            {
                var settings = new CommandLineParserSettings()
                {
                    ApplicationDescription = "Create a minidump of the specified process."
                };

                var parser = new CommandLineParser(settings);

                CommandLineOption parsed;
                if (!parser.ParseArgs(args, out parsed))
                {
                    return 1;
                }

                uint processId;

                if (parsed.Name != null && parsed.Pid == null)
                {
                    var candidates = Process.GetProcesses().Where(process => process.ProcessName.Contains(parsed.Name));
                    if (!candidates.Any())
                    {
                        Console.Error.WriteLine("The process with the specified name was not found.");
                        return 1;
                    }
                    else if (!(candidates.Count() == 1))
                    {
                        Console.Error.WriteLine("More than one process with the specified name was found.");
                        Console.Error.WriteLine(string.Join(Environment.NewLine,
                            candidates.Select(process => $"  {process.ProcessName} (pid={process.Id})")));
                        return 1;
                    }
                    processId = (uint)candidates.First().Id;
                }
                else if (parsed.Pid != null && parsed.Name == null)
                {
                    try
                    {
                        processId = uint.Parse(parsed.Pid);
                    }
                    catch
                    {
                        Console.Error.WriteLine("Specify the process ID numerically.");
                        return 1;
                    }
                }
                else
                {
                    Console.Error.WriteLine("Specify either one of the options of --name or --pid.");
                    return 1;
                }


                if (!(!File.Exists(parsed.Out) || parsed.Force))
                {
                    Console.Error.WriteLine("File exists in the output path.");
                    return 1;
                }

                MiniDump.Write(processId, parsed.Out, MiniDump.DumpType.WithFullMemory);

                return 0;
            }
            catch (Exception e)
            {
                PrintException(e);
                return 1;
            }
        }

        private static void PrintException(Exception e)
        {
            Console.Error.WriteLine($"[ERROR] {e.Message}");
            Console.Error.WriteLine(string.Join(Environment.NewLine,
                "StackTrace:",
                e.StackTrace));
        }
    }
}
