﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;

namespace DecodeStackTrace
{
    internal class SourceInfo
    {
        public string functionName;
        public string fileName;
        public int lineNumber;
    }

    internal class StackTraceDecoder
    {
        public StackTraceDecoder(string logFilePath, IEnumerable<string> nssFiles)
        {
            if (!File.Exists(logFilePath))
            {
                throw new FileNotFoundException("The specified log file is not found.", logFilePath);
            }

            this.logFilePath = logFilePath;

            if (!(nssFiles.Count() > 0))
            {
                throw new ArgumentException("One or more nss file is needed.");
            }

            var notExistFiles = nssFiles.Where(nssFile => !File.Exists(nssFile));

            if (!(notExistFiles.Count() == 0))
            {
                throw new FileNotFoundException("The specified nss file(s) is not found.", string.Join(", ", notExistFiles));
            }

            nssTable = nssFiles.ToDictionary(nssPath => Path.GetFileNameWithoutExtension(nssPath));

            if (!(nssFiles.Count() == nssTable.Count))
            {
                throw new ArgumentException("Cannnot distinguish the nss files.", "nssFiles");
            }

            symbolTable = nssFiles.ToDictionary(
                nssPath => Path.GetFileNameWithoutExtension(nssPath),
                nssPath => new SymbolTable(nssPath));
        }

        public void Decode(string outPath, bool force)
        {
            EnsurePathClearance(outPath, force);

            using (var sw = new StreamWriter(new FileStream(outPath, FileMode.Create, FileAccess.Write)))
            {
                foreach (var line in ReadLines(logFilePath))
                {
                    sw.WriteLine(line);

                    var m = kernelStackTracePattern.Match(line);

                    if (!m.Success)
                    {
                        continue;
                    }

                    var indent = new string(' ', m.Index + 2);

                    var module = m.Groups["module"].Value;

                    if (!nssTable.ContainsKey(module))
                    {
                        sw.WriteLine(indent + "(no nss)");
                        continue;
                    }

                    var offset = int.Parse(m.Groups["offset"].Value, NumberStyles.HexNumber);

                    var isReturnAddress = IsReturnAddress(nssTable[module], offset);

                    var sourceInfo = isReturnAddress
                        ? GetSourceInfo(nssTable[module], offset - 4) // 戻りアドレスのときは、ジャンプ命令のアドレスで検索する。
                        : GetSourceInfo(nssTable[module], offset);

                    sw.WriteLine(indent + sourceInfo.functionName);
                    sw.WriteLine(indent + sourceInfo.fileName + ":" + (sourceInfo.lineNumber > 0 ? sourceInfo.lineNumber.ToString() : "?"));

                    if (!IsAddressIncludedInSymbol(offset, symbolTable[module]))
                    {
                        sw.WriteLine(indent + "(out of symbol)");
                    }
                    else if (!isReturnAddress)
                    {
                        sw.WriteLine(indent + "(not return address)");
                    }
                }
            }
        }

        private static void EnsurePathClearance(string path, bool force)
        {
            if (File.Exists(path))
            {
                if (!force)
                {
                    throw new ArgumentException("File already exists in the path.", path);
                }

                File.Delete(path);
            }
        }

        private static IEnumerable<string> ReadLines(string path)
        {
            using (var sr = new StreamReader(path))
            {
                while (!sr.EndOfStream)
                {
                    yield return sr.ReadLine();
                }
            }
        }

        private static SourceInfo GetSourceInfo(string nssFilePath, long address)
        {
            var output = ProcessInvoker.Invoke(
                EnvironmentInfo.Addr2LinePath,
                $" --demangle --functions --exe={nssFilePath} 0x{address:X}");

            var lines = output.Split(new string[] { Environment.NewLine }, StringSplitOptions.RemoveEmptyEntries);

            var m = Regex.Match(output, $@"(?<functionName>.*){Environment.NewLine}(?<fileName>.*):(?<lineNumber>(\d+|\?))", RegexOptions.Multiline);
            Debug.Assert(m.Success);

            return new SourceInfo()
            {
                fileName = m.Groups["fileName"].Value,
                lineNumber = m.Groups["lineNumber"].Value == "?" ? -1 : int.Parse(m.Groups["lineNumber"].Value),
                functionName = m.Groups["functionName"].Value,
            };
        }

        private bool IsAddressIncludedInSymbol(int offset, SymbolTable symbolTable)
        {
            try
            {
                var symbol = symbolTable.QuerySymbol(offset);
                return true;
            }
            catch
            {
                return false;
            }
        }

        private static bool IsReturnAddress(string nssFilePath, long address)
        {
            var dasmString = ProcessInvoker.Invoke(
                EnvironmentInfo.ObjdumpPath,
                $"--disassemble --demangle --start-address=0x{address - 4:X} --stop-address=0x{address:X} {nssFilePath}"); // TODO: thumb 命令を考慮する。

            return blInstructionPattern.Match(dasmString).Success || blrInstructionPattern.Match(dasmString).Success;
        }

        private string logFilePath;

        private readonly Dictionary<string, string> nssTable;
        private readonly Dictionary<string, SymbolTable> symbolTable;

        private static readonly Regex kernelStackTracePattern = new Regex(@"0x(?<address>[0-9a-f]{8,16}) \[\s*(?<module>[\w\.]+) \+ \s*(?<offset>[0-9a-f]+)\]");

        private static readonly Regex blInstructionPattern = new Regex(@"\s+(?<offset>[0-9a-f]+):\s+(?<assembly>[0-9a-f]+)\s+(?<mnemonic>bl)\s+(?<operand>[0-9a-f]+)\s+<(?<destination>.+)>");
        private static readonly Regex blrInstructionPattern = new Regex(@"\s+(?<offset>[0-9a-f]+):\s+(?<assembly>[0-9a-f]+)\s+(?<mnemonic>blr)\s+(?<operand>[\w]+)");
    }
}
