﻿using Nintendo.Foundation.IO;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Threading;

namespace DecodeStackTrace
{
    internal class CommandArgument
    {
        [CommandLineOption('l', "log", Description = "A log includes stack trace.", IsRequired = true)]
        public string Log { get; set; }

        [CommandLineOption('o', "out", Description = "An output path of the log decoded stack trace.", IsRequired = true)]
        public string Out { get; set; }

        [CommandLineOption('f', "force", Description = "Force over written the out put file.", IsRequired = false, DefaultValue = false)]
        public bool Force { get; set; }

        [CommandLineOption('n', "nss", Description = "A nss file.", IsRequired = false)]
        public string Nss { get; set; }

        [CommandLineOption('d', "directory", Description = "A directory includes nss file.", IsRequired = false)]
        public string Directory { get; set; }
    }

    internal class CommandInterface
    {
        public static int Run(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

            try
            {
                var settings = new CommandLineParserSettings()
                {
                    ApplicationDescription = "Decode stack trace included in the log using the specified nss file."
                };

                var parser = new CommandLineParser(settings);

                CommandArgument parsed;
                if (!parser.ParseArgs(args, out parsed))
                {
                    return 1;
                }

                var nssFiles = new List<string>();

                if (parsed.Nss != null)
                {
                    nssFiles.Add(parsed.Nss);
                }

                if (parsed.Directory != null)
                {
                    nssFiles.AddRange(Directory.GetFiles(parsed.Directory, "*.nss", SearchOption.AllDirectories));
                }

                if (!(nssFiles.Count > 0))
                {
                    Console.Error.WriteLine("You must specify one or more elf files.");
                    return 1;
                }

                if (!(!File.Exists(parsed.Out) || parsed.Force))
                {
                    Console.Error.WriteLine("File exists in the output path.");
                    return 1;
                }

                if (!Directory.Exists(Path.GetDirectoryName(parsed.Out)))
                {
                    Console.Error.WriteLine("Output directory is not exist.");
                    return 1;
                }

                var decoder = new StackTraceDecoder(parsed.Log, nssFiles);
                decoder.Decode(parsed.Out, parsed.Force);

                return 0;
            }
            catch (Exception e)
            {
                PrintException(e);
                return 1;
            }
        }

        private static void PrintException(Exception e)
        {
            Console.Error.WriteLine(string.Join(Environment.NewLine,
                $"[ERROR] {e.Message}",
                "StackTrace:",
                e.StackTrace));
        }
    }
}
