﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ToolFoundation.Collections;
using NintendoWare.Spy.Collections;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace NintendoWare.Spy.Collections.Tests
{
    [TestClass()]
    public class ReadOnlyObservableDictionaryTests
    {
        [TestMethod()]
        public void ReadOnlyObservableDictionaryTest()
        {
            var dictionary = new ObservableDictionary<string, int>()
            {
                { "a", 1 },
                { "b", 2 },
                { "c", 3 },
            };

            var readOnlyDictionary = new ReadOnlyObservableDictionary<string, int>(dictionary);
            int value;

            // 普通にアクセスできるか。
            Assert.AreEqual(3, readOnlyDictionary.Count);

            Assert.AreEqual(1, readOnlyDictionary["a"]);
            Assert.AreEqual(2, readOnlyDictionary["b"]);
            Assert.AreEqual(3, readOnlyDictionary["c"]);
            TestException<KeyNotFoundException>(() => readOnlyDictionary["d"]);

            Assert.IsTrue(readOnlyDictionary.TryGetValue("c", out value));
            Assert.AreEqual(3, value);
            Assert.IsFalse(readOnlyDictionary.TryGetValue("d", out value));

            Assert.IsTrue(readOnlyDictionary.ContainsKey("c"));
            Assert.IsFalse(readOnlyDictionary.ContainsKey("d"));

            // Keys に普通にアクセスできるか。
            Assert.IsTrue(readOnlyDictionary.Keys.Contains("c"));
            Assert.IsFalse(readOnlyDictionary.Keys.Contains("d"));

            // Values に普通にアクセスできるか。
            Assert.IsTrue(readOnlyDictionary.Values.Contains(1));
            Assert.IsFalse(readOnlyDictionary.Values.Contains(4));

            // ReadOnly になっているか。
            TestException<NotSupportedException>(() => (readOnlyDictionary as IDictionary<string, int>).Add("d", 4));
            TestException<NotSupportedException>(() => (readOnlyDictionary as ICollection<KeyValuePair<string, int>>).Add(new KeyValuePair<string, int>("d", 4)));
            TestException<NotSupportedException>(() => readOnlyDictionary.Keys.Add("d"));
            TestException<NotSupportedException>(() => readOnlyDictionary.Values.Add(4));

            // イベントを受け取れるか。
            bool collectionChangedEventReceived = false;
            readOnlyDictionary.CollectionChanged += (sender, args) =>
            {
                Assert.AreEqual(System.Collections.Specialized.NotifyCollectionChangedAction.Add, args.Action);
                Assert.AreEqual(readOnlyDictionary, sender);
                collectionChangedEventReceived = true;
            };

#if false // ObservableDictionary が INotifyPropertyChanged を実装していない。
            bool propertyChangedEventReceived = false;
            readOnlyDictionary.PropertyChanged += (sender, args) =>
            {
                Assert.AreEqual("Count", args.PropertyName);
                propertyChangedEventReceived = true;
            };
#endif

            dictionary.Add("d", 4);
            Assert.IsTrue(collectionChangedEventReceived);
#if false // ObservableDictionary が INotifyPropertyChanged を実装していない。
            Assert.IsTrue(propertyChangedEventReceived);
#endif

            // 変更の結果は反映されているか。
            Assert.AreEqual(4, readOnlyDictionary.Count);
            Assert.AreEqual(4, readOnlyDictionary["d"]);
            Assert.IsTrue(readOnlyDictionary.TryGetValue("d", out value));
            Assert.AreEqual(4, value);
            Assert.IsTrue(readOnlyDictionary.Keys.Contains("d"));
            Assert.IsTrue(readOnlyDictionary.Values.Contains(4));

            // Dispose() したらイベントを受け取らなくなるか。
            collectionChangedEventReceived = false;
#if false // ObservableDictionary が INotifyPropertyChanged を実装していない。
            propertyChangedEventReceived = false;
#endif
            readOnlyDictionary.Dispose();
            readOnlyDictionary.Dispose(); // Dispose() は何回呼んでも良い。
            Assert.IsTrue(readOnlyDictionary.IsDisposed);
            dictionary.Add("e", 5);

            Assert.IsFalse(collectionChangedEventReceived);
#if false // ObservableDictionary が INotifyPropertyChanged を実装していない。
            Assert.IsFalse(propertyChangedEventReceived);
#endif
        }

        private static void TestException<TException>(Action action)
            where TException : Exception
        {
            Assert.AreNotEqual(typeof(TException), typeof(Exception));
            Assert.AreNotEqual(typeof(TException), typeof(AssertFailedException));

            try
            {
                action();
                Assert.Fail($"Exception '{nameof(TException)}' is expected, but not thrown.");
            }
            catch (TException)
            {
                // ok
            }
        }

        private static void TestException<TException>(Func<object> func)
            where TException : Exception
        {
            Assert.AreNotEqual(typeof(TException), typeof(Exception));
            Assert.AreNotEqual(typeof(TException), typeof(AssertFailedException));

            try
            {
                func();
                Assert.Fail($"Exception '{nameof(TException)}' is expected, but not thrown.");
            }
            catch (TException)
            {
                // ok
            }
        }
    }
}
