﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.CodecTool;
using System.Linq;

namespace AudioCodec.Test
{
    [TestClass]
    public class OpusEncoderUnitTest
    {
        [TestMethod]
        public void InitializeWithValidSampleRateTest()
        {
            int[] ValidSampleRates = new int[] { 8000, 12000, 16000, 24000, 48000, };
            foreach (var sampleRate in ValidSampleRates)
            {
                try
                {
                    var encoder = new Nintendo.CodecTool.OpusEncoder(sampleRate, 1, 0x1);
                    Assert.AreEqual(sampleRate, encoder.SampleRate);
                }
                catch (System.Exception)
                {
                    Assert.Fail("Unexpected Behaviour (sampleRate = {0})", sampleRate);
                }
            }
        }

        [TestMethod]
        public void InitializeWithInvalidSampleRateTest()
        {
            int[] InvalidSampleRates = new int[] { -1, 0, 32000, 44100, 88200, 96000, 192000, };
            foreach (var sampleRate in InvalidSampleRates)
            {
                try
                {
                    new OpusEncoder(sampleRate, 2, 0);
                }
                catch (System.ArgumentException)
                {
                    continue;
                }

                Assert.Fail("Unexpected Behaviour (sampleRate = {0})", sampleRate);
            }
        }

        [TestMethod]
        public void InitializeWithValidChannelCountTest()
        {
            int[] ValidChannelCounts = new int[] { 1, 2, 255, };
            foreach (var channelCount in ValidChannelCounts)
            {
                try
                {
                    var encoder = new OpusEncoder(48000, channelCount, 0xffffffff >> (32 - channelCount));
                    Assert.AreEqual(channelCount, encoder.ChannelCount);
                }
                catch (System.Exception)
                {
                    Assert.Fail("Unexpected Behaviour (channelCount = {0})", channelCount);
                }
            }
        }

        [TestMethod]
        public void InitializeWithInvalidChannelCountTest()
        {
            int[] InvalidChannelCounts = new int[] { -1, 0, 256, };
            foreach (var channelCount in InvalidChannelCounts)
            {
                try
                {
                    new OpusEncoder(48000, channelCount, 0);
                }
                catch (System.ArgumentException)
                {
                    continue;
                }

                Assert.Fail("Unexpected Behaviour (channelCount = {0})", channelCount);
            }
        }

        [TestMethod]
        public void InitializeWithValidChannelMaskTest()
        {
            for (int channelCount = 1; channelCount <= 32; ++channelCount)
            {
                uint channelMask = 0xffffffff >> (32 - channelCount);
                try
                {
                    new OpusEncoder(48000, channelCount, channelMask);
                }
                catch (System.Exception)
                {
                    Assert.Fail("Unexpected Behaviour");
                }
            }
        }

        [TestMethod]
        public void BitRateTest()
        {
            foreach (var channelCount in Enumerable.Range(1, 32))
            {
                var encoder = new OpusEncoder(48000, channelCount, 0xffffffff >> (32 - channelCount));
                int[] BitRates = new int[] { 6000, 96000, 255000 * channelCount, };
                foreach (var bitRate in BitRates)
                {
                    encoder.BitRate = bitRate;
                    Assert.AreEqual(encoder.BitRate, bitRate);
                }
            }
        }

        [TestMethod]
        public void BitRateControlSuccessTest()
        {
            foreach (var channelCount in Enumerable.Range(1, 32))
            {
                var encoder = new OpusEncoder(48000, channelCount, 0xffffffff >> (32 - channelCount));
                var BitRateControls = new OpusEncoder.OpusBitRateControl[] { OpusEncoder.OpusBitRateControl.Constant, OpusEncoder.OpusBitRateControl.ConstrainedVariable, OpusEncoder.OpusBitRateControl.Variable, };
                foreach (var control in BitRateControls)
                {
                    encoder.BitRateControl = control;
                    Assert.AreEqual(encoder.BitRateControl, control);
                }
            }
        }

#if false
        // TODO: OpusEncoder.OpusBitRateControl.Invalid を設定した場合、現状、エラーにもならず、値も反映されない
        //       OpusEncoder.exe 経由で利用する場合においては実害はないが、後日検討が必要かもしれない
        [TestMethod]
        public void BitRateControlFailureTest()
        {
            foreach (var channelCount in Enumerable.Range(1, 32))
            {
                var encoder = new OpusEncoder(48000, channelCount, 0xffffffff >> (32 - channelCount));
                var BitRateControls = new OpusEncoder.OpusBitRateControl[] { OpusEncoder.OpusBitRateControl.Invalid, };
                foreach (var control in BitRateControls)
                {
                    encoder.BitRateControl = control;
                    Assert.AreEqual(encoder.BitRateControl, control);
                }
            }
        }
#endif

        [TestMethod]
        public void CodingModeSuccessTest()
        {
            var CodingModes = new OpusEncoder.OpusCodingMode[] { OpusEncoder.OpusCodingMode.Celt, OpusEncoder.OpusCodingMode.Silk, OpusEncoder.OpusCodingMode.Invalid, };
            foreach (var mode in CodingModes)
            {
                var encoder = new OpusEncoder(48000, 1, 0x1);
                encoder.CodingMode = mode;
                Assert.AreEqual(mode, encoder.CodingMode);
            }
        }

        [TestMethod]
        public void PreSkipSampleCountSuccessTest()
        {
            int[] SampleRates = new int[] { 8000, 12000, 16000, 24000, 48000, };
            foreach (var sampleRate in SampleRates)
            {
                var encoder = new OpusEncoder(sampleRate, 1, 0x1);
                Assert.IsTrue(encoder.PreSkipSampleCount > 0);
            }
        }

        [TestMethod]
        public void TotalStreamSuccessTest()
        {
            foreach (var channelCount in Enumerable.Range(1, 32))
            {
                var encoder = new OpusEncoder(48000, channelCount, 0xffffffff >> (32 - channelCount));
                Assert.IsTrue(encoder.TotalStreamCount > 0);
            }
        }

        [TestMethod]
        public void EncodeSuccessTest()
        {
            int[] SampleRates = new int[] { 8000, 12000, 16000, 24000, 48000, };
            foreach (var sampleRate in SampleRates)
            {
                foreach (var channelCount in Enumerable.Range(1, 32))
                {
                    var input = new short[960 * channelCount];

                    var CodingModes = new OpusEncoder.OpusCodingMode[] { OpusEncoder.OpusCodingMode.Celt, OpusEncoder.OpusCodingMode.Silk, };
                    foreach (var mode in CodingModes)
                    {
                        var BitRateControls = new OpusEncoder.OpusBitRateControl[] { OpusEncoder.OpusBitRateControl.Constant, OpusEncoder.OpusBitRateControl.ConstrainedVariable, OpusEncoder.OpusBitRateControl.Variable, };
                        foreach (var control in BitRateControls)
                        {
                            var BitRates = new int[] { 96000, };
                            foreach (var bitRate in BitRates)
                            {
                                var SampleCounts = new int[] { sampleRate / 50, sampleRate / 100, };
                                if (mode == OpusEncoder.OpusCodingMode.Celt)
                                {
                                    SampleCounts = SampleCounts.Concat(new int[] { sampleRate / 200, sampleRate / 400, }).ToArray();
                                }
                                foreach (var sampleCount in SampleCounts)
                                {
                                    var encoder = new OpusEncoder(sampleRate, channelCount, 0xffffffff >> (32 - channelCount));

                                    encoder.CodingMode = mode;
                                    encoder.BitRateControl = control;
                                    encoder.BitRate = bitRate;

                                    // TODO: ビットレートが小さい場合に OpusEncoder.UnexpectedCodingModeException や OpusEncoder.UnexpectedInternalErrorException が返ることがある模様
                                    //       ひとまずビットレートを大きくしての retry で事なきを得ることにするが、何とかしないといけない
                                    try
                                    {
                                        encoder.Encode(input, sampleCount);
                                    }
                                    catch (OpusEncoder.UnexpectedCodingModeException)
                                    {
                                        encoder.BitRate = bitRate * 5;
                                        encoder.Encode(input, sampleCount);
                                    }
                                    catch (OpusEncoder.UnexpectedInternalErrorException)
                                    {
                                        encoder.BitRate = bitRate * 5;
                                        encoder.Encode(input, sampleCount);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        [TestMethod]
        public void EncodeFailureTest()
        {
            int[] SampleRates = new int[] { 8000, 12000, 16000, 24000, 48000, };
            foreach (var sampleRate in SampleRates)
            {
                foreach (var channelCount in Enumerable.Range(1, 32))
                {
                    var encoder = new OpusEncoder(sampleRate, channelCount, 0xffffffff >> (32 - channelCount));
                    var input = new short[960 * channelCount];

                    var CodingModes = new OpusEncoder.OpusCodingMode[] { OpusEncoder.OpusCodingMode.Celt, OpusEncoder.OpusCodingMode.Silk, };
                    foreach (var mode in CodingModes)
                    {
                        var BitRateControls = new OpusEncoder.OpusBitRateControl[] { OpusEncoder.OpusBitRateControl.Constant, OpusEncoder.OpusBitRateControl.ConstrainedVariable, OpusEncoder.OpusBitRateControl.Variable, };
                        foreach (var control in BitRateControls)
                        {
                            var BitRates = new int[] { 96000, };
                            foreach (var bitRate in BitRates)
                            {
                                var SampleCounts = new int[] { -1, };
                                foreach (var sampleCount in SampleCounts)
                                {
                                    encoder.CodingMode = mode;
                                    encoder.BitRateControl = control;
                                    encoder.BitRate = bitRate;

                                    try
                                    {
                                        encoder.Encode(input, sampleCount);
                                    }
                                    catch (System.ArgumentException)
                                    {
                                        continue;
                                    }
                                    catch (System.Exception)
                                    {
                                        Assert.Fail("Unexpected Behaviour");
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}
