﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TestUtility;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using CommandUtility;
using System.Xml;
using System.Xml.Linq;

namespace CupTest
{
    [TestClass]
    public class CupTest
    {
        public TestContext TestContext { get; set; }

        [TestInitialize]
        public void Initialize()
        {
            TempHolder = new ScopedTemporaryFileHolder(TestContext, afterDelete: false);
        }

        [TestCleanup]
        public void Cleanup()
        {
            TempHolder.Dispose();
        }

        protected ScopedTemporaryFileHolder TempHolder;

        [TestMethod]
        public void TestBasic()
        {
            // CUPのデータを作る
            FileInfo cupNsp = MakeTestCup(0x0000000000000001, 1);

            // テストアプリを作る
            FileInfo testAppNsp = MakeTestApp();

            // カードを焼く
            WriteCardWithCup(testAppNsp, cupNsp);

            // Cupするテストを実行する
            RunCupTest();

            // Cup後の条件をテストする
            RunTestAfterCup();
        }

        private FileInfo MakeTestCup(UInt64 id, UInt64 version)
        {
            // テスト用のシステムデータを作る
            List<FileInfo> systemDataArchives = MakeTestSystemDataArchives();

            // テスト用の nfa を作る
            FileInfo cupNfa = MakeTestFirmwareArchive(id, version, systemDataArchives);

            // テスト用の cup の nsp を作る
            return MakeTestCupNsp(id, version, cupNfa);
        }

        private FileInfo MakeTestFirmwareArchive(UInt64 id, UInt64 version, List<FileInfo> systemDataArchives)
        {
            var nfa = TempHolder.CreateTemporaryFilePath(string.Format("SystemUpdate-{0:X16}-ver{1}-", id, version), extension: ".cup.nfa");

            var baseYaml = @"
Name: TestFirmwareArchive

Attributes:
- Name: SYSTEM
  Path:
{0}
";
            var yaml = string.Format(baseYaml,
                string.Join("\n", (from dataNsp in systemDataArchives select string.Format("  - {0}", dataNsp.FullName)))
                );

            var nfaSetting = TempHolder.CreateTemporaryFile(yaml, extension: ".yml");

            SdkTool.Execute(SdkPath.FindToolPath("MakeFirmwareArchive", "MakeFirmwareArchive/MakeFirmwareArchive.exe", "MakeFirmwareArchive/MakeFirmwareArchive.exe"),
                new string[] { "createnfa", "-o", nfa.FullName, "-i", nfaSetting.FullName });

            return nfa;
        }

        private FileInfo MakeTestCupNsp(UInt64 id, UInt64 version, FileInfo cupNfa)
        {
            FileInfo cupNsp = TempHolder.CreateTemporaryFilePath("cup", ".nsp");

            SdkTool.Execute("MakeCupArchive.exe",
                new string[] {
                    "-o", cupNsp.FullName,
                    "-i", cupNfa.FullName,
                    "--system-meta-id", string.Format("0x{0:X16}", id),
                    "--system-meta-version", version.ToString(),
                    "--keyconfig", Path.Combine(SdkPath.FindSdkRoot(), "Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml")
                });

            return cupNsp;
        }

        private List<FileInfo> MakeTestSystemDataArchives(int numSystemData = 1, UInt64 baseId = 0x000000001000000)
        {
            List<FileInfo> ret = new List<FileInfo>();

            for (UInt64 i = 0; i < (UInt64)numSystemData; i++)
            {
                var id = baseId + i;
                ret.Add(MakeTestSystemDataArchive(string.Format("SystemData-0x{0:X16}-ver{1}-", id, 0), i, 0L));
            }

            return ret;
        }

        private FileInfo MakeTestSystemDataArchive(string name, UInt64 id, Int64 version = 0)
        {
            DirectoryInfo contentDirectory = TempHolder.CreateTemporaryDirectory(name + "-dir-");

            TempHolder.CreateTemporaryFile(contentDirectory, "content.txt", name);

            FileInfo contentMeta = MakeTestSystemDataContentMeta(name, id, version);

            return MakeSystemDataNsp(name, id, version, contentMeta, contentDirectory);
        }

        private FileInfo MakeSystemDataNsp(string name, UInt64 id, Int64 version, FileInfo contentMeta, DirectoryInfo contentDirectory)
        {
            var nspFile = TempHolder.CreateTemporaryFilePath(name, extension: ".nsp");

            SdkTool.Execute(SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe"),
                new string[] {
                    "creatensp",
                    "-o", nspFile.FullName,
                    "--type", "SystemData",
                    "--meta", contentMeta.FullName,
                    "--data", contentDirectory.FullName,
                    "--keyconfig", Path.Combine(SdkPath.FindSdkRoot(), "Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml")
                }
            );

            if(!File.Exists(nspFile.FullName))
            {
                throw new Exception(string.Format("Not create nsp: {0}", nspFile.FullName));
            }

            return nspFile;
        }

        private FileInfo MakeTestSystemDataContentMeta(string name, UInt64 id, Int64 version = 0)
        {
            var file = TempHolder.CreateTemporaryFilePath(name + "content-meta", ".xml");

            XmlUtility.WriteXml(
                file,
                new XElement("Meta",
                    new XElement("Core",
                        new XElement("SystemDataId", string.Format("0x{0:X16}", id)),
                        new XElement("Version", string.Format("{0}", version))
                    )
                )
            );

            return file;
        }

        private FileInfo MakeTestApp()
        {
            var appDirectory = TempHolder.CreateTemporaryDirectory("testApp");

            var makeTestAppPath = SdkPath.FindToolPath("MakeTestApplication.exe", rootRelativePath: "Tests/Outputs/AnyCPU/Tools/MakeTestApplication/MakeTestApplication/Release/MakeTestApplication.exe");

            SdkTool.Execute(
                makeTestAppPath,
                new string[] { "-o", appDirectory.FullName },
                workingDirectory: makeTestAppPath.Directory.FullName);

            var appfiles = appDirectory.EnumerateFiles().ToList();

            if(appfiles.Find(fileInfo => fileInfo.Extension == ".nsp") == null)
            {
                throw new Exception("Failed to create nsp in " + appDirectory.FullName);
            }

            return appfiles.First(file => file.Extension == ".nsp");
        }

        private void WriteCardWithCup(FileInfo testAppNsp, FileInfo cupNsp)
        {
            var xciWriterPath = Path.Combine(SdkPath.FindSdkRoot(), "Tests/Outputs/NX-NXFP2-a64/Tests/testFs_Sample_XciWriter/Develop/testFs_Sample_XciWriter.nca");

            SdkTool.Execute("RunOnTarget.exe", xciWriterPath, "--", testAppNsp.FullName.Replace("\\","/"), cupNsp.FullName.Replace("\\", "/"));
        }

        private void RunCupTest()
        {
            throw new NotImplementedException();
        }

        private void RunTestAfterCup()
        {
            throw new NotImplementedException();
        }
    }
}
