﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <string>
#include <algorithm>
#include "testRepair_Utility.h"
#include <nn/repair/repair_IFile.h>
#include <memory>
#include <nn/fs.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_Authentication.h>
#include <nn/crypto.h>
#include <random>
#include <functional>
#include <nn/repair/repair_ProtectedFile.h>
#include <nn/manu/manu_Api.h>
#include <nn/spl/spl_Api.h>
#include <nn/crypto/crypto_RsaOaepSha256Decoder.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>

using namespace nnt::repair;
using namespace nn::repair;

namespace {
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

class IntegratedAuthenticationTest : public ::testing::Test
{
protected:
    virtual void SetUp()
    {
        if (GetTestEnv().IsManuEnabled())
        {
            m_FileSystem = GetTestEnv().GetManuFileSystem();
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::repair::CreateProtectedFileEncryptor(&m_Encryptor, "Spl"));
        }
        else
        {
            m_FileSystem = GetTestEnv().GetNnfsFileSystem();
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::repair::CreateProtectedFileEncryptor(&m_Encryptor, "FixedKey"));
        }

        filename = GetTestEnv().GetWorkingDirectory() + "/test.bin";
        nn::fs::DeleteFile(filename.c_str());
    }

    virtual void TearDown()
    {

    }

    std::shared_ptr<FileSystem> m_FileSystem;
    std::shared_ptr<nn::repair::IProtectedFileEncryptor> m_Encryptor;

    std::string filename;
};

TEST_F(IntegratedAuthenticationTest, WholeSequence)
{
    std::string wd = nnt::repair::GetTestEnv().GetWorkingDirectory();
    std::string protectedFilePath = wd + "/Archive.bin";
    std::string backupRequestPath = wd + "/BackupRequest.bin";
    std::string backupResponsePath = wd + "/BackupResponse.bin";
    std::string restoreRequestPath = wd + "/RestoreRequest.bin";
    std::string restoreResponsePath = wd + "/RestoreResponse.bin";

    nn::fs::DeleteFile(protectedFilePath.c_str());
    nn::fs::DeleteFile(backupRequestPath.c_str());
    nn::fs::DeleteFile(backupResponsePath.c_str());
    nn::fs::DeleteFile(restoreRequestPath.c_str());
    nn::fs::DeleteFile(restoreResponsePath.c_str());

    // Backup sequence
    {
        Key128 protectedFileKey = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_Encryptor->GenerateEncryptedKey(&protectedFileKey));

        AuthenticationArchiveContent archiveContent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            RequestBackup(&archiveContent, protectedFileKey, backupRequestPath, backupResponsePath, m_FileSystem));

        std::shared_ptr<ProtectedFile> file;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            CreateProtectedFile(&file, m_FileSystem, protectedFilePath.c_str(), m_Encryptor, protectedFileKey, archiveContent));
        NN_UTIL_SCOPE_EXIT{
            file->Close();
        };

        uint32_t data = 0x12345678;
        NNT_ASSERT_RESULT_SUCCESS(
            file->Write(0, &data, sizeof(data), true));

        NNT_ASSERT_RESULT_SUCCESS(
            file->Finish());
    }

    // Restore sequence
    {
        AuthenticationArchiveContent content;
        NNT_ASSERT_RESULT_SUCCESS(
            LoadEncryptedKey(&content, protectedFilePath, m_FileSystem));

        Key128 protectedFileKey = {};
        NNT_ASSERT_RESULT_SUCCESS(
            RequestRestore(&protectedFileKey, content, restoreRequestPath, restoreResponsePath, m_FileSystem));

        std::shared_ptr<ProtectedFile> file;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            OpenProtectedFile(&file, m_FileSystem, protectedFilePath.c_str(), m_Encryptor, protectedFileKey));
        NN_UTIL_SCOPE_EXIT{
            file->Close();
        };

        uint32_t data = 0;
        size_t readSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            file->Read(&readSize, 0, &data, sizeof(data)));
        EXPECT_EQ(4, readSize);
        EXPECT_EQ(0x12345678, data);
    }
}

#endif
