﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using CmacFileTool;
using System.Text.RegularExpressions;
using CommandUtility;

namespace CmacFileToolTest
{
    [TestClass]
    public class CmacFileToolTest
    {
        private static Func<string, byte[]> bytes = b => BinaryUtility.FromHexString(b);

        [TestMethod]
        public void TestLoadKey()
        {
            CollectionAssert.AreEqual(
                new byte[] { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
                CmacFileTool.Commands.AppendCmacCommand.LoadKeyFromArgument("0123456789ABCDEF"));
        }

        [TestMethod]
        public void TestCryptUtility_GenerateKey()
        {
            var key = CryptUtility.GenerateKey();
            Assert.AreEqual(16, key.Length);

            CollectionAssert.AreEqual(
                bytes("FFFFFFFFFFFFFFFF"),
                CryptUtility.EncryptedKey(
                    bytes("A5A5A5AA5A5A5A5A"),
                    bytes("5A5A5A55A5A5A5A5")));
        }

        [TestMethod]
        public void TestCryptUtility()
        {
            // NIST SP800-38B のデータを利用

            var testKey = bytes("2b7e151628aed2a6abf7158809cf4f3c");

            var k1 = CmacUtility.CalculateK1(testKey);
            CollectionAssert.AreEqual(
                bytes("fbeed618357133667c85e08f7236a8de"),
                k1);

            var k2 = CmacUtility.CalculateK2(testKey);
            CollectionAssert.AreEqual(
                bytes("f7ddac306ae266ccf90bc11ee46d513b"),
                k2);

            var emptyDataCmac = CmacUtility.CalculateCmac(bytes(""), testKey);
            CollectionAssert.AreEqual(
                bytes("bb1d6929e95937287fa37d129b756746"),
                emptyDataCmac);

            CmacUtility.Verify(bytes(""), bytes("bb1d6929e95937287fa37d129b756746"), testKey);

            var example2 = CmacUtility.CalculateCmac(bytes("6bc1bee22e409f96e93d7e117393172a"), testKey);
            CollectionAssert.AreEqual(
                bytes("070a16b46b4d4144f79bdd9dd04a287c"),
                example2);

            CmacUtility.Verify(
                bytes("6bc1bee22e409f96e93d7e117393172a"),
                bytes("070a16b46b4d4144f79bdd9dd04a287c"),
                testKey);

            var example3 = CmacUtility.CalculateCmac(bytes("6bc1bee22e409f96e93d7e117393172aae2d8a571e03ac9c9eb76fac45af8e5130c81c46a35ce411"), testKey);
            CollectionAssert.AreEqual(
                bytes("dfa66747de9ae63030ca32611497c827"),
                example3);

            CmacUtility.Verify(
                bytes("6bc1bee22e409f96e93d7e117393172aae2d8a571e03ac9c9eb76fac45af8e5130c81c46a35ce411"),
                bytes("dfa66747de9ae63030ca32611497c827"),
                testKey);

            var example4 = CmacUtility.CalculateCmac(bytes("6bc1bee22e409f96e93d7e117393172aae2d8a571e03ac9c9eb76fac45af8e5130c81c46a35ce411e5fbc1191a0a52eff69f2445df4f9b17ad2b417be66c3710"), testKey);
            CollectionAssert.AreEqual(
                bytes("51f0bebf7e3b9d92fc49741779363cfe"),
                example4);

        }
    }
}
