﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/http.h>
#include <nn/http/http_JsonResponse.h>
#include <nn/ssl.h>
#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/crypto.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

using namespace nn;

namespace {
    static int64_t s_HttpBuffer[2 * 1024 * 1024 / sizeof(int64_t)];
    NN_ALIGNAS(4096) static uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];


    class GoManager
    {
    public:
        static const int MaxThreads = 32;
        static const size_t StackSize = 64 * 1024;
        typedef std::function<void()> GoFunction;

        class ThreadContainer
        {
            ThreadContainer()
            {
            }

        public:
            os::ThreadType thread;
            void* stack;
            GoFunction entry;
            GoManager* manager;

            static ThreadContainer* Allocate()
            {
                ThreadContainer* p = new ThreadContainer;
                if (!p)
                {
                    return nullptr;
                }

#ifdef NN_BUILD_CONFIG_OS_WIN
                p->stack = _aligned_malloc(StackSize, os::ThreadStackAlignment);
#else
                p->stack = memalign(os::ThreadStackAlignment, StackSize);
#endif
                if (!p->stack)
                {
                    free(p);
                    return nullptr;
                }
                return p;
            }

            static void Free(ThreadContainer* p)
            {
                if (p->stack)
                {
#ifdef NN_BUILD_CONFIG_OS_WIN
                    _aligned_free(p->stack);
#else
                    free(p->stack);
#endif
                }
                delete(p);
            }
        };


        GoManager()
            : m_MessageQueue(m_MessageQueueArray, MaxThreads)
        {
            m_pScavengerThread = ThreadContainer::Allocate();
            NN_ABORT_UNLESS_NOT_NULL(m_pScavengerThread);

            Result result = os::CreateThread(&m_pScavengerThread->thread, ScavengerEntry, this, m_pScavengerThread->stack, StackSize, os::DefaultThreadPriority);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            os::SetThreadNamePointer(&m_pScavengerThread->thread, "GoScavenger");
            os::StartThread(&m_pScavengerThread->thread);
        }

        ~GoManager()
        {
            Wait();
            os::DestroyThread(&m_pScavengerThread->thread);
            ThreadContainer::Free(m_pScavengerThread);
        }

        void Wait()
        {
            m_MessageQueue.Send(0);
            os::WaitThread(&m_pScavengerThread->thread);
        }

        Result GoWithResult(GoFunction func)
        {
            ThreadContainer* pContainer = ThreadContainer::Allocate();
            NN_ABORT_UNLESS_NOT_NULL(pContainer);

            Result result = CreateThread(pContainer, func);
            if (result.IsFailure())
            {
                ThreadContainer::Free(pContainer);
                return result;
            }

            m_MessageQueue.Send(reinterpret_cast<uintptr_t>(pContainer));

            os::SetThreadNamePointer(&pContainer->thread, "GoFunction");
            os::StartThread(&pContainer->thread);

            return ResultSuccess();
        }

        void Go(GoFunction func)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(GoWithResult(func));
        }

    protected:
        Result CreateThread(ThreadContainer* pContainer, GoFunction func)
        {
            pContainer->manager = this;
            pContainer->entry = func;
            return os::CreateThread(&pContainer->thread, GoEntry, pContainer, pContainer->stack, StackSize, os::DefaultThreadPriority);
        }

        static void FreeActivatedThread(ThreadContainer* pContainer)
        {
            os::WaitThread(&pContainer->thread);
            os::DestroyThread(&pContainer->thread);
            ThreadContainer::Free(pContainer);
        }

        static void GoEntry(void* p)
        {
            ThreadContainer* pContainer = static_cast<ThreadContainer*>(p);
            //GoManager* pThis = static_cast<GoManager*>(pContainer->manager);
            pContainer->entry();
        }

        static void ScavengerEntry(void* p)
        {
            GoManager* pThis = static_cast<GoManager*>(p);
            while (true)
            {
                uintptr_t out;
                pThis->m_MessageQueue.Receive(&out);
                if (out == 0)
                {
                    break;
                }

                ThreadContainer* pContainer = reinterpret_cast<ThreadContainer*>(out);
                FreeActivatedThread(pContainer);
            }
        }

    private:
        os::MessageQueue m_MessageQueue;
        uintptr_t m_MessageQueueArray[MaxThreads];
        ThreadContainer* m_pScavengerThread;

    };
}

class HttpBasicTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
        Result result;

        /* initialize the socket library, providing application use case specific amount of memory */
        result = nn::socket::Initialize(
                                s_SocketMemoryPoolBuffer,
                                sizeof(s_SocketMemoryPoolBuffer),
                                nn::socket::MinSocketAllocatorSize,
                                2);
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = nifm::Initialize();
        NNT_ASSERT_RESULT_SUCCESS(result);

        nifm::SubmitNetworkRequestAndWait();

        result = ssl::Initialize();
        NNT_ASSERT_RESULT_SUCCESS(result);

        http::Initialize(s_HttpBuffer, sizeof(s_HttpBuffer));
        NNT_ASSERT_RESULT_SUCCESS(result);

        std::srand(12345);
    }

    static void TearDownTestCase()
    {
        http::Finalize();
        ssl::Finalize();
        socket::Finalize();
    }
};

#if 1
TEST_F(HttpBasicTest, HttpExampleCom)
{
    Result result;
    http::ConnectionBroker conn;
    conn.Initialize();
//    conn.SetVerbose(true);
#ifdef NN_BUILD_CONFIG_OS_WIN
    conn.SetProxy("proxy.nintendo.co.jp", 8080);
#endif
    {
        http::Request req(conn);
        req.SetUrlPointer("http://www.example.com");
        req.AddHeaderFormat("User-Agent: %s", "curl");
        req.SetCookieFormat("tool=%s", "curl");
        req.SetVerbose(true);

        nn::http::Response res;
        result = req.GetResponse(&res);
        NNT_ASSERT_RESULT_SUCCESS(result);

        char buf[123];
        char html[2048] = { 0 };

        size_t readlen, htmllen = 0;
        do
        {
            result = res.ReadBody(&readlen, buf, sizeof(buf));
            if (result.IsFailure())
            {
                break;
            }
            memcpy(&html[htmllen], buf, readlen);
            htmllen += readlen;
        } while (readlen == sizeof(buf));
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = res.ReadBody(&readlen, buf, sizeof(buf));
        NNT_ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(readlen, 0);

        ASSERT_NE(strstr(html, "This domain is established to be used for illustrative examples in documents."), nullptr);
    }
}
#endif

TEST_F(HttpBasicTest, HttpThroughputAndProgress)
{
    Result result;
    http::ConnectionBroker conn;
    conn.Initialize();
    conn.SetSocketBufferSize(16 * 1024, 128 * 1024);
//    conn.SetVerbose(true);
#ifdef NN_BUILD_CONFIG_OS_WIN
    conn.SetProxy("proxy.nintendo.co.jp", 8080);
#endif
    {
        http::Request req(conn);
        req.SetUrlPointer("http://jk-test1.jaist.ac.jp/speedtest/random4000x4000.jpg");

        nn::http::Response res;
        nn::http::Progress progress;
        res.SetProgressTarget(&progress);
        result = req.GetResponse(&res);
        NNT_ASSERT_RESULT_SUCCESS(result);

        static char buf[234567];

        size_t readlen;
        uint64_t bytesTransferred = 0;
        do
        {
            result = res.ReadBody(&readlen, buf, sizeof(buf));
            if (result.IsFailure())
            {
                break;
            }

            if (bytesTransferred != progress.GetTransferredBytes())
            {
                bytesTransferred = progress.GetTransferredBytes();
                progress.PrintProgressBar();
            }

        } while (readlen == sizeof(buf));
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = res.ReadBody(&readlen, buf, sizeof(buf));
        NNT_ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(readlen, 0);
    }
}

#define GO_BEGIN(gman) (gman).Go([&]() -> void
#define GO_END         )

TEST_F(HttpBasicTest, HttpConcurrent)
{
    GoManager gm;
    http::ConnectionBroker conn;
    conn.Initialize();
    //conn.SetVerbose(true);
    conn.SetMaxTotalConnections(4);

#ifdef NN_BUILD_CONFIG_OS_WIN
    conn.SetProxy("proxy.nintendo.co.jp", 8080);
    conn.SetNoProxyHostsPointer("devsedci01.ncl.nintendo.co.jp");
#endif

    const int Concurrency = 10;
    int arrayWait[Concurrency];
    for (int i = 0; i < Concurrency; ++i)
    {
        arrayWait[i] = std::rand() % 500 + 1;
    }
    for (int i = 0; i < Concurrency; ++i)
    {
        gm.Go([&, i]() -> void
        {
            Result result;
            http::Request req(conn);

#if 0 //def NN_BUILD_CONFIG_OS_WIN
            const uint8_t filehash[crypto::Sha1Generator::HashSize] = {
                0xd1, 0xed, 0x93, 0xee, 0xae, 0xcb, 0x09, 0x65, 0xe2, 0xa1,
                0xad, 0xe7, 0x84, 0x32, 0x3c, 0xa6, 0xfa, 0x15, 0x07, 0xa3
            };
            const uint64_t filelen = 72156;
            req.SetUrlPointer("http://devsedci01.ncl.nintendo.co.jp/test/random.txt");
#elsif 0
            const uint8_t filehash[crypto::Sha1Generator::HashSize] = {
                0x2a, 0x5f, 0x17, 0xa6, 0xf7, 0x77, 0x02, 0xd4, 0xc6, 0xee,
                0x4e, 0xcc, 0x4a, 0x90, 0x8b, 0xd4, 0x8d, 0xb3, 0xf0, 0x60
            };
            const uint64_t filelen = 8826236;
            req.SetUrlPointer("https://www.nintendo.co.jp/wii/support/pdf/additional_function_ver3.0j.pdf");
#else
            const uint8_t filehash[crypto::Sha1Generator::HashSize] = {
                0x59, 0xda, 0xed, 0xdf, 0xea, 0x29, 0xe6, 0x02, 0xac, 0x37,
                0x60, 0x9c, 0xb0, 0xf9, 0xb5, 0x5b, 0x9e, 0xbc, 0xaa, 0x03
            };
            const uint64_t filelen = 245388;
            req.SetUrlPointer("http://jk-test1.jaist.ac.jp/speedtest/random350x350.jpg");

#endif
            os::SleepThread(TimeSpan::FromMilliSeconds(arrayWait[i]));

            nn::http::Response res;
            result = req.GetResponse(&res);
            NNT_ASSERT_RESULT_SUCCESS(result);

            ASSERT_EQ(res.GetStatusCode(), 200);
            ASSERT_EQ(res.GetContentLength(), filelen);

            crypto::Sha1Generator sha;
            sha.Initialize();

            char buf[1234];
            size_t readlen, totallen = 0;
            do
            {
                result = res.ReadBody(&readlen, buf, sizeof(buf));
                if (result.IsFailure())
                {
                    break;
                }
                sha.Update(buf, readlen);
                totallen += readlen;
            } while (readlen == sizeof(buf));
            NNT_ASSERT_RESULT_SUCCESS(result);

            ASSERT_EQ(totallen, filelen);

            uint8_t hash[crypto::Sha1Generator::HashSize];
            sha.GetHash(hash, sizeof(hash));
            ASSERT_EQ(memcmp(filehash, hash, crypto::Sha1Generator::HashSize), 0);
        });
    }
    gm.Wait();
    NN_LOG("leave\n");
}

#if 0
TEST_F(HttpBasicTest, HttpsJsonParse)
{
    Result result;
    http::ConnectionBroker conn;
    conn.Initialize();
    conn.SetSkipSslVerificationForDebug(true);
    conn.SetVerbose(true);
#ifdef NN_BUILD_CONFIG_OS_WIN
    uint64_t expire;
    char deviceToken[2048] ={};
    {
        http::Request req(conn);
        req.SetUrlPointer("https://devsedci01.ncl.nintendo.co.jp/test/test.json");

        char buf[5];
        nn::http::JsonResponse res(buf, sizeof(buf));
        result = req.GetResponse(&res);
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = res.ReadAndParse([&expire, &deviceToken](msgpack::JsonStreamParser& parser, msgpack::JsonStreamParser::Event ev) -> bool {
            if (ev == msgpack::JsonStreamParser::EVENT_KEY_NAME)
            {
                const char* pKeyName = parser.GetToken().buf;
                if (std::strcmp(pKeyName, "expires_in") == 0)
                {
                    ev = parser.Next();
                    if (!msgpack::IsSuccess(msgpack::JsonStreamParser::ToUint64(parser.GetToken(), &expire)))
                    {
                        return false;
                    }
                    return true;
                }
                if (std::strcmp(pKeyName, "device_auth_token") == 0)
                {
                    ev = parser.Next();
                    const msgpack::JsonStreamParser::Token& token = parser.GetToken();
                    if (ev != msgpack::JsonStreamParser::EVENT_STRING || token.token_toobig)
                    {
                        return false;
                    }
                    util::Strlcpy(deviceToken, token.buf, sizeof(deviceToken));
                    return true;
                }

            }
            return true;
        });
        NNT_ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(expire, 86400);
        ASSERT_EQ(strcmp(deviceToken,
            "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiIsImprdSI6Imh0dHBzOi8vZGNlcnQtZGQxLm5kYXMuc3J2"
            "Lm5pbnRlbmRvLm5ldC9rZXlzIiwia2lkIjoiOTQ3NjllNjYtY2ViNS00MjFiLWE4ZGYtODI0MGM4OTkx"
            "MWY1In0.eyJzdWIiOiI2MTAwMDAwMDAwMDAwMDQwIiwiaXNzIjoiZGF1dGgtZGQxLm5kYXMuc3J2Lm5p"
            "bnRlbmRvLm5ldCIsImF1ZCI6IjgzYjcyYjA1ZGMzMjc4ZDciLCJleHAiOjE0NjY4NjQyOTMsImlhdCI6"
            "MTQ2Njc3Nzg5MywianRpIjoiNjczMjI2ZjYtNTg2NS00NmZjLWJkNjAtYjc4NjQxZTEzZjM0Iiwibmlu"
            "dGVuZG8iOnsic24iOiJYQVYwMDAwMDAwMDY0MiIsInBjIjoiSEFDLVMtWlpaWlYoVVNaKSIsImR0Ijoi"
            "TlggVkRldiAxIn19.c4B2NAEt_OUlB4LFKTC1xBIcYNSvaEWWMTaWsApQSU31VzHfUkqlSiAbc3YLrXS"
            "ECMjROtTSFGpyAAYffkhznhqcbzHiAANT41txS-JA-rFuJRGoK76qZ9GvgGZCogjVAaAy4Iw36DP5GCv"
            "MfofMiqharAsCHY_u4WGqkb_MVguuc8UKuHVywt00OqJrcwCNmXVDwipLbQ0-RjUlHB9T-U4PADHcpfL"
            "wqxIKpEF5MenzbFopg3fL6whHaUhYmoZhF74hSTXw57YQhRZTnIz-pxvllSoB4cF4z-6iWbGXyBnLMq2"
            "TB9cBfrLQAyt05LNb3n3CSWpOt5paSibsHFA"),
            0
        );
    }
    conn.Finalize();
#endif
}

TEST_F(HttpBasicTest, HttpsDeviceToken)
{
    Result result;
    http::ConnectionBroker conn;
    conn.Initialize();
    conn.SetEnableClientCert(true);
    conn.SetSkipSslVerificationForDebug(true);
    conn.SetVerbose(true);
#ifdef NN_BUILD_CONFIG_OS_WIN
    conn.SetProxy("proxy.nintendo.co.jp", 8080);
#endif

    uint64_t expire;
    char deviceToken[2048] ={};
    {
        http::Request req(conn);
        req.SetUrlPointer("https://dauth-td1.ndas.srv.nintendo.net/v1/device_auth_token");

        req.SetMethodType(http::MethodType_Post);

        result = req.SetPostDataByFormatString("client_id=%016llx&system_version=00000001", 0x83b72b05dc3278d7ULL);
        NNT_ASSERT_RESULT_SUCCESS(result);

        char buf[5];
        nn::http::JsonResponse res(buf, sizeof(buf));
        req.GetResponse(&res);
        NNT_ASSERT_RESULT_SUCCESS(result);

        const http::JsonResponse::ParseRule parseRules[] = {
            { "expires_in",        nullptr, http::JsonResponse::ValueType_Int64,  &expire,     sizeof(expire) },
            { "device_auth_token", nullptr, http::JsonResponse::ValueType_String, deviceToken, sizeof(deviceToken) },
            {}
        };

        result = res.ReadAndParseWithRule(parseRules);
        NNT_ASSERT_RESULT_SUCCESS(result);

        ASSERT_GE(expire, 0);
        ASSERT_NE(strcmp(deviceToken, ""), 0);
    }
    conn.Finalize();

    http::Finalize();
    ssl::Finalize();
    socket::Finalize();
}
#endif
