﻿using Nintendo.Nact.Utility;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Xunit;

namespace PropertySheetsTest
{
    public class PropertySheetsTest
    {
        protected static SdkRootInfo SdkRootInfo { get; private set; }
        protected static BuildSystemTestVcProjectRepository ProjectRepository { get; private set; }

        static PropertySheetsTest()
        {
            SdkRootInfo = SdkRootInfo.SearchSdkRoot();
            ProjectRepository = new BuildSystemTestVcProjectRepository(SdkRootInfo);

            var vsInstance = VisualStudioCollection.Instance.GetMsBuild(VisualStudioCollection.VisualStudio15ChannelId);
            Environment.SetEnvironmentVariable("MSBUILD_EXE_PATH", null);
            Environment.SetEnvironmentVariable("VisualStudioVersion", @"15.0");
            Environment.SetEnvironmentVariable("VSINSTALLDIR", vsInstance?.InstallationPath);
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void AliceLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Alice", ProjectType.Library, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void ChrisLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Chris", ProjectType.Library, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void ErisLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Eris", ProjectType.Library, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void IrisLibraryTest(VcxprojConfiguration config)
        {
            AssertForProject("Iris", ProjectType.Library, config);
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void NintendoWareLibraryTest(VcxprojConfiguration config)
        {
            AssertForProject("NintendoWare", ProjectType.Library, config);
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void TestsLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Tests", ProjectType.Library, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void SamplesLibraryTest(VcxprojConfiguration config)
        {
            AssertForProject("Samples", ProjectType.Library, config);
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void UsersLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.User)
            {
                AssertForProject("Users", ProjectType.Library, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void ChrisDynamicLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Chris", ProjectType.DynamicLibrary, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void ErisDynamicLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Eris", ProjectType.DynamicLibrary, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void IrisDynamicLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Iris", ProjectType.DynamicLibrary, config);
            }
        }


        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void NintendoWareDynamicLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("NintendoWare", ProjectType.DynamicLibrary, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void TestsDynamicLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Tests", ProjectType.DynamicLibrary, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void SamplesDynamicLibraryTest(VcxprojConfiguration config)
        {
            AssertForProject("Samples", ProjectType.DynamicLibrary, config);
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void UsersDynamicLibraryTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.User)
            {
                AssertForProject("Users", ProjectType.DynamicLibrary, config);
            }
        }

        // Alice, Chris, Eris ではアプリケーションプログラムをビルドすることはできない

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void IrisProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Iris", ProjectType.Program, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void NintendoWareProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("NintendoWare", ProjectType.Program, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void TestsProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Tests", ProjectType.Program, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void SamplesProgramTest(VcxprojConfiguration config)
        {
            AssertForProject("Samples", ProjectType.Program, config);
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void UsersProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.User)
            {
                AssertForProject("Users", ProjectType.Program, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void SamplesProgramWithoutSdkNsoTest(VcxprojConfiguration config)
        {
            AssertForProject("Samples", ProjectType.ProgramWithoutSdkNso, config);
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void UsersProgramWithoutSdkNsoTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.User)
            {
                AssertForProject("Users", ProjectType.ProgramWithoutSdkNso, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void ChrisSystemProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Chris", ProjectType.SystemProgram, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void ErisSystemProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Eris", ProjectType.SystemProgram, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void IrisSystemProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Iris", ProjectType.SystemProgram, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void TestsSystemProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.Development)
            {
                AssertForProject("Tests", ProjectType.SystemProgram, config);
            }
        }

        [Theory]
        [MemberData(nameof(VcxprojConfigurationGenerator.GenerateNXPlatformConfigurations), MemberType = typeof(VcxprojConfigurationGenerator))]
        public void UsersSystemProgramTest(VcxprojConfiguration config)
        {
            if (SdkRootInfo.Environment == SdkEnvironment.User)
            {
                AssertForProject("Users", ProjectType.SystemProgram, config);
            }
        }

        private void AssertForProject(string subRootName, ProjectType projectType, VcxprojConfiguration config)
        {
            var conf = new TestConfiguration(subRootName, projectType, SdkRootInfo.Environment, config);
            var vcProject = GetVcProject(conf);
            var ep = vcProject[conf.ConfigurationPair];
            var expander = CreateExpander(conf, ep);

            Assert.True(ArePlatformPropertySheetsImported(ep, conf),
                string.Format("プラットフォーム {0} のプロパティシートをインポートできませんでした。", conf.Platform));

            InvokeAssertions(conf, ep, expander);
        }

        protected virtual VcProject GetVcProject(TestConfiguration conf)
        {
            return ProjectRepository.GetVcProject(conf);
        }

        protected virtual VcProject GetDefaultVcProject(TestConfiguration conf)
        {
            return ProjectRepository.GetDefaultVcProject(conf);
        }

        protected virtual void InvokeAssertions(TestConfiguration conf, EvaluatedVcProject ep, Expander expander)
        {
        }

        /// <summary>
        /// プラットフォーム固有のプロパティシートが（インストールされていて）インポートされているかどうかを返します。
        /// </summary>
        protected static bool ArePlatformPropertySheetsImported(EvaluatedVcProject ep, TestConfiguration conf)
        {
            string platformDirectory = string.Format(@"\Platforms\{0}\", conf.Platform);
            return ep.ImportFullPaths.Any(x => x.Contains(platformDirectory));
        }

        /// <summary>
        /// MsBuild の文字列のリスト xs と ys について、xs に ys の要素が1つ以上現れることをテストします。
        /// ys を Expand します。
        /// </summary>
        protected static void AssertMsBuildValueContainsAnyElement(Expander expander, string xs, string ys)
        {
            AssertUtil.AssertSequenceContainsAnyElement(
                Util.SplitMsBuildValue(xs),
                Util.SplitMsBuildValue(expander.Expand(ys)));
        }

        /// <summary>
        /// MsBuild の文字列のリスト xs と ys について、xs に ys の要素が現れることをテストします。
        /// ys を Expand します。
        /// </summary>
        protected static void AssertMsBuildValueContainsElements(Expander expander, string xs, string ys)
        {
            AssertUtil.AssertSequenceContainsElements(
                Util.SplitMsBuildValue(xs),
                Util.SplitMsBuildValue(expander.Expand(ys)));
        }

        /// <summary>
        /// MsBuild のパスのリスト xs と ys について、xs に ys の要素が順番に現れることをテストします。
        /// ys を Expand します。
        /// </summary>
        protected static void AssertMsBuildValueContainsPathsInOrder(Expander expander, string xs, string ys)
        {
            AssertUtil.AssertSequenceContainsElementsInOrder(
                Util.SplitMsBuildValue(xs),
                Util.SplitMsBuildValue(expander.Expand(ys)),
                PathComparer.Default);
        }

        /// <summary>
        /// MsBuild のパスのリスト xs と ys について、xs と ys が一致することをテストします。
        /// ys を Expand します。
        /// </summary>
        protected static void AssertMsBuildValueEqualsPaths(Expander expander, string xs, string ys)
        {
            AssertUtil.AssertSequenceEqual(
                Util.SplitMsBuildValue(xs),
                Util.SplitMsBuildValue(expander.Expand(ys)),
                PathComparer.Default);
        }

        /// <summary>
        /// パス expected と actual が等しいことを確認します。
        /// expected を Expand します。
        /// </summary>
        protected static void AssertArePathsEqual(Expander expander, string expected, string actual)
        {
            string expandedExpected = expander.Expand(expected);
            if (!PathComparer.Default.Equals(expandedExpected, actual))
            {
                Assert.True(false, string.Format("パスが等しくありません。\n expected: {0}\n actual:   {1}", expandedExpected, actual));
            }
        }

        /// <summary>
        /// 展開用の Expander を生成します。
        /// </summary>
        private Expander CreateExpander(TestConfiguration conf, EvaluatedVcProject ep)
        {
            var map = new Dictionary<string, string>()
            {
                { "SdkRoot", SdkRootInfo.SdkRoot },
                { "SubRootName", conf.SubRootName },
                { "SubRootRelativeDir", conf.SubRootRelativeDir },
                { "Spec", conf.Spec },
                { "BuildType", conf.BuildType },
                { "PlatformToolset", conf.PlatformToolset },
                { "Configuration", conf.Configuration },
                { "Platform", conf.Platform },
                { "BuildTarget", conf.BuildTarget },
                { "ProjectDir", Path.GetDirectoryName(ep.Path) }
            };

            // プロジェクトのプロパティ
            // * 参照したいものだけを追加する
            // * 追加したプロパティ自体のアサートを行ったうえで参照する
            var projectPropertyNames = new string[] {
                "NintendoSdkProgramCategory",
                "TargetName",
                "IntDir",
                "OutDir",
            };
            foreach (var x in projectPropertyNames)
            {
                map.Add(x, Util.GetValueOrDefault(ep.Properties, x, string.Empty));
            }

            return new Expander(map);
        }
    }
}
