﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PropertySheetsTest
{
    /// <summary>
    /// スペック、サブルート、プロジェクト種類（ライブラリ、プログラム）毎のテストプロジェクトが登録されていて、それを取得することができるクラス。
    /// </summary>
    public class BuildSystemTestVcProjectRepository
    {
        private class VcProjectKey : IEquatable<VcProjectKey>
        {
            public string Spec { get; private set; }
            public string SubRootName { get; private set; }
            public string TargetName { get; private set; }
            public ProjectType ProjectType { get; private set; }
            public string VsVersion { get; private set; }
            public IReadOnlyDictionary<string, string> GlobalProperties { get; private set; }

            public VcProjectKey(
                string spec,
                string subRootName,
                string targetName,
                ProjectType projectType,
                string vsVersion,
                IDictionary<string, string> globalProperties = null)
            {
                Spec = spec;
                SubRootName = subRootName;
                TargetName = targetName;
                ProjectType = projectType;
                VsVersion = vsVersion;
                GlobalProperties = (globalProperties != null) ? new Dictionary<string, string>(globalProperties) : null;
            }

            #region IEquatable
            public override bool Equals(object obj)
            {
                if (obj == null)
                {
                    return false;
                }
                else if (obj is VcProjectKey)
                {
                    return Equals((VcProjectKey)obj);
                }
                else
                {
                    return false;
                }
            }
            public bool Equals(VcProjectKey other)
            {
                if (other == null)
                {
                    return false;
                }

                if (Spec != other.Spec
                    || SubRootName != other.SubRootName
                    || TargetName != other.TargetName
                    || ProjectType != other.ProjectType
                    || VsVersion != other.VsVersion)
                {
                    return false;
                }

                if (GlobalProperties == null && other.GlobalProperties == null)
                {
                    return true;
                }
                else if (GlobalProperties != null && other.GlobalProperties != null)
                {
                    return GlobalProperties.Count == other.GlobalProperties.Count
                        && !GlobalProperties.Except(other.GlobalProperties).Any()
                        && !other.GlobalProperties.Except(GlobalProperties).Any();
                }
                else
                {
                    return false;
                }
            }
            public static bool operator ==(VcProjectKey lhs, VcProjectKey rhs)
            {
                if ((object)lhs == null || (object)rhs == null)
                {
                    return object.ReferenceEquals(lhs, rhs);
                }
                else
                {
                    return lhs.Equals(rhs);
                }
            }
            public static bool operator !=(VcProjectKey lhs, VcProjectKey rhs)
            {
                if ((object)lhs == null || (object)rhs == null)
                {
                    return !object.ReferenceEquals(lhs, rhs);
                }
                else
                {
                    return !lhs.Equals(rhs);
                }
            }
            public override int GetHashCode()
            {
                var specHash = Spec.GetHashCode();
                var subRootNameHash = SubRootName.GetHashCode();
                var targetNameHash = TargetName.GetHashCode();
                var projectTypeHash = ProjectType.GetHashCode();
                var vsVersionHash = VsVersion?.GetHashCode() ?? 0;
                var globalPropertiesHash = 0;
                if (GlobalProperties != null)
                {
                    globalPropertiesHash = GlobalProperties.Aggregate(0, (seed, kv) => seed ^ kv.Key.GetHashCode() ^ kv.Value.GetHashCode());
                }

                return specHash ^ subRootNameHash ^ targetNameHash ^ projectTypeHash ^ vsVersionHash ^ globalPropertiesHash;
            }
            #endregion
        }

        private class ProjectEntry
        {
            public object Lock { get; } = new object();
            public VcProject Project { get; set; }
        }

        private SdkRootInfo sdkRootInfo;
        private Dictionary<VcProjectKey, ProjectEntry> vcProjects = new Dictionary<VcProjectKey, ProjectEntry>();

        public BuildSystemTestVcProjectRepository(SdkRootInfo SdkRootInfo)
        {
            sdkRootInfo = SdkRootInfo;
        }

        private string GetLibraryRelativePath(string spec, string targetName)
        {
            return $@"Sources\Libraries\{targetName}\{targetName}-spec.{spec}.autogen.vcxproj";
        }

        private string GetProcessRelativePath(string spec, string targetName)
        {
            return $@"Sources\Processes\{targetName}\{targetName}-spec.{spec}.autogen.vcxproj";
        }

        private string GetTargetToolRelativePath(string spec, string targetName)
        {
            return $@"Sources\TargetTools\{targetName}\{targetName}-spec.{spec}.autogen.vcxproj";
        }

        private string GetApplicationRelativePath(string spec, string targetName)
        {
            return $@"Sources\Applications\{targetName}\{targetName}-spec.{spec}.autogen.vcxproj";
        }

        private string GetTestApplicationRelativePath(string spec, string targetName)
        {
            return $@"Sources\Tests\BuildSystem\{targetName}\{targetName}-spec.{spec}.autogen.vcxproj";
        }

        public VcProject GetVcProject(
            TestConfiguration conf,
            IDictionary<string, string> globalProperties = null)
        {
            var key = GetVcProjectKey(conf, globalProperties);
            return GetVcProjectImpl(key, GetVcProjectPath(conf), conf, globalProperties);
        }

        public VcProject GetDefaultVcProject(
            TestConfiguration conf,
            IDictionary<string, string> globalProperties = null)
        {
            var key = GetDefaultVcProjectKey(conf, globalProperties);
            return GetVcProjectImpl(key, GetDefaultVcProjectPath(conf), conf, globalProperties);
        }

        private VcProject GetVcProjectImpl(
            VcProjectKey key,
            string path,
            TestConfiguration conf,
            IDictionary<string, string> globalProperties)
        {
            var entry = GetOrAddEntry(key);
            // 同じ構成の VcProject が複数回生成されないようにエントリの読み出しも排他する
            lock (entry.Lock)
            {
                if (entry.Project == null)
                {
                    entry.Project = new VcProject(path, conf.VsVersion, globalProperties);
                }
            }
            return entry.Project;
        }

        private ProjectEntry GetOrAddEntry(VcProjectKey key)
        {
            ProjectEntry entry;
            lock (vcProjects)
            {
                if (vcProjects.TryGetValue(key, out entry))
                {
                    return entry;
                }
                else
                {
                    return vcProjects[key] = new ProjectEntry();
                }
            }
        }

        private VcProjectKey GetVcProjectKey(
            TestConfiguration conf,
            IDictionary<string, string> globalProperties = null)
        {
            return new VcProjectKey(
                conf.Spec, conf.SubRootName, conf.TargetName, conf.ProjectType, conf.VsVersion, globalProperties);
        }

        private VcProjectKey GetDefaultVcProjectKey(
            TestConfiguration conf,
            IDictionary<string, string> globalProperties = null)
        {
            // サブルート及び TargetName による分岐が存在しない、 PropertySheetsTest 専用のプロジェクト
            return new VcProjectKey(
                conf.Spec, "PropertySheetsTest", "PropertySheetsTest", conf.ProjectType, conf.VsVersion, globalProperties);
        }

        private string GetVcProjectPath(
            TestConfiguration conf)
        {
            switch (conf.SubRootName)
            {
                case "Alice":
                    return Path.Combine(sdkRootInfo.SdkRoot, "Programs", conf.SubRootName, GetLibraryRelativePath(conf.Spec, conf.TargetName));
                case "Chris":
                case "Eris":
                    switch (conf.ProjectType)
                    {
                        case ProjectType.Library:
                        case ProjectType.DynamicLibrary:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Programs", conf.SubRootName, GetLibraryRelativePath(conf.Spec, conf.TargetName));
                        case ProjectType.Program:
                        case ProjectType.ProgramWithoutSdkNso:
                            throw new NotImplementedException("テスト未実装です。");
                        case ProjectType.SystemProgram:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Programs", conf.SubRootName, GetProcessRelativePath(conf.Spec, conf.TargetName));
                        default:
                            throw new ArgumentException($"未知の ProjectType です: {conf.ProjectType}", nameof(conf.ProjectType));
                    }
                case "Iris":
                case "NintendoWare":
                    // プログラムをビルドするには Iris が必要
                    switch (conf.ProjectType)
                    {
                        case ProjectType.Library:
                            switch (sdkRootInfo.Environment)
                            {
                                case SdkEnvironment.Development:
                                    return Path.Combine(sdkRootInfo.SdkRoot, "Programs", conf.SubRootName, GetLibraryRelativePath(conf.Spec, conf.TargetName));
                                case SdkEnvironment.User:
                                    return Path.Combine(sdkRootInfo.SdkRoot, GetLibraryRelativePath(conf.Spec, conf.TargetName));
                                default:
                                    throw new NotImplementedException($"未知の SdkEnvironment です: {sdkRootInfo.Environment}");
                            }
                        case ProjectType.DynamicLibrary:
                            switch (sdkRootInfo.Environment)
                            {
                                case SdkEnvironment.Development:
                                    return Path.Combine(sdkRootInfo.SdkRoot, "Programs", conf.SubRootName, GetLibraryRelativePath(conf.Spec, conf.TargetName));
                                case SdkEnvironment.User:
                                    throw new NotImplementedException("SDK 利用者に手動ロード DLL プロジェクトを提供することは想定していません。");
                                default:
                                    throw new NotImplementedException($"未知の SdkEnvironment です: {sdkRootInfo.Environment}");
                            }
                        case ProjectType.Program:
                            switch (sdkRootInfo.Environment)
                            {
                                case SdkEnvironment.Development:
                                    return Path.Combine(sdkRootInfo.SdkRoot, "Programs", conf.SubRootName, GetTargetToolRelativePath(conf.Spec, conf.TargetName));
                                case SdkEnvironment.User:
                                    throw new NotImplementedException("SDK 利用者にターゲットツールを提供することは想定していません。");
                                default:
                                    throw new NotImplementedException($"未知の SdkEnvironment です: {sdkRootInfo.Environment}");
                            }
                        case ProjectType.ProgramWithoutSdkNso:
                            throw new NotImplementedException("テスト未実装です。");
                        case ProjectType.SystemProgram:
                            switch (sdkRootInfo.Environment)
                            {
                                case SdkEnvironment.Development:
                                    return Path.Combine(sdkRootInfo.SdkRoot, "Programs", conf.SubRootName, GetProcessRelativePath(conf.Spec, conf.TargetName));
                                case SdkEnvironment.User:
                                    throw new NotImplementedException("SDK 利用者にシステムプロセスを提供することは想定していません。");
                                default:
                                    throw new NotImplementedException($"未知の SdkEnvironment です: {sdkRootInfo.Environment}");
                            }
                        default:
                            throw new ArgumentException($"未知の ProjectType です: {conf.ProjectType}", nameof(conf.ProjectType));
                    }
                case "Samples":
                    switch (conf.ProjectType)
                    {
                        case ProjectType.Library:
                        case ProjectType.DynamicLibrary:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Samples", GetLibraryRelativePath(conf.Spec, conf.TargetName));
                        case ProjectType.Program:
                        case ProjectType.ProgramWithoutSdkNso:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Samples", GetApplicationRelativePath(conf.Spec, conf.TargetName));
                        case ProjectType.SystemProgram:
                            throw new NotImplementedException("テスト未実装です。");
                        default:
                            throw new ArgumentException($"不明な ProjectType です: {conf.ProjectType}", nameof(conf.ProjectType));
                    }
                case "Users":
                    switch (conf.ProjectType)
                    {
                        case ProjectType.Library:
                        case ProjectType.DynamicLibrary:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Users", $@"Sources\Libraries\{conf.TargetName}\{conf.TargetName}-spec.{conf.Spec}.vcxproj");
                        case ProjectType.Program:
                        case ProjectType.ProgramWithoutSdkNso:
                        case ProjectType.SystemProgram:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Users", $@"Sources\Applications\{conf.TargetName}\{conf.TargetName}-spec.{conf.Spec}.vcxproj");
                        default:
                            throw new ArgumentException($"不明な ProjectType です: {conf.ProjectType}", nameof(conf.ProjectType));
                    }
                case "Tests":
                    switch (conf.ProjectType)
                    {
                        case ProjectType.Library:
                        case ProjectType.DynamicLibrary:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Tests", GetLibraryRelativePath(conf.Spec, conf.TargetName));
                        case ProjectType.Program:
                        case ProjectType.SystemProgram:
                            return Path.Combine(sdkRootInfo.SdkRoot, "Tests", GetTestApplicationRelativePath(conf.Spec, conf.TargetName));
                        case ProjectType.ProgramWithoutSdkNso:
                            throw new NotImplementedException("テスト未実装です。");
                        default:
                            throw new ArgumentException($"不明な ProjectType です: {conf.ProjectType}", nameof(conf.ProjectType));
                    }
                default:
                    throw new ArgumentException($"不明な SubRootName です: {conf.ProjectType}", nameof(conf.ProjectType));
            }
        }

        private string GetDefaultVcProjectPath(TestConfiguration conf)
        {
            switch (conf.ProjectType)
            {
                case ProjectType.Library:
                    return Path.Combine(Environment.CurrentDirectory, $@"DefaultProjects\DefaultLibrary-spec.{conf.Spec}.vcxproj");
                case ProjectType.Program:
                    return Path.Combine(Environment.CurrentDirectory, $@"DefaultProjects\DefaultProgram-spec.{conf.Spec}.vcxproj");
                case ProjectType.DynamicLibrary:
                case ProjectType.ProgramWithoutSdkNso:
                case ProjectType.SystemProgram:
                    throw new NotImplementedException("テスト未実装です。");
                default:
                    throw new ArgumentException($"未知の ProjectType です: {conf.ProjectType}", nameof(conf.ProjectType));
            }

        }
    }
}
