﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <random>
#include <nnt.h>
#include <nne/audio/audio.h>
#include <nn/nn_Log.h>
#include <nn/i2c/i2c.h>
#include <nn/nn_TimeSpan.h>

#include "testAudioDriver_CodecTestData.h"

namespace  {
auto g_IsAdspLogEnabled = false;
const auto defaultIterationCount = 32;

class CodecRegisterChecker
{
    NN_DISALLOW_COPY( CodecRegisterChecker );
    NN_DISALLOW_MOVE( CodecRegisterChecker );
protected:
    nn::i2c::I2cSession m_I2c;
    bool m_IsOpend;

protected:
    void Write(const uint8_t reg, const uint16_t value) NN_NOEXCEPT
    {
        const nn::i2c::TransactionOption StartStopOption =
                static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition |
                                                        nn::i2c::TransactionOption_StopCondition );

        const nn::TimeSpan I2cAccessRetryInterval = nn::TimeSpan::FromMilliSeconds(5);
        const int I2cAccessRetryCountMax = 200 * 10; // 10 seconds

        uint8_t buffer[3];

        buffer[0] = reg;
        buffer[1] = value >> 8;
        buffer[2] = value & 0xff;

        nn::Result result;

        for (int i = 0; i < I2cAccessRetryCountMax; ++i)
        {
            result = nn::i2c::Send(m_I2c, &buffer,
                    3 * sizeof(uint8_t), StartStopOption);

            if (nn::i2c::ResultBusBusy::Includes(result))
            {
                nn::os::SleepThread(I2cAccessRetryInterval);
                continue;
            }

            else if (nn::i2c::ResultNoAck::Includes(result))
            {
                nn::os::SleepThread(I2cAccessRetryInterval);
                continue;
            }

            else
            {
                break;
            }
        }

        if (result.IsFailure())
        {
            NN_LOG("[audio] CodecRegisterChecker::Write failed. reg(%u) value(%u)\n", reg, value);
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    uint16_t ReadPublic(uint8_t reg) NN_NOEXCEPT
    {
        const nn::i2c::TransactionOption StartStopOption =
                static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition |
                                                        nn::i2c::TransactionOption_StopCondition );
        const nn::i2c::TransactionOption StartOption =
                static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition);

        const nn::TimeSpan I2cAccessRetryInterval = nn::TimeSpan::FromMilliSeconds(5);
        const int I2cAccessRetryCountMax = 200 * 10; // 10 seconds


        uint8_t buffer[2];
        uint8_t commandList[nn::i2c::CommandListLengthCountMax];
        nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
        commandListFormatter.EnqueueSendCommand(StartOption, &reg, sizeof(reg) );
        commandListFormatter.EnqueueReceiveCommand(StartStopOption, 2 * sizeof(uint8_t) );

        nn::Result result;

        for (int i = 0; i < I2cAccessRetryCountMax; ++i)
        {
            result = nn::i2c::ExecuteCommandList(buffer, 2 * sizeof(uint8_t),
                    m_I2c, commandList, commandListFormatter.GetCurrentLength());

            if (nn::i2c::ResultBusBusy::Includes(result))
            {
                nn::os::SleepThread(I2cAccessRetryInterval);
                continue;
            }

            else if (nn::i2c::ResultNoAck::Includes(result))
            {
                nn::os::SleepThread(I2cAccessRetryInterval);
                continue;
            }

            else
            {
                break;
            }
        }

        if (result.IsFailure())
        {
            NN_LOG("[audio] CodecRegisterChecker::Read failed. reg(%u)\n", reg);
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        return buffer[0] << 8 | buffer[1];
    }

    uint16_t ReadPriv(const uint8_t reg) NN_NOEXCEPT
    {
        // 0x6a: ALC5639_MX6A_PRIV_INDEX
        // 0x6c: ALC5639_MX6C_PRIV_DATA
        Write(0x6a, reg);
        return Read(0x6c);
    }

public:
    CodecRegisterChecker() NN_NOEXCEPT
        : m_IsOpend(false)
    {

    }

    ~CodecRegisterChecker() NN_NOEXCEPT
    {
        if(m_IsOpend)
        {
            Close();
        }
    }

    void Open() NN_NOEXCEPT
    {
        m_IsOpend = true;
        nn::i2c::Initialize();
        nn::i2c::OpenSession(&m_I2c, nn::i2c::I2cDevice_Alc5639);
    }

    uint16_t Read(uint8_t reg, bool isPrivate = false) NN_NOEXCEPT
    {
        return isPrivate ? ReadPriv(reg) : ReadPublic(reg);
    }

    void Close() NN_NOEXCEPT
    {
        nn::i2c::CloseSession(m_I2c);
        nn::i2c::Finalize();

        m_IsOpend = false;
    }

    void DumpAll(bool isTestPatternFormat = false) NN_NOEXCEPT
    {
        // Dump registers
        if(isTestPatternFormat)
        {
            for(const auto reg : RegistersListForAlc5639)
            {
                NN_LOG("{ { %s, 0x%02X }, 0x%04X },\n",
                        reg.isPrivate ? "true" : "false",
                        reg.address,
                        Read(reg.address, reg.isPrivate));
            }
        }
        else
        {
            for(const auto reg : RegistersListForAlc5639)
            {
                NN_LOG("[%s-%02X] 0x%04X\n",
                        reg.isPrivate ? "PR" : "MX",
                        reg.address,
                        Read(reg.address, reg.isPrivate));
            }
        }
    }

    void checkTestPattern(const TestPatternForAlc5639* patterns, int count) NN_NOEXCEPT
    {
        for(int i = 0; i < count; ++i)
        {
            const auto& pattern = patterns[i];
            const auto value = Read(pattern.reg.address, pattern.reg.isPrivate);
            EXPECT_EQ(value, pattern.expectValue) << (pattern.reg.isPrivate ? "PR-" : "MX-") << pattern.reg.address;
        }
    }
};

} // anonymous namespace

extern const char adsp_os_bin_begin[];
extern const char adsp_os_bin_end[];
extern const char vector_bin_begin[];
extern const char vector_bin_end[];

TEST(AdspInitializeAndFinalize, Success)
{
    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                    reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                    g_IsAdspLogEnabled);
        nne::audio::adsp::Finalize();
    }
}

TEST(GmixInitializeAndFinalize, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::gmix::Initialize();
        nne::audio::gmix::Finalize();
    }

    nne::audio::adsp::Finalize();
}

TEST(AhubInitializeAndFinalize, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);

    nne::audio::gmix::Initialize();
    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::ahub::Initialize();
        nne::audio::ahub::Finalize();
    }
    nne::audio::gmix::Finalize();

    nne::audio::adsp::Finalize();
}

TEST(HdaInitializeAndFinalize, Success)
{
    const auto sampleRateList = {nne::audio::hda::AUDIO_HDA_SAMPLERATE_32000,
                                 nne::audio::hda::AUDIO_HDA_SAMPLERATE_44100,
                                 nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
                                 nne::audio::hda::AUDIO_HDA_SAMPLERATE_96000,
                                 nne::audio::hda::AUDIO_HDA_SAMPLERATE_192000};

    const auto channelCountList = {nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2,
                                   nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_4,
                                   nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_6,
                                   nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_8};

    const auto bitsPerSampleList = {nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
                                    nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_20,
                                    nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_24};

    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        for(const auto& sampleRate : sampleRateList)
        {
            for(const auto& channelCount : channelCountList)
            {
                for(const auto& bitsPerSample : bitsPerSampleList)
                {
                    nne::audio::hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
                                                sampleRate,
                                                bitsPerSample,
                                                channelCount);

                    bool connected;
                    int maxChannels;
                    nne::audio::hda::GetHotplugState(&connected, &maxChannels);
                    // NN_LOG("connected(%d) maxChannels(%d)\n", connected, maxChannels);
                    nne::audio::hda::Finalize();
                }
            }
        }
    }
    nne::audio::adsp::Finalize();
}

TEST(GmixSessionOpenClose, Success)
{
    const auto defaultDeviceOutList = {nne::audio::gmix::Session::Name::Hda, nne::audio::gmix::Session::Name::Ahub};
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::gmix::Initialize();

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        for(const auto& defaultDeviceOut : defaultDeviceOutList)
        {
            nne::audio::gmix::SetDefaultDeviceOut(defaultDeviceOut);

            nne::audio::gmix::Session* pAhubGmixSession = nullptr;
            nne::audio::gmix::Session* pHdaGmixSession = nullptr;

            nne::audio::gmix::OpenSession(&pAhubGmixSession,
                                        nne::audio::gmix::Session::Name::Ahub, // session name
                                        0, // pin
                                        nne::audio::gmix::Session::Name::Default, // destination
                                        nne::audio::gmix::Session::Format::Stereo, // format stereo, sround
                                        nne::audio::gmix::Session::Mode::Default); // mode mute, solo
            nne::audio::gmix::OpenSession(&pHdaGmixSession,
                                        nne::audio::gmix::Session::Name::Hda,
                                        0,
                                        nne::audio::gmix::Session::Name::Default,
                                        nne::audio::gmix::Session::Format::Stereo,
                                        nne::audio::gmix::Session::Mode::Default);

            nne::audio::gmix::CloseSession(pAhubGmixSession);
            nne::audio::gmix::CloseSession(pHdaGmixSession);
        }
    }

    nne::audio::gmix::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(GmixSessionStartStop, Success)
{
    const auto defaultDeviceOutList = {nne::audio::gmix::Session::Name::Hda, nne::audio::gmix::Session::Name::Ahub};

    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::gmix::Initialize();

    for(const auto& defaultDeviceOut : defaultDeviceOutList)
    {
        nne::audio::gmix::SetDefaultDeviceOut(defaultDeviceOut);
        nne::audio::gmix::Session* pAhubGmixSession = nullptr;
        nne::audio::gmix::Session* pHdaGmixSession = nullptr;
        nne::audio::gmix::OpenSession(&pAhubGmixSession,
                                    nne::audio::gmix::Session::Name::Ahub,
                                    0,
                                    nne::audio::gmix::Session::Name::Default,
                                    nne::audio::gmix::Session::Format::Stereo,
                                    nne::audio::gmix::Session::Mode::Default);
        nne::audio::gmix::OpenSession(&pHdaGmixSession,
                                    nne::audio::gmix::Session::Name::Hda,
                                    0,
                                    nne::audio::gmix::Session::Name::Default,
                                    nne::audio::gmix::Session::Format::Stereo,
                                    nne::audio::gmix::Session::Mode::Default);

        for(auto i = 0; i < defaultIterationCount; ++i)
        {
            pAhubGmixSession->Start();
            pHdaGmixSession->Start();
            pAhubGmixSession->Stop();
            pHdaGmixSession->Stop();
        }

        nne::audio::gmix::CloseSession(pAhubGmixSession);
        nne::audio::gmix::CloseSession(pHdaGmixSession);
    }

    nne::audio::gmix::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(DeviceSessionOpenClose, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::gmix::Initialize();
    nne::audio::ahub::Initialize();
    nne::audio::hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
                                nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
                                nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
                                nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2);

    nne::audio::device::Session* pAhubOutDeviceSession = nullptr;
    nne::audio::device::Session* pAhubInDeviceSession = nullptr;
    nne::audio::device::Session* pHdaDeviceSession = nullptr;

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::ahub::OpenSession(&pAhubOutDeviceSession, 0);
        nne::audio::ahub::OpenSession(&pAhubInDeviceSession, 1);
        nne::audio::hda::OpenSession(&pHdaDeviceSession, 0);

        nne::audio::hda::CloseSession(pHdaDeviceSession);
        nne::audio::ahub::CloseSession(pAhubInDeviceSession);
        nne::audio::ahub::CloseSession(pAhubOutDeviceSession);
    }

    nne::audio::hda::Finalize();
    nne::audio::gmix::Finalize();
    nne::audio::ahub::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(DeviceSessionStartStop, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::ahub::Initialize();
    nne::audio::hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
                                nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
                                nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
                                nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2);

    nne::audio::device::Session* pAhubDeviceSession = nullptr;
    nne::audio::device::Session* pHdaDeviceSession = nullptr;
    nne::audio::ahub::OpenSession(&pAhubDeviceSession, 0);
    nne::audio::hda::OpenSession(&pHdaDeviceSession, 0);

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        pAhubDeviceSession->Start();
        pHdaDeviceSession->Start();
        pAhubDeviceSession->Stop();
        pHdaDeviceSession->Stop();
    }

    nne::audio::ahub::CloseSession(pAhubDeviceSession);
    nne::audio::hda::CloseSession(pHdaDeviceSession);
    nne::audio::ahub::Finalize();
    nne::audio::hda::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(DeviceSessionPlayback, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::ahub::Initialize();
    nne::audio::hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
                                nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
                                nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
                                nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2);

    nne::audio::device::Session* pAhubDeviceSession = nullptr;
    nne::audio::device::Session* pHdaDeviceSession = nullptr;
    nne::audio::ahub::OpenSession(&pAhubDeviceSession, 0);
    nne::audio::hda::OpenSession(&pHdaDeviceSession, 0);
    pAhubDeviceSession->Start();
    pHdaDeviceSession->Start();

    nne::audio::codec::SetSpeakerMute(true);

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        NN_LOG("[ahub] GetBufferSize() (%zu)\n", pAhubDeviceSession->GetBufferSize());
        NN_LOG("[hda]  GetBufferSize() (%zu)\n", pHdaDeviceSession->GetBufferSize());
        NN_LOG("[ahub] GetPosition() (%zu)\n", pAhubDeviceSession->GetPosition());
        NN_LOG("[hda]  GetPosition() (%zu)\n", pHdaDeviceSession->GetPosition());
        NN_LOG("[ahub] GetBufferAddress() (%p)\n", pAhubDeviceSession->GetBufferAddress());
        NN_LOG("[hda]  GetBufferDeviceAddress() (%p)\n", pHdaDeviceSession->GetBufferAddress());

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    pAhubDeviceSession->Stop();
    pHdaDeviceSession->Stop();
    nne::audio::ahub::CloseSession(pAhubDeviceSession);
    nne::audio::hda::CloseSession(pHdaDeviceSession);
    nne::audio::ahub::Finalize();
    nne::audio::hda::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(AdspDeviceSessionCpuStat, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::ahub::Initialize();
    nne::audio::hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
                                nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
                                nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
                                nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2);

    nne::audio::device::Session* pAhubDeviceSession = nullptr;
    nne::audio::device::Session* pHdaDeviceSession = nullptr;
    nne::audio::ahub::OpenSession(&pAhubDeviceSession, 0);
    nne::audio::hda::OpenSession(&pHdaDeviceSession, 0);
    pAhubDeviceSession->Start();
    pHdaDeviceSession->Start();

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::adsp::EnableAdspCpuStat();
        const auto adspUsage = nne::audio::adsp::getAdspUsage();
        NN_LOG("adspUsage %u\n", adspUsage);
        nne::audio::adsp::DisableAdspCpuStat();
    }

    pAhubDeviceSession->Stop();
    pHdaDeviceSession->Stop();
    nne::audio::ahub::CloseSession(pAhubDeviceSession);
    nne::audio::hda::CloseSession(pHdaDeviceSession);
    nne::audio::ahub::Finalize();
    nne::audio::hda::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(Codec, Success)
{
    nne::audio::codec::Initialize();

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::codec::SetSpeakerMute(false);
        nne::audio::codec::SetOutVolume(0);
        nne::audio::codec::SetOutVolume(0x94);

        nne::audio::codec::SetSpeakerMute(true);
        nne::audio::codec::SetOutVolume(0);
        nne::audio::codec::SetOutVolume(0xaf);
    }

    nne::audio::codec::Finalize();
}

TEST(Codec, CheckParameter)
{
    CodecRegisterChecker checker;
    checker.Open();
    nne::audio::codec::Initialize();

    // 全レジスタダンプ
    // checker.DumpAll(true);

    // 初期化直後のレジスタ値チェック
    checker.checkTestPattern(testPattern_00, sizeof(testPattern_00) / sizeof(testPattern_00[0]));

    // Headphones
    {
        nne::audio::codec::SetSpeakerMute(true);
        checker.checkTestPattern(testPattern_01, sizeof(testPattern_01) / sizeof(testPattern_01[0]));

        // Volume min
        nne::audio::codec::SetOutVolume(0);
        checker.checkTestPattern(testPattern_02, sizeof(testPattern_02) / sizeof(testPattern_02[0]));

        // Volume max
        nne::audio::codec::SetOutVolume(0xaa);
        checker.checkTestPattern(testPattern_03, sizeof(testPattern_03) / sizeof(testPattern_03[0]));
    }

    // Spaker
    {
        nne::audio::codec::SetSpeakerMute(false);
        checker.checkTestPattern(testPattern_04, sizeof(testPattern_04) / sizeof(testPattern_04[0]));

        // Volume min
        nne::audio::codec::SetOutVolume(0);
        checker.checkTestPattern(testPattern_05, sizeof(testPattern_05) / sizeof(testPattern_05[0]));

        // Volume max
        nne::audio::codec::SetOutVolume(0x94);
        checker.checkTestPattern(testPattern_06, sizeof(testPattern_06) / sizeof(testPattern_06[0]));
    }

    nne::audio::codec::Finalize();
    checker.Close();
}

TEST(GmixNarationVolume, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::ahub::Initialize();
    nne::audio::hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
                                nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
                                nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
                                nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2);
    nne::audio::gmix::Initialize();
    nne::audio::gmix::SetDefaultDeviceOut(nne::audio::gmix::Session::Name::Hda);
    nne::audio::gmix::SetDefaultDeviceOut(nne::audio::gmix::Session::Name::Ahub);

    nne::audio::gmix::Session* pAhubGmixSession = nullptr;
    nne::audio::gmix::Session* pHdaGmixSession = nullptr;

    nne::audio::gmix::OpenSession(&pAhubGmixSession,
                                nne::audio::gmix::Session::Name::Ahub, // session name
                                0, // pin
                                nne::audio::gmix::Session::Name::Default, // destination
                                nne::audio::gmix::Session::Format::Stereo, // format stereo, sround
                                nne::audio::gmix::Session::Mode::Default); // mode mute, solo
    nne::audio::gmix::OpenSession(&pHdaGmixSession,
                                nne::audio::gmix::Session::Name::Hda,
                                0,
                                nne::audio::gmix::Session::Name::Default,
                                nne::audio::gmix::Session::Format::Stereo,
                                nne::audio::gmix::Session::Mode::Default);

    nne::audio::device::Session* pAhubDeviceSession = nullptr;
    nne::audio::device::Session* pHdaDeviceSession = nullptr;
    nne::audio::ahub::OpenSession(&pAhubDeviceSession, 0);
    nne::audio::hda::OpenSession(&pHdaDeviceSession, 0);
    pAhubDeviceSession->Start();
    pHdaDeviceSession->Start();

    // TODO: Confirm max and min values
    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::gmix::SetNearVoiceNarrationVolume(0, 0u);

        nne::audio::gmix::SetNearVoiceNarrationVolume(256, 0u);
    }

    pAhubDeviceSession->Stop();
    pHdaDeviceSession->Stop();
    nne::audio::gmix::CloseSession(pAhubGmixSession);
    nne::audio::gmix::CloseSession(pHdaGmixSession);
    nne::audio::gmix::Finalize();
    nne::audio::ahub::CloseSession(pAhubDeviceSession);
    nne::audio::hda::CloseSession(pHdaDeviceSession);
    nne::audio::ahub::Finalize();
    nne::audio::hda::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(GmixStats, Success)
{
    nne::audio::adsp::Initialize(reinterpret_cast<uintptr_t>(adsp_os_bin_begin), adsp_os_bin_end - adsp_os_bin_begin,
                                 reinterpret_cast<uintptr_t>(vector_bin_begin), vector_bin_end - vector_bin_begin,
                                 g_IsAdspLogEnabled);
    nne::audio::ahub::Initialize();
    nne::audio::hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM,
                                nne::audio::hda::AUDIO_HDA_SAMPLERATE_48000,
                                nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE_16,
                                nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS_2);
    nne::audio::gmix::Initialize();
    nne::audio::gmix::SetDefaultDeviceOut(nne::audio::gmix::Session::Name::Ahub);

    nne::audio::gmix::Session* pAhubGmixSession = nullptr;
    nne::audio::gmix::Session* pHdaGmixSession = nullptr;

    nne::audio::gmix::OpenSession(&pAhubGmixSession,
                                nne::audio::gmix::Session::Name::Ahub, // session name
                                0, // pin
                                nne::audio::gmix::Session::Name::Default, // destination
                                nne::audio::gmix::Session::Format::Stereo, // format stereo, sround
                                nne::audio::gmix::Session::Mode::Default); // mode mute, solo
    nne::audio::gmix::OpenSession(&pHdaGmixSession,
                                nne::audio::gmix::Session::Name::Hda,
                                0,
                                nne::audio::gmix::Session::Name::Default,
                                nne::audio::gmix::Session::Format::Stereo,
                                nne::audio::gmix::Session::Mode::Default);
    nne::audio::device::Session* pAhubDeviceSession = nullptr;
    nne::audio::device::Session* pHdaDeviceSession = nullptr;
    nne::audio::ahub::OpenSession(&pAhubDeviceSession, 0);
    nne::audio::hda::OpenSession(&pHdaDeviceSession, 0);
    pAhubDeviceSession->Start();
    pHdaDeviceSession->Start();

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::gmix::stats_t statistics;
        nne::audio::gmix::Stats(&statistics);
        // TODO: Dump more values
        NN_LOG("AudioOutCycles (%llu), AudioOutPasses (%u)\n",
                statistics.Process.AudioOutCycles, statistics.Process.AudioOutPasses);
    }

    pAhubDeviceSession->Stop();
    pHdaDeviceSession->Stop();
    nne::audio::gmix::CloseSession(pAhubGmixSession);
    nne::audio::gmix::CloseSession(pHdaGmixSession);
    nne::audio::gmix::Finalize();
    nne::audio::ahub::CloseSession(pAhubDeviceSession);
    nne::audio::hda::CloseSession(pHdaDeviceSession);
    nne::audio::ahub::Finalize();
    nne::audio::hda::Finalize();
    nne::audio::adsp::Finalize();
}

TEST(IovaInitializeAndFinalize, Success)
{
    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        nne::audio::iova::Initialize();
        nne::audio::iova::Finalize();
    }
}

TEST(IovaMapUnmap, Success)
{
    size_t bufferSize = 1024;
    auto buffer = malloc(bufferSize);
    ASSERT_FALSE(buffer == nullptr);

    nne::audio::iova::Initialize();

    for(auto i = 0; i < defaultIterationCount; ++i)
    {
        auto iovaAddr = nne::audio::iova::Map(buffer, bufferSize);
        EXPECT_NE(iovaAddr, 0);
        nne::audio::iova::Unmap(buffer, bufferSize, iovaAddr);
    }
    nne::audio::iova::Finalize();

    free(buffer);
}
