﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/util/util_BitUtil.h>
#include <nn/nn_Common.h>

#include "../../../../../Programs/Eris/Sources/Libraries/audio/common/audio_WorkBufferAllocator.h"
#include "../../../../../Programs/Eris/Sources/Libraries/audio/common/audio_Util.h"

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(WorkBufferAllocator, AllocateWithAlign0)
{
    const uintptr_t address = 0xABCDEFAB;
    const size_t size = 1024;
    nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);

    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(1, 0), "");
}
#endif

TEST(WorkBufferAllocator, AllocateWithSize0)
{
    const uintptr_t address = 0xABCDEFAB;
    const size_t size = 1024;
    nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);

    EXPECT_EQ(allocator.Allocate(0), nullptr);
    EXPECT_EQ(allocator.GetSize(), size);
    EXPECT_EQ(allocator.GetUsedSize(), 0);
    EXPECT_EQ(allocator.GetFreeSize(), size);
}

TEST(WorkBufferAllocator, AllocateSuccessWithDefaultAlignment)
{
    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1024;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        uintptr_t aligned = nn::util::align_up(0xABCDEFAB, nn::audio::common::InfoTypeFieldAlignSize);
        size_t alignedDiff = aligned - address;
        EXPECT_EQ(allocator.Allocate(1), reinterpret_cast<void*>(aligned));
        EXPECT_EQ(allocator.GetUsedSize(), 1 + alignedDiff);
        EXPECT_EQ(allocator.GetFreeSize(), size - 1 - alignedDiff);
    }

    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1024;
        ASSERT_TRUE((1024 % nn::audio::common::InfoTypeFieldAlignSize) == 0);
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        EXPECT_EQ(allocator.Allocate(1024), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 0);
        EXPECT_EQ(allocator.GetFreeSize(), 1024);
    }

    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1024;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        uintptr_t aligned = nn::util::align_up(0xABCDEFAB, nn::audio::common::InfoTypeFieldAlignSize);
        size_t alignedDiff = aligned - address;
        EXPECT_EQ(allocator.Allocate(512), reinterpret_cast<void*>(aligned));
        EXPECT_EQ(allocator.GetUsedSize(), 512 + alignedDiff);
        EXPECT_EQ(allocator.GetFreeSize(), 512 - alignedDiff);

        EXPECT_EQ(allocator.Allocate(512 - alignedDiff), reinterpret_cast<void*>(aligned + 512));
        EXPECT_EQ(allocator.GetUsedSize(), 1024);
        EXPECT_EQ(allocator.GetFreeSize(), 0);
    }

    {
        const uintptr_t address = nn::util::align_up(0xABCDEFAB, nn::audio::common::InfoTypeFieldAlignSize);
        const size_t size = 1024;
        ASSERT_TRUE((1024 % nn::audio::common::InfoTypeFieldAlignSize) == 0);
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        EXPECT_EQ(allocator.Allocate(1024), reinterpret_cast<void*>(address));
        EXPECT_EQ(allocator.GetUsedSize(), size);
        EXPECT_EQ(allocator.GetFreeSize(), 0);
    }
}

TEST(WorkBufferAllocator, AllocateSuccessWithAlignment)
{
    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1024;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        const auto allocatedAddress = allocator.Allocate(512, 512);
        EXPECT_EQ(allocatedAddress, reinterpret_cast<void*>(nn::util::align_up(address, 512)));

        const auto paddingSize = nn::util::align_up(address, 512) - address;
        EXPECT_EQ(allocator.GetUsedSize(), paddingSize + 512);
        EXPECT_EQ(allocator.GetFreeSize(), size - (paddingSize + 512));
    }

    {
        const uintptr_t address = nn::util::align_up(0xABCDEFAB, 512);
        const size_t size = 1024;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        const auto allocatedAddress = allocator.Allocate(1024, 512);
        EXPECT_EQ(allocatedAddress, reinterpret_cast<void*>(address));

        EXPECT_EQ(allocator.GetUsedSize(), 1024);
        EXPECT_EQ(allocator.GetFreeSize(), 0);
    }

    {
        const uintptr_t address = nn::util::align_up(0xABCDEFAB, 512);
        const size_t size = 1024;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        EXPECT_EQ(allocator.Allocate(100, 512), reinterpret_cast<void*>(address));

        const auto paddingSize = nn::util::align_up(address, 512) - address;
        EXPECT_EQ(allocator.GetUsedSize(), 100 + paddingSize);
        EXPECT_EQ(allocator.GetFreeSize(), size - (100 + paddingSize));

        EXPECT_EQ(allocator.Allocate(1, 512), reinterpret_cast<void*>(nn::util::align_up(static_cast<uintptr_t>(0xABCDEFAB + 100), 512)));
        EXPECT_EQ(allocator.GetUsedSize(), 513);
        EXPECT_EQ(allocator.GetFreeSize(), size - 513);
    }
}

TEST(WorkBufferAllocator, AllocateFailWithDefaultAlignment)
{
    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        EXPECT_EQ(allocator.Allocate(2), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 0);
        EXPECT_EQ(allocator.GetFreeSize(), 1);
    }

    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1024;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        EXPECT_EQ(allocator.Allocate(1025), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 0);
        EXPECT_EQ(allocator.GetFreeSize(), size);
    }

    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1024;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        EXPECT_EQ(allocator.Allocate(1025), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 0);
        EXPECT_EQ(allocator.GetFreeSize(), size);

        EXPECT_EQ(allocator.Allocate(1025), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 0);
        EXPECT_EQ(allocator.GetFreeSize(), size);
    }
}

TEST(WorkBufferAllocator, AllocateFailWithAlignment)
{
    {
        const uintptr_t address = 0xABCDEFAB;
        const size_t size = 1;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);
        EXPECT_EQ(allocator.Allocate(1, 1), reinterpret_cast<void*>(address));
        EXPECT_EQ(allocator.GetUsedSize(), 1);
        EXPECT_EQ(allocator.GetFreeSize(), 0);

        EXPECT_EQ(allocator.Allocate(1, 1), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 1);
        EXPECT_EQ(allocator.GetFreeSize(), 0);
    }


    {
        const uintptr_t address = nn::util::align_up(0xABCDEFAB, 512) + 1;
        const size_t size = 1;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);

        EXPECT_EQ(allocator.Allocate(1, 512), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 0);
        EXPECT_EQ(allocator.GetFreeSize(), size);
    }

    {
        const uintptr_t address = nn::util::align_up(0xABCDEFAB, nn::audio::common::InfoTypeFieldAlignSize);
        const size_t size = nn::audio::common::InfoTypeFieldAlignSize / 2;
        nn::audio::common::WorkBufferAllocator allocator(reinterpret_cast<void*>(address), size);

        // Succeed to allocate
        EXPECT_EQ(allocator.Allocate(2), reinterpret_cast<void*>(address));
        EXPECT_EQ(allocator.GetUsedSize(), 2);
        EXPECT_EQ(allocator.GetFreeSize(), size - 2);

        // Fail to allocate
        EXPECT_EQ(allocator.Allocate(2, nn::audio::common::InfoTypeFieldAlignSize), nullptr);
        EXPECT_EQ(allocator.GetUsedSize(), 2);
        EXPECT_EQ(allocator.GetFreeSize(), size - 2);
    }
}
