﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <cstdlib>
#include <limits>

#include <nnt.h>

#include <nn/audio.h>
#include <nn/util/util_ScopeExit.h>

namespace {

void GenerateSineWave(int16_t* buffer, int sampleRate, int frequency, int amplitude, int sampleCount)
{
    const auto Pi = 3.1415926535897932384626433f;
    for (auto i = 0; i < sampleCount; ++i)
    {
        buffer[i] = static_cast<int16_t>(amplitude * sinf(Pi / 2 + (2 * Pi * frequency * i / sampleRate)));
    }
}

}

TEST(GetRequiredBufferSizeForResampler, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForResampler(0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForResampler(nn::audio::ResamplerType::ChannelCountMax + 1), "");
}

TEST(GetResamplerOutputSampleCount, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetResamplerOutputSampleCount(nullptr, 0), "");
}

TEST(InitializeResampler, Precondition)
{
    auto resampler = nn::audio::ResamplerType();
    const auto channelCount = 1;
    const auto bufferSize = nn::audio::GetRequiredBufferSizeForResampler(channelCount);
    auto buffer = std::malloc(bufferSize);
    ASSERT_TRUE(buffer != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(buffer);
    };

    const auto sampleRate = 16;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(nullptr, buffer, bufferSize, sampleRate, sampleRate, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(&resampler, nullptr, bufferSize, sampleRate, sampleRate, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(&resampler, buffer, bufferSize - 1, sampleRate, sampleRate, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(&resampler, buffer, bufferSize, 0, sampleRate, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(&resampler, buffer, bufferSize, sampleRate, 0, channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(&resampler, buffer, bufferSize, sampleRate, sampleRate, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(&resampler, buffer, bufferSize, sampleRate, static_cast<int>(sampleRate * (nn::audio::ResamplerType::GetConvertRatioMin() / 2)), channelCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeResampler(&resampler, buffer, bufferSize, sampleRate, static_cast<int>(sampleRate * (nn::audio::ResamplerType::GetConvertRatioMax() * 2)), channelCount), "");
}

TEST(ProcessResamplerBuffer, PreconditionAndError)
{
    auto resampler = nn::audio::ResamplerType();
    const auto channelCount = 1;
    const auto bufferSize = nn::audio::GetRequiredBufferSizeForResampler(channelCount);
    auto buffer = std::malloc(bufferSize);
    ASSERT_TRUE(buffer != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(buffer);
    };

    const auto inputSampleRate = 48000 + 1;
    const auto outputSampleRate = 48000 - 1;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::InitializeResampler(&resampler, buffer, bufferSize, inputSampleRate, outputSampleRate, channelCount));

    const auto inputSampleCount = 32;
    const auto inputBufferSize = size_t(inputSampleCount * channelCount * sizeof(int16_t));
    const auto outputSampleCount = nn::audio::GetResamplerOutputSampleCount(&resampler, inputSampleCount);
    const auto outputBufferSize = size_t(outputSampleCount * channelCount * sizeof(int16_t));
    auto inputBuffer = static_cast<int16_t*>(std::malloc(inputBufferSize));
    auto outputBuffer = static_cast<int16_t*>(std::malloc(outputBufferSize));
    ASSERT_TRUE(inputBuffer != nullptr);
    ASSERT_TRUE(outputBuffer != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(inputBuffer);
        std::free(outputBuffer);
    };

    auto resamplerUninitialized = nn::audio::ResamplerType();
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ProcessResamplerBuffer(&resamplerUninitialized, nullptr, outputBuffer, outputBufferSize, inputBuffer, inputSampleCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ProcessResamplerBuffer(nullptr, nullptr, outputBuffer, outputBufferSize, inputBuffer, inputSampleCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ProcessResamplerBuffer(&resampler, nullptr, nullptr, outputBufferSize, inputBuffer, inputSampleCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ProcessResamplerBuffer(&resampler, nullptr, outputBuffer, outputBufferSize, nullptr, inputSampleCount), "");

    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultInsufficientBuffer, nn::audio::ProcessResamplerBuffer(&resampler, nullptr, outputBuffer, outputBufferSize / 2, inputBuffer, inputSampleCount));
}

TEST(Resampler, Success)
{
    const auto channelCount = 1;
    const auto frameRate = 100;

    const auto inputSampleRate = 44100;
    const auto outputSampleRate = 48000;
    const auto inputSampleCount = inputSampleRate / frameRate;
    const auto outputSampleCount = outputSampleRate / frameRate;
    const auto sineFrequency = 540;

    const auto inputBufferSize = inputSampleCount * channelCount * sizeof(int16_t);
    const auto referenceBufferSize = outputSampleCount * channelCount * sizeof(int16_t);
    const auto outputBufferSize = outputSampleCount * channelCount * sizeof(int16_t) + 128;  // Add on some extra so we have room
    const auto resamplerBufferSize = nn::audio::GetRequiredBufferSizeForResampler(channelCount);

    auto inputBuffer = static_cast<int16_t*>(std::malloc(inputBufferSize));
    auto referenceBuffer = static_cast<int16_t*>(std::malloc(referenceBufferSize));
    auto outputBuffer = static_cast<int16_t*>(std::malloc(outputBufferSize));
    auto resamplerBuffer = std::malloc(resamplerBufferSize);
    ASSERT_TRUE(inputBuffer != nullptr);
    ASSERT_TRUE(referenceBuffer != nullptr);
    ASSERT_TRUE(outputBuffer != nullptr);
    ASSERT_TRUE(resamplerBuffer != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(inputBuffer);
        std::free(referenceBuffer);
        std::free(outputBuffer);
        std::free(resamplerBuffer);
    };

    auto resampler = nn::audio::ResamplerType();
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::InitializeResampler(&resampler, resamplerBuffer, resamplerBufferSize, inputSampleRate, outputSampleRate, channelCount));

    const auto amplitude = std::numeric_limits<int16_t>::max();
    GenerateSineWave(inputBuffer, inputSampleRate, sineFrequency, amplitude, inputSampleCount);
    GenerateSineWave(referenceBuffer, outputSampleRate, sineFrequency, amplitude, outputSampleCount);

    auto actualOutputSampleCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::ProcessResamplerBuffer(&resampler, &actualOutputSampleCount, outputBuffer, outputBufferSize, inputBuffer, inputSampleCount));
    EXPECT_TRUE(actualOutputSampleCount > 0);

    const auto resamplerDelay = 8;
    auto maxDiff = 0;
    for (auto i = 0; i < actualOutputSampleCount - resamplerDelay; ++i)
    {
        auto diff = abs(outputBuffer[i + resamplerDelay] - referenceBuffer[i]);
        maxDiff = std::max(maxDiff, diff);
    }

    // Percent error should be less than 5% of amplitude
    const auto threshold = 0.05f;
    EXPECT_TRUE(maxDiff / amplitude < threshold);
}

