﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <utility>

// DspAddr のサイズを CPU レンダリング前提にするため
#define NN_AUDIO_ENABLE_CPU_RENDERER
#define NN_AUDIO_FORCE_SKIP_INFOTYPE_CHECK

#include "../../../../../Programs/Eris/Sources/Libraries/audio/server/audio_PerformanceMetricsManager.h"
#include "../../../../../Programs/Eris/Sources/Libraries/audio/server/audio_ServiceMemoryPoolInfo.h"

NN_STATIC_ASSERT(sizeof(nn::audio::DspAddr) == sizeof(void*));

class MyPoolMapper : public nn::audio::server::PoolMapper
{
public:
    MyPoolMapper(nn::dd::ProcessHandle processHandle, bool forceMapping = false) NN_NOEXCEPT
        : nn::audio::server::PoolMapper(processHandle, forceMapping)
    {

    }

    bool InitializeSystemPool(nn::audio::server::MemoryPoolInfo* pOutInfo, void* address, size_t size) const NN_NOEXCEPT
    {
        return nn::audio::server::PoolMapper::InitializeSystemPool(pOutInfo, address, size);
    }
};

class PerformanceManagerTester : public ::testing::TestWithParam<const char*>
{
private:
    void SetUp()
    {
        const auto revision = nn::audio::common::CreateMagic(GetParam());
        nn::audio::detail::AudioRendererParameterInternal parameter;
        parameter.sampleRate  =  48000;
        parameter.sampleCount = 240;
        parameter.mixBufferCount = 24;
        parameter.subMixCount = 4;
        parameter.voiceCount = 96;
        parameter.sinkCount = 1;
        parameter.effectCount = 0;
        parameter.performanceFrameCount = m_PerformanceFrameCount;
        parameter.isVoiceDropEnabled = true;
        parameter.renderingDevice = 0;
        parameter.executionMode = 0;
        parameter.splitterCount = 0;
        parameter.splitterSendChannelCount = 0;
        parameter._magic = 65536;

        nn::audio::server::BehaviorInfo behavior;
        behavior.SetUserLibRevision(revision);

        m_PerformanceMetricsBufferSizePerFrame = nn::audio::server::PerformanceManager::GetRequiredBufferSizeForPerformanceMetricsPerFrame(behavior, parameter);
        const auto size = m_PerformanceMetricsBufferSizePerFrame * parameter.performanceFrameCount;
        m_WorkBuffer = std::malloc(size);
        ASSERT_NE(m_WorkBuffer, nullptr);
        // 適当に 0 以外の値で埋めておく
        std::memset(m_WorkBuffer, 0xA5, size);

        nn::audio::server::MemoryPoolInfo memoryPool(nn::audio::server::MemoryPoolInfo::Location_Service);

        const auto clientProcessHandle = nn::os::InvalidNativeHandle;
        MyPoolMapper(clientProcessHandle).InitializeSystemPool(&memoryPool, m_WorkBuffer, size);

        m_PerformanceManager.Initialize(
            m_WorkBuffer,
            size,
            parameter,
            behavior,
            memoryPool);
    }

    void TearDown()
    {
        std::free(m_WorkBuffer);
        m_WorkBuffer = nullptr;
    }

public:
    nn::audio::server::PerformanceManager& GetPerformanceManager()
    {
        return m_PerformanceManager;
    }

    size_t GetPerformanceMetricsBufferSizePerFrame() const
    {
        return m_PerformanceMetricsBufferSizePerFrame;
    }

    int GetPerformanceFrameCount() const
    {
        return m_PerformanceFrameCount;
    }

private:
    void* m_WorkBuffer = nullptr;
    size_t m_PerformanceMetricsBufferSizePerFrame = 0u;
    nn::audio::server::PerformanceManager m_PerformanceManager;
    const int m_PerformanceFrameCount = 3;
};

INSTANTIATE_TEST_CASE_P(Revision,
                        PerformanceManagerTester,
                        ::testing::ValuesIn({"REV4", "REV5"}));

TEST(PerformanceManager, GetRequiredBufferSizeForPerformanceMetricsPerFrame)
{
    using ConditionType = std::pair<const char*, size_t>;
    const ConditionType conditions[] = {{"REV4", 3256}, {"REV5", 4896}};
    for(const auto& condition : conditions)
    {
        const auto revision = nn::audio::common::CreateMagic(condition.first);
        nn::audio::detail::AudioRendererParameterInternal parameter;
        parameter.sampleRate  =  48000;
        parameter.sampleCount = 240;
        parameter.mixBufferCount = 24;
        parameter.subMixCount = 4;
        parameter.voiceCount = 96;
        parameter.sinkCount = 1;
        parameter.effectCount = 0;
        parameter.performanceFrameCount = 24;
        parameter.isVoiceDropEnabled = true;
        parameter.renderingDevice = 0;
        parameter.executionMode = 0;
        parameter.splitterCount = 0;
        parameter.splitterSendChannelCount = 0;
        parameter._magic = 65536;

        nn::audio::server::BehaviorInfo behavior;
        behavior.SetUserLibRevision(revision);

        const auto size = nn::audio::server::PerformanceManager::GetRequiredBufferSizeForPerformanceMetricsPerFrame(behavior, parameter);
        EXPECT_EQ(condition.second,  size);
    }
}

TEST_P(PerformanceManagerTester, Initialize)
{
    auto& performanceManager = GetPerformanceManager();
    EXPECT_TRUE(performanceManager.IsInitialized());
}

TEST_P(PerformanceManagerTester, TapFrame)
{
    auto& performanceManager = GetPerformanceManager();

    performanceManager.TapFrame(false, 0, 0);
    performanceManager.TapFrame(true, 0, 0);
}

TEST_P(PerformanceManagerTester, SetDetailTarget)
{
    auto& performanceManager = GetPerformanceManager();

    const auto target1 = nn::audio::PerformanceDetailType_PcmInt16DataSource;
    const auto target2 = nn::audio::PerformanceDetailType_AdpcmDataSource;
    EXPECT_NE(target1, target2);
    performanceManager.SetDetailTarget(target1);
    EXPECT_TRUE(performanceManager.IsDetailTarget(target1));
    EXPECT_FALSE(performanceManager.IsDetailTarget(target2));
}

TEST_P(PerformanceManagerTester, GetNextEntry)
{
    auto& performanceManager = GetPerformanceManager();

    nn::audio::server::PerformanceEntryAddresses block;

    const auto entryType = nn::audio::PerformanceEntryType_Voice;
    const auto targetId = 1;
    EXPECT_TRUE(performanceManager.GetNextEntry(&block, entryType, targetId));

    const auto detailType = nn::audio::PerformanceDetailType_PcmInt16DataSource;
    EXPECT_TRUE(performanceManager.GetNextEntry(&block, detailType, entryType, targetId));

    if(nn::util::string_view(GetParam()) != "REV4")
    {
        uint32_t* pEstimatedProcessingTime = nullptr;
        const auto sysDetailType = nn::audio::PerformanceSysDetailType_PcmInt16DataSource;
        EXPECT_TRUE(performanceManager.GetNextEntry(&block, &pEstimatedProcessingTime, sysDetailType, targetId));
        EXPECT_TRUE(pEstimatedProcessingTime != nullptr);
    }

    while(performanceManager.GetNextEntry(&block, detailType, entryType, targetId));
    EXPECT_FALSE(performanceManager.GetNextEntry(&block, detailType, entryType, targetId));
}

TEST_P(PerformanceManagerTester, CopyHistoriesWithInvalidArgs)
{
    auto& performanceManager = GetPerformanceManager();
    const auto historyFrameCount = GetPerformanceFrameCount();
    const auto historyBufferSize = GetPerformanceMetricsBufferSizePerFrame() * historyFrameCount;
    auto historyBuffer = malloc(historyBufferSize);

    ASSERT_NE(historyBuffer, nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        free(historyBuffer);
        historyBuffer = nullptr;
    };

    EXPECT_EQ(performanceManager.CopyHistories(nullptr, 1), 0u);
    EXPECT_EQ(performanceManager.CopyHistories(historyBuffer, 0u), 0u);
}

TEST_P(PerformanceManagerTester, CopyHistories)
{
    auto& performanceManager = GetPerformanceManager();
    const auto historyFrameCount = GetPerformanceFrameCount();
    const auto historyBufferSize = GetPerformanceMetricsBufferSizePerFrame() * historyFrameCount;
    auto historyBuffer = malloc(historyBufferSize);

    ASSERT_NE(historyBuffer, nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        free(historyBuffer);
        historyBuffer = nullptr;
    };


    // 何も記録されてないので CopyHistories() は 0 を返す
    EXPECT_EQ(performanceManager.CopyHistories(historyBuffer, historyBufferSize), 0u);

    // コマンド生成時に呼ばれる関数群
    const auto entryTypeBegin = nn::audio::PerformanceEntryType_Voice;
    const auto detailTypeBegin = nn::audio::PerformanceDetailType_PcmInt16DataSource;
    const auto targetIdBegin = 1;
    const auto entryCount = 4;
    const auto startTimeBegin = 123;
    const auto processingTimeBegin = 456;

    for(auto i = 0; i < entryCount; ++i)
    {
        const auto entryType = static_cast<nn::audio::PerformanceEntryType>(entryTypeBegin + i);
        const auto targetId = targetIdBegin + i;

        nn::audio::server::PerformanceEntryAddresses block;
        EXPECT_TRUE(performanceManager.GetNextEntry(&block, entryType, targetId));

        // DSP 側で書き込む処理部分
        auto base = static_cast<uintptr_t>(block.dspFrameBase);
        *(reinterpret_cast<int32_t*>(base + block.offsetToStartTime)) = startTimeBegin + i;
        *(reinterpret_cast<int32_t*>(base + block.offsetToProcessingTime)) = processingTimeBegin + i;
        *(reinterpret_cast<int32_t*>(base + block.offsetToCount)) += 1;
    }

    const auto detailCount = 5;
    for(int i = 0; i < detailCount; ++i)
    {
        const auto entryType = static_cast<nn::audio::PerformanceEntryType>(entryTypeBegin + i);
        const auto detailType = static_cast<nn::audio::PerformanceDetailType>(detailTypeBegin + i);
        const auto targetId = targetIdBegin + i;

        nn::audio::server::PerformanceEntryAddresses block;
        EXPECT_TRUE(performanceManager.GetNextEntry(&block, detailType, entryType, targetId));

        // DSP 側で書き込む処理部分
        auto base = static_cast<uintptr_t>(block.dspFrameBase);
        *(reinterpret_cast<int32_t*>(base + block.offsetToStartTime)) = startTimeBegin + i;
        *(reinterpret_cast<int32_t*>(base + block.offsetToProcessingTime)) = processingTimeBegin + i;
        *(reinterpret_cast<int32_t*>(base + block.offsetToCount)) += 1;
    }

    const auto isRenderingTimeLimitExceeded = true;
    const auto voiceDropCount = 789;
    const auto renderingStartTime = 111;
    performanceManager.TapFrame(isRenderingTimeLimitExceeded, voiceDropCount, renderingStartTime);

    // Update 時に呼ばれる部分
    EXPECT_GT(performanceManager.CopyHistories(historyBuffer, historyBufferSize), 0u);

    const auto revision = nn::util::string_view(GetParam());
    if(revision == "REV4")
    {
        const auto pHeader = static_cast<nn::audio::common::PerformanceFrameHeaderVersion1*>(historyBuffer);
        ASSERT_NE(pHeader, nullptr);
        EXPECT_EQ(pHeader->frameMagic, nn::audio::common::CreateMagic(nn::audio::common::PerformanceFrameHeaderMagic));
        EXPECT_EQ(pHeader->frameIndex, 0u);
        EXPECT_EQ(pHeader->entryCount, entryCount);
        EXPECT_EQ(pHeader->detailCount, detailCount);

        const auto entries = reinterpret_cast<nn::audio::common::PerformanceEntryVersion1*>(pHeader + 1);

        for(int i = 0; i < entryCount; ++i)
        {
            EXPECT_EQ(entries[i].startTime, startTimeBegin + i);
            EXPECT_EQ(entries[i].processingTime, processingTimeBegin + i);
        }

        const auto details = reinterpret_cast<nn::audio::common::PerformanceDetailVersion1*>(entries + entryCount);

        for(int i = 0; i < detailCount; ++i)
        {
            EXPECT_EQ(details[i].startTime, startTimeBegin + i);
            EXPECT_EQ(details[i].processingTime, processingTimeBegin + i);
        }
    }
    else if(revision == "REV5")
    {
        const auto pHeader = static_cast<nn::audio::common::PerformanceFrameHeaderVersion2*>(historyBuffer);
        EXPECT_EQ(pHeader->frameMagic, nn::audio::common::CreateMagic(nn::audio::common::PerformanceFrameHeaderMagic));
        EXPECT_EQ(pHeader->frameIndex, 0u);
        EXPECT_EQ(pHeader->entryCount, entryCount);
        EXPECT_EQ(pHeader->detailCount, detailCount);
        EXPECT_EQ(pHeader->isRenderingTimeLimitExceeded, isRenderingTimeLimitExceeded);
        EXPECT_EQ(pHeader->voiceDropCount, voiceDropCount);
        EXPECT_EQ(pHeader->startTime, renderingStartTime);

        const auto entries = reinterpret_cast<nn::audio::common::PerformanceEntryVersion2*>(pHeader + 1);

        for(int i = 0; i < entryCount; ++i)
        {
            EXPECT_EQ(entries[i].startTime, startTimeBegin + i);
            EXPECT_EQ(entries[i].processingTime, processingTimeBegin + i);
        }

        const auto details = reinterpret_cast<nn::audio::common::PerformanceDetailVersion2*>(entries + entryCount);

        for(int i = 0; i < detailCount; ++i)
        {
            EXPECT_EQ(details[i].startTime, startTimeBegin + i);
            EXPECT_EQ(details[i].processingTime, processingTimeBegin + i);
        }
    }
    else
    {
        EXPECT_TRUE(false) << "Unknown Revision.";
    }

    // 記録した全てのパフォーマンス情報を取得したので CopyHistories() は 0 を返す
    EXPECT_EQ(performanceManager.CopyHistories(historyBuffer, historyBufferSize), 0u);
}

