﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory> //unique_ptr

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <cstdlib>
#include <cmath>


#define NN_TEST_AUDIO_EXPECT_RANGE(value, begin, end)       \
    {                                                       \
        auto nn_test_audio_expect_range_value = (value);    \
        EXPECT_GE(nn_test_audio_expect_range_value, begin); \
        EXPECT_LT(nn_test_audio_expect_range_value, end);   \
    }

namespace nnt {
namespace audio {

class ScopedRenderer
{
protected:
    nn::audio::AudioRendererParameter m_Parameter;
    size_t m_SizeForEffect;
    size_t m_WorkBufferSize;
    size_t m_PerfBufferSize;
    void*  m_pWorkBuffer;
    void*  m_pConfigBuffer;
    void*  m_PerfBuffer[2];
    int    m_PerfBufferIndex;
    int    m_DumpCount;
    int    m_TotalDumpCount;
    int    m_ChannelCount;
    int8_t m_MainBus[2];
    int8_t m_SubBus[2];
    int8_t m_AuxSubBus[2];
    int m_SineSampleRate;
    int m_SineSampleCount;
    int m_SineFrequency;
    int m_VoiceAvg;
    int m_SubMixAvg;
    int m_FinalMixAvg;
    int m_SinkAvg;
    int m_TotalAvg;
    int m_ProcessingTime;
    int m_AuxDspAvg;
    int64_t m_RendererUpdateAvg;
    int m_RendererUpdateCount;
    int64_t m_AuxProcessingTotal;
    int m_AuxProcessingCount;

    int64_t m_AuxReadTotal;
    int64_t m_AuxWriteTotal;
    int m_AuxReadWriteCount;

    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;
    nn::audio::FinalMixType m_FinalMix;
    nn::audio::SubMixType m_SubMix;
    nn::audio::DeviceSinkType m_DeviceSink;
    nn::audio::VoiceType m_Voice;
    nn::audio::VoiceType* m_AdpcmVoice;
    nn::audio::VoiceType* m_PcmVoice;
    nn::audio::WaveBuffer m_WaveBuffer;
    nn::audio::WaveBuffer m_AdpcmWaveBuffers[4];
    nn::audio::WaveBuffer m_SineWaveBuffer;
    nn::audio::AdpcmHeaderInfo* m_Header;
    nn::audio::AdpcmHeaderInfo* m_AdpcmHeaders[4];
    nn::mem::StandardAllocator m_AllocatorInstance;
    nn::mem::StandardAllocator& m_Allocator;
    nn::mem::StandardAllocator& m_WaveBufferAllocator;
    nn::mem::StandardAllocator& m_EffectBufferAllocator;
    nn::os::SystemEvent m_SystemEvent;

    nn::audio::DelayType* m_Delay;
    size_t* m_DelaySize;
    void** m_DelayBuffer;
    nn::audio::ReverbType* m_Reverb;
    size_t* m_ReverbSize;
    void** m_ReverbBuffer;
    nn::audio::I3dl2ReverbType* m_I3dl2Reverb;
    size_t* m_I3dl2ReverbSize;
    void** m_I3dl2ReverbBuffer;

    nn::audio::AuxI3dl2ReverbType m_AuxI3dl2Reverb;
    size_t m_AuxI3dl2ReverbSize;
    void* m_AuxI3dl2ReverbBuffer;
    nn::audio::AuxType m_Aux;
    size_t m_AuxBufferSize;
    void* m_SendBuffer;
    void* m_ReturnBuffer;
    int32_t* m_ReadBuffer;

public:
    explicit ScopedRenderer(nn::mem::StandardAllocator& allocator, nn::mem::StandardAllocator& waveBufferAllocator, nn::mem::StandardAllocator& effectBufferAllocator);
    virtual ~ScopedRenderer();
    void RendererSetup(const nn::audio::AudioRendererParameter* param, const int subMixBufferCount = 1, const int finalMixBufferCount = 1);
    void Start();
    void Update();
    void Wait();
    void Stop(int delayCount, int reverbCount, int i3dl2ReverbCount);
    void RendererShutdown(int delayCount, int reverbCount, int i3dl2ReverbCount, bool auxEnabled);
    void RendererAdpcmVoiceShutdown();
    void RendererPcmVoiceShutdown();
    void UpdatePerformanceBuffer(int frames = 100);
    void ProcessAuxEffects(int channelCount);
    std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename);
    nn::audio::AudioRendererHandle GetHandle();
    nn::audio::AudioRendererConfig* GetConfig();
    nn::audio::AudioRendererParameter* GetParam();
    nn::audio::FinalMixType* GetFinalMix();
    nn::audio::SubMixType* GetSubMix();
    nn::audio::VoiceType* GetVoice();
    nn::audio::VoiceType* GetAdpcmVoice(int index);
    nn::audio::VoiceType* GetPcmVoice(int index);
    nn::audio::WaveBuffer* GetWaveBuffer();
    nn::audio::WaveBuffer* GetAdpcmWaveBuffer(int index);
    nn::audio::WaveBuffer* GetSineWaveBuffer();
    int8_t* GetMainBus();
    int8_t* GetSubBus();
    int GetTotalDumpCount();
    void DecrementTotalDumpCount();
    int GetVoiceAvg();
    int GetSubMixAvg();
    int GetFinalMixAvg();
    int GetSinkAvg();
    int GetAuxDspAvg();
    int GetTotalAvg();
    int GetProcessingTime();
    int64_t GetRendererUpdateAvg();
    int64_t GetAuxProcessingAverage();
    int64_t GetAuxReadAverage();
    int64_t GetAuxWriteAverage();
    void InitializeVoice(bool biquadEnabled);
    void InitializeAdpcmVoice(int voiceCount);
    void InitializePcmVoice(int voiceCount);
    std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount);
    void AddAdpcmVoice(int index);
    void AddPcmVoice(int index);
    void AddAuxEffects(int channelCount);
    void AddDelays(int delayCount, int channelCount);
    void AddReverbs(int reverbCount, int channelCount);
    void AddI3dl2Reverbs(int reverbCount, int channelCount);
};

} // namespace audio
} // namespace nnt
