﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_ScopedRenderer.h"
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <memory> // std::unique_ptr

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_EffectCommon.h" // QF_FRACTIONAL_BIT_COUNT
#include <cstdlib>
#include <cmath>

namespace nnt {
namespace audio {

ScopedRenderer::ScopedRenderer(nn::mem::StandardAllocator& allocator, nn::mem::StandardAllocator& waveBufferAllocator, nn::mem::StandardAllocator& effectBufferAllocator)
    : m_Allocator(allocator), m_WaveBufferAllocator(waveBufferAllocator), m_EffectBufferAllocator(effectBufferAllocator)
{
};

ScopedRenderer::~ScopedRenderer()
{
};

void ScopedRenderer::RendererSetup(const nn::audio::AudioRendererParameter* param, const int subMixBufferCount, const int finalMixBufferCount)
{
    m_MainBus[0] = 2;
    m_MainBus[1] = 3;
    m_SubBus[0] = 0;
    m_SubBus[1] = 1;
    m_AuxSubBus[0] = 0;
    m_AuxSubBus[1] = 1;

    m_AuxProcessingTotal = 0;
    m_AuxProcessingCount = 0;
    m_AuxReadTotal = 0;
    m_AuxWriteTotal = 0;
    m_AuxReadWriteCount = 0;

    // gather parameter.
    m_Parameter = *param;
    m_WorkBufferSize = GetAudioRendererWorkBufferSize(m_Parameter);
    m_pWorkBuffer = m_Allocator.Allocate(m_WorkBufferSize, nn::os::MemoryPageSize);
    NN_ASSERT(m_pWorkBuffer != nullptr);
    // instantiate renderer.
    NN_ASSERT(OpenAudioRenderer(&m_Handle, &m_SystemEvent, m_Parameter, m_pWorkBuffer, m_WorkBufferSize).IsSuccess());
    size_t configBufferSize = GetAudioRendererConfigWorkBufferSize(m_Parameter);
    m_pConfigBuffer = m_Allocator.Allocate(configBufferSize);
    NN_ASSERT(m_pConfigBuffer != nullptr);
    InitializeAudioRendererConfig(&m_Config, m_Parameter, m_pConfigBuffer, configBufferSize);
    NN_ASSERT(nn::audio::AcquireFinalMix(&m_Config, &m_FinalMix, finalMixBufferCount));
    NN_ASSERT(nn::audio::AcquireSubMix(&m_Config, &m_SubMix, param->sampleRate, subMixBufferCount));
    NN_ASSERT(nn::audio::AddDeviceSink(&m_Config, &m_DeviceSink, &m_FinalMix, m_MainBus, 2, "MainAudioOut").IsSuccess());
    nn::audio::SetSubMixDestination(&m_Config, &m_SubMix, &m_FinalMix);
    nn::audio::SetSubMixMixVolume(&m_SubMix, &m_FinalMix, 0.5f, m_SubBus[0], m_MainBus[0]);
    nn::audio::SetSubMixMixVolume(&m_SubMix, &m_FinalMix, 0.5f, m_SubBus[1], m_MainBus[1]);

    if( m_Parameter.performanceFrameCount > 0 )
    {
        m_PerfBufferSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(m_Parameter);
        m_PerfBuffer[0] = m_Allocator.Allocate(m_PerfBufferSize);
        NN_ASSERT_NOT_NULL(m_PerfBuffer[0]);
        m_PerfBuffer[1] = m_Allocator.Allocate(m_PerfBufferSize);
        NN_ASSERT_NOT_NULL(m_PerfBuffer[1]);
        nn::audio::SetPerformanceDetailTarget(&m_Config, &m_FinalMix);
        m_PerfBufferIndex = 0;
        m_DumpCount = 0;
        m_TotalDumpCount = 0;
        m_VoiceAvg = 0;
        m_SubMixAvg = 0;
        m_FinalMixAvg = 0;
        m_SinkAvg = 0;
        m_TotalAvg = 0;
        m_RendererUpdateAvg = 0;
        m_RendererUpdateCount = 0;
        m_ProcessingTime = 0;
    }
};

void ScopedRenderer::Start()
{
    NN_ASSERT(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config).IsSuccess());
    NN_ASSERT(nn::audio::StartAudioRenderer(m_Handle).IsSuccess());
}

void ScopedRenderer::Update()
{
    int64_t startTime = nn::os::GetSystemTick().ToTimeSpan().GetMicroSeconds();
    NN_ASSERT(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config).IsSuccess());
    int64_t endTime = nn::os::GetSystemTick().ToTimeSpan().GetMicroSeconds();
    m_RendererUpdateAvg += endTime - startTime;
    m_RendererUpdateCount++;
}

void ScopedRenderer::Wait()
{
    m_SystemEvent.Wait();
}

void ScopedRenderer::Stop(int delayCount, int reverbCount, int i3dl2ReverbCount)
{
    nn::audio::StopAudioRenderer(m_Handle);
    for( int i = 0; i < delayCount; i++ )
    {
        nn::audio::SetDelayEnabled(&m_Delay[i], false);
    }
    for( int i = 0; i < reverbCount; i++ )
    {
        nn::audio::SetReverbEnabled(&m_Reverb[i], false);
    }
    for (int i = 0; i < i3dl2ReverbCount; i++)
    {
        nn::audio::SetI3dl2ReverbEnabled(&m_I3dl2Reverb[i], false);
    }
}

void ScopedRenderer::RendererShutdown(int delayCount, int reverbCount, int i3dl2ReverbCount, bool auxEnabled)
{
    if( nn::audio::AudioRendererState_Started == nn::audio::GetAudioRendererState(m_Handle) )
    {
        nn::audio::StopAudioRenderer(m_Handle);
    }
    nn::audio::CloseAudioRenderer(m_Handle);
    m_Allocator.Free(m_pConfigBuffer);
    m_pConfigBuffer = nullptr;
    m_Allocator.Free(m_pWorkBuffer);
    m_pWorkBuffer = nullptr;
    m_Allocator.Free(m_PerfBuffer[0]);
    m_PerfBuffer[0] = nullptr;
    m_Allocator.Free(m_PerfBuffer[1]);
    m_PerfBuffer[1] = nullptr;
    m_WaveBufferAllocator.Free(m_Header);

    for( int i = 0; i < delayCount; i++ )
    {
        NN_ASSERT(nn::audio::RemoveDelay(&m_Config, &m_Delay[i], &m_FinalMix) == m_DelayBuffer[i]);
        m_EffectBufferAllocator.Free(m_DelayBuffer[i]);
    }
    if( delayCount > 0 )
    {
        delete [] m_Delay;
        delete [] m_DelaySize;
        delete [] m_DelayBuffer;
    }
    for( int i = 0; i < reverbCount; i++ )
    {
        NN_ASSERT(nn::audio::RemoveReverb(&m_Config, &m_Reverb[i], &m_FinalMix) == m_ReverbBuffer[i]);
        m_EffectBufferAllocator.Free(m_ReverbBuffer[i]);
    }
    if( reverbCount > 0 )
    {
        delete [] m_Reverb;
        delete [] m_ReverbSize;
        delete [] m_ReverbBuffer;
    }
    for( int i = 0; i < i3dl2ReverbCount; i++ )
    {
        NN_ASSERT(nn::audio::RemoveI3dl2Reverb(&m_Config, &m_I3dl2Reverb[i], &m_FinalMix) == m_I3dl2ReverbBuffer[i]);
        m_EffectBufferAllocator.Free(m_I3dl2ReverbBuffer[i]);
    }
    if( i3dl2ReverbCount > 0 )
    {
        delete[] m_I3dl2Reverb;
        delete[] m_I3dl2ReverbSize;
        delete[] m_I3dl2ReverbBuffer;
    }
    if( auxEnabled )
    {
        if( m_SendBuffer )
        {
            m_EffectBufferAllocator.Free(m_SendBuffer);
            m_SendBuffer = nullptr;
        }
        if( m_ReturnBuffer )
        {
            m_EffectBufferAllocator.Free(m_ReturnBuffer);
            m_ReturnBuffer = nullptr;
        }
        if( m_ReadBuffer )
        {
            m_EffectBufferAllocator.Free(m_ReadBuffer);
            m_ReadBuffer = nullptr;
        }
        if( m_AuxI3dl2ReverbBuffer )
        {
            m_EffectBufferAllocator.Free(m_AuxI3dl2ReverbBuffer);
            m_AuxI3dl2ReverbBuffer = nullptr;
        }
    }
};

void ScopedRenderer::RendererAdpcmVoiceShutdown()
{
    if( nn::audio::AudioRendererState_Started == nn::audio::GetAudioRendererState(m_Handle) )
    {
        nn::audio::StopAudioRenderer(m_Handle);
    }
    nn::audio::CloseAudioRenderer(m_Handle);
    m_Allocator.Free(m_pConfigBuffer);
    m_pConfigBuffer = nullptr;
    m_Allocator.Free(m_pWorkBuffer);
    m_pWorkBuffer = nullptr;
    m_Allocator.Free(m_PerfBuffer[0]);
    m_PerfBuffer[0] = nullptr;
    m_Allocator.Free(m_PerfBuffer[1]);
    m_PerfBuffer[1] = nullptr;
    for( int i = 0; i < 4; ++i )
    {
        m_WaveBufferAllocator.Free(m_AdpcmHeaders[i]);
        m_AdpcmHeaders[i] = nullptr;
    }
}

void ScopedRenderer::RendererPcmVoiceShutdown()
{
    if( nn::audio::AudioRendererState_Started == nn::audio::GetAudioRendererState(m_Handle) )
    {
        nn::audio::StopAudioRenderer(m_Handle);
    }
    nn::audio::CloseAudioRenderer(m_Handle);
    m_Allocator.Free(m_pConfigBuffer);
    m_pConfigBuffer = nullptr;
    m_Allocator.Free(m_pWorkBuffer);
    m_pWorkBuffer = nullptr;
    m_Allocator.Free(m_PerfBuffer[0]);
    m_PerfBuffer[0] = nullptr;
    m_Allocator.Free(m_PerfBuffer[1]);
    m_PerfBuffer[1] = nullptr;
}

void ScopedRenderer::UpdatePerformanceBuffer(int frames)
{
    void* lastBuffer = nn::audio::SetPerformanceFrameBuffer(&m_Config, m_PerfBuffer[m_PerfBufferIndex], m_PerfBufferSize);
    m_PerfBufferIndex++;
    m_PerfBufferIndex %= 2;
    nn::audio::PerformanceInfo performanceInfo;
    if( lastBuffer == 0 || lastBuffer == nullptr || performanceInfo.SetBuffer(lastBuffer, m_PerfBufferSize) == false )
    {
        return;
    }
    do
    {
        int entryCount = 0;
        const nn::audio::PerformanceEntry* entries = performanceInfo.GetEntries(&entryCount);

        // このサンプルでは frames オーディオフレーム毎の平均値を取ったログ出力します。
        for( int i = 0; i < entryCount; ++i )
        {
            const char* name = nullptr;
            switch( entries[i].entryType )
            {
            case nn::audio::PerformanceEntryType_Voice:
                name = "Voice   ";
                if( m_DumpCount > frames )
                {
                    m_VoiceAvg += entries[i].processingTime;
                }
                break;
            case nn::audio::PerformanceEntryType_SubMix:
                name = "SubMix  ";
                if( m_DumpCount > frames )
                {
                    m_SubMixAvg += entries[i].processingTime;
                }
                break;
            case nn::audio::PerformanceEntryType_FinalMix:
                name = "FinalMix";
                if( m_DumpCount > frames )
                {
                    m_FinalMixAvg += entries[i].processingTime;
                }
                break;
            case nn::audio::PerformanceEntryType_Sink:
                name = "Sink    ";
                if( m_DumpCount > frames )
                {
                    m_SinkAvg += entries[i].processingTime;
                }
                break;
            case nn::audio::PerformanceEntryType_Unknown:
                name = "None    ";
                break;
            default:
                name = "Unknown ";
                break;
            }
        }

        int detailCount = 0;
        const nn::audio::PerformanceDetail* details = performanceInfo.GetDetails(&detailCount);
        for( int i = 0; i < detailCount; ++i )
        {
            const char* name = nullptr;
            switch( details[i].detailType )
            {
            case nn::audio::PerformanceDetailType_Aux:
                name = "Aux     ";
                if( m_DumpCount > frames )
                {
                    m_AuxDspAvg += details[i].processingTime;
                }
                break;
            default:
                name = "Unknown ";
                break;
            }
        }

        if( m_DumpCount++ > frames )
        {
            m_TotalDumpCount++;
            m_TotalAvg += performanceInfo.GetTotalProcessingTime();
            m_ProcessingTime = performanceInfo.GetTotalProcessingTime();
            m_DumpCount = 0;
        }
    } while( performanceInfo.MoveToNextFrame() );
}

void ScopedRenderer::ProcessAuxEffects(int channelCount)
{
    const auto beginTickRead = nn::os::GetSystemTick();
    int32_t readCount = nn::audio::ReadAuxSendBuffer(&m_Aux, m_ReadBuffer, nn::audio::GetAuxSampleCount(&m_Aux));
    m_AuxReadTotal += nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - beginTickRead).GetMicroSeconds();
    int samplesPerFrame = m_Parameter.sampleCount;
    int frameCount = readCount / (samplesPerFrame * channelCount);
    for( auto frame = 0; frame < frameCount; ++frame )
    {
        int32_t* bufferBase = m_ReadBuffer + frame * samplesPerFrame * channelCount;
        const auto beginTickProcess = nn::os::GetSystemTick();
        nn::audio::ProcessAuxI3dl2Reverb(&m_AuxI3dl2Reverb, bufferBase, bufferBase, samplesPerFrame);
        m_AuxProcessingTotal += nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - beginTickProcess).GetMicroSeconds();
        m_AuxProcessingCount++;
    }
    const auto beginTickWrite = nn::os::GetSystemTick();
    nn::audio::WriteAuxReturnBuffer(&m_Aux, m_ReadBuffer, readCount);
    m_AuxWriteTotal += nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - beginTickWrite).GetMicroSeconds();
    m_AuxReadWriteCount++;
}

std::size_t ScopedRenderer::ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = m_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void ScopedRenderer::InitializeVoice(bool biquadEnabled)
{
    void* data;
    const char* filename = "asset:/AudioCommon/SampleSe3.adpcm";

    m_Header = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(m_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
    std::size_t dataSize = ReadAdpcmFile(m_Header, &data, filename);
    m_WaveBuffer.buffer = data;
    m_WaveBuffer.size = dataSize;
    m_WaveBuffer.startSampleOffset = 0;
    m_WaveBuffer.endSampleOffset = m_Header->sampleCount;
    m_WaveBuffer.loop = false;
    m_WaveBuffer.isEndOfStream = false;
    m_WaveBuffer.pContext = &m_Header->loopContext;
    m_WaveBuffer.contextSize = sizeof(nn::audio::AdpcmContext);
    NN_ABORT_UNLESS(nn::audio::AcquireVoiceSlot(&m_Config, &m_Voice, m_Header->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &m_Header->parameter, sizeof(nn::audio::AdpcmParameter)));
    nn::audio::SetVoiceDestination(&m_Config, &m_Voice, &m_FinalMix);
    nn::audio::AppendWaveBuffer(&m_Voice, &m_WaveBuffer);
    nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(&m_Voice, &m_FinalMix, 0.707f / 2, 0, m_SubBus[0]);
    nn::audio::SetVoiceMixVolume(&m_Voice, &m_FinalMix, 0.707f / 2, 0, m_SubBus[1]);

    if( biquadEnabled )
    {
        nn::audio::BiquadFilterParameter filter = { true, {14041, -28083, 14041}, {29547, -13563} };
        nn::audio::SetVoiceBiquadFilterParameter(&m_Voice, 1, filter);
    }
}

void ScopedRenderer::InitializeAdpcmVoice(int voiceCount)
{
    void* data[4];
    std::size_t dataSize[4];
    const char* filenames[4] = { "asset:/AudioCommon/SampleSe0.adpcm", "asset:/AudioCommon/SampleSe1.adpcm", "asset:/AudioCommon/SampleSe2.adpcm", "asset:/AudioCommon/SampleSe3.adpcm" };

    for( int i = 0; i < 4; ++i )
    {
        m_AdpcmHeaders[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(m_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        dataSize[i] = ReadAdpcmFile(m_AdpcmHeaders[i], &data[i], filenames[i]);
        m_AdpcmWaveBuffers[i].buffer = data[i];
        m_AdpcmWaveBuffers[i].size = dataSize[i];
        m_AdpcmWaveBuffers[i].startSampleOffset = 0;
        m_AdpcmWaveBuffers[i].endSampleOffset = m_AdpcmHeaders[i]->sampleCount;
        m_AdpcmWaveBuffers[i].loop = false;
        m_AdpcmWaveBuffers[i].isEndOfStream = false;
        m_AdpcmWaveBuffers[i].pContext = &m_AdpcmHeaders[i]->loopContext;
        m_AdpcmWaveBuffers[i].contextSize = sizeof(nn::audio::AdpcmContext);
    }

    m_AdpcmVoice = new nn::audio::VoiceType[voiceCount];
}

void ScopedRenderer::InitializePcmVoice(int voiceCount)
{
    m_SineSampleRate = 48000;
    m_SineSampleCount = 32000;
    m_SineFrequency = 880;
    void* dataSine;
    std::size_t size = GenerateSineWave(&dataSine, m_SineSampleRate, m_SineFrequency, m_SineSampleCount);

    m_SineWaveBuffer.buffer = dataSine;
    m_SineWaveBuffer.size = size;
    m_SineWaveBuffer.startSampleOffset = 0;
    m_SineWaveBuffer.endSampleOffset = m_SineSampleCount;
    m_SineWaveBuffer.loop = 0;
    m_SineWaveBuffer.isEndOfStream = false;

    m_PcmVoice = new nn::audio::VoiceType[voiceCount];
}

std::size_t ScopedRenderer::GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    int16_t* p = static_cast<int16_t*>(m_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for( auto i = 0; i < sampleCount; ++i )
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

void ScopedRenderer::AddAdpcmVoice(int index)
{
    for( int i = 0; i < index; i++ )
    {
        if( i == index - 1 )
        {
            NN_ABORT_UNLESS(nn::audio::AcquireVoiceSlot(&m_Config, &m_AdpcmVoice[i], m_AdpcmHeaders[i % 4]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &m_AdpcmHeaders[i % 4]->parameter, sizeof(nn::audio::AdpcmParameter)));
            nn::audio::SetVoiceDestination(&m_Config, &m_AdpcmVoice[i], &m_FinalMix);
            nn::audio::AppendWaveBuffer(&m_AdpcmVoice[i], &m_AdpcmWaveBuffers[i % 4]);
            nn::audio::SetVoicePlayState(&m_AdpcmVoice[i], nn::audio::VoiceType::PlayState_Play);
        }
        nn::audio::SetVoiceMixVolume(&m_AdpcmVoice[i], &m_FinalMix, 0.99f / index, 0, m_SubBus[0]);
        nn::audio::SetVoiceMixVolume(&m_AdpcmVoice[i], &m_FinalMix, 0.99f / index, 0, m_SubBus[1]);
    }
}

void ScopedRenderer::AddPcmVoice(int index)
{
    float pitch;
    for( int i = 0; i < index; i++ )
    {
        if( i == index - 1 )
        {
            if( i > 0 )
            {
                pitch = 1.0f / i;
            }
            else
            {
                pitch = 0.0f;
            }
            NN_ABORT_UNLESS(nn::audio::AcquireVoiceSlot(&m_Config, &m_PcmVoice[i], m_SineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
            nn::audio::SetVoiceDestination(&m_Config, &m_PcmVoice[i], &m_FinalMix);
            nn::audio::AppendWaveBuffer(&m_PcmVoice[i], &m_SineWaveBuffer);
            nn::audio::SetVoicePlayState(&m_PcmVoice[i], nn::audio::VoiceType::PlayState_Play);
            nn::audio::SetVoicePitch(&m_PcmVoice[i], 1.0f + pitch);
        }
        nn::audio::SetVoiceMixVolume(&m_PcmVoice[i], &m_FinalMix, 0.99f / index, 0, m_SubBus[0]);
        nn::audio::SetVoiceMixVolume(&m_PcmVoice[i], &m_FinalMix, 0.99f / index, 0, m_SubBus[1]);
    }
}

void ScopedRenderer::AddAuxEffects(int channelCount)
{
    // Setup AuxType
    m_AuxBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&m_Parameter, 8, channelCount);
    m_SendBuffer = static_cast<int32_t*>(m_EffectBufferAllocator.Allocate(m_AuxBufferSize, nn::audio::BufferAlignSize));
    m_ReturnBuffer = static_cast<int32_t*>(m_EffectBufferAllocator.Allocate(m_AuxBufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(&m_Config, &m_Aux, &m_FinalMix, m_SendBuffer, m_ReturnBuffer, m_AuxBufferSize);
    nn::audio::SetAuxInputOutput(&m_Aux, m_SubBus, m_SubBus, channelCount);
    m_ReadBuffer = static_cast<int32_t*>(m_EffectBufferAllocator.Allocate(m_AuxBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(m_ReadBuffer);
    memset(m_ReadBuffer, 0, m_AuxBufferSize);

    nn::audio::I3dl2ReverbParameterSet i3dl2Parameter;
    nn::audio::LoadI3dl2ReverbPreset(&i3dl2Parameter, nn::audio::I3dl2ReverbType::Preset_CavernousCathedral);

    m_AuxI3dl2ReverbSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(m_Parameter.sampleRate, channelCount);
    m_AuxI3dl2ReverbBuffer = m_EffectBufferAllocator.Allocate(m_AuxI3dl2ReverbSize);
    NN_ASSERT_NOT_NULL(m_AuxI3dl2ReverbBuffer);

    nn::audio::InitializeAuxI3dl2Reverb(&m_AuxI3dl2Reverb, m_AuxI3dl2ReverbBuffer, m_AuxI3dl2ReverbSize, m_Parameter.sampleRate, channelCount);
    nn::audio::SetAuxI3dl2ReverbParameters(&m_AuxI3dl2Reverb, &i3dl2Parameter);
    nn::audio::SetAuxI3dl2ReverbInputOutput(&m_AuxI3dl2Reverb, m_AuxSubBus, m_AuxSubBus, channelCount);
}

void ScopedRenderer::AddDelays(int delayCount, int channelCount)
{
    m_Delay = new nn::audio::DelayType[delayCount];
    m_DelaySize = new size_t[delayCount];
    m_DelayBuffer = new void*[delayCount];
    const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
    for( int i = 0; i < delayCount; i++ )
    {
        m_DelaySize[i] = nn::audio::GetRequiredBufferSizeForDelay(delayTime, m_Parameter.sampleRate, channelCount);
        m_DelayBuffer[i] = m_EffectBufferAllocator.Allocate(m_DelaySize[i], nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(m_DelayBuffer[i]);
        NN_ABORT_UNLESS(nn::audio::AddDelay(&m_Config, &m_Delay[i], m_DelayBuffer[i], m_DelaySize[i], &m_FinalMix, delayTime, channelCount).IsSuccess());
        nn:: audio::SetDelayInputOutput(&m_Delay[i], m_SubBus, m_SubBus, 2);
        nn::audio::SetDelayTime(&m_Delay[i], nn::TimeSpan::FromMilliSeconds(100));
        nn::audio::SetDelayChannelSpread(&m_Delay[i], 0.6f);
        nn::audio::SetDelayDryGain(&m_Delay[i], 0.8f);
        nn::audio::SetDelayFeedbackGain(&m_Delay[i], 0.6f);
        nn::audio::SetDelayInGain(&m_Delay[i], 0.8f);
        nn::audio::SetDelayLowPassAmount(&m_Delay[i], 0.1f);
        nn::audio::SetDelayEnabled(&m_Delay[i], true);
    }
}

void ScopedRenderer::AddReverbs(int reverbCount, int channelCount)
{
    m_Reverb = new nn::audio::ReverbType[reverbCount];
    m_ReverbSize = new size_t[reverbCount];
    m_ReverbBuffer = new void*[reverbCount];
    for( int i = 0; i < reverbCount; ++i )
    {
        m_ReverbSize[i] = nn::audio::GetRequiredBufferSizeForReverb(m_Parameter.sampleRate, channelCount);
        m_ReverbBuffer[i] = m_EffectBufferAllocator.Allocate(m_ReverbSize[i], nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(m_ReverbBuffer[i]);
        NN_ABORT_UNLESS(nn::audio::AddReverb(&m_Config, &m_Reverb[i], m_ReverbBuffer[i], m_ReverbSize[i], &m_FinalMix, channelCount).IsSuccess());
        nn::audio::SetReverbInputOutput(&m_Reverb[i], m_SubBus, m_SubBus, 2);

        nn::audio::SetReverbEarlyMode(&m_Reverb[i], nn::audio::ReverbType::EarlyMode_Hall);
        nn::audio::SetReverbEarlyGain(&m_Reverb[i], 0.8f);
        nn::audio::SetReverbPredelayTime(&m_Reverb[i], nn::TimeSpan::FromMilliSeconds(10));
        nn::audio::SetReverbLateMode(&m_Reverb[i], nn::audio::ReverbType::LateMode_Hall);
        nn::audio::SetReverbLateGain(&m_Reverb[i], 0.5f);
        nn::audio::SetReverbDecayTime(&m_Reverb[i], nn::TimeSpan::FromSeconds(1));
        nn::audio::SetReverbHighFrequencyDecayRatio(&m_Reverb[i], 0.5f);
        nn::audio::SetReverbColoration(&m_Reverb[i], 0.0f);
        nn::audio::SetReverbReverbGain(&m_Reverb[i], 0.8f);
        nn::audio::SetReverbOutGain(&m_Reverb[i], 1.0f);
        nn::audio::SetReverbEnabled(&m_Reverb[i], true);
    }
}

void ScopedRenderer::AddI3dl2Reverbs(int reverbCount, int channelCount)
{
    m_I3dl2Reverb = new nn::audio::I3dl2ReverbType[reverbCount];
    m_I3dl2ReverbSize = new size_t[reverbCount];
    m_I3dl2ReverbBuffer = new void*[reverbCount];
    for (int i = 0; i < reverbCount; ++i)
    {
        nn::audio::I3dl2ReverbParameterSet paramSet;

        m_I3dl2ReverbSize[i] = nn::audio::GetRequiredBufferSizeForI3dl2Reverb(m_Parameter.sampleRate, channelCount);
        m_I3dl2ReverbBuffer[i] = m_EffectBufferAllocator.Allocate(m_I3dl2ReverbSize[i], nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(m_I3dl2ReverbBuffer[i]);
        NN_ABORT_UNLESS(nn::audio::AddI3dl2Reverb(&m_Config, &m_I3dl2Reverb[i], m_I3dl2ReverbBuffer[i], m_I3dl2ReverbSize[i], &m_FinalMix, channelCount).IsSuccess());
        nn::audio::SetI3dl2ReverbInputOutput(&m_I3dl2Reverb[i], m_SubBus, m_SubBus, 2);

        nn::audio::LoadI3dl2ReverbPreset(&paramSet, nn::audio::I3dl2ReverbType::Preset_CavernousCathedral);
        nn::audio::SetI3dl2ReverbParameters(&m_I3dl2Reverb[i], &paramSet);
        nn::audio::SetI3dl2ReverbEnabled(&m_I3dl2Reverb[i], true);
    }
}

nn::audio::AudioRendererHandle ScopedRenderer::GetHandle()
{
    return m_Handle;
};

nn::audio::AudioRendererConfig* ScopedRenderer::GetConfig()
{
    return &m_Config;
};

nn::audio::AudioRendererParameter* ScopedRenderer::GetParam()
{
    return &m_Parameter;
};

nn::audio::FinalMixType* ScopedRenderer::GetFinalMix()
{
    return &m_FinalMix;
}

nn::audio::SubMixType* ScopedRenderer::GetSubMix()
{
    return &m_SubMix;
}

nn::audio::VoiceType* ScopedRenderer::GetVoice()
{
    return &m_Voice;
}

nn::audio::VoiceType* ScopedRenderer::GetAdpcmVoice(int index)
{
    return &m_AdpcmVoice[index];
}

nn::audio::VoiceType* ScopedRenderer::GetPcmVoice(int index)
{
    return &m_PcmVoice[index];
}

nn::audio::WaveBuffer* ScopedRenderer::GetWaveBuffer()
{
    return &m_WaveBuffer;
}

nn::audio::WaveBuffer* ScopedRenderer::GetAdpcmWaveBuffer(int index)
{
    return &m_AdpcmWaveBuffers[index];
}

nn::audio::WaveBuffer* ScopedRenderer::GetSineWaveBuffer()
{
    return &m_SineWaveBuffer;
}

int8_t* ScopedRenderer::GetSubBus()
{
    return m_SubBus;
}

int8_t* ScopedRenderer::GetMainBus()
{
    return m_MainBus;
}

int ScopedRenderer::GetTotalDumpCount()
{
    return m_TotalDumpCount;
}

void ScopedRenderer::DecrementTotalDumpCount()
{
    m_TotalDumpCount--;
}

int ScopedRenderer::GetVoiceAvg()
{
    return m_VoiceAvg;
}

int ScopedRenderer::GetSubMixAvg()
{
    return m_SubMixAvg;
}

int ScopedRenderer::GetFinalMixAvg()
{
    return m_FinalMixAvg;
}

int ScopedRenderer::GetSinkAvg()
{
    return m_SinkAvg;
}

int ScopedRenderer::GetAuxDspAvg()
{
    return m_AuxDspAvg;
}

int ScopedRenderer::GetTotalAvg()
{
    return m_TotalAvg;
}

int ScopedRenderer::GetProcessingTime()
{
    return m_ProcessingTime;
}

int64_t ScopedRenderer::GetRendererUpdateAvg()
{
    return m_RendererUpdateAvg / m_RendererUpdateCount;
}

int64_t ScopedRenderer::GetAuxProcessingAverage()
{
    return m_AuxProcessingTotal / m_AuxProcessingCount;
}

int64_t ScopedRenderer::GetAuxReadAverage()
{
    return m_AuxReadTotal / m_AuxReadWriteCount;
}

int64_t ScopedRenderer::GetAuxWriteAverage()
{
    return m_AuxWriteTotal / m_AuxReadWriteCount;
}

} // namespace audio
} // namespace nnt
