﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/audio.h>
#include <nn/nn_Log.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include "testAudio_Effect.h"

namespace {

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[2 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[1024 * 1024];
char g_RecorderBuffer[16 * 1024 * 1024];

class ScopedAudioRenderer : public nnt::audio::EffectScopedRenderer
{

public:
    explicit ScopedAudioRenderer(nn::mem::StandardAllocator& allocator)
        : EffectScopedRenderer(allocator)
    {
        nn::audio::AudioRendererParameter parameter;
        nn::audio::InitializeAudioRendererParameter(&parameter);
        parameter.mixBufferCount = 10;
        parameter.subMixCount = 1;
        parameter.effectCount = 1;

        Base::RendererSetup(&parameter, 1, 1);

        // Add BiquadFilterType
        nn::audio::AddBiquadFilter(&m_Config, &m_BiquadFilter, &m_FinalMix);
    };

    virtual ~ScopedAudioRenderer()
    {
        RendererShutdown();
    }

    nn::audio::BiquadFilterType& GetBiquadFilter()
    {
        return m_BiquadFilter;
    };

private:
    typedef nnt::audio::EffectScopedRenderer Base;
    nn::audio::BiquadFilterType m_BiquadFilter;
};

#if !defined(NN_SDK_BUILD_RELEASE)
nn::audio::BiquadFilterType g_InvalidBiquadFilter = { nullptr };

template<typename Mix>
void RunAddBiquadFilterPrecoditionTest()
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nnt::audio::EffectScopedRenderer renderer(allocator);

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.mixBufferCount = 10;
    parameter.subMixCount = 1;
    parameter.effectCount = 1;
    renderer.RendererSetup(&parameter, 1, 1);

    nn::audio::BiquadFilterType biquadFilter;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddBiquadFilter(nullptr, &biquadFilter, renderer.GetFinalMix()), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddBiquadFilter(renderer.GetConfig(), nullptr, renderer.GetFinalMix()), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddBiquadFilter(renderer.GetConfig(), &biquadFilter, static_cast<Mix*>(nullptr)), "");

    renderer.RendererShutdown();
}

template<typename Mix>
void RunRemoveBiquadFilterPrecoditionTest()
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    ScopedAudioRenderer scopedRenderer(allocator);
    auto& validBiquadFilter = scopedRenderer.GetBiquadFilter();
    auto pMix = scopedRenderer.GetMix<Mix>();

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveBiquadFilter(nullptr, &validBiquadFilter, pMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveBiquadFilter(scopedRenderer.GetConfig(), nullptr, pMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveBiquadFilter(nullptr, &validBiquadFilter, pMix), "");
}
#endif // #if !defined(NN_SDK_BUILD_RELEASE)

template<typename Mix>
void RunAddRemoveBiquadFilterSuccessTest()
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nnt::audio::EffectScopedRenderer renderer(allocator);
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.mixBufferCount = 10;
    parameter.subMixCount = 1;
    parameter.effectCount = 1;
    renderer.RendererSetup(&parameter, 1, 1);

    nn::audio::BiquadFilterType biquadFilter;
    auto pMix = renderer.GetMix<Mix>();

    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddBiquadFilter(renderer.GetConfig(), &biquadFilter, pMix));

    nn::audio::SetBiquadFilterEnabled(&biquadFilter, false);

    nn::audio::RemoveBiquadFilter(renderer.GetConfig(), &biquadFilter, pMix);

    renderer.RendererShutdown();
}

void RunWaveComparisonTest(int channelCount, const nn::audio::BiquadFilterParameter& filterParameter, const std::string& sourceFileName, const std::string& resultFileName)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + sourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + resultFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + resultFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(3);
    const int32_t RecorderBufferSize = 100;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, channelCount);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, channelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, static_cast<uint16_t>(channelCount), BitRate);

    // Renderer Setup
    nn::audio::BiquadFilterType biquadFilter;
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddBiquadFilter(renderer.GetConfig(), &biquadFilter, renderer.GetFinalMix()));
    const int8_t bus[nn::audio::BiquadFilterType::ChannelCountMax] = { 0, 1, 2, 3, 4, 5 };
    nn::audio::SetBiquadFilterInputOutput(&biquadFilter, bus, bus, channelCount);
    nn::audio::SetBiquadFilterParameter(&biquadFilter, filterParameter);

    // Setup AuxType
    nn::audio::AuxType aux;
    const auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, channelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer, returnBuffer, bufferSize));
    const int8_t inputIndex[nn::audio::BiquadFilterType::ChannelCountMax] = { 0, 1, 2, 3, 4, 5 };
    const int8_t outputIndex[nn::audio::BiquadFilterType::ChannelCountMax] = { 0, 1, 2, 3, 4, 5 };
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, channelCount);

    renderer.Update();

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    ASSERT_NE(dataShutter, nullptr);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 2);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 3);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 4);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 5);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while (recorder.Record())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);
}

bool IsSameParameter(const nn::audio::BiquadFilterParameter& parameterA, const nn::audio::BiquadFilterParameter& parameterB)
{
    return parameterA.enable == parameterB.enable &&
           parameterA.numerator[0] == parameterB.numerator[0] &&
           parameterA.numerator[1] == parameterB.numerator[1] &&
           parameterA.numerator[2] == parameterB.numerator[2] &&
           parameterA.denominator[0] == parameterB.denominator[0] &&
           parameterA.denominator[1] == parameterB.denominator[1];
}

} // namespace

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AddBiquadFilter, Precondition)
{
    RunAddBiquadFilterPrecoditionTest<nn::audio::FinalMixType>();
    RunAddBiquadFilterPrecoditionTest<nn::audio::SubMixType>();
}

TEST(RemoveBiquadFilter, Precondition)
{
    RunRemoveBiquadFilterPrecoditionTest<nn::audio::FinalMixType>();
    RunRemoveBiquadFilterPrecoditionTest<nn::audio::SubMixType>();
}

TEST(SetBiquadFilterInputOutput, Precondition)
{
    int8_t input[nn::audio::BiquadFilterType::ChannelCountMax];
    int8_t output[nn::audio::BiquadFilterType::ChannelCountMax];

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    ScopedAudioRenderer scopedRenderer(allocator);
    auto& validBiquadFilter = scopedRenderer.GetBiquadFilter();

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterInputOutput(nullptr, input, output, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterInputOutput(&g_InvalidBiquadFilter, input, output, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterInputOutput(&validBiquadFilter, nullptr, output, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterInputOutput(&validBiquadFilter, input, nullptr, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterInputOutput(&validBiquadFilter, input, output, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterInputOutput(&validBiquadFilter, input, output, nn::audio::BiquadFilterType::ChannelCountMax + 1), "");
}

TEST(GetBiquadFilterInputOutput, Precondition)
{
    int8_t input[nn::audio::BiquadFilterType::ChannelCountMax];
    int8_t output[nn::audio::BiquadFilterType::ChannelCountMax];
    int outCount;

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    ScopedAudioRenderer scopedRenderer(allocator);
    auto& validBiquadFilter = scopedRenderer.GetBiquadFilter();

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterInputOutput(nullptr, input, output, &outCount, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterInputOutput(&g_InvalidBiquadFilter, input, output, &outCount, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, nullptr, output, &outCount, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, input, nullptr, &outCount, nn::audio::BiquadFilterType::ChannelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, input, output, &outCount, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, input, output, &outCount, nn::audio::BiquadFilterType::ChannelCountMax + 1), "");
}

TEST(SetBiquadFilterParameter, Precondition)
{
    nn::audio::BiquadFilterParameter parameter;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterParameter(nullptr, parameter), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterParameter(&g_InvalidBiquadFilter, parameter), "");
}

TEST(GetBiquadFilterParameter, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterParameter(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetBiquadFilterParameter(&g_InvalidBiquadFilter), "");
}

TEST(SetBiquadFilterEnabled, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterEnabled(nullptr, false), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetBiquadFilterEnabled(&g_InvalidBiquadFilter, false), "");
}

TEST(IsBiquadFilterEnabled, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsBiquadFilterEnabled(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsBiquadFilterEnabled(&g_InvalidBiquadFilter), "");
}

TEST(IsBiquadFilterRemovable, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsBiquadFilterRemovable(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsBiquadFilterRemovable(&g_InvalidBiquadFilter), "");
}
#endif // #if !defined(NN_SDK_BUILD_RELEASE)

TEST(AddRemoveBiquadFilter, Success)
{
    RunAddRemoveBiquadFilterSuccessTest<nn::audio::SubMixType>();
    RunAddRemoveBiquadFilterSuccessTest<nn::audio::FinalMixType>();
}

TEST(SetGetBiquadFilterInputOutput, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    ScopedAudioRenderer scopedRenderer(allocator);
    auto& validBiquadFilter = scopedRenderer.GetBiquadFilter();

    // Check initial count
    {
        int outCount;
        int8_t input[nn::audio::BiquadFilterType::ChannelCountMax];
        int8_t output[nn::audio::BiquadFilterType::ChannelCountMax];
        nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, input, output, &outCount, nn::audio::BiquadFilterType::ChannelCountMax);
        EXPECT_EQ(outCount, 0);
    }

    // Case: originalCount == count
    {
        const int originalCount = nn::audio::BiquadFilterType::ChannelCountMax;

        const int8_t originalInput[originalCount]  = { 0, 1, 2, 3, 4, 5 };
        const int8_t originalOutput[originalCount] = { 5, 4, 3, 2, 1, 0 };
        nn::audio::SetBiquadFilterInputOutput(&validBiquadFilter, originalInput, originalOutput, originalCount);

        int outCount = 0xFF; // set dummy value
        const int count = originalCount;
        int8_t input[count];
        int8_t output[count];
        nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, input, output, &outCount, count);
        EXPECT_EQ(std::memcmp(originalInput, input, sizeof(input[0]) * outCount), 0);
        EXPECT_EQ(std::memcmp(originalOutput, output, sizeof(input[0]) * outCount), 0);
        EXPECT_EQ(outCount, std::min(count, originalCount));
    }

    // Case: originalCount > count
    {
        const int originalCount = nn::audio::BiquadFilterType::ChannelCountMax;

        const int8_t originalInput[originalCount]  = { 0, 1, 2, 3, 4, 5 };
        const int8_t originalOutput[originalCount] = { 5, 4, 3, 2, 1, 0 };
        nn::audio::SetBiquadFilterInputOutput(&validBiquadFilter, originalInput, originalOutput, originalCount);

        int outCount = 0xFF; // set dummy value
        const int count = originalCount - 1;
        int8_t input[count];
        int8_t output[count];
        nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, input, output, &outCount, count);
        EXPECT_EQ(std::memcmp(originalInput, input, sizeof(input[0]) * outCount), 0);
        EXPECT_EQ(std::memcmp(originalOutput, output, sizeof(output[0]) * outCount), 0);
        EXPECT_EQ(outCount, std::min(count, originalCount));
    }

    // Case: originalCount < count
    {
        const int originalCount = nn::audio::BiquadFilterType::ChannelCountMax - 1;

        const int8_t originalInput[originalCount]  = { 0, 1, 2, 3, 4};
        const int8_t originalOutput[originalCount] = { 5, 4, 3, 2, 1};
        nn::audio::SetBiquadFilterInputOutput(&validBiquadFilter, originalInput, originalOutput, originalCount);

        int outCount = 0xFF; // set dummy value
        const int count = originalCount + 1;
        int8_t input[count];
        int8_t output[count];
        nn::audio::GetBiquadFilterInputOutput(&validBiquadFilter, input, output, &outCount, count);
        EXPECT_EQ(std::memcmp(originalInput, input, sizeof(input[0]) * outCount), 0);
        EXPECT_EQ(std::memcmp(originalOutput, output, sizeof(output[0]) * outCount), 0);
        EXPECT_EQ(outCount, std::min(count, originalCount));
    }
}

TEST(SetGetBiquadFilterParameter, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    ScopedAudioRenderer scopedRenderer(allocator);
    auto& validBiquadFilter = scopedRenderer.GetBiquadFilter();

    // Check initial value
    {
        const auto parameter = nn::audio::GetBiquadFilterParameter(&validBiquadFilter);
        const nn::audio::BiquadFilterParameter zeroFilledBiquadFilterParameter = { true, { 0, 0, 0 }, { 0, 0 } };
        EXPECT_TRUE(IsSameParameter(parameter, zeroFilledBiquadFilterParameter));
    }

    // Set -> Get
    {
        const nn::audio::BiquadFilterParameter originalBiquadFilterParameter = { true, { 1, 2, 3 }, { 4, 5 } };
        nn::audio::SetBiquadFilterParameter(&validBiquadFilter, originalBiquadFilterParameter);
        const auto parameter = nn::audio::GetBiquadFilterParameter(&validBiquadFilter);
        EXPECT_TRUE(IsSameParameter(parameter, originalBiquadFilterParameter));
    }
}

TEST(SetIsBiquadFilterEnabled, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    ScopedAudioRenderer scopedRenderer(allocator);
    auto& validBiquadFilter = scopedRenderer.GetBiquadFilter();

    // Check initial value
    {
        EXPECT_TRUE(nn::audio::IsBiquadFilterEnabled(&validBiquadFilter));
    }

    // SetEnabled -> IsEnabled
    {
        nn::audio::SetBiquadFilterEnabled(&validBiquadFilter, false);
        EXPECT_EQ(nn::audio::IsBiquadFilterEnabled(&validBiquadFilter), false);
    }

    // SetParameter -> IsEnabled
    {
        const nn::audio::BiquadFilterParameter originalBiquadFilterParameter = { true, { 1, 2, 3 }, { 4, 5 } };
        nn::audio::SetBiquadFilterParameter(&validBiquadFilter, originalBiquadFilterParameter);
        EXPECT_TRUE(nn::audio::IsBiquadFilterEnabled(&validBiquadFilter));
    }
}

TEST(IsBiquadFilterRemovable, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    ScopedAudioRenderer scopedRenderer(allocator);
    auto validBiquadFilter = scopedRenderer.GetBiquadFilter();

    EXPECT_FALSE(nn::audio::IsBiquadFilterRemovable(&validBiquadFilter));
    nn::audio::SetBiquadFilterEnabled(&validBiquadFilter, false);
    EXPECT_TRUE(nn::audio::IsBiquadFilterRemovable(&validBiquadFilter));
}

TEST(BiquadFilter, WaveComparisonTest)
{
    const std::string sourceFileName = "/shutter.wav";

    // Enabled Lpf 2048 Hz
    {
        const nn::audio::BiquadFilterParameter lpf2048HzfilterParameter = { true, {720, 1439, 720}, {22684, -9350} };
        for(int channelCount = 1; channelCount <= nn::audio::BiquadFilterType::ChannelCountMax; ++channelCount)
        {
            std::string resultFileName = "/testAudio_Effect_BiquadFilterLpf";
            resultFileName += std::to_string(channelCount) + "chShutter.wav";

            RunWaveComparisonTest(channelCount, lpf2048HzfilterParameter, sourceFileName, resultFileName);
        }
    }

    // Disabled
    {
        const nn::audio::BiquadFilterParameter disabledFilterParameter = { false, {720, 1439, 720}, {22684, -9350} };
        for(int channelCount = 1; channelCount <= nn::audio::BiquadFilterType::ChannelCountMax; ++channelCount)
        {
            std::string resultFileName = "/testAudio_Effect_BiquadFilterPathThrough";
            resultFileName += std::to_string(channelCount) + "chShutter.wav";

            RunWaveComparisonTest(channelCount, disabledFilterParameter, sourceFileName, resultFileName);
        }
    }
}
