﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // max_element, min_element
#include <cstdlib>  // free, malloc
#include <numeric>  // accumulate

#include <nnt.h>

#include <nn/nn_Log.h>
#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

TEST(AudioSystemEventPeriod, AudioOut)
{
    auto allocatorSize = 1 * 1024 * 1024;
    auto allocatorBuffer = std::malloc(allocatorSize);
    EXPECT_TRUE(allocatorBuffer != nullptr);
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(allocatorBuffer, allocatorSize);

    for (auto frame = 1; frame <= 3; ++frame)
    {
        nn::audio::AudioOut audioOut;
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        nn::os::SystemEvent systemEvent;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter));
        const auto ChannelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
        const auto SampleCount = nn::audio::GetAudioOutSampleRate(&audioOut);
        const auto SampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
        ASSERT_TRUE(SampleFormat == nn::audio::SampleFormat_PcmInt16);
        const auto FrameSampleCount = SampleCount / 200 * frame;
        const size_t DataSize = FrameSampleCount * ChannelCount * nn::audio::GetSampleByteSize(SampleFormat);
        const size_t BufferSize = nn::util::align_up(DataSize, nn::audio::AudioOutBuffer::SizeGranularity);
        const auto BufferCount = 8;
        nn::audio::AudioOutBuffer audioOutBuffer[BufferCount];
        void* outBuffer[BufferCount];
        for (auto i = 0; i < BufferCount; ++i)
        {
            outBuffer[i] = allocator.Allocate(BufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
            ASSERT_TRUE(outBuffer[i] != nullptr);
            std::memset(outBuffer[i], 0, BufferSize);
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], BufferSize, DataSize);
            nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        }
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));

        nn::TimeSpan periods[1024];
        const auto Count = static_cast<int>(1024 / frame);
        systemEvent.Wait();
        auto begin = nn::os::GetSystemTick().ToTimeSpan();
        systemEvent.Wait();
        for (auto i = 0; i < Count; ++i)
        {
            auto end = nn::os::GetSystemTick().ToTimeSpan();
            periods[i] = end - begin;
            begin = end;
            auto pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            while (pAudioOutBuffer)
            {
                auto pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
                auto outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
                ASSERT_TRUE(outSize == FrameSampleCount * ChannelCount * nn::audio::GetSampleByteSize(SampleFormat));
                std::memset(pOutBuffer, 0, BufferSize);
                nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);
                pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            }
            systemEvent.Wait();
        }

        nn::audio::StopAudioOut(&audioOut);
        nn::audio::CloseAudioOut(&audioOut);
        nn::os::DestroySystemEvent(systemEvent.GetBase());
        for (auto i = 0; i < BufferCount; ++i)
        {
            allocator.Free(outBuffer[i]);
        }

        auto max = std::max_element(periods, periods + Count)->GetMicroSeconds();
        auto min = std::min_element(periods, periods + Count)->GetMicroSeconds();
        auto avg = std::accumulate(periods, periods + Count, nn::TimeSpan(0)).GetMicroSeconds() / Count;

        NN_LOG("##teamcity[buildStatisticValue key='AudioOutSystemEventPeriod%02dMax(usec)' value='%lld']\n", 5 * frame, max);
        NN_LOG("##teamcity[buildStatisticValue key='AudioOutSystemEventPeriod%02dMin(usec)' value='%lld']\n", 5 * frame, min);
        NN_LOG("##teamcity[buildStatisticValue key='AudioOutSystemEventPeriod%02dAvg(usec)' value='%lld']\n", 5 * frame, avg);
    }

    allocator.Finalize();
    std::free(allocatorBuffer);
}
