﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <limits>
#include <random>

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_Volume.h"

TEST(DspVolume, CompareX1X2AndX4)
{
    const int SampleCount = 1024;

    int32_t input[SampleCount];
    int32_t output0[SampleCount] = { 0 };
    int32_t output1[SampleCount] = { 0 };
    int32_t output2[SampleCount] = { 0 };

    std::mt19937 mt(0);
    for (auto i = 0; i < SampleCount; ++i)
    {
        const auto a = std::numeric_limits<uint16_t>::max();
        const auto b = std::numeric_limits<int16_t>::min();
        input[i] = (mt() % a) + b;
        output0[i] = output1[i] = output2[i] = ( mt() % a) + b;
    }

    const int32_t volumes[] = { 32767, 12345, 0, -12345, -32768 };
    for (const auto volume : volumes)
    {
        nn::audio::dsp::ApplyUniformGain1(output0, input, volume, SampleCount);
        nn::audio::dsp::ApplyUniformGain2(output1, input, volume, SampleCount);
        nn::audio::dsp::ApplyUniformGain4(output2, input, volume, SampleCount);
        for (auto i = 0; i < SampleCount; ++i)
        {
            EXPECT_EQ(output0[i], output1[i]);
            EXPECT_EQ(output0[i], output2[i]);
        }
    }
}

TEST(DspVolumeRamp, CompareX1X2AndX4)
{
    const int SampleCount = 1024;

    int32_t input[SampleCount];
    int32_t output0[SampleCount] = { 0 };
    int32_t output1[SampleCount] = { 0 };
    int32_t output2[SampleCount] = { 0 };

    std::mt19937 mt(0);
    for (auto i = 0; i < SampleCount; ++i)
    {
        const auto a = std::numeric_limits<uint16_t>::max();
        const auto b = std::numeric_limits<int16_t>::min();
        input[i] = (mt() % a) + b;
        output0[i] = output1[i] = output2[i] = ( mt() % a) + b;
    }

    const int32_t volumes[] = { 32767, 12345, 0, -12345, -32768 };
    const int32_t deltas[] = { -10, -1, 0, 1, 10 };
    for (const auto volume : volumes)
    {
        for (const auto delta : deltas)
        {
            nn::audio::dsp::ApplyLinearEnvelopeGain1(output0, input, volume, delta, SampleCount);
            nn::audio::dsp::ApplyLinearEnvelopeGain2(output1, input, volume, delta, SampleCount);
            nn::audio::dsp::ApplyLinearEnvelopeGain4(output2, input, volume, delta, SampleCount);
            for (auto i = 0; i < SampleCount; ++i)
            {
                EXPECT_EQ(output0[i], output1[i]);
                EXPECT_EQ(output0[i], output2[i]);
            }
        }
    }
}

#if defined(__ARM_NEON__) || defined(__ARM_NEON)
namespace generic {

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/detail/audio_Volume.generic.h"

}  // namespace negeric

namespace neon {

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/detail/audio_Volume.neon.h"

}  // namespace neon

TEST(DspVolume, CompareGenericAndNeon)
{
    const int SampleCount = 1024;

    int32_t input[SampleCount];
    int32_t output0[SampleCount] = { 0 };
    int32_t output1[SampleCount] = { 0 };
    int32_t output2[SampleCount] = { 0 };
    int32_t output3[SampleCount] = { 0 };
    std::mt19937 mt(0);
    for (auto i = 0; i < SampleCount; ++i)
    {
        const auto a = std::numeric_limits<uint16_t>::max();
        const auto b = std::numeric_limits<int16_t>::min();
        input[i] = (mt() % a) + b;
        output0[i] = output1[i] = output2[i] = output3[i] = ( mt() % a) + b;
    }

    const int32_t volumes[] = { 32767, 12345, 0, -12345, -32768 };
    for (const auto volume : volumes)
    {
        generic::nn::audio::dsp::detail::ApplyUniformGain2(output0, input, volume, SampleCount);
        neon::nn::audio::dsp::detail::ApplyUniformGain2(output1, input, volume, SampleCount);
        generic::nn::audio::dsp::detail::ApplyUniformGain4(output2, input, volume, SampleCount);
        neon::nn::audio::dsp::detail::ApplyUniformGain4(output3, input, volume, SampleCount);
        for (auto i = 0; i < SampleCount; ++i)
        {
            EXPECT_EQ(output0[i], output1[i]);
            EXPECT_EQ(output2[i], output3[i]);
        }
    }
}

TEST(DspVolumeRamp, CompareGenericAndNeon)
{
    const int SampleCount = 1024;

    int32_t input[SampleCount];
    int32_t output0[SampleCount] = { 0 };
    int32_t output1[SampleCount] = { 0 };
    int32_t output2[SampleCount] = { 0 };
    int32_t output3[SampleCount] = { 0 };

    std::mt19937 mt(0);
    for (auto i = 0; i < SampleCount; ++i)
    {
        const auto a = std::numeric_limits<uint16_t>::max();
        const auto b = std::numeric_limits<int16_t>::min();
        input[i] = (mt() % a) + b;
        output0[i] = output1[i] = output2[i] = output3[i] = ( mt() % a) + b;
    }

    const int32_t volumes[] = { 32767, 12345, 0, -12345, -32768 };
    const int32_t deltas[] = { -10, -1, 0, 1, 10 };
    for (const auto volume : volumes)
    {
        for (const auto delta : deltas)
        {
            generic::nn::audio::dsp::detail::ApplyLinearEnvelopeGain2(output0, input, volume, delta, SampleCount);
            neon::nn::audio::dsp::detail::ApplyLinearEnvelopeGain2(output1, input, volume, delta, SampleCount);
            generic::nn::audio::dsp::detail::ApplyLinearEnvelopeGain4(output2, input, volume, delta, SampleCount);
            neon::nn::audio::dsp::detail::ApplyLinearEnvelopeGain4(output3, input, volume, delta, SampleCount);
            for (auto i = 0; i < SampleCount; ++i)
            {
                EXPECT_EQ(output0[i], output1[i]);
                EXPECT_EQ(output2[i], output3[i]);
            }
        }
    }
}
#endif  // defined(__ARM_NEON__) || defined(__ARM_NEON)
