﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>     // std::make_unique
#include <cstring>    // std::memcmp
#include <cstdlib>    // std::aligned_alloc
#include <algorithm>  // std::max
#include <limits>     // std::numeric_limits
#include <utility>    // std::tuple
#include <vector>     // std::vector

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "testAudio_SimpleAudioRenderer.h"

namespace {

nn::audio::AudioRendererParameter GetDefaultParameter() NN_NOEXCEPT
{
    nn::audio::AudioRendererParameter parameter;

    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.mixBufferCount = 16;
    parameter.subMixCount = 1;
    parameter.voiceCount = 24;

    return parameter;
}

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;
    nn::audio::AudioRendererParameter m_Parameter;

    nn::mem::StandardAllocator m_Allocator;
    static const size_t AllocatorMemorySize = 2 * 1024 * 1024;
    void* m_AllocatorMemory;

public:
    NN_IMPLICIT ScopedAudioRenderer(const nn::audio::AudioRendererParameter& parameter = GetDefaultParameter(), nn::os::SystemEvent* pSystemEvent = nullptr)
        : m_Parameter(parameter)
    {
        m_AllocatorMemory = std::malloc(AllocatorMemorySize);
        EXPECT_TRUE(m_AllocatorMemory != nullptr);
        m_Allocator.Initialize(m_AllocatorMemory, AllocatorMemorySize);

        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

        if (pSystemEvent)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, pSystemEvent, parameter, m_WorkBuffer, workBufferSize));
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));
        }

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
    }

    ~ScopedAudioRenderer()
    {
        if(nn::audio::GetAudioRendererState(m_Handle) == nn::audio::AudioRendererState_Started)
        {
            nn::audio::StopAudioRenderer(m_Handle);
        }
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);
        m_Allocator.Finalize();

        std::free(m_AllocatorMemory);
    }

    nn::audio::AudioRendererHandle GetHandle() const
    {
        return m_Handle;
    }

    nn::audio::AudioRendererConfig& GetConfig()
    {
        return m_Config;
    }

    nn::audio::AudioRendererParameter GetParameter() const
    {
        return m_Parameter;
    }
};

} // namespace anonymous

TEST(AcquireVoiceSlot, SingleAcquireAndRelease)
{
    auto parameter = GetDefaultParameter();
    parameter.voiceCount = nn::audio::VoiceType::ChannelCountMax;

    ScopedAudioRenderer sar(parameter);

    nn::audio::VoiceType voice;

    const int channelCounts[]                     = {1, nn::audio::VoiceType::ChannelCountMax / 2, nn::audio::VoiceType::ChannelCountMax};
    const int sampleRates[]                       = {1, 32000, 48000, 96000, std::numeric_limits<int>::max()};
    const int priorities[]                        = {nn::audio::VoiceType::PriorityLowest, (nn::audio::VoiceType::PriorityLowest - nn::audio::VoiceType::PriorityHighest) / 2, nn::audio::VoiceType::PriorityHighest};
    const nn::audio::SampleFormat sampleFormats[] = { nn::audio::SampleFormat_PcmInt16, nn::audio::SampleFormat_Adpcm };

    for(const auto channelCount : channelCounts)
    {
        for(const auto sampleRate : sampleRates)
        {
            for(const auto priority : priorities)
            {
                for(const auto sampleFormat : sampleFormats)
                {
                    NN_AUDIO_ALIGNAS_BUFFER_ALIGN nn::audio::AdpcmContext adpcmContext;
                    void* pParameter = nullptr;
                    size_t parameterSize = 0u;

                    if(sampleFormat == nn::audio::SampleFormat_Adpcm)
                    {
                        pParameter = &adpcmContext;
                        parameterSize = sizeof(adpcmContext);
                    }

                    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, sampleRate, channelCount, nn::audio::SampleFormat_PcmInt16, priority, pParameter, parameterSize));
                    EXPECT_TRUE(nn::audio::IsVoiceValid(&voice));
                    nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice);
                    EXPECT_TRUE(!nn::audio::IsVoiceValid(&voice));
                }
            }
        }
    }
}

TEST(AcquireVoiceSlot, MultiAcquire)
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    parameter.voiceCount = 1;

    ScopedAudioRenderer sar(parameter);

    nn::audio::VoiceType voice[3];
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice[0], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_FALSE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice[1], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_FALSE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice[2], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0));
    EXPECT_TRUE(nn::audio::IsVoiceValid(&voice[0]));
    EXPECT_FALSE(nn::audio::IsVoiceValid(&voice[1]));
    EXPECT_FALSE(nn::audio::IsVoiceValid(&voice[2]));
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AcquireVoiceSlot, PreCondition)
{
    const auto parameter = GetDefaultParameter();
    ScopedAudioRenderer sar(parameter);

    nn::audio::VoiceType voice;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(nullptr, &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), nullptr, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 0, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest - 1, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest + 1, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_Invalid, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, nn::audio::VoiceType::ChannelCountMax + 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0, nullptr), "");

    const nn::audio::VoiceType::BehaviorOptions options = { false, true }; // isPitchAndSrcSkipped enabled
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, parameter.sampleRate + 1, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0, &options), "");

    // These sample formats are not supported for now.
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt8, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt24, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt32, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmFloat, nn::audio::VoiceType::PriorityHighest, nullptr, 0), "");

}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ReleaseVoiceSlot, PreCondition)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseVoiceSlot(nullptr, &voice), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(IsVoiceValid, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsVoiceValid(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetVoiceSampleRate, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    int sampleRate = 48000;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, sampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_TRUE(nn::audio::GetVoiceSampleRate(&voice) == sampleRate);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoiceSampleRate, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceSampleRate(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetVoiceSampleFormat, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    nn::audio::SampleFormat sampleFormat = nn::audio::SampleFormat_PcmInt16;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, sampleFormat, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_TRUE(nn::audio::GetVoiceSampleFormat(&voice) == sampleFormat);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoiceSampleFormat, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceSampleFormat(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetVoicePriority, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    int priority = 64;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, priority, nullptr, 0));
    EXPECT_TRUE(nn::audio::GetVoicePriority(&voice) == priority);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoicePriority, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoicePriority(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoicePlayState, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoicePlayState(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoicePlayState, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    nn::audio::VoiceType::PlayState playState = nn::audio::VoiceType::PlayState_Play;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::SetVoicePlayState(&voice, playState);
    EXPECT_TRUE(nn::audio::GetVoicePlayState(&voice) == playState);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetVoicePlayState, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoicePlayState(nullptr, nn::audio::VoiceType::PlayState_Play), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoicePitch, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoicePitch(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoicePitch, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::SetVoicePitch(&voice, nn::audio::VoiceType::GetPitchMin());
    EXPECT_TRUE(nn::audio::GetVoicePitch(&voice) == nn::audio::VoiceType::GetPitchMin());
    nn::audio::SetVoicePitch(&voice, nn::audio::VoiceType::GetPitchMax());
    EXPECT_TRUE(nn::audio::GetVoicePitch(&voice) == nn::audio::VoiceType::GetPitchMax());
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetVoicePitch, PreCondition)
{
    {
        ScopedAudioRenderer sar;
        nn::audio::VoiceType voice;
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoicePitch(nullptr, 1.0f), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoicePitch(&voice, nn::audio::VoiceType::GetPitchMin() - 1e-3f), "");
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoicePitch(&voice, nn::audio::VoiceType::GetPitchMax() + 1e-3f), "");
    }

    {
        ScopedAudioRenderer sar;
        nn::audio::VoiceType voice;
        const nn::audio::VoiceType::BehaviorOptions options = { false, true }; // isPitchAndSrcSkipped enabled
        ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, sar.GetParameter().sampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0, &options));
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoicePitch(&voice, nn::audio::VoiceType::GetPitchMin()), "");
    }
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoiceVolume, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceVolume(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoiceVolume, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::SetVoiceVolume(&voice, nn::audio::VoiceType::GetVolumeMin());
    EXPECT_TRUE(nn::audio::GetVoiceVolume(&voice) == nn::audio::VoiceType::GetVolumeMin());
    nn::audio::SetVoiceVolume(&voice, nn::audio::VoiceType::GetVolumeMax());
    EXPECT_TRUE(nn::audio::GetVoiceVolume(&voice) == nn::audio::VoiceType::GetVolumeMax());
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetVoiceVolume, PreCondition)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceVolume(nullptr, 1.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceVolume(&voice, nn::audio::VoiceType::GetVolumeMin() - 1e-3f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceVolume(&voice, nn::audio::VoiceType::GetVolumeMax() + 1e-3f), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoiceVolume, OutputWaveformCheck)
{
    const std::size_t size = 128 * 1024 * 1024;
    void* buffer = std::malloc(size);
    ASSERT_TRUE(buffer != nullptr);

    {
        SimpleAudioRenderer renderer(buffer, size);

        // preparation: prepare constant value short waveform
        const auto InputValue = std::numeric_limits<int16_t>::max();
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t input[nn::audio::MemoryPoolType::SizeGranularity / sizeof(int16_t)];
        for (auto i = 0; i < static_cast<int>(sizeof(input) / sizeof(input[0])); ++i)
        {
            input[i] = InputValue;
        }
        nn::audio::MemoryPoolType dataMemoryPool;
        EXPECT_TRUE(nn::audio::AcquireMemoryPool(&renderer.GetConfig(), &dataMemoryPool, input, nn::util::align_up(sizeof(input), nn::audio::MemoryPoolType::SizeGranularity)));
        EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&dataMemoryPool));

        // testcase: Setting VoiceVolume to 0.0f
        // expected: Final output is zero
        {
            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetVoiceVolume(0.0f);

            const auto LoopCount = 64;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    EXPECT_TRUE(data[i] == 0);
                }
            }
        }

        // testcase: Setting VoiceVolume to 0.5f
        // expected: max of Final output is in (InputValue - Delta, InputValue + Delta)
        {
//             const auto Delta = 32;

            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetVoiceVolume(0.5f);

            int16_t max = 0;
            const auto LoopCount = 128;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    max = std::max(max, data[i]);
                }
            }
//             EXPECT_TRUE(max >= std::numeric_limits<int16_t>::max() / 2 - Delta);
//             EXPECT_TRUE(max <= std::numeric_limits<int16_t>::max() / 2 + Delta);
            NN_LOG("max = %d\n", max);
        }
    }

    std::free(buffer);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoiceMixVolume, PreCondition)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, nn::audio::VoiceType::ChannelCountMax, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 6));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceMixVolume(nullptr, &finalMix, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceMixVolume(&voice, static_cast<nn::audio::FinalMixType*>(nullptr), 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceMixVolume(&voice, &finalMix, -1, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceMixVolume(&voice, &finalMix, nn::audio::GetVoiceChannelCount(&voice), 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceMixVolume(&voice, &finalMix, 0, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceMixVolume(&voice, &finalMix, 0, nn::audio::GetFinalMixBufferCount(&finalMix)), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoiceMixVolume, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, nn::audio::VoiceType::ChannelCountMax, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::SubMixType subMix;
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 6));


    // destination が SubMix の場合
     nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &subMix);
    nn::audio::SetVoiceMixVolume(&voice, &subMix, nn::audio::VoiceType::GetVolumeMin(), 0, 0);
    EXPECT_TRUE(nn::audio::GetVoiceMixVolume(&voice, &subMix, 0, 0) == nn::audio::VoiceType::GetVolumeMin());
    nn::audio::SetVoiceMixVolume(&voice, &subMix, nn::audio::VoiceType::GetVolumeMax(), nn::audio::GetVoiceChannelCount(&voice) - 1, 0);
    EXPECT_TRUE(nn::audio::GetVoiceMixVolume(&voice, &subMix, nn::audio::GetVoiceChannelCount(&voice) - 1, 0) == nn::audio::VoiceType::GetVolumeMax());

    // destination が FinalMix の場合
     nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, nn::audio::VoiceType::GetVolumeMin(), 0, 0);
    EXPECT_TRUE(nn::audio::GetVoiceMixVolume(&voice, &finalMix, 0, 0) == nn::audio::VoiceType::GetVolumeMin());
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, nn::audio::VoiceType::GetVolumeMax(), 0, nn::audio::GetVoiceChannelCount(&voice) - 1);
    EXPECT_TRUE(nn::audio::GetVoiceMixVolume(&voice, &finalMix, 0, nn::audio::GetVoiceChannelCount(&voice) - 1) == nn::audio::VoiceType::GetVolumeMax());
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetVoiceMixVolume, PreCondition)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 6));

    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(nullptr, &finalMix, 1.0f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, static_cast<nn::audio::FinalMixType*>(nullptr), 1.0f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, &finalMix, nn::audio::VoiceType::GetVolumeMin() - 1e-3f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, &finalMix, nn::audio::VoiceType::GetVolumeMax() + 1e-3f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, -1, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, nn::audio::GetFinalMixBufferCount(&finalMix)), "");
    nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice);

    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, nn::audio::VoiceType::ChannelCountMax, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, -1, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, nn::audio::GetVoiceChannelCount(&voice), 0), "");
    nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice);
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoiceMixVolume, OutputWaveformCheck)
{
    const std::size_t size = 128 * 1024 * 1024;
    void* buffer = std::malloc(size);
    ASSERT_TRUE(buffer != nullptr);

    {
        SimpleAudioRenderer renderer(buffer, size);

        // preparation: prepare constant value short waveform
        const auto InputValue = std::numeric_limits<int16_t>::max();
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t input[nn::audio::MemoryPoolType::SizeGranularity / sizeof(int16_t)];
        for (auto i = 0; i < static_cast<int>(sizeof(input) / sizeof(input[0])); ++i)
        {
            input[i] = InputValue;
        }
        nn::audio::MemoryPoolType dataMemoryPool;
        EXPECT_TRUE(nn::audio::AcquireMemoryPool(&renderer.GetConfig(), &dataMemoryPool, input, nn::util::align_up(sizeof(input), nn::audio::MemoryPoolType::SizeGranularity)));
        EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&dataMemoryPool));

        // testcase: Setting VoiceMixVolume to 0.0f
        // expected: Final output is zero
        {
            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetVoiceMixVolume(0.0f);

            const auto LoopCount = 64;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    EXPECT_TRUE(data[i] == 0);
                }
            }
        }

        // testcase: Setting VoiceMixVolume to 0.5f
        // expected: max of Final output is in (InputValue - Delta, InputValue + Delta)
        {
//             const auto Delta = 64;

            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetVoiceMixVolume(0.5f);

            int16_t max = 0;
            const auto LoopCount = 128;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    max = std::max(max, data[i]);
                }
            }
//             EXPECT_TRUE(max >= std::numeric_limits<int16_t>::max() / 2 - Delta);
//             EXPECT_TRUE(max <= std::numeric_limits<int16_t>::max() / 2 + Delta);
            NN_LOG("max = %d\n", max);
        }
    }

    std::free(buffer);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoiceBiquadFilterParameter, PreCondition)
{
    ScopedAudioRenderer sar;
    nn::audio::VoiceType voice;
    ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceBiquadFilterParameter(nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceBiquadFilterParameter(&voice, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceBiquadFilterParameter(&voice, nn::audio::VoiceType::BiquadFilterCountMax), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoiceBiquadFilterParameter, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::BiquadFilterParameter biquadFilterParameter = { false, { 1, 2, 3 }, { 4, 5 } };
    nn::audio::SetVoiceBiquadFilterParameter(&voice, 1, biquadFilterParameter);
    auto obtained = nn::audio::GetVoiceBiquadFilterParameter(&voice, 1);
    EXPECT_EQ(obtained.enable, biquadFilterParameter.enable);
    EXPECT_EQ(obtained.numerator[0], biquadFilterParameter.numerator[0]);
    EXPECT_EQ(obtained.numerator[1], biquadFilterParameter.numerator[1]);
    EXPECT_EQ(obtained.numerator[2], biquadFilterParameter.numerator[2]);
    EXPECT_EQ(obtained.denominator[0], biquadFilterParameter.denominator[0]);
    EXPECT_EQ(obtained.denominator[1], biquadFilterParameter.denominator[1]);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetVoiceBiquadFilterParameter, PreCondition)
{
    ScopedAudioRenderer sar;
    nn::audio::VoiceType voice;
    ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));

    nn::audio::BiquadFilterParameter biquadFilterParameter;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceBiquadFilterParameter(nullptr, 0, biquadFilterParameter), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceBiquadFilterParameter(&voice, -1, biquadFilterParameter), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceBiquadFilterParameter(&voice, nn::audio::VoiceType::BiquadFilterCountMax, biquadFilterParameter), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(AppendAndGetReleasedWaveBuffer, Success)
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    nn::os::SystemEvent systemEvent;

    ScopedAudioRenderer sar(parameter, &systemEvent);

    nn::audio::VoiceType voice;
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
    nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    nn::audio::WaveBuffer waveBuffer;
    waveBuffer.buffer = pcmBuffer;
    waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
    waveBuffer.loop = 0;
    waveBuffer.isEndOfStream = false;


    EXPECT_TRUE(nn::audio::AppendWaveBuffer(&voice, &waveBuffer));
    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == nullptr);

    systemEvent.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    // TODO: Sometimes tests fail, probably due to lack of "wait"
    //       For now I increase the number of waits from 2 to 10
//     // First signal comes before processing, second one comes after
//     for (auto i = 0; i < 2; ++i)
    for (auto i = 0; i < 10; ++i)
    {
        systemEvent.Wait();
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == &waveBuffer);
    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == nullptr);

    EXPECT_TRUE(nn::audio::GetVoicePlayedSampleCount(&voice) == waveBuffer.endSampleOffset);

    waveBuffer.isEndOfStream = true;
    EXPECT_TRUE(nn::audio::AppendWaveBuffer(&voice, &waveBuffer));

    systemEvent.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    // First signal comes before processing, second one comes after
    for (auto i = 0; i < 2; ++i)
    {
        systemEvent.Wait();
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == &waveBuffer);
    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == nullptr);

    EXPECT_TRUE(nn::audio::GetVoicePlayedSampleCount(&voice) == 0);

    //// "Stop" ステートの確認
    // Append new WaveBuffers
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    waveBuffer.isEndOfStream = false;
    for (auto i = 0; i < nn::audio::VoiceType::WaveBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendWaveBuffer(&voice, &waveBuffer));
    }
    systemEvent.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    // set "Stop"
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Stop);
    systemEvent.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    systemEvent.Wait(); // 最低でも一度 GenerateCommand() が完了するのを待つ。
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    // Stop にすると有効な WaveBuffer は空になる。
    EXPECT_TRUE(nn::audio::GetWaveBufferCount(&voice) == 0);

    // 突っ込んだ WaveBuffer はすべて回収される。
    for (auto i = 0; i < nn::audio::VoiceType::WaveBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == &waveBuffer);
    }
    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == nullptr);

    // PlayedSampleCount はクリアされる。
    EXPECT_TRUE(nn::audio::GetVoicePlayedSampleCount(&voice) == 0);


    // Voice が PlayState_Stop の状態のまま、再度 WaveBuffer を入れる。
    EXPECT_TRUE(nn::audio::AppendWaveBuffer(&voice, &waveBuffer));

    systemEvent.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    for (auto i = 0; i < 10; ++i)
    {
        systemEvent.Wait();
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    // Stop 状態を維持しており、Stop に変更したわけではないので WaveBuffer は維持される。
    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) == nullptr);
    EXPECT_TRUE(nn::audio::GetWaveBufferCount(&voice) == 1);

    //// "Pause" ステートの確認
    for (auto i = 0; i < nn::audio::VoiceType::WaveBufferCountMax - 1; ++i) // already one WaveBuffer is added.
    {
        EXPECT_TRUE(nn::audio::AppendWaveBuffer(&voice, &waveBuffer));
    }

    // 再度 Play にして、問題なく再生されるか確認。
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    // すべてではない、いくらかの WaveBuffer が再生完了するまで待つ。
    for (auto i = 0; i < 2; ++i)
    {
        systemEvent.Wait();
    }
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Pause);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    systemEvent.Wait(); // 完全に一時停止するまで待つ。
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    auto prevWaveBufferCount = nn::audio::GetWaveBufferCount(&voice);
    auto prevPlayedSampleCount = nn::audio::GetVoicePlayedSampleCount(&voice);

    EXPECT_GT(prevPlayedSampleCount, 0);

    for (auto i = 0; i < 10; ++i)
    {
        systemEvent.Wait();
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), prevWaveBufferCount);
    EXPECT_EQ(nn::audio::GetVoicePlayedSampleCount(&voice), prevPlayedSampleCount);

    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    systemEvent.Wait();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    for (auto i = 0; i < 10; ++i)
    {
        systemEvent.Wait();
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

    EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), 0);
    for (auto i = 0; i < nn::audio::VoiceType::WaveBufferCountMax; ++i)
    {
        EXPECT_EQ(nn::audio::GetReleasedWaveBuffer(&voice), &waveBuffer);
    }
    EXPECT_EQ(nn::audio::GetReleasedWaveBuffer(&voice), nullptr);

    EXPECT_EQ(nn::audio::GetVoicePlayedSampleCount(&voice), waveBuffer.endSampleOffset * 4);
} // NOLINT(readability/fn_size)

#if !defined(NN_SDK_BUILD_RELEASE)
namespace {

void DeathTestHelperForAppendWaveBuffer(nn::audio::WaveBuffer* pWaveBuffer, nn::audio::SampleFormat sampleFormat = nn::audio::SampleFormat_PcmInt16, int channelCount = 1)
{
    ScopedAudioRenderer sar;
    nn::audio::VoiceType voice;
    NN_AUDIO_ALIGNAS_BUFFER_ALIGN nn::audio::AdpcmContext adpcmContext;

    void* pContext = nullptr;
    size_t contextSize = 0u;

    if(sampleFormat == nn::audio::SampleFormat_Adpcm)
    {
        pContext = &adpcmContext;
        contextSize = sizeof(adpcmContext);
    }

    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, channelCount, sampleFormat, nn::audio::VoiceType::PriorityHighest, pContext, contextSize));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AppendWaveBuffer(&voice, pWaveBuffer), "");
}

} // anonymous namespace

TEST(AppendWaveBuffer, PreCondition)
{
    auto buffer = std::make_unique<int8_t[]>(nn::audio::MemoryPoolType::SizeGranularity + nn::audio::MemoryPoolType::AddressAlignment - 1);
    auto pcmBuffer = reinterpret_cast<int16_t*>(nn::util::align_up(reinterpret_cast<uintptr_t>(buffer.get()), nn::audio::MemoryPoolType::AddressAlignment));

    const nn::audio::WaveBuffer defaultWaveBuffer = {
        reinterpret_cast<void*>(pcmBuffer),
        sizeof(pcmBuffer[0]),
        0,
        static_cast<int32_t>(sizeof(pcmBuffer) / sizeof(pcmBuffer[0])),
        0,
        false
    };

    // pVoice is nullptr.
    {
        nn::audio::WaveBuffer waveBuffer;
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::AppendWaveBuffer(nullptr, &waveBuffer), "");
    }

    // pVoice is uninitialized.
    {
        nn::audio::VoiceType uninitializedVoice = { 0 };
        nn::audio::WaveBuffer waveBuffer;
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::AppendWaveBuffer(&uninitializedVoice, &waveBuffer), "");
    }

    // pWaveBuffer is nullptr.
    DeathTestHelperForAppendWaveBuffer(nullptr);

    // Unaligned buffer.
    {
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        waveBuffer.buffer = reinterpret_cast<void*>(nn::audio::BufferAlignSize + 1);
        DeathTestHelperForAppendWaveBuffer(&waveBuffer);
    }

    // Unaligned pContext.
    {
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        waveBuffer.pContext = reinterpret_cast<void*>(nn::audio::BufferAlignSize + 1);
        waveBuffer.contextSize = nn::audio::BufferAlignSize;
        DeathTestHelperForAppendWaveBuffer(&waveBuffer);
    }

    // contextSize is 0 when voice is created for ADPCM format and pContext is not nullptr.
    {
        NN_AUDIO_ALIGNAS_BUFFER_ALIGN nn::audio::AdpcmContext context;
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        waveBuffer.pContext = &context;
        waveBuffer.contextSize = 0;
        DeathTestHelperForAppendWaveBuffer(&waveBuffer);
    }

    // startSampleOffset is negative.
    {
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        waveBuffer.startSampleOffset = -1;
        DeathTestHelperForAppendWaveBuffer(&waveBuffer);
    }

    // endSampleOffset is negative.
    {
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        waveBuffer.endSampleOffset = -1;
        DeathTestHelperForAppendWaveBuffer(&waveBuffer);
    }

    // startSampleOffset < endSampleOffset.
    {
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        waveBuffer.startSampleOffset = 1;
        waveBuffer.endSampleOffset = 0;
        DeathTestHelperForAppendWaveBuffer(&waveBuffer);
    }

    // Insufficient buffer size. (PcmInt16 format, 2 ch)
    {
        const auto channelCount = 2;
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(int16_t) / channelCount + 1);
        DeathTestHelperForAppendWaveBuffer(&waveBuffer, nn::audio::SampleFormat_PcmInt16, channelCount);
    }

    // Insufficient buffer size. (ADPCM format, 1 ch)
    {
        NN_AUDIO_ALIGNAS_BUFFER_ALIGN nn::audio::AdpcmContext context;
        nn::audio::WaveBuffer waveBuffer = defaultWaveBuffer;
        const auto adpcmFrameCount = nn::util::align_up(waveBuffer.size, nn::audio::AdpcmFrameSize);
        const auto sampleCount = static_cast<int32_t>(adpcmFrameCount * nn::audio::AdpcmFrameSampleCount);
        waveBuffer.endSampleOffset = sampleCount + 1;
        waveBuffer.pContext = &context;
        waveBuffer.contextSize = sizeof(context);
        DeathTestHelperForAppendWaveBuffer(&waveBuffer);
    }
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(AppendWaveBuffer, RegressionTestForInvalidWaveBuffer) // SIGLO-77512
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();

    const bool isEndOfStreamFlags[] = { false, true };
    const bool loopFlags[] = { false, true };
    const nn::audio::SampleFormat formats[] = { nn::audio::SampleFormat_PcmInt16, nn::audio::SampleFormat_Adpcm };

    for(const auto isEndOfStreamFlag : isEndOfStreamFlags)
    {
        for(const auto loopFlag : loopFlags)
        {
            for(const auto format : formats)
            {
                nn::os::SystemEvent systemEvent;
                ScopedAudioRenderer sar(parameter, &systemEvent);

                nn::audio::VoiceType voice;
                nn::audio::FinalMixType finalMix;
                nn::audio::MemoryPoolType memoryPool;
                NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

                // 波形用バッファで代用
                auto pParameter = (format == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
                auto parameterSize = (format == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmParameter);
                auto pContext = (format == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
                auto contextSize = (format == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmContext);

                EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, format, nn::audio::VoiceType::PriorityHighest, pParameter, parameterSize));
                EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
                nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

                nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);


                ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
                ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

                nn::audio::WaveBuffer waveBuffer;
                waveBuffer.buffer = pcmBuffer;
                waveBuffer.size = 0; // Set 0 to size.
                waveBuffer.startSampleOffset = 0;
                waveBuffer.endSampleOffset = 0;
                waveBuffer.loop = loopFlag;
                waveBuffer.isEndOfStream = isEndOfStreamFlag;
                waveBuffer.pContext = pContext;
                waveBuffer.contextSize = contextSize;

                // Append WaveBuffer
                nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

                // Wait for 5 audio frames
                for (auto j = 0; j < 5; ++j)
                {
                    systemEvent.Wait();
                    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                }

                if(!loopFlag)
                {
                    EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voice) != nullptr);
                }
            }
        }
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetReleasedWaveBuffer, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetReleasedWaveBuffer(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoicePlayedSampleCount, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoicePlayedSampleCount(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetVoicePlayedSampleCount, Success)
{
    // isPlayedSampleCountResetAtLoopPoint, isOldStyleApiUsed, SampleFormat
    const std::vector< std::tuple<bool, bool, nn::audio::SampleFormat> > testParameters = {
        std::make_tuple(false, false, nn::audio::SampleFormat_PcmInt16),
        std::make_tuple(false, false, nn::audio::SampleFormat_Adpcm),
        std::make_tuple(true,  false, nn::audio::SampleFormat_PcmInt16),
        std::make_tuple(true,  false, nn::audio::SampleFormat_Adpcm),
        std::make_tuple(false, true,  nn::audio::SampleFormat_PcmInt16),
        std::make_tuple(false, true,  nn::audio::SampleFormat_Adpcm)
    };

    for(const auto testParameter : testParameters)
    {
        const auto isPlayedSampleCountResetAtLoopPoint = std::get<0>(testParameter);
        const auto isOldStyleApiUsed = std::get<1>(testParameter);
        const auto format = std::get<2>(testParameter);

        nn::os::SystemEvent systemEvent;
        ScopedAudioRenderer sar(GetDefaultParameter(), &systemEvent);

        nn::audio::VoiceType voice;
        nn::audio::FinalMixType finalMix;
        nn::audio::MemoryPoolType memoryPool;
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

        // 波形用バッファで代用
        auto pParameter = (format == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
        auto parameterSize = (format == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmParameter);
        auto pContext = (format == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
        auto contextSize = (format == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmContext);

        if(isOldStyleApiUsed)
        {
            EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, format, nn::audio::VoiceType::PriorityHighest, pParameter, parameterSize));
        }
        else
        {
            const nn::audio::VoiceType::BehaviorOptions behaviorOptions = { isPlayedSampleCountResetAtLoopPoint, false };
            EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, format, nn::audio::VoiceType::PriorityHighest, pParameter, parameterSize, &behaviorOptions));
        }

        EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);


        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = sizeof(pcmBuffer);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = 1;
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;
        waveBuffer.pContext = pContext;
        waveBuffer.contextSize = contextSize;

        // Append WaveBuffer
        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

        // Wait for 5 audio frames
        for (auto j = 0; j < 5; ++j)
        {
            systemEvent.Wait();
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        }

        const auto playedSampleCount = nn::audio::GetVoicePlayedSampleCount(&voice);

        if(isOldStyleApiUsed || isPlayedSampleCountResetAtLoopPoint == false)
        {
            EXPECT_GT(playedSampleCount, waveBuffer.endSampleOffset);
        }
        else
        {
            EXPECT_LE(playedSampleCount, waveBuffer.endSampleOffset);
        }
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetVoiceDestination, Precondition)
{
    // MEMO: 現状は Mix のサンプルレート == オーディオレンダラのレンダリングサンプルレートなので、VoiceType::BehaviorOptions.isPitchAndSrcSkipped が有効な時の SetVoiceDestination() の事前条件テストができない。
    // (AcquireVoiceSlot() の方の事前条件に先に引っかかる)

    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    nn::audio::SubMixType subMix;
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Destination が SubMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceDestination(nullptr, &voice, &subMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceDestination(&sar.GetConfig(), nullptr, &subMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, static_cast<nn::audio::SubMixType*>(nullptr)), "");

    // Destination が FinalMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceDestination(nullptr, &voice, &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceDestination(&sar.GetConfig(), nullptr, &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, static_cast<nn::audio::FinalMixType*>(nullptr)), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoiceDestination, Success)
{
    ScopedAudioRenderer sar;

    nn::audio::VoiceType voice;
    nn::audio::SubMixType subMix;
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &subMix);
    nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);
}

TEST(GetVoiceNodeId, Success)
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    parameter.voiceCount = 2;

    ScopedAudioRenderer sar(parameter);

    nn::audio::VoiceType voice[2];
    for (int32_t i = 0; i <= std::numeric_limits<uint16_t>::max(); ++i)
    {
        ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice[0], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice[1], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        ASSERT_EQ(nn::audio::GetVoiceNodeId(&voice[0]), 0x10000000 + i );
        ASSERT_EQ(nn::audio::GetVoiceNodeId(&voice[1]), 0x10010000 + i );
        nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice[0]);
        nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice[1]);
    }

    {
        ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice[0], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice[1], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        ASSERT_EQ(nn::audio::GetVoiceNodeId(&voice[0]), 0x10000000); // variation が一周してリセットされていることの確認
        ASSERT_EQ(nn::audio::GetVoiceNodeId(&voice[1]), 0x10010000);
        nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice[0]);
        nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice[1]);
    }

}

TEST(GetWaveBufferCount, Success)
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    nn::os::SystemEvent systemEvent;

    ScopedAudioRenderer sar(parameter, &systemEvent);

    nn::audio::VoiceType voice;
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
    nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    nn::audio::WaveBuffer waveBuffer;
    waveBuffer.buffer = pcmBuffer;
    waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = 1;
    waveBuffer.loop = 0;
    waveBuffer.isEndOfStream = false;

    for(int i = 0; i < 100; ++i)
    {
        // Append 4 wavebuffers
        EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), 0);
        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), 1);
        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), 2);
        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), 3);
        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), 4);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

        // Wait for 20 audio frames
        int releasedWaveBufferCount = 0;
        for (auto j = 0; j < 20; ++j)
        {
            systemEvent.Wait();
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            if(nn::audio::GetReleasedWaveBuffer(&voice) != nullptr)
            {
                ++releasedWaveBufferCount;

                if(releasedWaveBufferCount == 4)
                {
                    break;
                }
            }
        }

        EXPECT_EQ(nn::audio::GetWaveBufferCount(&voice), 0);
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetVoicePriority, Precondition)
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();

    ScopedAudioRenderer sar(parameter);

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoicePriority(&voice, nn::audio::VoiceType::PriorityLowest + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetVoicePriority(&voice, nn::audio::VoiceType::PriorityHighest - 1), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetVoicePriority, Success)
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();

    ScopedAudioRenderer sar(parameter);

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_EQ(nn::audio::GetVoicePriority(&voice), nn::audio::VoiceType::PriorityHighest);

    nn::audio::SetVoicePriority(&voice, nn::audio::VoiceType::PriorityLowest);
    EXPECT_EQ(nn::audio::GetVoicePriority(&voice), nn::audio::VoiceType::PriorityLowest);

    nn::audio::SetVoicePriority(&voice, nn::audio::VoiceType::PriorityHighest);
    EXPECT_EQ(nn::audio::GetVoicePriority(&voice), nn::audio::VoiceType::PriorityHighest);
}

TEST(IsVoiceDropFlagOn, Success)
{
    ScopedAudioRenderer sar(GetDefaultParameter());

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));

    EXPECT_FALSE(nn::audio::IsVoiceDroppedFlagOn(&voice));
}

TEST(ResetVoiceDroppedFlag, Success)
{
    ScopedAudioRenderer sar(GetDefaultParameter());

    nn::audio::VoiceType voice;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));

    nn::audio::ResetVoiceDroppedFlag(&voice);
}

#if defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
TEST(VoiceDrop, Success)
{
    const auto voiceCount = 300;
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    parameter.voiceCount = voiceCount;
    parameter.performanceFrameCount = 10;
    parameter.isVoiceDropEnabled = true;
    nn::os::SystemEvent systemEvent;
    ScopedAudioRenderer sar(parameter, &systemEvent);

    // Setup FinalMix
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Setup MemoryPool
    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    // Setup Voice
    nn::audio::VoiceType voices[voiceCount];
    for(auto& voice : voices)
    {
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    }

    // Performance buffer settings
    const auto perfSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter);
    const int bufferCount = 2;
    void* perfBuffer[bufferCount] = { nullptr };
    void* prevBuffer = nullptr;
    int perfBufferIndex = 0;
    for (auto i = 0; i < bufferCount; ++i)
    {
        perfBuffer[i] = malloc(perfSize);
        ASSERT_NE(perfBuffer[i], nullptr);
    }
    prevBuffer = perfBuffer[perfBufferIndex];
    ASSERT_EQ(nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[perfBufferIndex], perfSize), nullptr);
    perfBufferIndex = (perfBufferIndex + 1) % bufferCount;

    // Wait for 5 audio frames...
    for(int i = 0; i < 5; ++i)
    {
        // Sync with AudioRenderer
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        systemEvent.Wait();

        // Check that IsRenderingTimeLimitExceeded() returns false when voice drop is enabled.
        auto consumedBuffer = nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[perfBufferIndex], perfSize);
        prevBuffer = perfBuffer[perfBufferIndex];
        perfBufferIndex = (perfBufferIndex + 1) % bufferCount;

        nn::audio::PerformanceInfo perfInfo;
        if(perfInfo.SetBuffer(consumedBuffer, perfSize))
        {
            do
            {
                EXPECT_FALSE(perfInfo.IsRenderingTimeLimitExceeded());
            } while(perfInfo.MoveToNextFrame());
        }
    }

    // Check the amount of dropped voice
    {
        auto voiceDropCount = 0;
        for(auto& voice : voices)
        {
            if(nn::audio::IsVoiceDroppedFlagOn(&voice))
            {
                ++voiceDropCount;
            }
        }

        EXPECT_GT(voiceDropCount, 0);
    }

    // Check FIFO voice drop policy
    {
        EXPECT_FALSE(nn::audio::IsVoiceDroppedFlagOn(&voices[voiceCount - 1]));
        EXPECT_TRUE(nn::audio::IsVoiceDroppedFlagOn(&voices[0]));
    }

    // Set NoDrop priority but this voice's voice drop flag is still on.
    nn::audio::SetVoicePriority(&voices[0], nn::audio::VoiceType::PriorityHighest);

    // Wait for 5 audio frames...
    for(int i = 0; i < 5; ++i)
    {
        // Sync with AudioRenderer
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        systemEvent.Wait();
    }

    // this voice's voice drop flag should be on because the flag hasn't been reset.
    EXPECT_TRUE(nn::audio::IsVoiceDroppedFlagOn(&voices[0]));

    // Reset and check a voice drop flag
    nn::audio::ResetVoiceDroppedFlag(&voices[0]);
    EXPECT_FALSE(nn::audio::IsVoiceDroppedFlagOn(&voices[0]));

    // Wait for 5 audio frames...
    for(int i = 0; i < 5; ++i)
    {
        // Sync with AudioRenderer
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        systemEvent.Wait();
    }

    // this voice's voice drop flag should be off because the flag has been reset and priority is no drop.
    EXPECT_FALSE(nn::audio::IsVoiceDroppedFlagOn(&voices[0]));

    for (auto i = 0; i < bufferCount; ++i)
    {
        if(perfBuffer[i])
        {
            free(perfBuffer[i]);
            perfBuffer[i] = nullptr;
        }
    }
}
#endif // #if defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)

// マルチチャンネル再生時に誤ったアサート条件に違反する (SIGLO-46584) 不具合のテスト
TEST(RegressionTestForInvalidAssertInMultiChannelVoiceProcessing, Success)
{
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    parameter.voiceCount = 1 + nn::audio::VoiceType::ChannelCountMax;

    nn::os::SystemEvent systemEvent;
    ScopedAudioRenderer sar(parameter, &systemEvent);

    // Setup FinalMix
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Setup MemoryPool
    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    nn::audio::VoiceType voice;
    nn::audio::WaveBuffer waveBuffer;

    const auto iterationCount = 10; // 適当な回数

    for(int i = 0; i < iterationCount; ++i)
    {
        // Play mono
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);

        // Wait for 5 audio frames (1 オーディオフレーム以上待てば良いはずだが念のため 5 フレーム分)
        for(int j = 0; j < 5; ++j)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();
        }

        // Release voice
        nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice);

        // Play multichannel (2 ch ～ nn::audio::VoiceType::ChannelCountMax ch)
        const auto channelCount = 2 + i % (nn::audio::VoiceType::ChannelCountMax - 1);
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0])) / channelCount;
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;

        // Wait for 5 audio frames (1 オーディオフレーム以上待てば良いはずだが念のため 5 フレーム分)
        for(int j = 0; j < 5; ++j)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();
        }

        // Release voice
        nn::audio::ReleaseVoiceSlot(&sar.GetConfig(), &voice);
    }
}

TEST(LargeSampleRate, Success)
{
    const auto voiceCount = 2;
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    parameter.voiceCount = voiceCount;
    parameter.isVoiceDropEnabled = true;
    nn::os::SystemEvent systemEvent;
    ScopedAudioRenderer sar(parameter, &systemEvent);

    // Setup FinalMix
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Setup MemoryPool
    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    // Setup Voice
    nn::audio::VoiceType voices[voiceCount];

    for(int i = 0; i < voiceCount; ++i)
    {
        auto& voice = voices[i];
        // Set large sample rate.
        const auto sampleRate = 48000 * 1000;
        const auto format = (i == 0) ? nn::audio::SampleFormat_PcmInt16 : nn::audio::SampleFormat_Adpcm;

        // 波形用バッファで代用
        auto pParameter = (format == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
        auto parameterSize = (format == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmParameter);
        auto pContext = (format == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
        auto contextSize = (format == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmContext);

        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, sampleRate, 1, format, nn::audio::VoiceType::PriorityLowest, pParameter, parameterSize));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;
        waveBuffer.pContext = pContext;
        waveBuffer.contextSize = contextSize;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    }

    // Wait for 5 audio frames...
    for(int i = 0; i < 5; ++i)
    {
        // Sync with AudioRenderer
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        systemEvent.Wait();
    }

}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(FlushWaveBuffers, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::FlushWaveBuffers(nullptr), "");
}
#endif

TEST(FlushWaveBuffers, Success)
{
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
    ScopedAudioRenderer sar(parameter, &systemEvent);

    // Setup FinalMix
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Setup MemoryPool
    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    nn::audio::SampleFormat sampleFormats[] = { nn::audio::SampleFormat_PcmInt16, nn::audio::SampleFormat_Adpcm };
    int channelCounts[] = { 1, 2, 6 };

    for(const auto sampleFormat : sampleFormats)
    {
        for(const auto channelCount : channelCounts)
        {
            // ADPCM voice supports only mono channel so skip a test.
            if(sampleFormat == nn::audio::SampleFormat_Adpcm && channelCount != 1)
            {
                continue;
            }

            nn::audio::VoiceType voice;
            const auto pParameter = (sampleFormat == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
            const auto parameterSize = (sampleFormat == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmParameter);
            EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, channelCount, sampleFormat, nn::audio::VoiceType::PriorityHighest, pParameter, parameterSize));

            // Setup WaveBuffers
            const auto waveBufferCount = nn::audio::VoiceType::WaveBufferCountMax;
            nn::audio::WaveBuffer waveBuffers[waveBufferCount];
            for(int i = 0; i < waveBufferCount; ++i)
            {
                auto& waveBuffer = waveBuffers[i];
                const auto pLoopContext = (sampleFormat == nn::audio::SampleFormat_PcmInt16) ? nullptr : pcmBuffer;
                const auto loopContextSize = (sampleFormat == nn::audio::SampleFormat_PcmInt16) ? 0 : sizeof(nn::audio::AdpcmContext);

                waveBuffer.buffer = pcmBuffer;
                waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
                waveBuffer.startSampleOffset = 0;
                waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]) / channelCount);
                waveBuffer.loop = true; // Loop
                waveBuffer.isEndOfStream = false;
                waveBuffer.pContext = pLoopContext;
                waveBuffer.contextSize = loopContextSize;
            }

            // Flush -> Append(4) -> Flush
            {
                // Flush WaveBuffers (Voice has not been appended any buffer.)
                nn::audio::FlushWaveBuffers(&voice);

                // Append WaveBuffers
                for(int i = 0; i < waveBufferCount; ++i)
                {
                    nn::audio::AppendWaveBuffer(&voice, &waveBuffers[i]);
                }

                // Set Voice's PlayState to Play
                nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

                // Wait for 5 audio frames...
                for(int i = 0; i < 5; ++i)
                {
                    // Sync with AudioRenderer
                    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                    systemEvent.Wait();
                }

                // Expect to return nullptr by nn::audio::GetReleasedWaveBuffer() because there is no flushed WaveBuffer and waveBuffer's loop is true.
                EXPECT_EQ(nn::audio::GetReleasedWaveBuffer(&voice), nullptr);

                // Flush WaveBuffers (Voice has been appended WaveBuffers)
                nn::audio::FlushWaveBuffers(&voice);

                // Wait for 5 audio frames...
                for(int i = 0; i < 5; ++i)
                {
                    // Sync with AudioRenderer
                    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                    systemEvent.Wait();
                }

                // Count up released WaveBuffers.
                {
                    int releasedWaveBufferCount = 0;

                    while(nn::audio::GetReleasedWaveBuffer(&voice) != nullptr)
                    {
                        ++releasedWaveBufferCount;
                    }

                    // Expect all appended WaveBuffer is released from Voice.
                    EXPECT_EQ(waveBufferCount, releasedWaveBufferCount);
                }
            }

            // Append(2) -> Flush -> Append(2) -> Flush
            {
                // Append WaveBuffers (2)
                for(int i = 0; i < 2; ++i)
                {
                    nn::audio::AppendWaveBuffer(&voice, &waveBuffers[i]);
                }

                // Flush WaveBuffers (Voice has been appended WaveBuffers(2))
                nn::audio::FlushWaveBuffers(&voice);

                // Re-Append WaveBuffers (2)
                for(int i = 0; i < 2; ++i)
                {
                    nn::audio::AppendWaveBuffer(&voice, &waveBuffers[i]);
                }

                // Wait for 5 audio frames...
                for(int i = 0; i < 5; ++i)
                {
                    // Sync with AudioRenderer
                    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                    systemEvent.Wait();
                }

                // Count up released WaveBuffers.
                {
                    int releasedWaveBufferCount = 0;

                    while(nn::audio::GetReleasedWaveBuffer(&voice) != nullptr)
                    {
                        ++releasedWaveBufferCount;
                    }

                    // Expect all appended WaveBuffer is released from Voice.
                    EXPECT_EQ(2, releasedWaveBufferCount);
                }

                // Flush WaveBuffers (Voice has been appended WaveBuffers(2))
                nn::audio::FlushWaveBuffers(&voice);

                // Wait for 5 audio frames...
                for(int i = 0; i < 5; ++i)
                {
                    // Sync with AudioRenderer
                    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                    systemEvent.Wait();
                }

                // Count up released WaveBuffers.
                {
                    int releasedWaveBufferCount = 0;

                    while(nn::audio::GetReleasedWaveBuffer(&voice) != nullptr)
                    {
                        ++releasedWaveBufferCount;
                    }

                    // Expect all appended WaveBuffer is released from Voice.
                    EXPECT_EQ(2, releasedWaveBufferCount);
                }
            }

            // Append(4) -> Flush -> RequestUpdateAudioRenderer() * 2
            {
                // Flush WaveBuffers (Voice has not been appended any buffer.)
                nn::audio::FlushWaveBuffers(&voice);

                // Append WaveBuffers
                for(int i = 0; i < waveBufferCount; ++i)
                {
                    nn::audio::AppendWaveBuffer(&voice, &waveBuffers[i]);
                }

                // Set Voice's PlayState to Play
                nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

                // Wait for 5 audio frames...
                for(int i = 0; i < 5; ++i)
                {
                    // Sync with AudioRenderer
                    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                    systemEvent.Wait();
                }

                // Expect to return nullptr by nn::audio::GetReleasedWaveBuffer() because there is no flushed WaveBuffer and waveBuffer's loop is true.
                EXPECT_EQ(nn::audio::GetReleasedWaveBuffer(&voice), nullptr);

                // Flush WaveBuffers (Voice has been appended WaveBuffers)
                nn::audio::FlushWaveBuffers(&voice);

                // Call RequestUpdateAudioRenderer() twice
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

                // Wait for 5 audio frames...
                for(int i = 0; i < 5; ++i)
                {
                    // Sync with AudioRenderer
                    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                    systemEvent.Wait();
                }

                // Count up released WaveBuffers.
                {
                    int releasedWaveBufferCount = 0;

                    while(nn::audio::GetReleasedWaveBuffer(&voice) != nullptr)
                    {
                        ++releasedWaveBufferCount;
                    }

                    // Expect all appended WaveBuffer is released from Voice.
                    EXPECT_EQ(waveBufferCount, releasedWaveBufferCount);
                }
            }
        }
    }

    nn::os::DestroySystemEvent(systemEvent.GetBase());
} // NOLINT(impl/function_size)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetVoiceBehaviorOptions, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetVoiceBehaviorOptions(nullptr), "");
}
#endif

TEST(GetVoiceBehaviorOptions, Success)
{
    const auto parameter = GetDefaultParameter();

    for(const auto isPlayedSampleCountResetAtLoopPoint : { true, false })
    {
        for(const auto isPitchAndSrcSkipped : { true, false })
        {
            const nn::audio::VoiceType::BehaviorOptions originalOptions = { isPlayedSampleCountResetAtLoopPoint, isPitchAndSrcSkipped };
            ScopedAudioRenderer sar(parameter);
            nn::audio::VoiceType voice;

            ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, parameter.sampleRate, 1, nn::audio::SampleFormat_PcmInt16, 0, nullptr, 0, &originalOptions));

            const auto options = nn::audio::GetVoiceBehaviorOptions(&voice);
            EXPECT_EQ(originalOptions.isPlayedSampleCountResetAtLoopPoint, options.isPlayedSampleCountResetAtLoopPoint);
            EXPECT_EQ(originalOptions.isPitchAndSrcSkipped, options.isPitchAndSrcSkipped);
        }
    }
}
