﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>    // std::memcmp
#include <cstdlib>    // std::aligned_alloc
#include <algorithm>  // std::max
#include <limits>     // std::numeric_limits
#include <memory>

#include <nnt.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_NodeAllocator.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_BitFlagSet.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#endif

namespace
{
struct StdAllocatorDeleter
{
    static void SetupDeleter(nn::mem::StandardAllocator* pAllocator)
    {
        m_Allocator = pAllocator;
    }
    static nn::mem::StandardAllocator* m_Allocator;
    void operator()(void* ptr) const
    {
        NN_ABORT_UNLESS_NOT_NULL(m_Allocator);
        m_Allocator->Free(ptr);
    }
};
nn::mem::StandardAllocator* StdAllocatorDeleter::m_Allocator = nullptr;
using StdUniquePtr = std::unique_ptr<void, StdAllocatorDeleter>;

const int L = nn::audio::ChannelMapping_FrontLeft;
const int R = nn::audio::ChannelMapping_FrontRight;
const int C = nn::audio::ChannelMapping_FrontCenter;
const int RL = nn::audio::ChannelMapping_RearLeft;
const int RR = nn::audio::ChannelMapping_RearRight;
const int LF = nn::audio::ChannelMapping_LowFrequency;

}


namespace {
void PrepareTestWaveBuffer(nn::audio::WaveBuffer& waveBuffer, void* buffer, size_t bufferSize, int16_t currentValue, int channelCount)
{
    auto buf = reinterpret_cast<int16_t*>(buffer);
    for (uint32_t i = 0; i < bufferSize / sizeof(int16_t); ++i)
    {
        buf[i] = currentValue;
    }
    waveBuffer.buffer = buf;
    waveBuffer.size = bufferSize;
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(buf[0]) / channelCount);
    waveBuffer.loop = 1;
    waveBuffer.isEndOfStream = false;
};
}

struct TestSource
{
    int index;
    nn::audio::WaveBuffer* waveBuffer;
};

struct ExpectedValue
{
    int nodeIndex;
    int channelIndex;
    int answer;
    int acceptableError;
};

const int NodeCountMax = 10;
const int ProbeCountMax = 10;

void SplitterTestBase(
    const nn::audio::AudioRendererParameter testConfigurationParam,
    const nnt::audio::util::NodeDescription nodeInfo[], int nodeCount,
    const nnt::audio::util::EdgeDescription edges[], int edgeCount,
    const TestSource sources[], int sourceCount,
    const ExpectedValue expects[], int expectsCount)
{
    NN_ABORT_UNLESS(nodeCount <= NodeCountMax);
    NN_ABORT_UNLESS(expectsCount <= ProbeCountMax);
    NN_ABORT_UNLESS(expectsCount <= testConfigurationParam.effectCount);

    const size_t workBufferSize = 10 * 1024 * 1024;
    auto workBuffer = malloc(workBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);
    auto alignedWorkBuffer = nn::util::BytePtr(workBuffer).AlignUp(nn::audio::MemoryPoolType::AddressAlignment).Get();
    nn::mem::StandardAllocator allocator(alignedWorkBuffer, workBufferSize - nn::audio::MemoryPoolType::SizeGranularity);
    StdAllocatorDeleter::SetupDeleter(&allocator);

    {
        nn::os::SystemEvent systemEvent;
        nn::audio::AudioRendererParameter param;
        nn::audio::InitializeAudioRendererParameter(&param);
        param.sampleRate = testConfigurationParam.sampleRate;
        param.sampleCount = testConfigurationParam.sampleCount;
        param.mixBufferCount = testConfigurationParam.mixBufferCount;
        param.subMixCount = testConfigurationParam.subMixCount;
        param.voiceCount = testConfigurationParam.voiceCount;
        param.sinkCount = testConfigurationParam.sinkCount;
        param.effectCount = testConfigurationParam.effectCount;
        param.splitterCount = testConfigurationParam.splitterCount;
        param.splitterSendChannelCount = testConfigurationParam.splitterSendChannelCount;
        param.performanceFrameCount = testConfigurationParam.performanceFrameCount;
        param.isVoiceDropEnabled = testConfigurationParam.isVoiceDropEnabled;

        nnt::audio::util::ScopedAudioRenderer sar(param, &systemEvent);
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::os::Event event(nn::os::EventClearMode_AutoClear);
        nn::audio::SetFastRenderingMode(true, &event);
        EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif
        StdUniquePtr voiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::Voice) * 10));
        StdUniquePtr multiDestVoiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationVoice) * 10));
        StdUniquePtr splitterBuffer(allocator.Allocate(sizeof(nnt::audio::util::Splitter) * 10));
        StdUniquePtr subMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::SubMix) * 10));
        StdUniquePtr multiDestSubMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationSubMix) * 10));
        StdUniquePtr finalMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::FinalMix) * 1));
        nnt::audio::util::AudioNodeAllocator nodeAllocator(
            voiceBuffer.get(), NodeCountMax,
            multiDestVoiceBuffer.get(), NodeCountMax,
            splitterBuffer.get(), NodeCountMax,
            subMixBuffer.get(), NodeCountMax,
            multiDestSubMixBuffer.get(), NodeCountMax,
            finalMixBuffer.get(), 1);
        nnt::audio::util::AudioNode* node[NodeCountMax];

        // Node construction
        for (auto i = 0; i < nodeCount; ++i)
        {
            const auto& info = nodeInfo[i];
            ASSERT_TRUE(info.index < nodeCount);
            switch (info.type)
            {
            case nnt::audio::util::AudioNode::NodeTypeInfo::Voice:
                node[info.index] = nodeAllocator.AllocateVoice(&sar.GetConfig(), info.sampleRate, info.channel, nn::audio::SampleFormat_PcmInt16, 0, nullptr, 0);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationVoice:
                node[info.index] = nodeAllocator.AllocateMultiDestinationVoice(&sar.GetConfig(), info.sampleRate, info.channel, nn::audio::SampleFormat_PcmInt16, 0, nullptr, 0, info.destinationCount);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::Splitter:
                node[info.index] = nodeAllocator.AllocateSplitter(&sar.GetConfig(), info.sampleRate, info.channel, info.destinationCount);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::SubMix:
                node[info.index] = nodeAllocator.AllocateSubMix(&sar.GetConfig(), info.sampleRate, info.channel);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationSubMix:
                node[info.index] = nodeAllocator.AllocateMultiDestinationSubMix(&sar.GetConfig(), info.sampleRate, info.channel, info.destinationCount);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix:
                node[info.index] = nodeAllocator.AllocateFinalMix(&sar.GetConfig(), info.channel);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            ASSERT_NE(node[info.index], nullptr) << "node allocation error. index:" << info.index;
        }

        // Graph construction
        for (auto i = 0; i < edgeCount; ++i)
        {
            const auto& edge = edges[i];
            ASSERT_TRUE(edge.sourceNodeIndex < nodeCount);
            auto srcNode = node[edge.sourceNodeIndex];
            switch (srcNode->GetType())
            {
            case nnt::audio::util::AudioNode::NodeTypeInfo::Voice:
                srcNode
                    ->As<nnt::audio::util::Voice>()
                    ->SetDestination(&sar.GetConfig(), node[edge.destNodeIndex])
                    ->SetMixVolume(edge.volume, edge.sourceChannel, edge.destChannel);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::Splitter:
                srcNode
                    ->As<nnt::audio::util::Splitter>()
                    ->SetDestination(&sar.GetConfig(), edge.sourceDestIndex, node[edge.destNodeIndex])
                    ->SetMixVolume(edge.sourceDestIndex, edge.volume, edge.sourceChannel, edge.destChannel);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::SubMix:
                srcNode
                    ->As<nnt::audio::util::SubMix>()
                    ->SetDestination(&sar.GetConfig(), node[edge.destNodeIndex])
                    ->SetMixVolume(edge.volume, edge.sourceChannel, edge.destChannel);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationVoice:
                srcNode
                    ->As<nnt::audio::util::MultiDestinationVoice>()
                    ->SetDestination(&sar.GetConfig(), edge.sourceDestIndex, node[edge.destNodeIndex])
                    ->SetMixVolume(edge.sourceDestIndex, edge.volume, edge.sourceChannel, edge.destChannel);
                break;
            case nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationSubMix:
                srcNode
                    ->As<nnt::audio::util::MultiDestinationSubMix>()
                    ->SetDestination(&sar.GetConfig(), edge.sourceDestIndex, node[edge.destNodeIndex])
                    ->SetMixVolume(edge.sourceDestIndex, edge.volume, edge.sourceChannel, edge.destChannel);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        // Source construction
        nn::audio::MemoryPoolType memoryPool;
        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, alignedWorkBuffer, workBufferSize - 4096));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));
        auto bufferSize = param.sampleCount * sizeof(int16_t) * nn::audio::VoiceType::ChannelCountMax;
        StdUniquePtr pcmBuffer(allocator.Allocate(bufferSize));
        for (auto i = 0; i < sourceCount; ++i)
        {
            const auto& src = sources[i];
            ASSERT_TRUE(src.index < nodeCount);
            if (src.waveBuffer)
            {
                node[src.index] // voice node
                    ->As<nnt::audio::util::Voice>()
                    ->AppendWaveBuffer(src.waveBuffer)
                    ->Start();
            }
            else
            {
                nn::audio::WaveBuffer SimpleDCSource;
                if (node[src.index]->As<nnt::audio::util::Voice>())
                {
                    PrepareTestWaveBuffer(SimpleDCSource, pcmBuffer.get(), bufferSize, 100,
                        node[src.index]->As<nnt::audio::util::Voice>()->GetChannelCount());
                    node[src.index] // voice node
                        ->As<nnt::audio::util::Voice>()
                        ->AppendWaveBuffer(&SimpleDCSource)
                        ->Start();
                }
                else if (node[src.index]->As<nnt::audio::util::MultiDestinationVoice>())
                {
                    PrepareTestWaveBuffer(SimpleDCSource, pcmBuffer.get(), bufferSize, 100,
                        node[src.index]->As<nnt::audio::util::MultiDestinationVoice>()->GetChannelCount());
                    node[src.index] // voice node
                        ->As<nnt::audio::util::MultiDestinationVoice>()
                        ->AppendWaveBuffer(&SimpleDCSource)
                        ->Start();
                }
            }
        }

        // Setup probe
        const int auxInputChannels = 1;
        const int probeFrameCount = 20;
        auto auxBufferSize = nnt::audio::util::Aux::GetWorkBufferSize(&param, auxInputChannels, probeFrameCount);
        nnt::audio::util::Aux aux[ProbeCountMax];
        StdUniquePtr sendBuffers[ProbeCountMax];
        StdUniquePtr returnBuffers[ProbeCountMax];
        StdUniquePtr readBuffers[ProbeCountMax];
        for (auto i = 0; i < expectsCount; ++i)
        {
            auto& expected = expects[i];
            const int8_t inout[auxInputChannels] = { static_cast<int8_t>(expected.channelIndex) };
            StdUniquePtr sendBuffer(allocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize));
            StdUniquePtr returnBuffer(allocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize));
            StdUniquePtr readBuffer(allocator.Allocate(auxBufferSize));
            aux[i].SetupBuffers(sendBuffer.get(), returnBuffer.get(), auxBufferSize);
            aux[i].AddTo(&sar.GetConfig(), node[expected.nodeIndex])->SetInOut(inout, inout, auxInputChannels);
            sendBuffers[i] = std::move(sendBuffer);
            returnBuffers[i] = std::move(returnBuffer);
            readBuffers[i] = std::move(readBuffer);
        }

        sar.Start();
        int loopCount = probeFrameCount * 2;
        nn::util::BitFlagSet<128> retryFlag; // 128 はおよそ十分な Probe の最大数
        retryFlag.Reset();

        while (loopCount-- > 0 || retryFlag.IsAnyOn())
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            nn::audio::TriggerRendering();
            event.Wait();
#endif
            systemEvent.Wait();
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

            retryFlag.Reset();
            for (auto i = 0; i < expectsCount; ++i)
            {
                auto rBuf = reinterpret_cast<int32_t*>(readBuffers[i].get());
                memset(rBuf, 0, auxBufferSize);
                auto read = aux[i].Read(rBuf, aux[i].GetSampleCount());
                if (read <= 0)
                {
                    retryFlag[i] = true;
                    continue;
                }
                aux[i].Write(rBuf, read);
            }
        }

        // test result
        for (auto i = 0; i < expectsCount; ++i)
        {
            auto& expected = expects[i];
            auto rBuf = reinterpret_cast<int32_t*>(readBuffers[i].get());
            EXPECT_NEAR(rBuf[0], expected.answer, expected.acceptableError);
        }
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::SetFastRenderingMode(false, nullptr);
        EXPECT_FALSE(nn::audio::GetFastRenderingMode());
#endif

    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100)); // wait adsp stopped
    allocator.Finalize();
    StdAllocatorDeleter::SetupDeleter(nullptr);
    free(workBuffer);
} // NOLINT(impl/function_size)

TEST(SplitterType, SimpleDiamond)
{
    // Structure
    // =========
    // src = mv3 -+-->s1 -> f0 = probe
    //            |         ^
    //            |         |
    //            +-->s2----+
    //
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleCount = 240;
    param.sampleRate = 48000;
    param.voiceCount = 1;
    param.splitterCount = 1;
    param.splitterSendChannelCount = param.splitterCount * /* input ch */ 1 * /* dest count */ 2 * 2;
    param.mixBufferCount = 2 + 2 * 2;
    param.subMixCount = 2;
    param.effectCount = 2;
    param.sinkCount = 1;

    const nnt::audio::util::NodeDescription nodeInfo[] =
    {
        {0, nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix,                 2},
        {1, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {2, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {3, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationVoice,    1, param.sampleRate, 2},
    };
    const nnt::audio::util::EdgeDescription edges[] =
    {
        // s1
        {1, -1, 0, 0, 0, 1.0f},
        {1, -1, 1, 0, 1, 1.0f},
        // s2
        {2, -1, 0, 0, 0, 1.0f},
        {2, -1, 1, 0, 1, 1.0f},
        // s3 | multi dest voice
        {3,  0, 0, 1, 0, 0.5f},
        {3,  0, 0, 1, 1, 1.0f},
        {3,  1, 0, 2, 0, 0.5f},
        {3,  1, 0, 2, 1, 1.0f},
    };
    const TestSource sources[] =
    {
        {3 , nullptr},
    };

    const ExpectedValue expected[] =
    {
        { 0, 0, 100, 2 },
        { 0, 1, 200, 2 },
    };

    SplitterTestBase(
        param,
        nodeInfo, sizeof(nodeInfo) / sizeof(nnt::audio::util::NodeDescription),
        edges, sizeof(edges) / sizeof(nnt::audio::util::EdgeDescription),
        sources, sizeof(sources) / sizeof(TestSource),
        expected, sizeof(expected) / sizeof(ExpectedValue));
}

TEST(SplitterType, PassThrough2ch)
{
    // Structure
    // =========
    // src = v2 -+--> ms1 -> f0 = probe
    //
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleCount = 240;
    param.sampleRate = 48000;
    param.voiceCount = 2;
    param.splitterCount = 2;
    param.splitterSendChannelCount = param.splitterCount * /* input ch */ 1 * /* dest count */ 2 * 2;
    param.mixBufferCount = 2 + 2 * 2;
    param.subMixCount = 2;
    param.effectCount = 5;
    param.sinkCount = 1;

    const nnt::audio::util::NodeDescription nodeInfo[] =
    {
        {0, nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix,                  2},
        {1, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationSubMix,    3, param.sampleRate, 1},
        {2, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationVoice,     2, param.sampleRate, 1},
    };
    const nnt::audio::util::EdgeDescription edges[] =
    {
        // ms1
        {1,  0, L, 0, L, 4.0f},
        {1,  0, R, 0, R, 2.0f},
        {1,  0, C, 0, L, 0.5f},
        {1,  0, C, 0, R, 0.5f},
        // v2
        {2,  0, L, 1, L, 0.25f},
        {2,  0, L, 1, R, 0.5f },
        {2,  0, L, 1, C, 1.0f },
    };

    const TestSource sources[] =
    {
        {2 , nullptr},
    };

    const ExpectedValue expected[] =
    {
        { 0, L, 150, 3 },
        { 0, R, 150, 3 },

        { 1, L,  25, 2 },
        { 1, R,  50, 2 },
        { 1, C, 100, 2 },
    };

    SplitterTestBase(
        param,
        nodeInfo, sizeof(nodeInfo) / sizeof(nnt::audio::util::NodeDescription),
        edges, sizeof(edges) / sizeof(nnt::audio::util::EdgeDescription),
        sources, sizeof(sources) / sizeof(TestSource),
        expected, sizeof(expected) / sizeof(ExpectedValue));
}

TEST(SplitterType, PassThrough6ch)
{
    // Structure
    // =========
    // src = v2 -+--> ms1 -> f0 = probe
    //
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleCount = 240;
    param.sampleRate = 48000;
    param.voiceCount = 6;
    param.splitterCount = 2;
    param.splitterSendChannelCount = 36 * 2;
    param.mixBufferCount = 6 + 6;
    param.subMixCount = 1;
    param.effectCount = 6;
    param.sinkCount = 1;

    const nnt::audio::util::NodeDescription nodeInfo[] =
    {
        {0, nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix,                  6},
        {1, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationSubMix,    6, param.sampleRate, 1},
        {2, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationVoice,     6, param.sampleRate, 1},
    };
    const nnt::audio::util::EdgeDescription edges[] =
    {
        // ms1
        {1,  0,  L, 0,  L, 1.0f / 1.0f},
        {1,  0,  R, 0,  R, 1.0f / 2.0f},
        {1,  0,  C, 0,  C, 1.0f / 3.0f},
        {1,  0, RL, 0, RL, 1.0f / 4.0f},
        {1,  0, RR, 0, RR, 1.0f / 5.0f},
        {1,  0, LF, 0, LF, 1.0f / 6.0f},

        // v2
        {2,  0,  L, 1,  L, 1.f},
        {2,  0,  R, 1,  R, 2.f},
        {2,  0,  C, 1,  C, 3.f},
        {2,  0, RL, 1, RL, 4.f},
        {2,  0, RR, 1, RR, 5.f},
        {2,  0, LF, 1, LF, 6.f},
    };

    const TestSource sources[] =
    {
        {2 , nullptr},
    };

    const ExpectedValue expected[] =
    {
        { 0, L,  100, 3 },
        { 0, R,  100, 3 },
        { 0, C,  100, 3 },
        { 0, RL, 100, 3 },
        { 0, RR, 100, 3 },
        { 0, LF, 100, 3 },
    };

    SplitterTestBase(
        param,
        nodeInfo, sizeof(nodeInfo) / sizeof(nnt::audio::util::NodeDescription),
        edges, sizeof(edges) / sizeof(nnt::audio::util::EdgeDescription),
        sources, sizeof(sources) / sizeof(TestSource),
        expected, sizeof(expected) / sizeof(ExpectedValue));
}

TEST(SplitterType, CascadeDiamond)
{
    // Structure
    // =========
    //                src2 = v7
    //                       |   = probe2
    //                       v  /
    // src1 = mv6 -+-> s4 -> ms3 -+-> s1 -> f0 = probe1
    //             |         ^    |         ^
    //             |         |    |         |
    //             +-> s5 ---+    +-> s2 ---+
    //
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleCount = 160;
    param.sampleRate = 32000;
    param.voiceCount = 2;
    param.splitterCount = 2;
    param.splitterSendChannelCount = /* input ch */ 1 * /* dest count */ 2 * /* dest ch count */ 2
                                   + /* input ch */ 2 * /* dest count */ 2 * /* dest ch count */ 2;
    param.mixBufferCount = /* FinalMix */ 2 + /* SubMix */ 2 * 6;
    param.subMixCount = 6;
    param.effectCount = 4;
    param.sinkCount = 1;

    const nnt::audio::util::NodeDescription nodeInfo[] =
    {
        {0, nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix,                 2},
        {1, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {2, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {3, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationSubMix,   2, param.sampleRate, 2},
        {4, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {5, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {6, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationVoice,    1, param.sampleRate, 2},
        {7, nnt::audio::util::AudioNode::NodeTypeInfo::Voice,                    1, param.sampleRate, 1},
    };
    const nnt::audio::util::EdgeDescription edges[] =
    {
        // s1
        {1, -1, 0, 0, 0, 1.0f},
        {1, -1, 1, 0, 1, 1.0f},
        // s2
        {2, -1, 0, 0, 0, 1.0f},
        {2, -1, 1, 0, 1, 1.0f},
        // ms3 | multi dest subMix
        {3,  0, 0, 1, 0, 1.0f},
        {3,  0, 1, 1, 1, 1.0f},
        {3,  1, 0, 2, 0, 1.0f},
        {3,  1, 1, 2, 1, 1.0f},
        // s4
        {4, -1, 0, 3, 0, 1.0f},
        {4, -1, 1, 3, 1, 1.0f},
        // s5
        {5, -1, 0, 3, 0, 1.0f},
        {5, -1, 1, 3, 1, 1.0f},
        // mv6 | multi dest voice
        {6,  0, 0, 4, 0, 0.5f},
        {6,  0, 0, 4, 1, 1.0f},
        {6,  1, 0, 5, 0, 0.5f},
        {6,  1, 0, 5, 1, 1.0f},
        // v7
        {7, -1, 0, 3, 0, 0.5f},
        {7, -1, 0, 3, 1, 1.0f},
    };
    const TestSource sources[] =
    {
        {6 , nullptr},
        {7 , nullptr},
    };

    const ExpectedValue expected[] =
    {
        // probe1
        { 0, 0, 300, 6 },
        { 0, 1, 600, 6 },
        // probe2
        { 3, 0, 150, 3 },
        { 3, 1, 300, 3 },
    };

    SplitterTestBase(
        param,
        nodeInfo, sizeof(nodeInfo) / sizeof(nnt::audio::util::NodeDescription),
        edges, sizeof(edges) / sizeof(nnt::audio::util::EdgeDescription),
        sources, sizeof(sources) / sizeof(TestSource),
        expected, sizeof(expected) / sizeof(ExpectedValue));
}

TEST(SplitterType, VoiceParaOut)
{
    // Structure
    // =========
    // src = mv5 -+   s1 -> s2 -> s3 -> s4 -> f0 = probe
    //            |   ^     ^     ^     ^     ^
    //            |   |     |     |     |     |
    //            +---+-----+-----+-----+-----+
    //
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleCount = 240;
    param.sampleRate = 48000;
    param.voiceCount = 1;
    param.splitterCount = 10; // SIGLO-80635 のチェック用に、 Instance 数を増やしておく
    param.splitterSendChannelCount = /* input ch */ 1 * /* dest count */ 5 * 2;
    param.mixBufferCount = 2 + 2 * 5;
    param.subMixCount = 4;
    param.effectCount = 1;
    param.sinkCount = 1;

    const nnt::audio::util::NodeDescription nodeInfo[] =
    {
        {0, nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix,                 2},
        {1, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {2, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {3, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {4, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {5, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationVoice,    1, param.sampleRate, 5},
    };
    const nnt::audio::util::EdgeDescription edges[] =
    {
        // s1
        {1, -1, 0, 2, 0, 1.0f},
        {1, -1, 1, 2, 1, 1.0f},
        // s2
        {2, -1, 0, 3, 0, 1.0f},
        {2, -1, 1, 3, 1, 1.0f},
        // s3
        {3, -1, 0, 4, 0, 1.0f},
        {3, -1, 1, 4, 1, 1.0f},
        // s4
        {4, -1, 0, 0, 0, 1.0f},
        {4, -1, 1, 0, 1, 1.0f},
        // s5 | multi dest voice
        {5,  0, 0, 1, 0, 1.0f},
        {5,  0, 0, 1, 1, 1.0f},
        {5,  1, 0, 2, 0, 1.0f},
        {5,  1, 0, 2, 1, 1.0f},
        {5,  2, 0, 3, 0, 1.0f},
        {5,  2, 0, 3, 1, 1.0f},
        {5,  3, 0, 4, 0, 1.0f},
        {5,  3, 0, 4, 1, 1.0f},
        {5,  4, 0, 0, 0, 1.0f},
        {5,  4, 0, 0, 1, 1.0f},
    };
    const TestSource sources[] =
    {
        {5 , nullptr},
    };

    const ExpectedValue expected[] =
    {
        { 0, 0, 500, 5 },
    };

    SplitterTestBase(
        param,
        nodeInfo, sizeof(nodeInfo) / sizeof(nnt::audio::util::NodeDescription),
        edges, sizeof(edges) / sizeof(nnt::audio::util::EdgeDescription),
        sources, sizeof(sources) / sizeof(TestSource),
        expected, sizeof(expected) / sizeof(ExpectedValue));
}

TEST(SplitterType, SubMixParaOut)
{
    // Structure
    // =========
    // src = v5 -+-> ms1    s2 -> s3 -> s4 -> f0 = probe
    //                |     ^     ^     ^     ^
    //                |     |     |     |     |
    //                +-----+-----+-----+-----+
    //
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleCount = 240;
    param.sampleRate = 48000;
    param.voiceCount = 1;
    param.splitterCount = 1;
    param.splitterSendChannelCount = param.splitterCount * /* input ch */ 2 * /* dest count */ 5 * 2;
    param.mixBufferCount = 2 + 2 * 5;
    param.subMixCount = 4;
    param.effectCount = 1;
    param.sinkCount = 1;

    const nnt::audio::util::NodeDescription nodeInfo[] =
    {
        {0, nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix,                 2},
        {1, nnt::audio::util::AudioNode::NodeTypeInfo::MultiDestinationSubMix,   2, param.sampleRate, 4},
        {2, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {3, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {4, nnt::audio::util::AudioNode::NodeTypeInfo::SubMix,                   2, param.sampleRate, 1},
        {5, nnt::audio::util::AudioNode::NodeTypeInfo::Voice,                    1, param.sampleRate, 1},
    };
    const nnt::audio::util::EdgeDescription edges[] =
    {
        // s1 | multi dest voice
        {1,  0, 0, 2, 0, 1.0f},
        {1,  0, 1, 2, 1, 1.0f},
        {1,  1, 0, 3, 0, 1.0f},
        {1,  1, 1, 3, 1, 1.0f},
        {1,  2, 0, 4, 0, 1.0f},
        {1,  2, 1, 4, 1, 1.0f},
        {1,  3, 0, 0, 0, 1.0f},
        {1,  3, 1, 0, 1, 1.0f},
        // s2
        {2, -1, 0, 3, 0, 1.0f},
        {2, -1, 1, 3, 1, 1.0f},
        // s3
        {3, -1, 0, 4, 0, 1.0f},
        {3, -1, 1, 4, 1, 1.0f},
        // s4
        {4, -1, 0, 0, 0, 1.0f},
        {4, -1, 1, 0, 1, 1.0f},
        // s5
        {5, -1, 0, 1, 0, 1.0f},
        {5, -1, 0, 1, 1, 1.0f},
    };
    const TestSource sources[] =
    {
        {5 , nullptr},
    };

    const ExpectedValue expected[] =
    {
        { 0, 0, 400, 4 },
    };

    SplitterTestBase(
        param,
        nodeInfo, sizeof(nodeInfo) / sizeof(nnt::audio::util::NodeDescription),
        edges, sizeof(edges) / sizeof(nnt::audio::util::EdgeDescription),
        sources, sizeof(sources) / sizeof(TestSource),
        expected, sizeof(expected) / sizeof(ExpectedValue));
}

TEST(SplitterType, MultiVoice)
{
    // Structure
    // =========
    //
    // src = v0 -|
    //       v1 -+- sp ->s1 -> f0 = probe
    //       v2 -|
    //
    //
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleCount = 240;
    param.sampleRate = 48000;
    param.voiceCount = 3;
    param.splitterCount = 1;
    param.splitterSendChannelCount = param.splitterCount * /* input ch */ 1 * /* dest count */ 2 * 3;
    param.mixBufferCount = 2 + 2 * 2;
    param.subMixCount = 2;
    param.effectCount = 2;
    param.sinkCount = 1;

    const nnt::audio::util::NodeDescription nodeInfo[] =
    {
        {0, nnt::audio::util::AudioNode::NodeTypeInfo::FinalMix,                 2},
        {1, nnt::audio::util::AudioNode::NodeTypeInfo::Splitter,                 2, param.sampleRate, 2},
        {2, nnt::audio::util::AudioNode::NodeTypeInfo::Voice,                    1, param.sampleRate, 1},
        {3, nnt::audio::util::AudioNode::NodeTypeInfo::Voice,                    1, param.sampleRate, 1},
        {4, nnt::audio::util::AudioNode::NodeTypeInfo::Voice,                    1, param.sampleRate, 1},
    };
    const nnt::audio::util::EdgeDescription edges[] =
    {
        // Splitter -> FinalMix
        {1, -1, 0, 0, 0, 1.0f},
        {1, -1, 1, 0, 1, 0.5f},

        // Voice(0) -> Splitter
        {2,  0, 0, 1, 0, 1.0f},
        {2,  0, 0, 1, 1, 1.0f},

        // Voice(1) -> Splitter
        {3,  0, 0, 1, 0, 1.0f},
        {3,  0, 0, 1, 1, 1.0f},

        // Voice(2) -> Splitter
        {4,  0, 0, 1, 0, 1.0f},
        {4,  0, 0, 1, 1, 1.0f},
    };
    const TestSource sources[] =
    {
        {2 , nullptr},
        {3 , nullptr},
        {4 , nullptr},
    };

    const ExpectedValue expected[] =
    {
        { 0, 0, 300, 10 },
        { 0, 1, 150, 10 },
    };

    SplitterTestBase(
        param,
        nodeInfo, sizeof(nodeInfo) / sizeof(nnt::audio::util::NodeDescription),
        edges, sizeof(edges) / sizeof(nnt::audio::util::EdgeDescription),
        sources, sizeof(sources) / sizeof(TestSource),
        expected, sizeof(expected) / sizeof(ExpectedValue));
}


TEST(SplitterType, LoopDetection)
{
    const size_t workBufferSize = 10 * 1024 * 1024;
    auto workBuffer = malloc(workBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);
    auto alignedWorkBuffer = nn::util::BytePtr(workBuffer).AlignUp(nn::audio::MemoryPoolType::AddressAlignment).Get();
    nn::mem::StandardAllocator allocator(alignedWorkBuffer, workBufferSize - nn::audio::MemoryPoolType::SizeGranularity);
    StdAllocatorDeleter::SetupDeleter(&allocator);

    {
        nn::os::SystemEvent systemEvent;
        nn::audio::AudioRendererParameter param;
        nn::audio::InitializeAudioRendererParameter(&param);
        param.sampleRate = 48000;
        param.sampleCount = 240;
        param.mixBufferCount = 6;
        param.subMixCount = 2;
        param.voiceCount = 1;
        param.sinkCount = 1;
        param.effectCount = 0;
        param.splitterCount = 1;
        param.splitterSendChannelCount = 10;
        param.performanceFrameCount = 0;
        param.isVoiceDropEnabled = false;

        nnt::audio::util::ScopedAudioRenderer sar(param, &systemEvent);

        StdUniquePtr voiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::Voice) * 10));
        StdUniquePtr multiDestVoiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationVoice) * 10));
        StdUniquePtr splitterBuffer(allocator.Allocate(sizeof(nnt::audio::util::Splitter) * 10));
        StdUniquePtr subMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::SubMix) * 10));
        StdUniquePtr multiDestSubMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationSubMix) * 10));
        StdUniquePtr finalMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::FinalMix) * 1));
        nnt::audio::util::AudioNodeAllocator nodeAllocator(
            voiceBuffer.get(), NodeCountMax,
            multiDestVoiceBuffer.get(), NodeCountMax,
            splitterBuffer.get(), NodeCountMax,
            subMixBuffer.get(), NodeCountMax,
            multiDestSubMixBuffer.get(), NodeCountMax,
            finalMixBuffer.get(), 1);

        nnt::audio::util::SubMix* subMix = nodeAllocator.AllocateSubMix(&sar.GetConfig(), param.sampleRate, 2);
        nnt::audio::util::MultiDestinationSubMix* multiDestSubMix = nodeAllocator.AllocateMultiDestinationSubMix(&sar.GetConfig(), param.sampleRate, 2, 2);
        nnt::audio::util::FinalMix* finalMix = nodeAllocator.AllocateFinalMix(&sar.GetConfig(), 2);
        subMix
            ->SetDestination(&sar.GetConfig(), multiDestSubMix)
            ->SetMixVolume(1.f, 0, 0)
            ->SetMixVolume(1.f, 1, 1);
        multiDestSubMix
            ->SetDestination(&sar.GetConfig(), 0, subMix)
            ->SetMixVolume(0, 1.f, 0, 0)
            ->SetMixVolume(0, 1.f, 1, 1);
        multiDestSubMix
            ->SetDestination(&sar.GetConfig(), 1, finalMix)
            ->SetMixVolume(1, 1.f, 0, 0)
            ->SetMixVolume(1, 1.f, 1, 1);

        sar.Start();
        auto result = nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig());
        EXPECT_TRUE(nn::audio::ResultCycleDetected::Includes(result));
    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100)); // wait adsp stopped
    allocator.Finalize();
    StdAllocatorDeleter::SetupDeleter(nullptr);
    free(workBuffer);
}

namespace {

void Loop(
    nnt::audio::util::ScopedAudioRenderer& sar,
    nn::os::SystemEvent& systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::os::Event& event,
#endif
    int loopCount,
    nnt::audio::util::Aux* probes, StdUniquePtr* readBuffers, int probeCount)
{
    nn::util::BitFlagSet<128> retryFlag; // 128 はおよそ十分な Probe の最大数
    retryFlag.Reset();

    while (loopCount-- > 0 || retryFlag.IsAnyOn())
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::TriggerRendering();
        event.Wait();
#endif
        systemEvent.Wait();
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));

        retryFlag.Reset();
        for (auto i = 0; i < probeCount; ++i)
        {
            auto rBuf = reinterpret_cast<int32_t*>(readBuffers[i].get());
            memset(rBuf, 0, sizeof(int32_t) * probes[i].GetSampleCount());
            auto read = probes[i].Read(rBuf, probes[i].GetSampleCount());
            if (read <= 0)
            {
                retryFlag[i] = true;
                continue;
            }
            probes[i].Write(rBuf, read);
        }
    }
}


}


TEST(SplitterType, DynamicRoutingVoice)
{
    const size_t workBufferSize = 10 * 1024 * 1024;
    auto workBuffer = malloc(workBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);
    auto alignedWorkBuffer = nn::util::BytePtr(workBuffer).AlignUp(nn::audio::MemoryPoolType::AddressAlignment).Get();
    nn::mem::StandardAllocator allocator(alignedWorkBuffer, workBufferSize - nn::audio::MemoryPoolType::SizeGranularity);
    StdAllocatorDeleter::SetupDeleter(&allocator);

    {
        nn::os::SystemEvent systemEvent;
        nn::audio::AudioRendererParameter param;
        nn::audio::InitializeAudioRendererParameter(&param);
        param.sampleRate = 48000;
        param.sampleCount = 240;
        param.mixBufferCount = 6;
        param.subMixCount = 2;
        param.voiceCount = 2;
        param.sinkCount = 1;
        param.effectCount = 2;
        param.splitterCount = 2;
        param.splitterSendChannelCount = 10;
        param.performanceFrameCount = 0;
        param.isVoiceDropEnabled = false;

        nnt::audio::util::ScopedAudioRenderer sar(param, &systemEvent);
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::os::Event event(nn::os::EventClearMode_AutoClear);
        nn::audio::SetFastRenderingMode(true, &event);
        EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif

        StdUniquePtr voiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::Voice) * 10));
        StdUniquePtr multiDestVoiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationVoice) * 10));
        StdUniquePtr splitterBuffer(allocator.Allocate(sizeof(nnt::audio::util::Splitter) * 10));
        StdUniquePtr subMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::SubMix) * 10));
        StdUniquePtr multiDestSubMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationSubMix) * 10));
        StdUniquePtr finalMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::FinalMix) * 1));
        nnt::audio::util::AudioNodeAllocator nodeAllocator(
            voiceBuffer.get(), NodeCountMax,
            multiDestVoiceBuffer.get(), NodeCountMax,
            splitterBuffer.get(), NodeCountMax,
            subMixBuffer.get(), NodeCountMax,
            multiDestSubMixBuffer.get(), NodeCountMax,
            finalMixBuffer.get(), 1);

        nnt::audio::util::Voice* voice = nodeAllocator.AllocateVoice(&sar.GetConfig(), param.sampleRate, 1, nn::audio::SampleFormat_PcmInt16, 0, nullptr, 0);
        nnt::audio::util::SubMix* subMix = nodeAllocator.AllocateSubMix(&sar.GetConfig(), param.sampleRate, 2);
        nnt::audio::util::FinalMix* finalMix = nodeAllocator.AllocateFinalMix(&sar.GetConfig(), 2);
        nnt::audio::util::MultiDestinationVoice* multiDestVoice = nodeAllocator.AllocateMultiDestinationVoice(&sar.GetConfig(), param.sampleRate, 1, nn::audio::SampleFormat_PcmInt16, 0, nullptr, 0, 2);
        nnt::audio::util::MultiDestinationSubMix* multiDestSubMix = nodeAllocator.AllocateMultiDestinationSubMix(&sar.GetConfig(), param.sampleRate, 2, 2);
        ASSERT_NE(voice, nullptr);
        ASSERT_NE(subMix, nullptr);
        ASSERT_NE(finalMix, nullptr);
        ASSERT_NE(multiDestVoice, nullptr);
        ASSERT_NE(multiDestSubMix, nullptr);

        // Setup probe
        const int ProveCount = 2;
        const int auxInputChannels = 1;
        const int probeFrameCount = 10;
        auto auxBufferSize = nnt::audio::util::Aux::GetWorkBufferSize(&param, auxInputChannels, probeFrameCount);
        nnt::audio::util::Aux aux[ProveCount];
        StdUniquePtr sendBuffers[ProveCount];
        StdUniquePtr returnBuffers[ProveCount];
        StdUniquePtr readBuffers[ProveCount];
        for (auto i = 0; i < ProveCount; ++i)
        {
            const int8_t inout[auxInputChannels] = { 0 };
            StdUniquePtr sendBuffer(allocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize));
            StdUniquePtr returnBuffer(allocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize));
            StdUniquePtr readBuffer(allocator.Allocate(auxBufferSize));
            aux[i].SetupBuffers(sendBuffer.get(), returnBuffer.get(), auxBufferSize);
            if (i % 2)
            {
                aux[i].AddTo(&sar.GetConfig(), subMix)->SetInOut(inout, inout, auxInputChannels);
            }
            else
            {
                aux[i].AddTo(&sar.GetConfig(), finalMix)->SetInOut(inout, inout, auxInputChannels);
            }
            sendBuffers[i] = std::move(sendBuffer);
            returnBuffers[i] = std::move(returnBuffer);
            readBuffers[i] = std::move(readBuffer);
        }

        // prepare source voice
        nn::audio::MemoryPoolType memoryPool;
        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, alignedWorkBuffer, workBufferSize - 4096));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));
        auto bufferSize = param.sampleCount * sizeof(int16_t);
        StdUniquePtr pcmBuffer(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
        nn::audio::WaveBuffer SimpleDCSource;
        PrepareTestWaveBuffer(SimpleDCSource, pcmBuffer.get(), bufferSize, 100, voice->GetChannelCount());

        // ================================================================================
        // * default connection
        //
        // voice -> subMix -> finalMix
        //
        subMix
            ->SetDestination(&sar.GetConfig(), finalMix)
            ->SetMixVolume(1.f, 0, 0)
            ->SetMixVolume(1.f, 1, 1);
        voice
            ->AppendWaveBuffer(&SimpleDCSource)
            ->SetDestination(&sar.GetConfig(), subMix)
            ->SetMixVolume(1.f, 0, 0)
            ->SetMixVolume(1.f, 0, 1)
            ->Start();


        sar.Start();
        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        int32_t* rBuf = nullptr;
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 100, 1);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 100, 1);

        // ================================================================================
        // * add multi destination voice
        //
        // voice -> subMix -> finalMix
        //             ^         ^
        //             |         |
        // multiVoice -+---------+
        //
        multiDestVoice
            ->AppendWaveBuffer(&SimpleDCSource)
            ->SetDestination(&sar.GetConfig(), 0, subMix)
            ->SetMixVolume(0, 1.f, 0, 0)
            ->SetMixVolume(0, 1.f, 0, 1)
            ->SetDestination(&sar.GetConfig(), 1, finalMix)
            ->SetMixVolume(1, 1.f, 0, 0)
            ->SetMixVolume(1, 1.f, 0, 1)
            ->Start();

        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 300, 3);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 200, 3);

        // ================================================================================
        // * Change destination
        //
        // voice -> subMix -> finalMix
        //                     ^ ^
        //                     | |
        // multiVoice -+-------+-+
        //
        multiDestVoice
            ->SetDestination(&sar.GetConfig(), 0, finalMix);

        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 300, 3);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 100, 1);


        // ================================================================================
        // * remove multi destination voice's splitter
        //
        // voice -> subMix -> finalMix
        //

        nn::audio::ReleaseSplitter(&sar.GetConfig(), multiDestVoice->GetSplitterBase());

        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 100, 1);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 100, 1);

#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::SetFastRenderingMode(false, nullptr);
        EXPECT_FALSE(nn::audio::GetFastRenderingMode());
#endif
    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100)); // wait adsp stopped
    allocator.Finalize();
    StdAllocatorDeleter::SetupDeleter(nullptr);
    free(workBuffer);
} // NOLINT(readability/fn_size)

TEST(SplitterType, DynamicRoutingSubMix)
{
    const size_t workBufferSize = 10 * 1024 * 1024;
    auto workBuffer = malloc(workBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);
    auto alignedWorkBuffer = nn::util::BytePtr(workBuffer).AlignUp(nn::audio::MemoryPoolType::AddressAlignment).Get();
    nn::mem::StandardAllocator allocator(alignedWorkBuffer, workBufferSize - nn::audio::MemoryPoolType::SizeGranularity);
    StdAllocatorDeleter::SetupDeleter(&allocator);

    {
        nn::os::SystemEvent systemEvent;
        nn::audio::AudioRendererParameter param;
        nn::audio::InitializeAudioRendererParameter(&param);
        param.sampleRate = 48000;
        param.sampleCount = 240;
        param.mixBufferCount = 6;
        param.subMixCount = 2;
        param.voiceCount = 2;
        param.sinkCount = 1;
        param.effectCount = 2;
        param.splitterCount = 2;
        param.splitterSendChannelCount = 10;
        param.performanceFrameCount = 0;
        param.isVoiceDropEnabled = false;

        nnt::audio::util::ScopedAudioRenderer sar(param, &systemEvent);
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::os::Event event(nn::os::EventClearMode_AutoClear);
        nn::audio::SetFastRenderingMode(true, &event);
        EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif

        StdUniquePtr voiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::Voice) * 10));
        StdUniquePtr multiDestVoiceBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationVoice) * 10));
        StdUniquePtr splitterBuffer(allocator.Allocate(sizeof(nnt::audio::util::Splitter) * 10));
        StdUniquePtr subMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::SubMix) * 10));
        StdUniquePtr multiDestSubMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::MultiDestinationSubMix) * 10));
        StdUniquePtr finalMixBuffer(allocator.Allocate(sizeof(nnt::audio::util::FinalMix) * 1));
        nnt::audio::util::AudioNodeAllocator nodeAllocator(
            voiceBuffer.get(), NodeCountMax,
            multiDestVoiceBuffer.get(), NodeCountMax,
            splitterBuffer.get(), NodeCountMax,
            subMixBuffer.get(), NodeCountMax,
            multiDestSubMixBuffer.get(), NodeCountMax,
            finalMixBuffer.get(), 1);

        nnt::audio::util::Voice* voice = nodeAllocator.AllocateVoice(&sar.GetConfig(), param.sampleRate, 1, nn::audio::SampleFormat_PcmInt16, 0, nullptr, 0);
        nnt::audio::util::SubMix* subMix = nodeAllocator.AllocateSubMix(&sar.GetConfig(), param.sampleRate, 2);
        nnt::audio::util::FinalMix* finalMix = nodeAllocator.AllocateFinalMix(&sar.GetConfig(), 2);
        nnt::audio::util::MultiDestinationVoice* multiDestVoice = nodeAllocator.AllocateMultiDestinationVoice(&sar.GetConfig(), param.sampleRate, 1, nn::audio::SampleFormat_PcmInt16, 0, nullptr, 0, 2);
        nnt::audio::util::MultiDestinationSubMix* multiDestSubMix = nodeAllocator.AllocateMultiDestinationSubMix(&sar.GetConfig(), param.sampleRate, 2, 2);
        ASSERT_NE(voice, nullptr);
        ASSERT_NE(subMix, nullptr);
        ASSERT_NE(finalMix, nullptr);
        ASSERT_NE(multiDestVoice, nullptr);
        ASSERT_NE(multiDestSubMix, nullptr);

        // Setup probe
        const int ProveCount = 2;
        const int auxInputChannels = 1;
        const int probeFrameCount = 20;
        auto auxBufferSize = nnt::audio::util::Aux::GetWorkBufferSize(&param, auxInputChannels, probeFrameCount);
        nnt::audio::util::Aux aux[ProveCount];
        StdUniquePtr sendBuffers[ProveCount];
        StdUniquePtr returnBuffers[ProveCount];
        StdUniquePtr readBuffers[ProveCount];
        for (auto i = 0; i < ProveCount; ++i)
        {
            const int8_t inout[auxInputChannels] = { 0 };
            StdUniquePtr sendBuffer(allocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize));
            StdUniquePtr returnBuffer(allocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize));
            StdUniquePtr readBuffer(allocator.Allocate(auxBufferSize));
            aux[i].SetupBuffers(sendBuffer.get(), returnBuffer.get(), auxBufferSize);
            if (i % 2)
            {
                aux[i].AddTo(&sar.GetConfig(), subMix)->SetInOut(inout, inout, auxInputChannels);
            }
            else
            {
                aux[i].AddTo(&sar.GetConfig(), finalMix)->SetInOut(inout, inout, auxInputChannels);
            }
            sendBuffers[i] = std::move(sendBuffer);
            returnBuffers[i] = std::move(returnBuffer);
            readBuffers[i] = std::move(readBuffer);
        }

        // prepare source voice
        nn::audio::MemoryPoolType memoryPool;
        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, alignedWorkBuffer, workBufferSize - 4096));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));
        auto bufferSize = param.sampleCount * sizeof(int16_t);
        StdUniquePtr pcmBuffer(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
        nn::audio::WaveBuffer SimpleDCSource;
        PrepareTestWaveBuffer(SimpleDCSource, pcmBuffer.get(), bufferSize, 100, voice->GetChannelCount());

        // ================================================================================
        // * default connection
        //
        // voice -> subMix -> finalMix
        //
        subMix
            ->SetDestination(&sar.GetConfig(), finalMix)
            ->SetMixVolume(1.f, 0, 0)
            ->SetMixVolume(1.f, 1, 1);
        voice
            ->AppendWaveBuffer(&SimpleDCSource)
            ->SetDestination(&sar.GetConfig(), subMix)
            ->SetMixVolume(1.f, 0, 0)
            ->SetMixVolume(1.f, 0, 1)
            ->Start();


        sar.Start();
        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        int32_t* rBuf = nullptr;
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 100, 1);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 100, 1);

        // ================================================================================
        // * Add multi destination subMix
        //
        // voice -> mutlDestSubMix -> subMix -> finalMix
        //                 |                       ^
        //                 |                       |
        //                 +-----------------------+

        voice
            ->SetDestination(&sar.GetConfig(), multiDestSubMix)
            ->SetMixVolume(1.f, 0, 0)
            ->SetMixVolume(1.f, 0, 1);
        multiDestSubMix
            ->SetDestination(&sar.GetConfig(), 0, subMix)
            ->SetMixVolume(0, 1.f, 0, 0)
            ->SetMixVolume(0, 1.f, 1, 1)
            ->SetDestination(&sar.GetConfig(), 1, finalMix)
            ->SetMixVolume(1, 1.f, 0, 0)
            ->SetMixVolume(1, 1.f, 1, 1);

        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        rBuf = nullptr;
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 200, 2);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 100, 1);

        // ================================================================================
        // * Change destination
        //
        // voice -> mutlDestSubMix -> subMix -> finalMix
        //                 |            ^
        //                 |            |
        //                 +------------+

        multiDestSubMix
            ->SetDestination(&sar.GetConfig(), 1, subMix);

        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        rBuf = nullptr;
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 200, 2);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 200, 2);

        // ================================================================================
        // * Remove multi destination subMix's splitter
        //
        // voice -> mutlDestSubMix    subMix -> finalMix
        //

        nn::audio::ReleaseSplitter(&sar.GetConfig(), multiDestSubMix->GetSplitterBase());

        Loop(
            sar,
            systemEvent,
#if defined(NN_BUILD_CONFIG_OS_WIN)
            event,
#endif
            probeFrameCount * 2,
            aux, readBuffers, ProveCount);

        // test result
        rBuf = nullptr;
        rBuf = reinterpret_cast<int32_t*>(readBuffers[0].get());
        EXPECT_NEAR(rBuf[0], 0, 0);
        rBuf = reinterpret_cast<int32_t*>(readBuffers[1].get());
        EXPECT_NEAR(rBuf[0], 0, 0);
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::SetFastRenderingMode(false, nullptr);
        EXPECT_FALSE(nn::audio::GetFastRenderingMode());
#endif

    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100)); // wait adsp stopped
    allocator.Finalize();
    StdAllocatorDeleter::SetupDeleter(nullptr);
    free(workBuffer);
} // NOLINT(impl/function_size)
