﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

namespace {

const nn::audio::AudioRendererParameter GetDefaultParameter()
{
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleRate = 48000;
    param.sampleCount = 240;
    param.mixBufferCount = 16;
    param.subMixCount = 0;
    param.voiceCount = 24;
    param.sinkCount = 1;
    param.effectCount = 0;
    param.performanceFrameCount = 0;
    return param;
}

NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024];

}

/**
 * @brief Open-close success test with different parameters
 *
 * This test is valid even when the system accepts single instance only
 */
TEST(MultiRendererInstance, OpenCloseSuccessTest)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererHandle handles[nn::audio::AudioRendererCountMax];
    void* workBuffers[nn::audio::AudioRendererCountMax];
    nn::os::SystemEvent systemEvent[nn::audio::AudioRendererCountMax];

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        nn::audio::AudioRendererParameter parameter = GetDefaultParameter();
        if (i == 0)  // change sample rate for the first instance
        {
            parameter.sampleRate = 32000;
            parameter.sampleCount = 160;
        }
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        workBuffers[i] = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

        if (i == 1)  // initialize the second instance with nn::os::SystemEvent
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handles[i], &systemEvent[i], parameter, workBuffers[i], workBufferSize));
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handles[i], parameter, workBuffers[i], workBufferSize));
        }
    }

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        nn::audio::CloseAudioRenderer(handles[i]);
        allocator.Free(workBuffers[i]);
    }
}

/**
 * @brief Boundary test for nn::audio::AudioRendererCountMax
 *
 * This test is valid even when the system accepts single instance only
 */
TEST(MultiRendererInstance, CountBoundaryTest)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererHandle handles[nn::audio::AudioRendererCountMax + 1];
    void* workBuffers[nn::audio::AudioRendererCountMax + 1];

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(GetDefaultParameter());
        workBuffers[i] = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handles[i], GetDefaultParameter(), workBuffers[i], workBufferSize));
    }

    // should fail if # of instances is over the limit
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(GetDefaultParameter());
    workBuffers[nn::audio::AudioRendererCountMax] = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOutOfResource, nn::audio::OpenAudioRenderer(&handles[nn::audio::AudioRendererCountMax], GetDefaultParameter(), workBuffers[nn::audio::AudioRendererCountMax], workBufferSize));
    allocator.Free(workBuffers[nn::audio::AudioRendererCountMax]);

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        nn::audio::CloseAudioRenderer(handles[i]);
        allocator.Free(workBuffers[i]);
    }
}

/**
 * @brief Open-start-stop-close success test (with the same parameter)
 *
 * This test is valid even when the system accepts single instance only
 */
TEST(MultiRendererInstance, StartStopSuccessTest)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererHandle handles[nn::audio::AudioRendererCountMax];
    void* workBuffers[nn::audio::AudioRendererCountMax];

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(GetDefaultParameter());
        workBuffers[i] = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handles[i], GetDefaultParameter(), workBuffers[i], workBufferSize));
    }

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handles[i]));
    }

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        nn::audio::StopAudioRenderer(handles[i]);
    }

    for (int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        nn::audio::CloseAudioRenderer(handles[i]);
        allocator.Free(workBuffers[i]);
    }
}

TEST(MultiRendererInstance, AppendVoiceBuffers)
{
    const int rendererCount = nn::audio::AudioRendererCountMax;

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererHandle handles[rendererCount];
    nn::audio::AudioRendererConfig configs[rendererCount];
    nn::audio::VoiceType voices[rendererCount];
    nn::audio::FinalMixType finalMixes[rendererCount];
    nn::audio::WaveBuffer waveBuffers[rendererCount];
    void* workBuffers[rendererCount];
    nn::os::SystemEvent systemEvent[rendererCount];
    void* configBuffers[rendererCount];

    nn::audio::MemoryPoolType memoryPools[rendererCount];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[rendererCount][nn::audio::MemoryPoolType::SizeGranularity];

    for (int i = 0; i < rendererCount; ++i)
    {
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(GetDefaultParameter());
        workBuffers[i] = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handles[i], &systemEvent[i], GetDefaultParameter(), workBuffers[i], workBufferSize));

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(GetDefaultParameter());
        configBuffers[i] = allocator.Allocate(configBufferSize, nn::audio::BufferAlignSize);
        nn::audio::InitializeAudioRendererConfig(&configs[i], GetDefaultParameter(), configBuffers[i], configBufferSize);
    }

    for (int i = 0; i < rendererCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handles[i]));
    }

    for (int i = 0; i < rendererCount; ++i)
    {
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&configs[i], &voices[i], 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        EXPECT_TRUE(nn::audio::AcquireFinalMix(&configs[i], &finalMixes[i], 1));
        nn::audio::SetVoiceDestination(&configs[i], &voices[i], &finalMixes[i]);
        nn::audio::SetVoicePlayState(&voices[i], nn::audio::VoiceType::PlayState_Play);

        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&configs[i], &memoryPools[i], pcmBuffer[i], sizeof(pcmBuffer[i])));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPools[i]));

        waveBuffers[i].buffer = pcmBuffer[i];
        waveBuffers[i].size = GetDefaultParameter().sampleCount * sizeof(pcmBuffer[i][0]);
        waveBuffers[i].startSampleOffset = 0;
        waveBuffers[i].endSampleOffset = static_cast<int32_t>(waveBuffers[i].size / sizeof(pcmBuffer[i][0]));
        waveBuffers[i].loop = 0;
        waveBuffers[i].isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voices[i], &waveBuffers[i]);
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == nullptr);

        systemEvent[i].Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
    }

    for (auto i = 0; i < 10; ++i)
    {
        systemEvent[0].Wait();
    }

    for (int i = 0; i < rendererCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == &waveBuffers[i]);
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == nullptr);
        EXPECT_TRUE(nn::audio::GetVoicePlayedSampleCount(&voices[i]) == waveBuffers[i].endSampleOffset);

        waveBuffers[i].isEndOfStream = true;
        nn::audio::AppendWaveBuffer(&voices[i], &waveBuffers[i]);

        systemEvent[i].Wait();
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
    }

    for (auto i = 0; i < 10; ++i)
    {
        systemEvent[0].Wait();
    }

    for (int i = 0; i < rendererCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handles[i], &configs[i]));
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == &waveBuffers[i]);
        EXPECT_TRUE(nn::audio::GetReleasedWaveBuffer(&voices[i]) == nullptr);

        EXPECT_TRUE(nn::audio::GetVoicePlayedSampleCount(&voices[i]) == 0);
    }

    for (int i = 0; i < rendererCount; ++i)
    {
        nn::audio::StopAudioRenderer(handles[i]);
    }

    for (int i = 0; i < rendererCount; ++i)
    {
        nn::audio::CloseAudioRenderer(handles[i]);
        allocator.Free(workBuffers[i]);
    }

}
