﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::max
#include <limits>     // std::numeric_limits

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../../Programs/Eris/Sources/Libraries/audio/audio_MixManager.h"

#include "testAudio_SimpleAudioRenderer.h"

namespace {
NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024];  // 4096 == nn::os::MemoryPageSize

void SetDefaultParameter(nn::audio::AudioRendererParameter* pOutParameter)
{
    nn::audio::InitializeAudioRendererParameter(pOutParameter);
    pOutParameter->mixBufferCount = 16;
    pOutParameter->voiceCount = 24;
}
}

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;

    nn::mem::StandardAllocator m_Allocator;

public:
    ScopedAudioRenderer(nn::audio::AudioRendererParameter& parameter, nn::os::SystemEvent* pSystemEvent = nullptr)
    {
        m_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

        if (pSystemEvent)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, pSystemEvent, parameter, m_WorkBuffer, workBufferSize));
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));
        }

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
    }
    ~ScopedAudioRenderer()
    {
        nn::audio::StopAudioRenderer(m_Handle);
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);
    }
    nn::audio::AudioRendererHandle GetHandle() const
    {
        return m_Handle;
    }
    nn::audio::AudioRendererConfig& GetConfig()
    {
        return m_Config;
    }
};

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AcquireFinalMix, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireFinalMix(nullptr, &finalMix, parameter.mixBufferCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireFinalMix(&sar.GetConfig(), nullptr, parameter.mixBufferCount), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 0), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(AcquireFinalMix, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ReleaseFinalMix, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);


    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    // nn::audio::FinalMixType invalidFinalMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseFinalMix(nullptr, &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseFinalMix(&sar.GetConfig(), nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, nn::audio::MixBufferCountMax + 1), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseFinalMix(&sar.GetConfig(), &invalidFinalMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(ReleaseFinalMix, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    nn::audio::ReleaseFinalMix(&sar.GetConfig(), &finalMix);
}

TEST(AcquireReleaseFinalMix, AcquireFailAfterAcquire)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    EXPECT_FALSE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    nn::audio::ReleaseFinalMix(&sar.GetConfig(), &finalMix);
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetFinalMixBufferCount, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);


    ScopedAudioRenderer sar(parameter);

    // nn::audio::FinalMixType invalidFinalMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetFinalMixBufferCount(nullptr), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetFinalMixBufferCount(&invalidFinalMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetFinalMixBufferCount, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);


    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    EXPECT_TRUE(nn::audio::GetFinalMixBufferCount(&finalMix) == parameter.mixBufferCount);
    nn::audio::ReleaseFinalMix(&sar.GetConfig(), &finalMix);
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
    EXPECT_TRUE(nn::audio::GetFinalMixBufferCount(&finalMix) == 1);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetFinalMixVolume, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);


    ScopedAudioRenderer sar(parameter);
    // nn::audio::FinalMixType invalidFinalMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetFinalMixVolume(nullptr), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetFinalMixVolume(&invalidFinalMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetFinalMixVolume, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);


    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    // nn::audio::FinalMixType invalidFinalMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetFinalMixVolume(nullptr, 0.0f), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetFinalMixVolume(&invalidFinalMix, 0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetFinalMixVolume(&finalMix, nn::audio::FinalMixType::GetVolumeMin() - 1e-3f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetFinalMixVolume(&finalMix, nn::audio::FinalMixType::GetVolumeMax() + 1e-3f), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetFinalMixVolume, OutputWaveformCheck)
{
    const std::size_t size = 128 * 1024 * 1024;
    void* buffer = std::malloc(size);
    ASSERT_TRUE(buffer != nullptr);

    {
        SimpleAudioRenderer renderer(buffer, size);

        // preparation: prepare constant value short waveform
        const auto InputValue = std::numeric_limits<int16_t>::max();
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t input[nn::audio::MemoryPoolType::SizeGranularity / sizeof(int16_t)];
        for (auto i = 0; i < static_cast<int>(sizeof(input) / sizeof(input[0])); ++i)
        {
            input[i] = InputValue;
        }
        nn::audio::MemoryPoolType dataMemoryPool;
        EXPECT_TRUE(nn::audio::AcquireMemoryPool(&renderer.GetConfig(), &dataMemoryPool, input, nn::util::align_up(sizeof(input), nn::audio::MemoryPoolType::SizeGranularity)));
        EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&dataMemoryPool));

        // testcase: Setting FinalMixVolume to 0.0f
        // expected: Final output is zero
        {
            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetFinalMixVolume(0.0f);

            const auto LoopCount = 64;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    EXPECT_TRUE(data[i] == 0);
                }
            }
        }

        // testcase: Setting FinalMixVolume to 0.5f
        // expected: max of Final output is in (InputValue - Delta, InputValue + Delta)
        {
//             const auto Delta = 32;

            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetFinalMixVolume(0.5f);

            int16_t max = 0;
            const auto LoopCount = 128;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    max = std::max(max, data[i]);
                }
            }
//             EXPECT_TRUE(max >= std::numeric_limits<int16_t>::max() / 2 - Delta);
//             EXPECT_TRUE(max <= std::numeric_limits<int16_t>::max() / 2 + Delta);
            NN_LOG("max = %d\n", max);
        }
    }

    std::free(buffer);
}

TEST(GetFinalMixVolume, DefaultValue)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);


    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    EXPECT_TRUE(nn::audio::GetFinalMixVolume(&finalMix) == 1.0f);
}

TEST(GetSetFinalMixVolume, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);


    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));

    float volumes[] = { 1.5f, 0.0f, nn::audio::FinalMixType::GetVolumeMax(), nn::audio::FinalMixType::GetVolumeMin() };
    for (auto volume : volumes)
    {
        nn::audio::SetFinalMixVolume(&finalMix, volume);
        EXPECT_TRUE(nn::audio::GetFinalMixVolume(&finalMix) == volume);
    }
}
