﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::max
#include <limits>     // std::numeric_limits

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../../Programs/Eris/Sources/Libraries/audio/common/audio_NodeIdManager.h"
#include "../../Programs/Eris/Sources/Libraries/audio/audio_MixManager.h"

#include "testAudio_SimpleAudioRenderer.h"

namespace {
NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024];  // 4096 == nn::os::MemoryPageSize

void SetDefaultParameter(nn::audio::AudioRendererParameter* pOutParameter)
{
    nn::audio::InitializeAudioRendererParameter(pOutParameter);
    pOutParameter->mixBufferCount = 16;
    pOutParameter->voiceCount = 24;
}
}

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;

    nn::mem::StandardAllocator m_Allocator;

public:
    ScopedAudioRenderer(nn::audio::AudioRendererParameter& parameter, nn::os::SystemEvent* pSystemEvent = nullptr)
    {
        m_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

        if (pSystemEvent)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, pSystemEvent, parameter, m_WorkBuffer, workBufferSize));
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));
        }

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
    }
    ~ScopedAudioRenderer()
    {
        nn::audio::StopAudioRenderer(m_Handle);
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);
    }
    nn::audio::AudioRendererHandle GetHandle() const
    {
        return m_Handle;
    }
    nn::audio::AudioRendererConfig& GetConfig()
    {
        return m_Config;
    }
};

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ReleaseFinalMixPrivate, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    nn::audio::MixInfo invalidMixInfo;
    nn::audio::FinalMixType invalidFinalMix = { reinterpret_cast<nn::audio::MixInfo*>(&invalidMixInfo) };

    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseFinalMix(nullptr, &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseFinalMix(&sar.GetConfig(), nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, nn::audio::MixBufferCountMax + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseFinalMix(&sar.GetConfig(), &invalidFinalMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetFinalMixNodeId, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;

    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 2));

    EXPECT_EQ(nn::audio::GetFinalMixNodeId(&finalMix), (static_cast<uint32_t>(nn::audio::common::NodeIdManager::NodeIdType::Mix) << nn::audio::common::NodeIdManager::TypeShift | finalMix._pMixInfo->mixId << nn::audio::common::NodeIdManager::BaseShift));
}
